// Variables used by Scriptable.
// These must be at the very top of the file. Do not edit.
// icon-color: purple; icon-glyph: magic;
const API_BASE_URL = "https://www.dyxcloud.com/getinfo/api";
const DEBUG = true;
const AUTO_UPDATE_INTERVAL = 5 * 60 * 1000; // 5分钟
const QUICK_TEST = false; // 设置为true只做快速测试，不更新数据
const TEST_SINGLE_URL = ''; // 设置为要测试的订阅链接，例如 'https://example.com/subscribe'

const COMPONENT_CONFIG = {
    small: { maxCount: 3, showDetails: false },
    medium: { maxCount: 8, showDetails: true },
    large: { maxCount: 15, showDetails: true }
};

let cachedData = null;
let lastFetchTime = 0;
let lastUpdateTime = 0;
let lastUpdateResult = null; // 存储最后一次更新的结果
let lastUpdateError = null; // 存储最后一次更新的错误信息
const CACHE_DURATION = 5 * 60 * 1000; // 延长缓存时间，减少频繁请求

function debugLog(...args) {
    if (DEBUG) {
        // 在Scriptable App中同时输出到console和console.log
        const message = args.join(' ');
        console.log(message);

        // 如果有重要信息，也显示在脚本运行结果中
        if (message.includes('📊') || message.includes('🔋') || message.includes('✅') || message.includes('❌')) {
            console.log(`[流量监控] ${message}`);
        }
    }
}

function debugError(...args) {
    const message = args.join(' ');
    console.log(`❌ [错误] ${message}`);
    if (DEBUG) {
        console.log(`[Traffic Monitor ERROR] ${message}`);
    }
}

function debugSuccess(...args) {
    const message = args.join(' ');
    console.log(`✅ [成功] ${message}`);
    if (DEBUG) {
        console.log(`[Traffic Monitor SUCCESS] ${message}`);
    }
}

function debugInfo(...args) {
    const message = args.join(' ');
    console.log(`ℹ️  [信息] ${message}`);
    if (DEBUG) {
        console.log(`[Traffic Monitor INFO] ${message}`);
    }
}

function dynamicColor(lightColor, darkColor) {
    return Color.dynamic(new Color(lightColor), new Color(darkColor));
}

async function run() {
    debugLog('🚀 ===== 流量监控脚本启动 =====');
    debugLog(`⚙️  调试模式: ${DEBUG ? '开启' : '关闭'}`);
    debugLog(`📱 运行环境: ${config.runsInApp ? 'Scriptable App' : 'Widget'}`);

    try {
        // 在Scriptable App中运行时的处理
        if (config.runsInApp) {
            console.log('\n🎯 ===== 流量监控详细报告 =====');
            console.log(`⏰ 运行时间: ${formatDate(new Date(), 'yyyy-MM-dd HH:mm:ss')}`);

            // 强制更新数据（无论是否为调试模式）
            if (!QUICK_TEST) {
                console.log('🔄 开始更新流量数据...');
                cachedData = null;
                lastFetchTime = 0;
                await performAutoUpdate();
                console.log('✅ 流量数据更新完成');
            } else {
                console.log('🧪 快速测试模式：跳过数据更新');
            }

            // 显示详细统计信息
            console.log('\n📊 显示最新统计信息...');
            // 清除缓存以确保获取最新数据
            cachedData = null;
            await showDetailedTrafficStats();

            // 显示Widget预览
            console.log('\n📱 生成Widget预览...');
            const largeWidget = await renderLarge();
            await largeWidget.presentLarge();

            // 显示总结信息
            console.log('\n📋 ===== 运行总结 =====');
            if (!QUICK_TEST) {
                console.log('✅ 数据已更新到最新状态');
                if (lastUpdateResult) {
                    console.log(`📊 更新结果: ${lastUpdateResult.message}`);
                }
                console.log(`⏰ 完成时间: ${formatDate(new Date(), 'HH:mm:ss')}`);
            } else {
                console.log('🧪 快速测试完成（未更新数据）');
            }
            console.log('🎯 ===== 报告完成 =====\n');
            return; // 在App中运行直接返回，不设置Widget

        }

        // Widget模式：强制更新数据并设置Widget
        debugLog('🔧 Widget模式：开始设置...');
        if (!QUICK_TEST) {
            debugLog('强制更新流量数据...');
            cachedData = null;
            lastFetchTime = 0;
            await performAutoUpdate();
        } else {
            debugLog('🧪 快速测试模式：不更新数据');
        }

        // 设置Widget
        const widget = await createWidget();
        Script.setWidget(widget);
        debugLog('✅ Widget设置完成');
    } catch (error) {
        debugLog('run() error:', error);
        const errorWidget = createErrorWidget(error);
        Script.setWidget(errorWidget);

        if (config.runsInApp) {
            const alert = new Alert();
            alert.title = "运行错误";
            alert.message = `错误: ${error.message}`;
            alert.addAction("确定");
            await alert.present();
        }
    }

    Script.complete();
}

async function checkAndUpdateIfNeeded() {
    const now = Date.now();

    // 如果是首次运行或者距离上次更新超过10分钟，则执行自动更新
    if (lastUpdateTime === 0 || (now - lastUpdateTime) >= AUTO_UPDATE_INTERVAL) {
        debugLog('triggering auto update');
        await performAutoUpdate();
        lastUpdateTime = now;
    } else {
        debugLog('skip auto update, last update:', new Date(lastUpdateTime));
    }
}

async function performAutoUpdate() {
    const startTime = Date.now();
    debugLog('🔄 performAutoUpdate() 开始');

    try {
        // 性能优化：不要总是清空缓存，只有在强制更新时才清空
        if (cachedData) {
            debugLog('💾 使用现有缓存，仅更新流量数据');
        } else {
            debugLog('🗑️  清空缓存，准备重新获取数据');
            lastFetchTime = 0;
        }

        // 获取最新的订阅数据
        debugLog('📡 获取订阅信息...');
        const subscriptions = await fetchTrafficData();
        debugLog(`📋 获取到 ${subscriptions.length} 个订阅`);

        if (subscriptions.length === 0) {
            throw new Error('没有找到任何订阅信息');
        }

        // 调用后端API触发流量更新
        debugLog('🔄 开始更新所有订阅的流量...');
        const updateResult = await updateAllTraffic();

        // 记录更新成功结果
        lastUpdateResult = updateResult;
        lastUpdateError = null;
        lastUpdateTime = Date.now(); // 记录成功更新的时间
        debugLog('✅ 自动更新成功:', updateResult.message);

        // 清除缓存，强制重新获取最新数据
        cachedData = null;
        const freshData = await fetchTrafficData();
        debugLog(`🔄 更新完成，重新获取到 ${freshData.length} 个订阅信息`);

        const duration = Date.now() - startTime;
        debugLog(`⏱️  自动更新完成，总耗时: ${duration}ms`);

        return updateResult;

    } catch (error) {
        const duration = Date.now() - startTime;
        debugError(`❌ performAutoUpdate() 失败，耗时 ${duration}ms:`, error.message);

        // 记录更新失败信息
        lastUpdateResult = null;
        lastUpdateError = {
            message: error.message,
            timestamp: new Date()
        };

        // 显示更详细的错误信息
        debugError('🔍 详细错误:', error);
        if (error.stack) {
            debugError('📍 错误堆栈:', error.stack);
        }

        throw error; // 重新抛出错误，让上层处理
    }
}


async function createWidget() {
    const family = config.widgetFamily || 'small';
    debugLog('creating widget for family:', family);

    try {
        switch (family) {
            case 'medium':
                return await renderMedium();
            case 'large':
                return await renderLarge();
            default:
                return await renderSmall();
        }
    } catch (error) {
        debugLog('createWidget error:', error.message);
        return createErrorWidget(error);
    }
}

function createErrorWidget(error) {
    const widget = new ListWidget();
    widget.backgroundColor = dynamicColor("#e5e7eb", "#0f172a");
    widget.setPadding(25, 18, 25, 18);

    // 标题
    const titleStack = widget.addStack();
    titleStack.centerAlignContent();
    const titleText = titleStack.addText("流量监控");
    titleText.font = Font.boldSystemFont(22);
    titleText.textColor = dynamicColor("#1e293b", "#f1f5f9");
    widget.addSpacer(12);

    // 分隔线
    const separator = widget.addStack();
    separator.backgroundColor = dynamicColor("#e2e8f0", "#334155");
    separator.size = new Size(0, 1);
    widget.addSpacer(20);

    // 错误内容卡片
    const errorCard = widget.addStack();
    errorCard.backgroundColor = dynamicColor("#fef2f2", "#1e293b");
    errorCard.cornerRadius = 10;
    errorCard.setPadding(20, 20, 20, 20);
    errorCard.centerAlignContent();
    errorCard.layoutVertically();

    const icon = SFSymbol.named("exclamationmark.triangle.fill");
    const iconImg = errorCard.addImage(icon.image);
    iconImg.tintColor = dynamicColor("#dc2626", "#ef4444");
    iconImg.imageSize = new Size(36, 36);

    errorCard.addSpacer(12);

    const errorText = errorCard.addText("网络连接失败");
    errorText.font = Font.semiboldSystemFont(18);
    errorText.textColor = dynamicColor("#dc2626", "#ef4444");
    errorText.centerAlignText();

    errorCard.addSpacer(8);

    const detailText = errorCard.addText(String(error.message || "请检查网络连接"));
    detailText.font = Font.systemFont(14);
    detailText.textColor = dynamicColor("#991b1b", "#fca5a5");
    detailText.centerAlignText();
    detailText.lineLimit = 3;

    errorCard.addSpacer(12);

    const retryText = errorCard.addText("点击重试");
    retryText.font = Font.mediumSystemFont(14);
    retryText.textColor = dynamicColor("#4CAF50", "#66BB6A");
    retryText.centerAlignText();
    retryText.url = "scriptable:///run";

    return widget;
}

async function renderSmall() {
    const widget = new ListWidget();
    widget.backgroundColor = dynamicColor("#f3f8ff", "#1e1e22");
    widget.setPadding(8, 8, 8, 8);

    const data = await fetchTrafficData();
    const config = COMPONENT_CONFIG.small;
    const items = data.slice(0, config.maxCount);

    if (items.length === 0) {
        return createEmptyWidget(widget, "暂无流量数据");
    }

    await addHeader(widget, "流量监控", false);

    // 添加简单的状态指示器
    await addUpdateStatusIndicatorSmall(widget);

    for (let i = 0; i < items.length; i++) {
        if (i > 0) widget.addSpacer(4);
        await addTrafficItemSmall(widget, items[i], i + 1);
    }

    widget.addSpacer(4);
    await addFooter(widget, data.length);

    return widget;
}

async function renderMedium() {
    const widget = new ListWidget();
    widget.backgroundColor = dynamicColor("#f3f8ff", "#1e1e22");
    widget.setPadding(10, 10, 10, 10);

    const data = await fetchTrafficData();
    const config = COMPONENT_CONFIG.medium;
    const items = data.slice(0, config.maxCount);

    if (items.length === 0) {
        return createEmptyWidget(widget, "暂无流量数据");
    }

    await addHeader(widget, "流量监控", true);

    // 添加简化的更新状态显示
    await addUpdateStatusInfoCompact(widget);

    const leftStack = widget.addStack();
    leftStack.layoutVertically();
    const rightStack = widget.addStack();
    rightStack.layoutVertically();

    for (let i = 0; i < items.length; i++) {
        const targetStack = i % 2 === 0 ? leftStack : rightStack;
        if (i > 1) targetStack.addSpacer(6);
        await addTrafficItemMedium(targetStack, items[i], i + 1);
    }

    widget.addSpacer(8);
    await addFooter(widget, data.length);

    return widget;
}

async function renderLarge() {
    const widget = new ListWidget();
    widget.backgroundColor = dynamicColor("#e5e7eb", "#0f172a");
    widget.setPadding(25, 18, 25, 18);

    const data = await fetchTrafficData();
    const config = COMPONENT_CONFIG.large;
    const items = data.slice(0, config.maxCount);

    if (items.length === 0) {
        return createEmptyWidget(widget, "暂无流量数据");
    }

    // 标题和更新时间 - 同一行
    const headerStack = widget.addStack();
    headerStack.centerAlignContent();

    // 左侧：标题
    const titleText = headerStack.addText("流量监控");
    titleText.font = Font.boldSystemFont(22);
    titleText.textColor = dynamicColor("#1e293b", "#f1f5f9");

    headerStack.addSpacer(); // 推到右边

    // 右侧：更新时间和状态
    const timeStack = headerStack.addStack();
    timeStack.layoutVertically();
    timeStack.centerAlignContent();

    // 更新时间 - 显示上次更新成功的时间
    const updateTimeText = timeStack.addText(`更新于: ${getLastSuccessfulUpdateTime()}`);
    updateTimeText.font = Font.mediumSystemFont(13);
    updateTimeText.textColor = dynamicColor("#475569", "#cbd5e1");
    updateTimeText.textOpacity = 0.9;

    // 更新状态指示器
    await addUpdateStatusIndicator(timeStack);

    widget.addSpacer(8);

    // 添加分隔线
    const separator = widget.addStack();
    separator.backgroundColor = dynamicColor("#e2e8f0", "#334155");
    separator.size = new Size(0, 1);
    widget.addSpacer(8);

    // 添加更新状态信息（如果有）
    await addUpdateStatusInfo(widget);

    // 内容区域
    const contentStack = widget.addStack();
    contentStack.layoutVertically();

    const itemCount = items.length;
    const isCompact = itemCount > 10;
    const spacing = isCompact ? 6 : 8;

    for (let i = 0; i < items.length; i++) {
        if (i > 0) contentStack.addSpacer(spacing);
        await addTrafficItemSimple(contentStack, items[i], itemCount);
    }

    widget.addSpacer(10);
    await addTotalFooter(widget, data);

    return widget;
}

function createEmptyWidget(widget, message) {
    widget.backgroundColor = dynamicColor("#e5e7eb", "#0f172a");

    const stack = widget.addStack();
    stack.centerAlignContent();
    stack.layoutVertically();

    const icon = SFSymbol.named("chart.bar.xaxis");
    const iconImg = stack.addImage(icon.image);
    iconImg.tintColor = dynamicColor("#94a3b8", "#64748b");
    iconImg.imageSize = new Size(40, 40);
    iconImg.imageOpacity = 0.6;

    stack.addSpacer(12);

    const text = stack.addText(message);
    text.font = Font.mediumSystemFont(16);
    text.textColor = dynamicColor("#64748b", "#94a3b8");
    text.centerAlignText();

    return widget;
}

async function addHeader(widget, title, showTime) {
    widget.addSpacer(8);

    const head = widget.addStack();
    head.setPadding(0, 8, 0, 8);
    head.centerAlignContent();

    const icon = SFSymbol.named('chart.bar');
    const img = head.addImage(icon.image);
    img.tintColor = dynamicColor("#4CAF50", "#66BB6A");
    img.imageSize = new Size(16, 16);

    head.addSpacer(6);

    const titleText = head.addText(title);
    titleText.font = Font.boldSystemFont(13);
    titleText.textColor = dynamicColor("#333333", "#ffffff");

    head.addSpacer();

    const statusIcon = "checkmark.circle";
    const statusSF = SFSymbol.named(statusIcon);
    const statusImg = head.addImage(statusSF.image);
    statusImg.tintColor = dynamicColor("#4CAF50", "#66BB6A");
    statusImg.imageSize = new Size(12, 12);

    if (showTime) {
        head.addSpacer(4);
        const timeText = head.addText(formatDate(new Date(), "HH:mm"));
        timeText.font = Font.systemFont(10);
        timeText.textColor = dynamicColor("#666666", "#999999");
        timeText.textOpacity = 0.7;
    }

    widget.addSpacer(4);
}

async function addFooter(widget, totalCount) {
    widget.addSpacer(8);

    const footer = widget.addStack();
    footer.setPadding(0, 8, 0, 8);
    footer.centerAlignContent();

    const stats = calculateStats(await fetchTrafficData());
    const totalText = footer.addText(`剩余: ${stats.remaining} | 已用: ${stats.monthlyUsed}`);
    totalText.font = Font.systemFont(10);
    totalText.textColor = dynamicColor("#4CAF50", "#66BB6A");

    footer.addSpacer();

    const status = footer.addText("监控");
    status.font = Font.systemFont(9);
    status.textColor = dynamicColor("#999999", "#666666");
    status.textOpacity = 0.6;

    footer.addSpacer(2);

    const updateTime = footer.addText(formatDate(new Date(), "HH:mm"));
    updateTime.font = Font.systemFont(9);
    updateTime.textColor = dynamicColor("#999999", "#666666");
    updateTime.textOpacity = 0.6;
}

async function addTrafficItemSmall(widget, item, index) {
    const stack = widget.addStack();
    stack.centerAlignContent();

    const idx = stack.addText(String(index));
    idx.font = Font.boldSystemFont(11);
    idx.textColor = getRankColor(index);
    idx.minimumScaleFactor = 0.8;

    stack.addSpacer(6);

    const infoStack = stack.addStack();
    infoStack.layoutVertically();

    const nameText = infoStack.addText(item.name);
    nameText.font = Font.mediumSystemFont(11);
    nameText.textColor = dynamicColor("#333333", "#ffffff");
    nameText.lineLimit = 1;
    nameText.minimumScaleFactor = 0.8;

    const trafficText = infoStack.addText(item.traffic);
    trafficText.font = Font.systemFont(9);
    trafficText.textColor = getTrafficColor(item.traffic);
    trafficText.textOpacity = 0.8;

    stack.addSpacer();

    if (item.traffic_change && item.traffic_change !== "0.00 GB") {
        const changeText = stack.addText(formatTrafficChange(item.traffic_change));
        changeText.font = Font.systemFont(8);
        changeText.textColor = getChangeColor(item.traffic_change);
        changeText.minimumScaleFactor = 0.8;
    }
}

async function addTrafficItemMedium(widget, item, index) {
    const stack = widget.addStack();
    stack.centerAlignContent();

    const idx = stack.addText(String(index));
    idx.font = Font.boldSystemFont(12);
    idx.textColor = getRankColor(index);
    idx.minimumScaleFactor = 0.8;

    stack.addSpacer(6);

    const infoStack = stack.addStack();
    infoStack.layoutVertically();

    const nameText = infoStack.addText(item.name);
    nameText.font = Font.mediumSystemFont(12);
    nameText.textColor = dynamicColor("#333333", "#ffffff");
    nameText.lineLimit = 1;
    nameText.minimumScaleFactor = 0.8;

    const detailStack = infoStack.addStack();

    const trafficText = detailStack.addText(item.traffic);
    trafficText.font = Font.systemFont(10);
    trafficText.textColor = getTrafficColor(item.traffic);

    if (item.traffic_change && item.traffic_change !== "0.00 GB") {
        detailStack.addSpacer(4);
        const changeText = detailStack.addText(formatTrafficChange(item.traffic_change));
        changeText.font = Font.systemFont(9);
        changeText.textColor = getChangeColor(item.traffic_change);
    }
}

async function addTrafficItemLarge(widget, item, index) {
    const stack = widget.addStack();
    stack.centerAlignContent();

    const idxStack = stack.addStack();
    idxStack.width = 25;
    const idx = idxStack.addText(String(index));
    idx.font = Font.boldSystemFont(16);
    idx.textColor = getRankColor(index);
    idx.centerAlignText();

    stack.addSpacer(10);

    const infoStack = stack.addStack();
    infoStack.layoutVertically();

    const nameText = infoStack.addText(item.name);
    nameText.font = Font.mediumSystemFont(16);
    nameText.textColor = dynamicColor("#333333", "#ffffff");
    nameText.lineLimit = 1;

    if (item.subscribe_url) {
        nameText.url = item.subscribe_url;
    }

    const trafficText = infoStack.addText(item.traffic);
    trafficText.font = Font.boldSystemFont(15);
    trafficText.textColor = getTrafficColor(item.traffic);

    stack.addSpacer();

    if (item.traffic_change && item.traffic_change !== "0.00 GB") {
        const changeStack = stack.addStack();
        changeStack.width = 70;
        changeStack.layoutVertically();

        const changeValue = changeStack.addText(formatTrafficChange(item.traffic_change));
        changeValue.font = Font.boldSystemFont(14);
        changeValue.textColor = getChangeColor(item.traffic_change);
        changeValue.rightAlignText();

        const changeLabel = changeStack.addText(getChangeLabel(item.traffic_change));
        changeLabel.font = Font.systemFont(10);
        changeLabel.textColor = getChangeColor(item.traffic_change);
        changeLabel.rightAlignText();
        changeLabel.textOpacity = 0.8;
    }
}

async function addTrafficItemLargeOptimized(widget, item, index, totalItems) {
    const stack = widget.addStack();
    stack.centerAlignContent();

    // 根据订阅数量调整字体大小
    const isManyItems = totalItems > 8;
    const isVeryManyItems = totalItems > 12;

    const idxSize = isVeryManyItems ? 12 : (isManyItems ? 14 : 16);
    const nameSize = isVeryManyItems ? 12 : (isManyItems ? 14 : 16);
    const trafficSize = isVeryManyItems ? 11 : (isManyItems ? 13 : 15);
    const changeSize = isVeryManyItems ? 10 : (isManyItems ? 12 : 14);

    const idxStack = stack.addStack();
    idxStack.width = isVeryManyItems ? 20 : 25;
    const idx = idxStack.addText(String(index));
    idx.font = Font.boldSystemFont(idxSize);
    idx.textColor = getRankColor(index);
    idx.centerAlignText();

    stack.addSpacer(isVeryManyItems ? 6 : 10);

    const infoStack = stack.addStack();
    infoStack.layoutVertically();

    const nameText = infoStack.addText(item.name);
    nameText.font = Font.mediumSystemFont(nameSize);
    nameText.textColor = dynamicColor("#333333", "#ffffff");
    nameText.lineLimit = 1;

    if (item.subscribe_url) {
        nameText.url = item.subscribe_url;
    }

    const trafficText = infoStack.addText(item.traffic);
    trafficText.font = Font.boldSystemFont(trafficSize);
    trafficText.textColor = getTrafficColor(item.traffic);

    stack.addSpacer();

    if (item.traffic_change && item.traffic_change !== "0.00 GB") {
        const changeStack = stack.addStack();
        changeStack.width = isVeryManyItems ? 50 : 70;
        changeStack.layoutVertically();

        const changeValue = changeStack.addText(formatTrafficChange(item.traffic_change));
        changeValue.font = Font.boldSystemFont(changeSize);
        changeValue.textColor = getChangeColor(item.traffic_change);
        changeValue.rightAlignText();

        const changeLabel = changeStack.addText(getChangeLabel(item.traffic_change));
        changeLabel.font = Font.systemFont(Math.max(8, changeSize - 4));
        changeLabel.textColor = getChangeColor(item.traffic_change);
        changeLabel.rightAlignText();
        changeLabel.textOpacity = 0.8;
    }
}

async function addTrafficItemSimple(widget, item, totalItems) {
    // 创建卡片背景
    const cardStack = widget.addStack();
    cardStack.backgroundColor = dynamicColor("#f9fafb", "#1e293b");
    cardStack.cornerRadius = 8;
    cardStack.setPadding(10, 14, 10, 14);
    cardStack.centerAlignContent();

    // 根据订阅数量调整字体大小
    const isManyItems = totalItems > 8;
    const isVeryManyItems = totalItems > 12;

    const nameSize = isVeryManyItems ? 15 : (isManyItems ? 17 : 19);
    const trafficSize = isVeryManyItems ? 14 : (isManyItems ? 16 : 18);

    // 左侧：标题
    const nameText = cardStack.addText(item.name);
    nameText.font = Font.mediumSystemFont(nameSize);
    nameText.textColor = dynamicColor("#1e293b", "#f1f5f9");
    nameText.lineLimit = 1;

    if (item.subscribe_url) {
        nameText.url = item.subscribe_url;
    }

    cardStack.addSpacer();

    // 右侧：剩余流量
    const trafficText = cardStack.addText(item.traffic);
    trafficText.font = Font.boldSystemFont(trafficSize);
    trafficText.textColor = dynamicColor("#3b82f6", "#60a5fa"); // 现代蓝色
}

async function addUpdateStatusIndicatorSmall(widget) {
    // 只在更新失败时显示错误信息，成功时不显示任何提示
    if (lastUpdateError) {
        const statusStack = widget.addStack();
        statusStack.centerAlignContent();

        // 小尺寸失败指示器
        const errorIcon = SFSymbol.named("exclamationmark.triangle.fill");
        const iconImg = statusStack.addImage(errorIcon.image);
        iconImg.tintColor = dynamicColor("#ef4444", "#f87171");
        iconImg.imageSize = new Size(8, 8);

        statusStack.addSpacer(3);
        const statusText = statusStack.addText("更新失败");
        statusText.font = Font.systemFont(9);
        statusText.textColor = dynamicColor("#ef4444", "#f87171");
        statusText.textOpacity = 0.8;

        widget.addSpacer(2);
    }
}

async function addUpdateStatusIndicator(containerStack) {
    // 右上角只显示更新时间，不显示状态图标和文字
    // 保持简洁的界面设计
}

async function addUpdateStatusInfoCompact(widget) {
    // 只在更新失败时显示错误信息，成功时不显示任何提示
    if (lastUpdateError) {
        // 简化的失败状态显示
        const statusStack = widget.addStack();
        statusStack.backgroundColor = dynamicColor("#fef2f2", "#450a0a");
        statusStack.cornerRadius = 6;
        statusStack.setPadding(8, 12, 8, 12);
        statusStack.centerAlignContent();

        const errorIcon = SFSymbol.named("exclamationmark.triangle.fill");
        const iconImg = statusStack.addImage(errorIcon.image);
        iconImg.tintColor = dynamicColor("#ef4444", "#f87171");
        iconImg.imageSize = new Size(12, 12);

        statusStack.addSpacer(6);
        const messageText = statusStack.addText(`自动更新失败`);
        messageText.font = Font.mediumSystemFont(11);
        messageText.textColor = dynamicColor("#991b1b", "#fca5a5");

        widget.addSpacer(6);
    }
}

async function addUpdateStatusInfo(widget) {
    // 只在更新失败时显示错误信息，成功时不显示任何提示
    if (lastUpdateError) {
        // 显示更新失败信息
        const errorCard = widget.addStack();
        errorCard.backgroundColor = dynamicColor("#fef2f2", "#450a0a");
        errorCard.cornerRadius = 8;
        errorCard.setPadding(12, 16, 12, 16);
        errorCard.layoutVertically();

        const headerStack = errorCard.addStack();
        headerStack.centerAlignContent();

        const errorIcon = SFSymbol.named("exclamationmark.triangle.fill");
        const iconImg = headerStack.addImage(errorIcon.image);
        iconImg.tintColor = dynamicColor("#ef4444", "#f87171");
        iconImg.imageSize = new Size(16, 16);

        headerStack.addSpacer(8);
        const titleText = headerStack.addText("自动更新失败");
        titleText.font = Font.semiboldSystemFont(14);
        titleText.textColor = dynamicColor("#991b1b", "#fca5a5");

        errorCard.addSpacer(4);

        const messageText = errorCard.addText(lastUpdateError.message);
        messageText.font = Font.systemFont(12);
        messageText.textColor = dynamicColor("#dc2626", "#f87171");
        messageText.lineLimit = 3;

        if (lastUpdateError.timestamp) {
            errorCard.addSpacer(4);
            const timeText = errorCard.addText(`失败时间: ${formatDate(lastUpdateError.timestamp, "HH:mm")}`);
            timeText.font = Font.systemFont(10);
            timeText.textColor = dynamicColor("#b91c1c", "#ef4444");
            timeText.textOpacity = 0.7;
        }

        widget.addSpacer(8);
    }
}

function getLastSuccessfulUpdateTime() {
    // 返回上次更新成功的时间
    if (lastUpdateTime > 0) {
        return formatDate(new Date(lastUpdateTime), "HH:mm");
    } else {
        return formatDate(new Date(), "HH:mm");
    }
}

async function showDetailedTrafficStats() {
    debugLog('\n📊 ===== 流量监控详细信息 =====');

    try {
        // 获取订阅数据
        const subscriptions = await fetchTrafficData();
        debugLog(`📡 从API获取到 ${subscriptions.length} 个订阅信息`);

        if (subscriptions.length === 0) {
            debugLog('⚠️  没有找到订阅数据');
            return;
        }

        // 显示总览信息
        debugLog('\n📈 === 总览统计 ===');
        const stats = calculateStats(subscriptions);
        debugLog(`🔋 总剩余流量: ${stats.remaining}`);
        debugLog(`📊 当月已使用: ${stats.monthlyUsed}`);

        // 详细分析每个订阅
        debugLog('\n📱 === 订阅详情 ===');
        for (let i = 0; i < subscriptions.length; i++) {
            const sub = subscriptions[i];
            debugLog(`\n--- 订阅 ${i + 1}: ${sub.name} ---`);
            debugLog(`🔗 订阅链接: ${sub.subscribe_url || '无'}`);
            debugLog(`🔋 当前剩余: ${sub.traffic}`);
            debugLog(`📈 上次剩余: ${sub.previous_traffic || '无记录'}`);
            debugLog(`📊 流量变化: ${sub.traffic_change || '无数据'}`);
            debugLog(`📅 当月使用: ${sub.monthly_used || '0.00 GB'}`);
            debugLog(`📅 今日使用: ${sub.daily_used || '0.00 GB'}`);
            debugLog(`🔄 重置日期: 每月${sub.monthly_reset_date || 1}号`);

            // 计算单个订阅的详细信息
            if (sub.traffic) {
                const currentTraffic = parseFloat(sub.traffic) || 0;
                const monthlyUsed = parseFloat(sub.monthly_used?.replace(' GB', '') || 0);
                const estimatedTotal = currentTraffic + monthlyUsed;
                const usageRate = estimatedTotal > 0 ? (monthlyUsed / estimatedTotal * 100).toFixed(1) : 0;

                debugLog(`💰 估算总量: ${estimatedTotal.toFixed(2)} GB`);
                debugLog(`📊 使用率: ${usageRate}%`);

                // 流量状态警告
                if (currentTraffic <= 10) {
                    debugLog('⚠️  警告: 剩余流量不足10GB');
                } else if (currentTraffic <= 50) {
                    debugLog('💛 提醒: 剩余流量不足50GB');
                } else {
                    debugLog('✅ 流量充足');
                }
            }
        }

        // 如果有订阅链接，显示更新状态
        const validSubscriptions = subscriptions.filter(sub => sub.subscribe_url);
        if (validSubscriptions.length > 0) {
            debugLog('\n🔄 === 自动更新状态 ===');
            debugLog(`🔧 可更新订阅: ${validSubscriptions.length}/${subscriptions.length}`);
            debugLog(`⏰ 上次更新: ${formatDate(new Date(lastUpdateTime), "yyyy-MM-dd HH:mm:ss")}`);
            debugLog(`📡 API地址: ${API_BASE_URL}`);
        }

        // 缓存状态
        debugLog('\n💾 === 缓存状态 ===');
        if (cachedData) {
            const cacheAge = Date.now() - lastFetchTime;
            const cacheMinutes = Math.floor(cacheAge / 60000);
            debugLog(`✅ 数据已缓存 (${cacheMinutes}分钟前)`);
        } else {
            debugLog('❌ 无缓存数据');
        }

        debugLog('\n🎯 ===== 调试信息完成 =====\n');

    } catch (error) {
        debugLog('❌ 获取流量统计失败:', error.message);
        debugLog('错误详情:', JSON.stringify(error, null, 2));
    }

    // 如果设置了测试单个URL，单独测试
    if (TEST_SINGLE_URL) {
        console.log('\n🧪 ===== 单个链接测试 =====');
        console.log(`🔗 测试链接: ${TEST_SINGLE_URL}`);

        try {
            const result = await testSingleSubscription(TEST_SINGLE_URL);
            if (result) {
                console.log(`✅ 测试成功: ${result}`);
            } else {
                console.log('❌ 测试失败: 无法解析流量信息');
            }
        } catch (error) {
            console.log(`❌ 测试出错: ${error.message}`);
        }

        console.log('🧪 ===== 单个链接测试完成 =====\n');
    }
}

async function testSingleSubscription(subscribeUrl) {
    console.log(`\n🔍 测试订阅链接: ${subscribeUrl}`);

    try {
        // 直接调用fetchTrafficInfo
        const traffic = await fetchTrafficInfo(subscribeUrl);
        return traffic;
    } catch (error) {
        console.log(`❌ 测试失败: ${error.message}`);
        throw error;
    }
}

async function addTotalFooter(widget, data) {
    // 添加分隔线
    widget.addSpacer(2);
    const separator = widget.addStack();
    separator.backgroundColor = dynamicColor("#e2e8f0", "#334155");
    separator.size = new Size(0, 1);
    widget.addSpacer(4);

    // 创建总计卡片 - 占满宽度
    const footerCard = widget.addStack();
    footerCard.backgroundColor = dynamicColor("#f3f4f6", "#1e293b");
    footerCard.cornerRadius = 8;
    footerCard.setPadding(14, 18, 14, 18); // 增加垂直padding，保持左右padding
    footerCard.centerAlignContent();

    const stats = calculateStats(data);

    // 左侧：剩余流量 - 占1/3空间
    const leftStack = footerCard.addStack();
    leftStack.layoutVertically();
    leftStack.centerAlignContent();
    leftStack.spacing = 2; // 减少标签和数值间的间距

    const leftLabel = leftStack.addText("剩余流量");
    leftLabel.font = Font.semiboldSystemFont(12);
    leftLabel.textColor = dynamicColor("#475569", "#94a3b8");

    const leftValue = leftStack.addText(stats.remaining);
    leftValue.font = Font.boldSystemFont(16);
    leftValue.textColor = dynamicColor("#10b981", "#34d399"); // 现代绿色
    leftValue.minimumScaleFactor = 0.8;
    leftValue.lineLimit = 1;

    // 弹性占位，让分布更均匀
    footerCard.addSpacer();

    // 中间分隔符 - 增加高度
    const middleSeparator = footerCard.addStack();
    middleSeparator.backgroundColor = dynamicColor("#d1d5db", "#4b5563");
    middleSeparator.size = new Size(1, 35);

    // 弹性占位
    footerCard.addSpacer();

    // 右侧：当月使用和今日使用 - 占剩余空间
    const rightStack = footerCard.addStack();
    rightStack.centerAlignContent();

    // 当月使用
    const monthlyStack = rightStack.addStack();
    monthlyStack.layoutVertically();
    monthlyStack.centerAlignContent();
    monthlyStack.spacing = 2;

    const monthlyLabel = monthlyStack.addText("当月使用");
    monthlyLabel.font = Font.semiboldSystemFont(12);
    monthlyLabel.textColor = dynamicColor("#475569", "#94a3b8");

    const monthlyValue = monthlyStack.addText(stats.monthlyUsed);
    monthlyValue.font = Font.boldSystemFont(16);
    monthlyValue.textColor = dynamicColor("#3b82f6", "#60a5fa"); // 现代蓝色
    monthlyValue.minimumScaleFactor = 0.8;
    monthlyValue.lineLimit = 1;

    // 分隔符 - 增加高度
    rightStack.addSpacer(10);
    const rightInnerSeparator = rightStack.addStack();
    rightInnerSeparator.backgroundColor = dynamicColor("#d1d5db", "#4b5563");
    rightInnerSeparator.size = new Size(1, 30);

    // 今日使用
    rightStack.addSpacer(10);
    const dailyStack = rightStack.addStack();
    dailyStack.layoutVertically();
    dailyStack.centerAlignContent();
    dailyStack.spacing = 2;

    const dailyLabel = dailyStack.addText("今日使用");
    dailyLabel.font = Font.semiboldSystemFont(12);
    dailyLabel.textColor = dynamicColor("#475569", "#94a3b8");

    const dailyValue = dailyStack.addText(stats.dailyUsed);
    dailyValue.font = Font.boldSystemFont(16);
    dailyValue.textColor = dynamicColor("#f59e0b", "#fbbf24"); // 现代橙色
    dailyValue.minimumScaleFactor = 0.8;
    dailyValue.lineLimit = 1;
}

function getRankColor(index) {
    if (index <= 3) return dynamicColor("#fe4f67", "#ff4757");
    if (index <= 6) return dynamicColor("#f5c94c", "#ffa502");
    return dynamicColor("#9195a3", "#a4b0be");
}

function getTrafficColor(traffic) {
    if (!traffic) return dynamicColor("#999999", "#666666");

    const value = parseFloat(traffic);
    if (value <= 10) return dynamicColor("#f44336", "#ef5350");
    if (value <= 50) return dynamicColor("#ff9800", "#ffa726");
    return dynamicColor("#4CAF50", "#66BB6A");
}

function getChangeColor(change) {
    if (!change) return dynamicColor("#999999", "#666666");

    if (change.startsWith("-")) {
        return dynamicColor("#f44336", "#ef5350");
    }
    return dynamicColor("#4CAF50", "#66BB6A");
}

function formatTrafficChange(change) {
    if (!change || change === "0.00 GB") return "";
    return change.replace(" GB", "G");
}

function getChangeLabel(change) {
    if (!change || change === "0.00 GB") return "";
    return change.startsWith("-") ? "消耗" : "增加";
}

function calculateStats(data) {
    let remaining = 0;
    let monthlyUsed = 0;
    let dailyUsed = 0;
    const details = [];

    debugLog('🧮 === 开始计算流量统计 ===');

    data.forEach((item, index) => {
        if (item.traffic) {
            const value = parseFloat(item.traffic) || 0;
            remaining += value;

            // 解析月使用量
            let monthlyValue = 0;
            if (item.monthly_used) {
                monthlyValue = parseFloat(item.monthly_used.replace(' GB', '')) || 0;
                monthlyUsed += monthlyValue;
            }

            // 解析今日使用量
            let dailyValue = 0;
            if (item.daily_used) {
                dailyValue = parseFloat(item.daily_used.replace(' GB', '')) || 0;
                dailyUsed += dailyValue;
            }

            // 记录每个订阅的详细信息
            details.push({
                name: item.name,
                remaining: value,
                monthlyUsed: monthlyValue,
                dailyUsed: dailyValue,
                hasUrl: !!item.subscribe_url
            });

            debugLog(`📊 ${index + 1}. ${item.name}:`);
            debugLog(`   剩余: ${value.toFixed(2)} GB`);
            debugLog(`   月使用: ${monthlyValue.toFixed(2)} GB`);
            debugLog(`   日使用: ${dailyValue.toFixed(2)} GB`);
            debugLog(`   订阅链接: ${item.subscribe_url ? '✅' : '❌'}`);
        }
    });

    debugLog('\n📈 === 统计结果 ===');
    debugLog(`🔋 总剩余流量: ${remaining.toFixed(2)} GB`);
    debugLog(`📊 当月已使用: ${monthlyUsed.toFixed(2)} GB`);
    debugLog(`📅 今日已使用: ${dailyUsed.toFixed(2)} GB`);

    const result = {
        remaining: remaining.toFixed(2) + " GB",
        monthlyUsed: monthlyUsed.toFixed(2) + " GB",
        dailyUsed: dailyUsed.toFixed(2) + " GB",
        details: details
    };

    debugLog(`✅ 计算完成: 剩余${result.remaining}, 月使用${result.monthlyUsed}, 日使用${result.dailyUsed}`);
    debugLog('🧮 === 流量统计计算完成 ===\n');

    return result;
}

async function updateAllTraffic() {
    debugLog('🔄 updateAllTraffic() 开始');

    try {
        // 获取当前订阅数据
        const subscriptions = await fetchTrafficData();
        debugLog(`📋 获取到 ${subscriptions.length} 个订阅`);

        if (subscriptions.length === 0) {
            throw new Error('没有找到任何订阅信息');
        }

        // 创建名称到ID的映射表（动态生成）
        const nameToIdMap = {};
        subscriptions.forEach(sub => {
            if (sub.id && sub.name) {
                nameToIdMap[sub.name] = sub.id;
                debugLog(`🗂️  ID映射: ${sub.name} -> ${sub.id}`);
            }
        });

        const updates = [];
        const errors = [];

        // 筛选有订阅链接的订阅
        const validSubscriptions = subscriptions.filter(sub => sub.subscribe_url);
        debugLog(`🔗 找到 ${validSubscriptions.length} 个有订阅链接的订阅`);

        if (validSubscriptions.length === 0) {
            debugError('⚠️  没有找到任何有订阅链接的订阅');
            return { success: true, message: '没有需要更新的订阅（无订阅链接）', updates: [] };
        }

        // 创建所有更新任务
        debugLog('🔄 开始并行更新...');
        const updatePromises = validSubscriptions.map(async (subscription, index) => {
            try {
                debugLog(`📱 [${index + 1}/${validSubscriptions.length}] 正在更新 ${subscription.name}...`);
                debugLog(`🔗 订阅链接: ${subscription.subscribe_url.substring(0, 50)}...`);
                debugLog(`🆔 使用数据库ID: ${subscription.id}`);

                const updatedTraffic = await fetchTrafficInfo(subscription.subscribe_url);
                if (updatedTraffic) {
                    debugLog(`✅ ${subscription.name}: 解析成功 -> ${updatedTraffic}`);

                    // 计算流量变化
                    const changeValue = calculateTrafficChange(
                        subscription.previous_traffic || subscription.traffic,
                        updatedTraffic
                    );

                    // 计算当日使用量（基于流量变化，传递当前当日使用量）
                    const dailyUsage = calculateDailyUsage(changeValue, subscription.daily_used);

                    // 计算当月使用量（需要检查是否需要重置）
                    const monthlyUsage = calculateMonthlyUsage(
                        subscription.monthly_used,
                        dailyUsage,
                        subscription.monthly_reset_date || 1
                    );

                    // 使用从数据库获取的真实ID
                    return {
                        netinfo_id: subscription.id, // 这是数据库中的真实ID
                        name: subscription.name,
                        traffic: updatedTraffic,
                        traffic_change: changeValue,
                        daily_used: dailyUsage,
                        monthly_used: monthlyUsage,
                        success: true
                    };
                } else {
                    debugError(`❌ ${subscription.name}: 无法解析流量信息`);
                    return {
                        netinfo_id: subscription.id,
                        name: subscription.name,
                        success: false,
                        error: '无法解析流量信息'
                    };
                }
            } catch (error) {
                debugError(`❌ ${subscription.name}: 更新失败 - ${error.message}`);
                return {
                    netinfo_id: subscription.id,
                    name: subscription.name,
                    success: false,
                    error: error.message
                };
            }
        });

        // 等待所有更新任务完成
        debugLog('⏳ 等待所有更新任务完成...');
        const results = await Promise.all(updatePromises);

        // 处理结果
        let successCount = 0;
        results.forEach(result => {
            if (result.success) {
                updates.push({
                    netinfo_id: result.netinfo_id,
                    name: result.name,
                    traffic: result.traffic,
                    traffic_change: result.traffic_change
                });
                debugLog(`✅ ${result.name}: 更新成功 -> ${result.traffic} (${result.traffic_change})`);
                successCount++;
            } else {
                errors.push(result);
                debugLog(`❌ ${result.name}: ${result.error}`);
            }
        });

        debugLog(`📊 更新统计: 成功 ${successCount}/${validSubscriptions.length}, 失败 ${errors.length}`);

        // 将更新结果推送到服务器
        if (updates.length > 0) {
            debugLog(`📤 准备推送 ${updates.length} 个更新结果到服务器...`);
            debugLog('📋 更新数据预览:');
            updates.forEach((update, index) => {
                debugLog(`  ${index + 1}. ${update.name} (ID: ${update.netinfo_id}): ${update.traffic}`);
            });

            const syncResult = await pushUpdatesToServer(updates);
            debugLog('🔄 同步结果:', JSON.stringify(syncResult, null, 2));

            return {
                success: true,
                message: `更新完成：成功 ${updates.length} 个，失败 ${errors.length} 个`,
                updates: updates,
                errors: errors,
                syncResult: syncResult
            };
        } else {
            debugError('❌ 所有订阅都更新失败');
            return {
                success: false,
                message: '所有订阅都更新失败',
                updates: [],
                errors: errors
            };
        }

    } catch (error) {
        debugError('❌ updateAllTraffic error:', error.message);
        throw error;
    }
}

async function fetchTrafficInfo(subscribeUrl) {
    debugLog(`🔗 fetchTrafficInfo: ${subscribeUrl.substring(0, 60)}...`);

    try {
        const request = new Request(subscribeUrl);
        request.headers = {
            'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language': 'zh-CN,zh;q=0.9,en;q=0.8',
            'Accept-Encoding': 'gzip, deflate, br',
            'DNT': '1',
            'Connection': 'keep-alive',
            'Upgrade-Insecure-Requests': '1',
        };
        request.timeoutInterval = 10; // 10秒超时

        debugLog('⏳ 发送HTTP请求...');
        const response = await request.loadString();
        debugLog(`📥 获取到响应，长度: ${response.length} 字符`);

        if (response.length === 0) {
            debugError('❌ 响应内容为空');
            throw new Error('订阅链接返回空内容');
        }

        if (DEBUG) {
            debugLog('📄 原始响应内容前300字符:');
            debugLog(response.substring(0, 300));

            // 检查是否包含常见的关键词
            const commonKeywords = ['traffic', 'remaining', '剩余', '流量', 'GB', 'MB', 'TB'];
            debugLog('🔍 响应包含的关键词:');
            commonKeywords.forEach(keyword => {
                if (response.toLowerCase().includes(keyword.toLowerCase())) {
                    debugLog(`  ✅ 包含 "${keyword}"`);
                }
            });
        }

        // 解码响应内容
        debugLog('🔓 开始解码响应内容...');
        let decodedText = decodeResponse(response);
        debugLog(`🔓 解码完成，解码后长度: ${decodedText.length} 字符`);

        if (DEBUG) {
            debugLog('🔓 解码后内容前300字符:');
            debugLog(decodedText.substring(0, 300));
        }

        // 解析流量信息
        debugLog('🎯 开始解析流量信息...');
        const traffic = parseTrafficInfo(decodedText);

        if (traffic) {
            debugLog(`✅ 解析成功: ${traffic}`);
        } else {
            debugError('❌ 无法解析流量信息');
            debugError('🔍 解码后内容完整长度:', decodedText.length);
            debugError('🔍 是否包含数字:', /\d/.test(decodedText));

            // 尝试找出可能的流量模式
            const numberMatches = decodedText.match(/(\d+(?:\.\d+)?)\s*([GMT]B)/gi);
            if (numberMatches) {
                debugLog('💡 找到的可能流量数值:', numberMatches.slice(0, 5));
            }
        }

        return traffic;

    } catch (error) {
        debugError(`❌ fetchTrafficInfo 失败: ${error.message}`);
        debugError('🔗 失败的URL:', subscribeUrl);

        // 分析错误类型
        if (error.message.includes('timeout')) {
            debugError('⏰ 错误类型: 请求超时');
        } else if (error.message.includes('network')) {
            debugError('🌐 错误类型: 网络错误');
        } else if (error.message.includes('404')) {
            debugError('🔍 错误类型: 链接无效 (404)');
        } else if (error.message.includes('403')) {
            debugError('🚫 错误类型: 访问被拒绝 (403)');
        }

        throw error;
    }
}

function decodeResponse(responseText) {
    // 性能优化：快速检查是否需要解码
    if (!responseText || responseText.length < 50) {
        return responseText; // 太短的文本可能不需要解码
    }

    debugLog(`🔍 开始解码，原始长度: ${responseText.length}`);
    debugLog(`📄 原始数据前100字符: ${responseText.substring(0, 100)}`);

    try {
        // 方法1: 尝试Base64解码
        const trimmedText = responseText.trim();
        const isLikelyBase64 = /^[A-Za-z0-9+/]*={0,2}$/.test(trimmedText) && trimmedText.length % 4 === 0;

        if (isLikelyBase64 && trimmedText.length > 20) {
            debugLog('🔍 检测到可能的Base64编码，尝试解码...');

            try {
                // 方法1: 使用Data.fromBase64String()专门处理Base64解码
                let decodedData;
                try {
                    const base64Data = Data.fromBase64String(trimmedText);
                    decodedData = base64Data.toRawString();
                    debugLog('✅ 使用Data.fromBase64String()解码成功');
                } catch (error1) {
                    debugLog('❌ Data.fromBase64String()失败，尝试Data.fromString()');
                    // 方法2: 备用方案 - 使用Data.fromString()但指定Base64
                    const base64Decoded = Data.fromString(trimmedText);
                    decodedData = base64Decoded.toRawString();
                }

                debugLog(`📊 Base64解码后长度: ${decodedData.length}`);
                debugLog(`📄 Base64解码后前100字符: ${decodedData.substring(0, 100)}`);

                // 检查解码是否成功
                if (decodedData !== trimmedText) {
                    debugLog('✅ Base64解码产生变化');

                    // 尝试对Base64解码结果进行URL解码
                    let finalResult = decodedData;
                    if (decodedData.includes('%')) {
                        try {
                            const urlDecoded = decodeURIComponent(decodedData);
                            debugLog(`✅ Base64+URL解码成功，最终长度: ${urlDecoded.length}`);
                            debugLog(`📄 最终解码前100字符: ${urlDecoded.substring(0, 100)}`);
                            finalResult = urlDecoded;
                        } catch (urlError) {
                            debugLog('⚠️ URL解码失败，使用Base64解码结果');
                        }
                    }

                    // 检查最终结果是否包含可识别内容
                    if (containsReadableContent(finalResult)) {
                        debugLog('✅ 解码成功，包含可读内容');
                        return finalResult;
                    } else {
                        debugLog('⚠️ 解码结果不包含可读内容，继续尝试其他方法');
                    }
                } else {
                    debugLog('⚠️ Base64解码后没有变化');
                }
            } catch (base64Error) {
                debugLog('❌ Base64解码失败:', base64Error.message);
            }
        }

        // 方法2: 直接URL解码
        if (responseText.includes('%')) {
            debugLog('🔍 检测到URL编码，尝试解码...');
            try {
                const urlDecoded = decodeURIComponent(responseText);
                debugLog(`✅ URL解码成功，长度: ${urlDecoded.length}`);
                debugLog(`📄 URL解码前100字符: ${urlDecoded.substring(0, 100)}`);

                if (containsReadableContent(urlDecoded)) {
                    debugLog('✅ URL解码结果包含可读内容');
                    return urlDecoded;
                }
            } catch (error) {
                debugLog('❌ URL解码失败:', error.message);
            }
        }

        // 方法3: 尝试多重解码
        debugLog('🔍 尝试多重解码...');
        const multiDecoded = deepDecodeResponse(responseText);
        if (multiDecoded !== responseText && containsReadableContent(multiDecoded)) {
            debugLog('✅ 多重解码成功');
            return multiDecoded;
        }

    } catch (error) {
        debugLog('❌ 所有解码方式都失败，使用原始文本:', error.message);
    }

    debugLog('⚠️ 使用原始文本');
    return responseText;
}

// 辅助函数：检查文本是否包含可读内容
function containsReadableContent(text) {
    if (!text || text.length < 10) return false;

    // 检查是否包含中文
    const hasChinese = /[\u4e00-\u9fff\u3400-\u4dbf]/.test(text);

    // 检查是否包含流量相关单位
    const hasTrafficUnits = /[0-9]+\s*(GB|MB|TB|KB|bytes?|B)/i.test(text);

    // 检查是否包含常见的订阅信息关键词
    const hasKeywords = /total|used|remaining|expire|valid|流量|剩余|总量|过期|有效/i.test(text);

    // 检查是否包含常见的协议信息
    const hasProtocols = /(vmess|vless|trojan|ss|ssr|hysteria)/i.test(text);

    const isReadable = hasChinese || hasTrafficUnits || hasKeywords || hasProtocols;

    if (isReadable) {
        debugLog(`✅ 可读内容检测: 中文=${hasChinese}, 流量单位=${hasTrafficUnits}, 关键词=${hasKeywords}, 协议=${hasProtocols}`);
    }

    return isReadable;
}

function deepDecodeResponse(encodedText) {
    try {
        let decoded = encodedText;
        let decodeCount = 0;
        const maxDecodes = 5; // 增加最大解码次数

        debugLog(`🔍 开始多重解码，初始长度: ${encodedText.length}`);

        while (decodeCount < maxDecodes) {
            const previouslyDecoded = decoded;

            // 尝试URL解码
            if (decoded.includes('%')) {
                try {
                    decoded = decodeURIComponent(decoded);
                    debugLog(`🔍 第${decodeCount + 1}次URL解码: 长度 ${decoded.length}`);
                } catch (error) {
                    debugLog(`❌ 第${decodeCount + 1}次URL解码失败: ${error.message}`);
                    break;
                }
            }

            // 如果URL解码没有变化，尝试Base64解码
            if (decoded === previouslyDecoded) {
                const trimmedDecoded = decoded.trim();
                const isLikelyBase64 = /^[A-Za-z0-9+/]*={0,2}$/.test(trimmedDecoded) && trimmedDecoded.length % 4 === 0;

                if (isLikelyBase64 && trimmedDecoded.length > 20) {
                    try {
                        let base64Data;
                        try {
                            const base64Decoded = Data.fromBase64String(trimmedDecoded);
                            base64Data = base64Decoded.toRawString();
                            debugLog(`✅ 第${decodeCount + 1}次Data.fromBase64String()解码成功`);
                        } catch (error1) {
                            debugLog(`❌ 第${decodeCount + 1}次Data.fromBase64String()失败，尝试Data.fromString()`);
                            const base64Decoded = Data.fromString(trimmedDecoded);
                            base64Data = base64Decoded.toRawString();
                        }
                        debugLog(`🔍 第${decodeCount + 1}次Base64解码: 长度 ${base64Data.length}`);
                        decoded = base64Data;
                    } catch (base64Error) {
                        debugLog(`❌ 第${decodeCount + 1}次Base64解码失败: ${base64Error.message}`);
                        break;
                    }
                } else {
                    // 无法继续解码
                    debugLog(`🔍 第${decodeCount + 1}次解码没有变化，停止`);
                    break;
                }
            }

            if (decoded === previouslyDecoded) {
                debugLog(`🔍 第${decodeCount + 1}次解码没有变化，停止`);
                break;
            }

            decodeCount++;
            debugLog(`📄 第${decodeCount}次解码后前100字符: ${decoded.substring(0, 100)}`);

            // 检查是否包含可读内容
            if (containsReadableContent(decoded)) {
                debugLog('✅ 多重解码成功，包含可读内容');
                return decoded;
            }
        }

        debugLog(`❌ 多重解码失败，使用原始文本`);
        return encodedText;

    } catch (error) {
        debugLog('❌ 多重解码失败，使用原始文本:', error.message);
        return encodedText;
    }
}

function parseTrafficInfo(text) {
    debugLog('🔍 开始解析流量信息...');
    debugLog(`📊 待解析文本长度: ${text.length}`);

    // 更全面的流量匹配模式 - 按优先级排序
    const patterns = [
        // 高优先级：明确的剩余流量模式
        { pattern: /(?:剩余流量|流量剩余|流量|remaining|left)\s*[:：]?\s*(\d+(?:\.\d+)?\s*(?:GB|MB|TB|KB))/i, priority: 1, name: "剩余流量模式" },
        { pattern: /(?:traffic|quota|data|limit)\s*[:：]?\s*(\d+(?:\.\d+)?\s*(?:GB|MB|TB|KB))/i, priority: 2, name: "英文流量模式" },

        // 中等优先级：包含关键词的模式
        { pattern: /(\d+(?:\.\d+)?\s*(?:GB|MB|TB|KB))\s*(?:剩余|left|remain|remaining)/i, priority: 3, name: "数值+剩余关键词" },
        { pattern: /(\d+(?:\.\d+)?\s*(?:GB|MB|TB|KB))\s*\/\s*[\d.]+/i, priority: 4, name: "比例格式 XXX/YYY" },

        // 低优先级：通用模式
        { pattern: /total\s*[:：]?\s*(\d+(?:\.\d+)?\s*(?:GB|MB|TB|KB))/i, priority: 5, name: "总量模式" },
        { pattern: /used\s*[:：]?\s*(\d+(?:\.\d+)?\s*(?:GB|MB|TB|KB))/i, priority: 6, name: "已用模式" },
        { pattern: /transfer\s*[:：]?\s*(\d+(?:\.\d+)?\s*(?:GB|MB|TB|KB))/i, priority: 7, name: "传输模式" },

        // 最后兜底：任意数字+单位
        { pattern: /(\d+(?:\.\d+)?\s*(?:GB|MB|TB|KB))/i, priority: 8, name: "任意数值模式" }
    ];

    // 检查更多行，避免遗漏
    const lines = text.split(/\r?\n/).slice(0, 15); // 增加检查行数
    debugLog(`📝 分析前 ${lines.length} 行文本`);

    let bestMatch = null;
    let bestPriority = 999; // 越小优先级越高

    // 逐行匹配
    for (let i = 0; i < lines.length; i++) {
        const line = lines[i].trim();
        if (!line) continue; // 跳过空行

        debugLog(`📄 第${i + 1}行 (${line.length}字符): ${line.substring(0, 80)}${line.length > 80 ? '...' : ''}`);

        for (let j = 0; j < patterns.length; j++) {
            const { pattern, name, priority } = patterns[j];
            const match = line.match(pattern);
            if (match) {
                const result = match[1] || match[0]; // 使用第一个捕获组或整个匹配
                const finalResult = result.trim();
                debugLog(`✅ 模式 "${name}" (优先级${priority}) 匹配成功: ${finalResult}`);

                // 保存优先级最高的匹配
                if (priority < bestPriority) {
                    bestMatch = finalResult;
                    bestPriority = priority;
                    debugLog(`🎯 新的最佳匹配 (优先级${priority}): ${finalResult}`);
                }
            }
        }
    }

    // 如果找到了最佳匹配，直接返回
    if (bestMatch) {
        debugLog(`🏆 返回最佳匹配结果: ${bestMatch}`);
        return bestMatch;
    }

    debugLog('❌ 逐行匹配失败，尝试整段文本匹配');

    // 整段文本匹配
    for (let i = 0; i < patterns.length; i++) {
        const { pattern, name, priority } = patterns[i];
        const match = text.match(pattern);
        if (match) {
            const result = match[1] || match[0];
            const finalResult = result.trim();
            debugLog(`✅ 整段文本 - 模式 "${name}" (优先级${priority}) 匹配成功: ${finalResult}`);
            return finalResult;
        }
    }

    // 最后尝试：查找所有可能的流量数值
    debugLog('❌ 所有模式都失败，查找可能的流量数值...');
    const allNumbers = text.match(/(\d+(?:\.\d+)?\s*(?:GB|MB|TB|KB))/gi);
    if (allNumbers && allNumbers.length > 0) {
        debugLog(`💡 找到 ${allNumbers.length} 个可能的流量值:`, allNumbers.slice(0, 10));

        // 尝试智能选择最合适的值
        let selectedValue;
        if (allNumbers.length === 1) {
            selectedValue = allNumbers[0].trim();
        } else {
            // 如果有多个值，选择看起来最像剩余流量的（通常是较小的GB值，或者包含明显特征的）
            selectedValue = allNumbers[0].trim();
        }

        debugLog(`🎯 选择值: ${selectedValue}`);
        return selectedValue;
    }

    // 完全失败
    debugError('❌ 完全无法解析到流量信息');
    if (DEBUG) {
        debugError('🔍 文本长度:', text.length);
        debugError('🔍 是否包含数字:', /\d/.test(text));
        debugError('🔍 是否包含单位:', /GB|MB|TB|KB/i.test(text));
        debugError('🔍 是否包含流量关键词:', /流量|剩余|traffic|remaining|quota/i.test(text));

        // 显示前800字符用于调试
        debugError('📄 前800字符内容:');
        for (let i = 0; i < Math.min(text.length, 800); i += 100) {
            const chunk = text.substring(i, i + 100);
            debugError(`  ${i}-${i + 99}: ${chunk}`);
        }
    }

    return null;
}

function calculateTrafficChange(oldTrafficStr, newTrafficStr) {
    if (!oldTrafficStr || oldTrafficStr === newTrafficStr) {
        return "0.00 GB";
    }

    const oldValue = extractTrafficValue(oldTrafficStr);
    const newValue = extractTrafficValue(newTrafficStr);

    const changeValue = oldValue - newValue;

    // 如果流量增加了（可能是重置），则不显示变化
    if (changeValue < 0) {
        return "0.00 GB";
    }

    return `${changeValue.toFixed(2)} GB`;
}

function calculateDailyUsage(trafficChangeStr, currentDailyUsed = "0.00 GB") {
    // 从流量变化字符串中提取数值
    const changeValue = extractTrafficValue(trafficChangeStr);

    // 当日使用量重置检查：如果是新的一天，重置为0
    const now = new Date();
    const currentHour = now.getHours();

    // 检查是否是0点后第一次更新（简单实现：0-6点之间都认为是新的一天）
    // 实际应该基于last_updated_date，但Scriptable无法访问后端的日期字段
    // 这里假设每次运行都是新的一天，更准确的逻辑在后端处理
    if (currentHour < 6) {
        debugLog(`🌅 检测到早晨更新，当日使用量从0开始计算`);
        if (changeValue > 0) {
            return `${changeValue.toFixed(2)} GB`;
        }
        return "0.00 GB";
    }

    // 提取当前当日使用量
    const currentDaily = extractTrafficValue(currentDailyUsed);

    // 当日使用量累加本次流量消耗
    if (changeValue > 0) {
        const newDaily = currentDaily + changeValue;
        debugLog(`📊 当日使用量累加: ${currentDaily.toFixed(2)} + ${changeValue.toFixed(2)} = ${newDaily.toFixed(2)} GB`);
        return `${newDaily.toFixed(2)} GB`;
    }

    return currentDailyUsed || "0.00 GB";
}

function calculateMonthlyUsage(currentMonthlyUsed, dailyUsage, resetDay = 1) {
    const now = new Date();
    const currentDate = now.getDate();

    // 提取数值
    const currentMonthly = extractTrafficValue(currentMonthlyUsed);
    const daily = extractTrafficValue(dailyUsage);

    // 检查是否是重置日（每月1号或指定的重置日期）
    if (currentDate === resetDay) {
        debugLog(`🔄 检测到重置日（${resetDay}号），当月使用量清空重置为当日使用量: ${dailyUsage}`);
        // 重置日：当月使用量从当日使用量开始累计
        return dailyUsage;
    }

    // 非重置日：当月使用量累加当日使用量
    const newMonthlyTotal = currentMonthly + daily;
    debugLog(`📈 当月使用量累计: ${currentMonthly.toFixed(2)} + ${daily.toFixed(2)} = ${newMonthlyTotal.toFixed(2)} GB`);
    return `${newMonthlyTotal.toFixed(2)} GB`;
}

function extractTrafficValue(trafficStr) {
    if (!trafficStr) return 0.0;

    const match = trafficStr.match(/(\d+(?:\.\d+)?)/);
    if (!match) return 0.0;

    let value = parseFloat(match[1]);

    if (trafficStr.toUpperCase().includes('MB')) {
        value = value / 1024.0;
    } else if (trafficStr.toUpperCase().includes('TB')) {
        value = value * 1024.0;
    } else if (trafficStr.toUpperCase().includes('KB')) {
        value = value / (1024.0 * 1024.0);
    }

    return value;
}

async function pushUpdatesToServer(updates) {
    debugLog('pushUpdatesToServer start, updates count:', updates.length);

    try {
        const url = `${API_BASE_URL}/update_all_traffic/`;
        debugLog('🌐 push URL:', url);

        // 验证输入数据
        debugLog('🔍 验证输入数据...');
        if (!Array.isArray(updates)) {
            throw new Error('updates参数必须是数组');
        }

        const validUpdates = updates.filter(update =>
            update &&
            update.netinfo_id &&
            update.name &&
            update.traffic
        );

        if (validUpdates.length !== updates.length) {
            debugLog(`⚠️  过滤了 ${updates.length - validUpdates.length} 个无效更新项`);
        }

        if (validUpdates.length === 0) {
            throw new Error('没有有效的更新数据');
        }

        const requestData = {
            updates: validUpdates
        };

        debugLog('📤 发送数据详情:');
        debugLog(JSON.stringify(requestData, null, 2));

        // 创建HTTP请求
        const request = new Request(url);
        request.method = 'POST';
        request.headers = {
            "User-Agent": "Scriptable-Traffic-Monitor-Auto/1.0",
            "Accept": "application/json",
            "Content-Type": "application/json"
        };

        const requestBody = JSON.stringify(requestData);
        debugLog(`📦 请求体长度: ${requestBody.length} 字符`);
        debugLog('📋 请求数据验证:');
        debugLog(`  - 更新项数量: ${requestData.updates.length}`);
        debugLog(`  - 数据格式: ${typeof requestBody}`);
        debugLog(`  - Content-Type: application/json`);

        request.body = requestBody;
        request.timeoutInterval = 15; // 增加超时时间

        debugLog('🚀 发送HTTP请求到服务器...');
        const response = await request.loadJSON();
        debugLog('📨 服务器响应:');
        debugLog(JSON.stringify(response, null, 2));

        // 详细分析响应
        if (response && typeof response === 'object') {
            if (response.success === true) {
                debugLog(`✅ 数据推送成功: ${response.message}`);
                if (response.success_count !== undefined) {
                    debugLog(`📊 推送统计: 成功 ${response.success_count} 个，失败 ${response.failed_count || 0} 个`);
                }
                return response;
            } else {
                debugLog('❌ 服务器返回失败状态');
                debugLog(`🔍 错误信息: ${response.error || response.message || '未知错误'}`);
                throw new Error(response?.error || response?.message || "数据推送失败");
            }
        } else {
            debugLog('❌ 服务器响应格式无效');
            debugLog(`🔍 响应类型: ${typeof response}`);
            debugLog(`🔍 响应内容: ${response}`);
            throw new Error("服务器响应格式无效");
        }
    } catch (error) {
        debugLog('❌ pushUpdatesToServer 发生异常:');
        debugLog(`🔍 错误类型: ${error.constructor?.name || 'Unknown'}`);
        debugLog(`🔍 错误消息: ${error.message}`);
        debugLog(`📍 错误堆栈: ${error.stack || '无堆栈信息'}`);

        // 根据错误类型提供更详细的错误信息
        let detailedError = '数据推送失败';
        if (error.message.includes('timeout')) {
            detailedError = '请求超时，服务器响应缓慢';
        } else if (error.message.includes('network')) {
            detailedError = '网络连接错误';
        } else if (error.message.includes('404')) {
            detailedError = 'API端点不存在 (404)';
        } else if (error.message.includes('403') || error.message.includes('401')) {
            detailedError = 'API访问被拒绝';
        } else if (error.message.includes('500')) {
            detailedError = '服务器内部错误';
        } else {
            detailedError = `数据推送失败: ${error.message}`;
        }

        throw new Error(detailedError);
    }
}

async function fetchTrafficData() {
    const now = Date.now();

    if (cachedData && (now - lastFetchTime) < CACHE_DURATION) {
        debugLog('💾 使用缓存数据');
        return cachedData;
    }

    debugLog('📡 从服务器获取新数据...');
    debugLog(`🌐 API地址: ${API_BASE_URL}`);

    try {
        const url = `${API_BASE_URL}/subscriptions/`;
        debugLog(`🔗 请求URL: ${url}`);

        const request = new Request(url);
        request.headers = {
            "User-Agent": "Scriptable-Traffic-Monitor-Auto/1.0",
            "Accept": "application/json"
        };
        request.timeoutInterval = 8; // 减少超时时间，提高响应速度

        debugLog('⏳ 发送请求...');
        const response = await request.loadJSON();
        debugLog('📨 收到API响应');

        if (response && response.success && response.data && Array.isArray(response.data)) {
            debugLog(`📋 响应成功: 获取到 ${response.data.length} 条记录`);

            // 详细分析响应数据
            const validData = response.data.filter(item => item && item.id && item.name);
            const invalidData = response.data.length - validData.length;

            if (invalidData > 0) {
                debugLog(`⚠️  过滤了 ${invalidData} 条无效记录`);
            }

            debugLog(`✅ 有效数据: ${validData.length} 条订阅`);

            // 统计有订阅链接的数据
            const withUrl = validData.filter(item => item.subscribe_url).length;
            const withMonthlyUsed = validData.filter(item => item.monthly_used && item.monthly_used !== '0.00 GB').length;
            const withDailyUsed = validData.filter(item => item.daily_used && item.daily_used !== '0.00 GB').length;

            debugLog(`🔗 有订阅链接: ${withUrl}/${validData.length}`);
            debugLog(`📊 有月使用数据: ${withMonthlyUsed}/${validData.length}`);
            debugLog(`📅 有日使用数据: ${withDailyUsed}/${validData.length}`);

            cachedData = validData;
            lastFetchTime = now;
            return cachedData;
        } else {
            debugLog('❌ 响应格式无效:', response);
            throw new Error(response?.error || "数据格式错误或获取失败");
        }
    } catch (error) {
        debugLog('❌ 获取数据失败:', error.message);
        debugLog('🔍 错误详情:', JSON.stringify(error, null, 2));

        if (cachedData) {
            const cacheAge = Math.floor((now - lastFetchTime) / 60000);
            debugLog(`🔄 使用过期缓存数据 (${cacheAge}分钟前)`);
            return cachedData;
        }

        // 在调试模式下提供测试数据
        if (DEBUG && config.runsInApp) {
            debugLog('🧪 DEBUG模式: 使用测试数据进行演示');
            return getTestData();
        }

        throw new Error(`API连接失败: ${error.message}\n地址: ${API_BASE_URL}`);
    }
}

function getTestData() {
    return [
        {
            id: 1,
            name: "M78星云",
            traffic: "180.25 GB",
            previous_traffic: "500.00 GB",
            traffic_change: "14.68 GB",
            monthly_used: "319.75 GB",  // 当月已使用
            daily_used: "8.23 GB",     // 今日已使用
            order: 0
        },
        {
            id: 2,
            name: "智联云",
            traffic: "712.76 GB",
            previous_traffic: "1024.00 GB",
            traffic_change: "0.05 GB",
            monthly_used: "311.24 GB",  // 当月已使用
            daily_used: "12.45 GB",     // 今日已使用
            order: 1
        },
        {
            id: 3,
            name: "蜂群",
            traffic: "840.72 GB",
            previous_traffic: "1000.00 GB",
            traffic_change: "20.09 GB",
            monthly_used: "159.28 GB",  // 当月已使用
            daily_used: "3.67 GB",      // 今日已使用
            order: 2
        }
    ];
}



function formatDate(date, fmt) {
    const df = new DateFormatter();
    df.dateFormat = fmt;
    return df.string(date);
}

await run();