// Variables used by Scriptable.
// These must be at the very top of the file. Do not edit.
// icon-color: purple; icon-glyph: magic;
const API_BASE_URL = "https://www.dyxcloud.com/getinfo/api";
const DEBUG = false;
const AUTO_UPDATE_INTERVAL = 10 * 60 * 1000; // 10分钟

const COMPONENT_CONFIG = {
    small: { maxCount: 3, showDetails: false },
    medium: { maxCount: 8, showDetails: true },
    large: { maxCount: 15, showDetails: true }
};

let cachedData = null;
let lastFetchTime = 0;
let lastUpdateTime = 0;
let lastUpdateResult = null; // 存储最后一次更新的结果
let lastUpdateError = null; // 存储最后一次更新的错误信息
const CACHE_DURATION = 5 * 60 * 1000; // 延长缓存时间，减少频繁请求

function debugLog(...args) {
    if (DEBUG) {
        console.log('[Traffic Monitor]', ...args);
    }
}

function dynamicColor(lightColor, darkColor) {
    return Color.dynamic(new Color(lightColor), new Color(darkColor));
}

async function run() {
    debugLog('run() start');

    try {
        // 调试模式：运行更新并显示解析数据
        if (config.runsInApp && DEBUG) {
            debugLog('=== 开始调试模式：运行更新并显示解析数据 ===');

            // 获取订阅数据
            const subscriptions = await fetchTrafficData();
            debugLog(`获取到 ${subscriptions.length} 个订阅`);

            // 逐个解析订阅数据
            for (let i = 0; i < subscriptions.length; i++) {
                const subscription = subscriptions[i];
                debugLog(`\n--- 订阅 ${i + 1}: ${subscription.name} ---`);
                debugLog(`订阅链接: ${subscription.subscribe_url}`);

                if (!subscription.subscribe_url) {
                    debugLog('❌ 没有订阅链接，跳过');
                    continue;
                }

                try {
                    debugLog('🔍 开始获取流量信息...');
                    const updatedTraffic = await fetchTrafficInfo(subscription.subscribe_url);

                    if (updatedTraffic) {
                        debugLog(`✅ 解析成功: ${updatedTraffic}`);

                        // 计算流量变化
                        const changeValue = calculateTrafficChange(
                            subscription.previous_traffic || subscription.traffic,
                            updatedTraffic
                        );
                        debugLog(`📊 流量变化: ${subscription.traffic} -> ${updatedTraffic} (${changeValue})`);
                    } else {
                        debugLog('❌ 解析失败: 无法解析流量信息');
                    }
                } catch (error) {
                    debugLog(`❌ 获取失败: ${error.message}`);
                }
            }

            debugLog('\n=== 调试完成 ===');

            // 显示Widget预览
            const largeWidget = await renderLarge();
            await largeWidget.presentLarge();

        } else {
            // 正常模式
            if (config.runsInApp) {
                debugLog('running in Scriptable app, forcing update and showing preview');
                cachedData = null;
                lastFetchTime = 0;
                await performAutoUpdate();

                const largeWidget = await renderLarge();
                await largeWidget.presentLarge();
            }

            if (!DEBUG) {
                await checkAndUpdateIfNeeded();
            }

            const widget = await createWidget();
            Script.setWidget(widget);
            debugLog('run() success');
        }
    } catch (error) {
        debugLog('run() error:', error);
        const errorWidget = createErrorWidget(error);
        Script.setWidget(errorWidget);

        if (config.runsInApp) {
            const alert = new Alert();
            alert.title = "运行错误";
            alert.message = `错误: ${error.message}`;
            alert.addAction("确定");
            await alert.present();
        }
    }

    Script.complete();
}

async function checkAndUpdateIfNeeded() {
    const now = Date.now();

    // 如果是首次运行或者距离上次更新超过10分钟，则执行自动更新
    if (lastUpdateTime === 0 || (now - lastUpdateTime) >= AUTO_UPDATE_INTERVAL) {
        debugLog('triggering auto update');
        await performAutoUpdate();
        lastUpdateTime = now;
    } else {
        debugLog('skip auto update, last update:', new Date(lastUpdateTime));
    }
}

async function performAutoUpdate() {
    const startTime = Date.now();
    debugLog('performAutoUpdate() start');

    try {
        // 性能优化：不要总是清空缓存，只有在强制更新时才清空
        if (cachedData) {
            debugLog('使用现有缓存，仅更新流量数据');
        } else {
            lastFetchTime = 0;
        }

        // 调用后端API触发流量更新
        debugLog('triggering local traffic update');
        const updateResult = await updateAllTraffic();

        // 记录更新成功结果
        lastUpdateResult = updateResult;
        lastUpdateError = null;
        lastUpdateTime = Date.now(); // 记录成功更新的时间
        debugLog('auto update success:', updateResult.message);

        // 性能优化：只有在缓存被清空时才重新获取数据
        if (!cachedData) {
            const freshData = await fetchTrafficData();
            debugLog(`auto update completed, fetched ${freshData.length} subscriptions`);
        }

        const duration = Date.now() - startTime;
        debugLog(`自动更新完成，耗时: ${duration}ms`);

    } catch (error) {
        const duration = Date.now() - startTime;
        debugLog(`performAutoUpdate() error after ${duration}ms:`, error);
        // 记录更新失败信息
        lastUpdateResult = null;
        lastUpdateError = {
            message: error.message,
            timestamp: new Date()
        };
    }
}


async function createWidget() {
    const family = config.widgetFamily || 'small';
    debugLog('creating widget for family:', family);

    try {
        switch (family) {
            case 'medium':
                return await renderMedium();
            case 'large':
                return await renderLarge();
            default:
                return await renderSmall();
        }
    } catch (error) {
        debugLog('createWidget error:', error.message);
        return createErrorWidget(error);
    }
}

function createErrorWidget(error) {
    const widget = new ListWidget();
    widget.backgroundColor = dynamicColor("#e5e7eb", "#0f172a");
    widget.setPadding(25, 18, 25, 18);

    // 标题
    const titleStack = widget.addStack();
    titleStack.centerAlignContent();
    const titleText = titleStack.addText("流量监控");
    titleText.font = Font.boldSystemFont(22);
    titleText.textColor = dynamicColor("#1e293b", "#f1f5f9");
    widget.addSpacer(12);

    // 分隔线
    const separator = widget.addStack();
    separator.backgroundColor = dynamicColor("#e2e8f0", "#334155");
    separator.size = new Size(0, 1);
    widget.addSpacer(20);

    // 错误内容卡片
    const errorCard = widget.addStack();
    errorCard.backgroundColor = dynamicColor("#fef2f2", "#1e293b");
    errorCard.cornerRadius = 10;
    errorCard.setPadding(20, 20, 20, 20);
    errorCard.centerAlignContent();
    errorCard.layoutVertically();

    const icon = SFSymbol.named("exclamationmark.triangle.fill");
    const iconImg = errorCard.addImage(icon.image);
    iconImg.tintColor = dynamicColor("#dc2626", "#ef4444");
    iconImg.imageSize = new Size(36, 36);

    errorCard.addSpacer(12);

    const errorText = errorCard.addText("网络连接失败");
    errorText.font = Font.semiboldSystemFont(18);
    errorText.textColor = dynamicColor("#dc2626", "#ef4444");
    errorText.centerAlignText();

    errorCard.addSpacer(8);

    const detailText = errorCard.addText(String(error.message || "请检查网络连接"));
    detailText.font = Font.systemFont(14);
    detailText.textColor = dynamicColor("#991b1b", "#fca5a5");
    detailText.centerAlignText();
    detailText.lineLimit = 3;

    errorCard.addSpacer(12);

    const retryText = errorCard.addText("点击重试");
    retryText.font = Font.mediumSystemFont(14);
    retryText.textColor = dynamicColor("#4CAF50", "#66BB6A");
    retryText.centerAlignText();
    retryText.url = "scriptable:///run";

    return widget;
}

async function renderSmall() {
    const widget = new ListWidget();
    widget.backgroundColor = dynamicColor("#f3f8ff", "#1e1e22");
    widget.setPadding(8, 8, 8, 8);

    const data = await fetchTrafficData();
    const config = COMPONENT_CONFIG.small;
    const items = data.slice(0, config.maxCount);

    if (items.length === 0) {
        return createEmptyWidget(widget, "暂无流量数据");
    }

    await addHeader(widget, "流量监控", false);

    // 添加简单的状态指示器
    await addUpdateStatusIndicatorSmall(widget);

    for (let i = 0; i < items.length; i++) {
        if (i > 0) widget.addSpacer(4);
        await addTrafficItemSmall(widget, items[i], i + 1);
    }

    widget.addSpacer(4);
    await addFooter(widget, data.length);

    return widget;
}

async function renderMedium() {
    const widget = new ListWidget();
    widget.backgroundColor = dynamicColor("#f3f8ff", "#1e1e22");
    widget.setPadding(10, 10, 10, 10);

    const data = await fetchTrafficData();
    const config = COMPONENT_CONFIG.medium;
    const items = data.slice(0, config.maxCount);

    if (items.length === 0) {
        return createEmptyWidget(widget, "暂无流量数据");
    }

    await addHeader(widget, "流量监控", true);

    // 添加简化的更新状态显示
    await addUpdateStatusInfoCompact(widget);

    const leftStack = widget.addStack();
    leftStack.layoutVertically();
    const rightStack = widget.addStack();
    rightStack.layoutVertically();

    for (let i = 0; i < items.length; i++) {
        const targetStack = i % 2 === 0 ? leftStack : rightStack;
        if (i > 1) targetStack.addSpacer(6);
        await addTrafficItemMedium(targetStack, items[i], i + 1);
    }

    widget.addSpacer(8);
    await addFooter(widget, data.length);

    return widget;
}

async function renderLarge() {
    const widget = new ListWidget();
    widget.backgroundColor = dynamicColor("#e5e7eb", "#0f172a");
    widget.setPadding(25, 18, 25, 18);

    const data = await fetchTrafficData();
    const config = COMPONENT_CONFIG.large;
    const items = data.slice(0, config.maxCount);

    if (items.length === 0) {
        return createEmptyWidget(widget, "暂无流量数据");
    }

    // 标题和更新时间 - 同一行
    const headerStack = widget.addStack();
    headerStack.centerAlignContent();

    // 左侧：标题
    const titleText = headerStack.addText("流量监控");
    titleText.font = Font.boldSystemFont(22);
    titleText.textColor = dynamicColor("#1e293b", "#f1f5f9");

    headerStack.addSpacer(); // 推到右边

    // 右侧：更新时间和状态
    const timeStack = headerStack.addStack();
    timeStack.layoutVertically();
    timeStack.centerAlignContent();

    // 更新时间 - 显示上次更新成功的时间
    const updateTimeText = timeStack.addText(`更新于: ${getLastSuccessfulUpdateTime()}`);
    updateTimeText.font = Font.mediumSystemFont(13);
    updateTimeText.textColor = dynamicColor("#475569", "#cbd5e1");
    updateTimeText.textOpacity = 0.9;

    // 更新状态指示器
    await addUpdateStatusIndicator(timeStack);

    widget.addSpacer(8);

    // 添加分隔线
    const separator = widget.addStack();
    separator.backgroundColor = dynamicColor("#e2e8f0", "#334155");
    separator.size = new Size(0, 1);
    widget.addSpacer(12);

    // 添加更新状态信息（如果有）
    await addUpdateStatusInfo(widget);

    // 内容区域
    const contentStack = widget.addStack();
    contentStack.layoutVertically();

    const itemCount = items.length;
    const isCompact = itemCount > 10;
    const spacing = isCompact ? 8 : 12;

    for (let i = 0; i < items.length; i++) {
        if (i > 0) contentStack.addSpacer(spacing);
        await addTrafficItemSimple(contentStack, items[i], itemCount);
    }

    widget.addSpacer(14);
    await addTotalFooter(widget, data);

    return widget;
}

function createEmptyWidget(widget, message) {
    widget.backgroundColor = dynamicColor("#e5e7eb", "#0f172a");

    const stack = widget.addStack();
    stack.centerAlignContent();
    stack.layoutVertically();

    const icon = SFSymbol.named("chart.bar.xaxis");
    const iconImg = stack.addImage(icon.image);
    iconImg.tintColor = dynamicColor("#94a3b8", "#64748b");
    iconImg.imageSize = new Size(40, 40);
    iconImg.imageOpacity = 0.6;

    stack.addSpacer(12);

    const text = stack.addText(message);
    text.font = Font.mediumSystemFont(16);
    text.textColor = dynamicColor("#64748b", "#94a3b8");
    text.centerAlignText();

    return widget;
}

async function addHeader(widget, title, showTime) {
    widget.addSpacer(8);

    const head = widget.addStack();
    head.setPadding(0, 8, 0, 8);
    head.centerAlignContent();

    const icon = SFSymbol.named('chart.bar');
    const img = head.addImage(icon.image);
    img.tintColor = dynamicColor("#4CAF50", "#66BB6A");
    img.imageSize = new Size(16, 16);

    head.addSpacer(6);

    const titleText = head.addText(title);
    titleText.font = Font.boldSystemFont(13);
    titleText.textColor = dynamicColor("#333333", "#ffffff");

    head.addSpacer();

    const statusIcon = "checkmark.circle";
    const statusSF = SFSymbol.named(statusIcon);
    const statusImg = head.addImage(statusSF.image);
    statusImg.tintColor = dynamicColor("#4CAF50", "#66BB6A");
    statusImg.imageSize = new Size(12, 12);

    if (showTime) {
        head.addSpacer(4);
        const timeText = head.addText(formatDate(new Date(), "HH:mm"));
        timeText.font = Font.systemFont(10);
        timeText.textColor = dynamicColor("#666666", "#999999");
        timeText.textOpacity = 0.7;
    }

    widget.addSpacer(4);
}

async function addFooter(widget, totalCount) {
    widget.addSpacer(8);

    const footer = widget.addStack();
    footer.setPadding(0, 8, 0, 8);
    footer.centerAlignContent();

    const stats = calculateStats(await fetchTrafficData());
    const totalText = footer.addText(`总计: ${stats.total} | 剩余: ${stats.remaining}`);
    totalText.font = Font.systemFont(10);
    totalText.textColor = dynamicColor("#4CAF50", "#66BB6A");

    footer.addSpacer();

    const status = footer.addText("监控");
    status.font = Font.systemFont(9);
    status.textColor = dynamicColor("#999999", "#666666");
    status.textOpacity = 0.6;

    footer.addSpacer(2);

    const updateTime = footer.addText(formatDate(new Date(), "HH:mm"));
    updateTime.font = Font.systemFont(9);
    updateTime.textColor = dynamicColor("#999999", "#666666");
    updateTime.textOpacity = 0.6;
}

async function addTrafficItemSmall(widget, item, index) {
    const stack = widget.addStack();
    stack.centerAlignContent();

    const idx = stack.addText(String(index));
    idx.font = Font.boldSystemFont(11);
    idx.textColor = getRankColor(index);
    idx.minimumScaleFactor = 0.8;

    stack.addSpacer(6);

    const infoStack = stack.addStack();
    infoStack.layoutVertically();

    const nameText = infoStack.addText(item.name);
    nameText.font = Font.mediumSystemFont(11);
    nameText.textColor = dynamicColor("#333333", "#ffffff");
    nameText.lineLimit = 1;
    nameText.minimumScaleFactor = 0.8;

    const trafficText = infoStack.addText(item.traffic);
    trafficText.font = Font.systemFont(9);
    trafficText.textColor = getTrafficColor(item.traffic);
    trafficText.textOpacity = 0.8;

    stack.addSpacer();

    if (item.traffic_change && item.traffic_change !== "0.00 GB") {
        const changeText = stack.addText(formatTrafficChange(item.traffic_change));
        changeText.font = Font.systemFont(8);
        changeText.textColor = getChangeColor(item.traffic_change);
        changeText.minimumScaleFactor = 0.8;
    }
}

async function addTrafficItemMedium(widget, item, index) {
    const stack = widget.addStack();
    stack.centerAlignContent();

    const idx = stack.addText(String(index));
    idx.font = Font.boldSystemFont(12);
    idx.textColor = getRankColor(index);
    idx.minimumScaleFactor = 0.8;

    stack.addSpacer(6);

    const infoStack = stack.addStack();
    infoStack.layoutVertically();

    const nameText = infoStack.addText(item.name);
    nameText.font = Font.mediumSystemFont(12);
    nameText.textColor = dynamicColor("#333333", "#ffffff");
    nameText.lineLimit = 1;
    nameText.minimumScaleFactor = 0.8;

    const detailStack = infoStack.addStack();

    const trafficText = detailStack.addText(item.traffic);
    trafficText.font = Font.systemFont(10);
    trafficText.textColor = getTrafficColor(item.traffic);

    if (item.traffic_change && item.traffic_change !== "0.00 GB") {
        detailStack.addSpacer(4);
        const changeText = detailStack.addText(formatTrafficChange(item.traffic_change));
        changeText.font = Font.systemFont(9);
        changeText.textColor = getChangeColor(item.traffic_change);
    }
}

async function addTrafficItemLarge(widget, item, index) {
    const stack = widget.addStack();
    stack.centerAlignContent();

    const idxStack = stack.addStack();
    idxStack.width = 25;
    const idx = idxStack.addText(String(index));
    idx.font = Font.boldSystemFont(16);
    idx.textColor = getRankColor(index);
    idx.centerAlignText();

    stack.addSpacer(10);

    const infoStack = stack.addStack();
    infoStack.layoutVertically();

    const nameText = infoStack.addText(item.name);
    nameText.font = Font.mediumSystemFont(16);
    nameText.textColor = dynamicColor("#333333", "#ffffff");
    nameText.lineLimit = 1;

    if (item.subscribe_url) {
        nameText.url = item.subscribe_url;
    }

    const trafficText = infoStack.addText(item.traffic);
    trafficText.font = Font.boldSystemFont(15);
    trafficText.textColor = getTrafficColor(item.traffic);

    stack.addSpacer();

    if (item.traffic_change && item.traffic_change !== "0.00 GB") {
        const changeStack = stack.addStack();
        changeStack.width = 70;
        changeStack.layoutVertically();

        const changeValue = changeStack.addText(formatTrafficChange(item.traffic_change));
        changeValue.font = Font.boldSystemFont(14);
        changeValue.textColor = getChangeColor(item.traffic_change);
        changeValue.rightAlignText();

        const changeLabel = changeStack.addText(getChangeLabel(item.traffic_change));
        changeLabel.font = Font.systemFont(10);
        changeLabel.textColor = getChangeColor(item.traffic_change);
        changeLabel.rightAlignText();
        changeLabel.textOpacity = 0.8;
    }
}

async function addTrafficItemLargeOptimized(widget, item, index, totalItems) {
    const stack = widget.addStack();
    stack.centerAlignContent();

    // 根据订阅数量调整字体大小
    const isManyItems = totalItems > 8;
    const isVeryManyItems = totalItems > 12;

    const idxSize = isVeryManyItems ? 12 : (isManyItems ? 14 : 16);
    const nameSize = isVeryManyItems ? 12 : (isManyItems ? 14 : 16);
    const trafficSize = isVeryManyItems ? 11 : (isManyItems ? 13 : 15);
    const changeSize = isVeryManyItems ? 10 : (isManyItems ? 12 : 14);

    const idxStack = stack.addStack();
    idxStack.width = isVeryManyItems ? 20 : 25;
    const idx = idxStack.addText(String(index));
    idx.font = Font.boldSystemFont(idxSize);
    idx.textColor = getRankColor(index);
    idx.centerAlignText();

    stack.addSpacer(isVeryManyItems ? 6 : 10);

    const infoStack = stack.addStack();
    infoStack.layoutVertically();

    const nameText = infoStack.addText(item.name);
    nameText.font = Font.mediumSystemFont(nameSize);
    nameText.textColor = dynamicColor("#333333", "#ffffff");
    nameText.lineLimit = 1;

    if (item.subscribe_url) {
        nameText.url = item.subscribe_url;
    }

    const trafficText = infoStack.addText(item.traffic);
    trafficText.font = Font.boldSystemFont(trafficSize);
    trafficText.textColor = getTrafficColor(item.traffic);

    stack.addSpacer();

    if (item.traffic_change && item.traffic_change !== "0.00 GB") {
        const changeStack = stack.addStack();
        changeStack.width = isVeryManyItems ? 50 : 70;
        changeStack.layoutVertically();

        const changeValue = changeStack.addText(formatTrafficChange(item.traffic_change));
        changeValue.font = Font.boldSystemFont(changeSize);
        changeValue.textColor = getChangeColor(item.traffic_change);
        changeValue.rightAlignText();

        const changeLabel = changeStack.addText(getChangeLabel(item.traffic_change));
        changeLabel.font = Font.systemFont(Math.max(8, changeSize - 4));
        changeLabel.textColor = getChangeColor(item.traffic_change);
        changeLabel.rightAlignText();
        changeLabel.textOpacity = 0.8;
    }
}

async function addTrafficItemSimple(widget, item, totalItems) {
    // 创建卡片背景
    const cardStack = widget.addStack();
    cardStack.backgroundColor = dynamicColor("#f9fafb", "#1e293b");
    cardStack.cornerRadius = 8;
    cardStack.setPadding(12, 14, 12, 14);
    cardStack.centerAlignContent();

    // 根据订阅数量调整字体大小
    const isManyItems = totalItems > 8;
    const isVeryManyItems = totalItems > 12;

    const nameSize = isVeryManyItems ? 15 : (isManyItems ? 17 : 19);
    const trafficSize = isVeryManyItems ? 14 : (isManyItems ? 16 : 18);

    // 左侧：标题
    const nameText = cardStack.addText(item.name);
    nameText.font = Font.mediumSystemFont(nameSize);
    nameText.textColor = dynamicColor("#1e293b", "#f1f5f9");
    nameText.lineLimit = 1;

    if (item.subscribe_url) {
        nameText.url = item.subscribe_url;
    }

    cardStack.addSpacer();

    // 右侧：剩余流量
    const trafficText = cardStack.addText(item.traffic);
    trafficText.font = Font.boldSystemFont(trafficSize);
    trafficText.textColor = dynamicColor("#3b82f6", "#60a5fa"); // 现代蓝色
}

async function addUpdateStatusIndicatorSmall(widget) {
    // 只在更新失败时显示错误信息，成功时不显示任何提示
    if (lastUpdateError) {
        const statusStack = widget.addStack();
        statusStack.centerAlignContent();

        // 小尺寸失败指示器
        const errorIcon = SFSymbol.named("exclamationmark.triangle.fill");
        const iconImg = statusStack.addImage(errorIcon.image);
        iconImg.tintColor = dynamicColor("#ef4444", "#f87171");
        iconImg.imageSize = new Size(8, 8);

        statusStack.addSpacer(3);
        const statusText = statusStack.addText("更新失败");
        statusText.font = Font.systemFont(9);
        statusText.textColor = dynamicColor("#ef4444", "#f87171");
        statusText.textOpacity = 0.8;

        widget.addSpacer(2);
    }
}

async function addUpdateStatusIndicator(containerStack) {
    // 右上角只显示更新时间，不显示状态图标和文字
    // 保持简洁的界面设计
}

async function addUpdateStatusInfoCompact(widget) {
    // 只在更新失败时显示错误信息，成功时不显示任何提示
    if (lastUpdateError) {
        // 简化的失败状态显示
        const statusStack = widget.addStack();
        statusStack.backgroundColor = dynamicColor("#fef2f2", "#450a0a");
        statusStack.cornerRadius = 6;
        statusStack.setPadding(8, 12, 8, 12);
        statusStack.centerAlignContent();

        const errorIcon = SFSymbol.named("exclamationmark.triangle.fill");
        const iconImg = statusStack.addImage(errorIcon.image);
        iconImg.tintColor = dynamicColor("#ef4444", "#f87171");
        iconImg.imageSize = new Size(12, 12);

        statusStack.addSpacer(6);
        const messageText = statusStack.addText(`自动更新失败`);
        messageText.font = Font.mediumSystemFont(11);
        messageText.textColor = dynamicColor("#991b1b", "#fca5a5");

        widget.addSpacer(6);
    }
}

async function addUpdateStatusInfo(widget) {
    // 只在更新失败时显示错误信息，成功时不显示任何提示
    if (lastUpdateError) {
        // 显示更新失败信息
        const errorCard = widget.addStack();
        errorCard.backgroundColor = dynamicColor("#fef2f2", "#450a0a");
        errorCard.cornerRadius = 8;
        errorCard.setPadding(12, 16, 12, 16);
        errorCard.layoutVertically();

        const headerStack = errorCard.addStack();
        headerStack.centerAlignContent();

        const errorIcon = SFSymbol.named("exclamationmark.triangle.fill");
        const iconImg = headerStack.addImage(errorIcon.image);
        iconImg.tintColor = dynamicColor("#ef4444", "#f87171");
        iconImg.imageSize = new Size(16, 16);

        headerStack.addSpacer(8);
        const titleText = headerStack.addText("自动更新失败");
        titleText.font = Font.semiboldSystemFont(14);
        titleText.textColor = dynamicColor("#991b1b", "#fca5a5");

        errorCard.addSpacer(4);

        const messageText = errorCard.addText(lastUpdateError.message);
        messageText.font = Font.systemFont(12);
        messageText.textColor = dynamicColor("#dc2626", "#f87171");
        messageText.lineLimit = 3;

        if (lastUpdateError.timestamp) {
            errorCard.addSpacer(4);
            const timeText = errorCard.addText(`失败时间: ${formatDate(lastUpdateError.timestamp, "HH:mm")}`);
            timeText.font = Font.systemFont(10);
            timeText.textColor = dynamicColor("#b91c1c", "#ef4444");
            timeText.textOpacity = 0.7;
        }

        widget.addSpacer(8);
    }
}

function getLastSuccessfulUpdateTime() {
    // 返回上次更新成功的时间
    if (lastUpdateTime > 0) {
        return formatDate(new Date(lastUpdateTime), "HH:mm");
    } else {
        return formatDate(new Date(), "HH:mm");
    }
}

async function addTotalFooter(widget, data) {
    // 添加分隔线
    widget.addSpacer(4);
    const separator = widget.addStack();
    separator.backgroundColor = dynamicColor("#e2e8f0", "#334155");
    separator.size = new Size(0, 1);
    widget.addSpacer(6);

    // 创建总计卡片
    const footerCard = widget.addStack();
    footerCard.backgroundColor = dynamicColor("#f3f4f6", "#1e293b");
    footerCard.cornerRadius = 10;
    footerCard.setPadding(10, 20, 10, 20); // 减少上下padding
    footerCard.centerAlignContent();

    const stats = calculateStats(data);

    // 总剩余流量文字 - 分开设置以便调整冒号大小
    const labelStack = footerCard.addStack();
    labelStack.centerAlignContent();

    const labelText = labelStack.addText("总剩余流量");
    labelText.font = Font.semiboldSystemFont(18);
    labelText.textColor = dynamicColor("#475569", "#94a3b8");

    // 冒号 - 更大字体
    const colonText = labelStack.addText("：");
    colonText.font = Font.semiboldSystemFont(20);
    colonText.textColor = dynamicColor("#475569", "#94a3b8");

    // 数字 - 适中字体，绿色
    const valueText = labelStack.addText(stats.remaining);
    valueText.font = Font.boldSystemFont(22);
    valueText.textColor = dynamicColor("#10b981", "#34d399"); // 现代绿色
}

function getRankColor(index) {
    if (index <= 3) return dynamicColor("#fe4f67", "#ff4757");
    if (index <= 6) return dynamicColor("#f5c94c", "#ffa502");
    return dynamicColor("#9195a3", "#a4b0be");
}

function getTrafficColor(traffic) {
    if (!traffic) return dynamicColor("#999999", "#666666");

    const value = parseFloat(traffic);
    if (value <= 10) return dynamicColor("#f44336", "#ef5350");
    if (value <= 50) return dynamicColor("#ff9800", "#ffa726");
    return dynamicColor("#4CAF50", "#66BB6A");
}

function getChangeColor(change) {
    if (!change) return dynamicColor("#999999", "#666666");

    if (change.startsWith("-")) {
        return dynamicColor("#f44336", "#ef5350");
    }
    return dynamicColor("#4CAF50", "#66BB6A");
}

function formatTrafficChange(change) {
    if (!change || change === "0.00 GB") return "";
    return change.replace(" GB", "G");
}

function getChangeLabel(change) {
    if (!change || change === "0.00 GB") return "";
    return change.startsWith("-") ? "消耗" : "增加";
}

function calculateStats(data) {
    let total = 0;
    let remaining = 0;

    data.forEach(item => {
        if (item.traffic) {
            const value = parseFloat(item.traffic) || 0;
            total += value;
            remaining += value;
        }
    });

    return {
        total: total.toFixed(1) + " GB",
        remaining: remaining.toFixed(1) + " GB"
    };
}

async function updateAllTraffic() {
    debugLog('updateAllTraffic() start');

    try {
        // 获取当前订阅数据
        const subscriptions = await fetchTrafficData();
        debugLog(`获取到 ${subscriptions.length} 个订阅`);

        const updates = [];

        // 并行更新所有订阅的流量信息（性能优化）
        const validSubscriptions = subscriptions.filter(sub => sub.subscribe_url);
        debugLog(`并行更新 ${validSubscriptions.length} 个订阅...`);

        // 创建所有更新任务
        const updatePromises = validSubscriptions.map(async (subscription) => {
            try {
                debugLog(`正在更新 ${subscription.name}...`);
                const updatedTraffic = await fetchTrafficInfo(subscription.subscribe_url);
                if (updatedTraffic) {
                    // 计算流量变化
                    const changeValue = calculateTrafficChange(
                        subscription.previous_traffic || subscription.traffic,
                        updatedTraffic
                    );

                    return {
                        netinfo_id: subscription.id,
                        name: subscription.name,
                        traffic: updatedTraffic,
                        traffic_change: changeValue,
                        success: true
                    };
                } else {
                    return {
                        netinfo_id: subscription.id,
                        name: subscription.name,
                        success: false,
                        error: '无法解析流量信息'
                    };
                }
            } catch (error) {
                debugLog(`${subscription.name}: 更新失败 - ${error.message}`);
                return {
                    netinfo_id: subscription.id,
                    name: subscription.name,
                    success: false,
                    error: error.message
                };
            }
        });

        // 等待所有更新任务完成（最多8秒）
        const results = await Promise.all(updatePromises);

        // 处理结果
        results.forEach(result => {
            if (result.success) {
                updates.push({
                    netinfo_id: result.netinfo_id,
                    name: result.name,
                    traffic: result.traffic,
                    traffic_change: result.traffic_change
                });
                debugLog(`${result.name}: 更新成功 -> ${result.traffic} (${result.traffic_change})`);
            } else {
                debugLog(`${result.name}: ${result.error}`);
            }
        });

        // 将更新结果推送到服务器
        if (updates.length > 0) {
            debugLog(`推送 ${updates.length} 个更新结果到服务器...`);
            const syncResult = await pushUpdatesToServer(updates);
            debugLog('同步结果:', syncResult);
            return syncResult;
        } else {
            debugLog('没有需要更新的订阅');
            return { success: true, message: '没有需要更新的订阅', updates: [] };
        }

    } catch (error) {
        debugLog('updateAllTraffic error:', error.message);
        throw error;
    }
}

async function fetchTrafficInfo(subscribeUrl) {
    debugLog('fetchTrafficInfo for:', subscribeUrl);

    try {
        const request = new Request(subscribeUrl);
        request.headers = {
            'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language': 'zh-CN,zh;q=0.9,en;q=0.8',
            'Accept-Encoding': 'gzip, deflate, br',
            'DNT': '1',
            'Connection': 'keep-alive',
            'Upgrade-Insecure-Requests': '1',
        };
        request.timeoutInterval = 8; // 减少超时时间，提高响应速度

        const response = await request.loadString();
        debugLog('📥 获取到响应，长度:', response.length);

        if (DEBUG && response.length > 0) {
            debugLog('📄 原始响应内容前200字符:');
            debugLog(response.substring(0, 200));
        }

        // 解码响应内容
        let decodedText = decodeResponse(response);
        debugLog('🔓 解码后内容前200字符:');
        debugLog(decodedText.substring(0, 200));

        // 解析流量信息
        const traffic = parseTrafficInfo(decodedText);
        debugLog('🎯 最终解析到的流量:', traffic);

        return traffic;

    } catch (error) {
        debugLog('❌ fetchTrafficInfo error:', error.message);
        throw error;
    }
}

function decodeResponse(responseText) {
    // 性能优化：快速检查是否需要解码
    if (!responseText || responseText.length < 50) {
        return responseText; // 太短的文本可能不需要解码
    }

    try {
        // 快速检查是否包含Base64字符（不包含空格、换行等）
        const isLikelyBase64 = /^[A-Za-z0-9+/=]+$/.test(responseText.trim());
        if (isLikelyBase64) {
            const base64Decoded = Data.fromString(responseText);
            const decodedData = base64Decoded.toRawString();
            const urlDecoded = decodeURIComponent(decodedData);
            debugLog('Base64 + URL解码成功');
            return urlDecoded;
        }
    } catch (error) {
        debugLog('解码失败，使用原始文本');
    }

    // 直接返回原始文本，避免不必要的处理
    return responseText;
}

function parseTrafficInfo(text) {
    debugLog('🔍 开始解析流量信息...');

    // 简化的流量匹配模式（性能优化）
    const patterns = [
        { pattern: /流量[:：]\s*(\d+(?:\.\d+)?\s*(?:GB|MB|TB))/i, name: "流量" },
        { pattern: /剩余[:：]\s*(\d+(?:\.\d+)?\s*(?:GB|MB|TB))/i, name: "剩余" },
        { pattern: /traffic[:：]\s*(\d+(?:\.\d+)?\s*(?:GB|MB|TB))/i, name: "traffic" },
        { pattern: /remaining[:：]\s*(\d+(?:\.\d+)?\s*(?:GB|MB|TB))/i, name: "remaining" },
        { pattern: /(\d+(?:\.\d+)?\s*(?:GB|MB|TB))\s*(?:剩余|left|remain)/i, name: "流量+后缀" },
        { pattern: /(\d+(?:\.\d+)?\s*(?:GB|MB|TB))\s*\/\s*\d+/i, name: "流量/总量" },
        { pattern: /(\d+(?:\.\d+)?\s*(?:GB|MB|TB))/i, name: "简单数字" }  // 最简单的模式，放在最后
    ];

    // 只检查前3行，提高性能
    const lines = text.split(/\r?\n/).slice(0, 3);

    debugLog(`📝 分析前 ${lines.length} 行文本`);

    for (let i = 0; i < lines.length; i++) {
        const line = lines[i];
        debugLog(`📄 第${i + 1}行 (${line.length}字符):`, line.substring(0, 50) + (line.length > 50 ? '...' : ''));

        for (let j = 0; j < patterns.length; j++) {
            const { pattern, name } = patterns[j];
            const match = line.match(pattern);
            if (match) {
                const result = match[1] || match[0]; // 使用第一个捕获组或整个匹配
                debugLog(`✅ 模式 "${name}" 匹配成功:`, result);
                return result.trim();
            }
        }
    }

    debugLog('❌ 行匹配失败，尝试整段文本匹配');

    // 如果行匹配失败，只对整段文本尝试最简单的模式
    const simpleMatch = text.match(/(\d+(?:\.\d+)?\s*(?:GB|MB|TB))/i);
    if (simpleMatch) {
        debugLog('✅ 整段文本匹配成功:', simpleMatch[1]);
        return simpleMatch[1].trim();
    }

    debugLog('❌ 未能解析到流量信息');
    if (DEBUG) {
        debugLog('🔍 完整文本内容:');
        debugLog(text);
    }
    return null;
}

function calculateTrafficChange(oldTrafficStr, newTrafficStr) {
    if (!oldTrafficStr || oldTrafficStr === newTrafficStr) {
        return "0.00 GB";
    }

    const oldValue = extractTrafficValue(oldTrafficStr);
    const newValue = extractTrafficValue(newTrafficStr);

    const changeValue = oldValue - newValue;

    // 如果流量增加了（可能是重置），则不显示变化
    if (changeValue < 0) {
        return "0.00 GB";
    }

    return `${changeValue.toFixed(2)} GB`;
}

function extractTrafficValue(trafficStr) {
    if (!trafficStr) return 0.0;

    const match = trafficStr.match(/(\d+(?:\.\d+)?)/);
    if (!match) return 0.0;

    let value = parseFloat(match[1]);

    if (trafficStr.toUpperCase().includes('MB')) {
        value = value / 1024.0;
    } else if (trafficStr.toUpperCase().includes('TB')) {
        value = value * 1024.0;
    } else if (trafficStr.toUpperCase().includes('KB')) {
        value = value / (1024.0 * 1024.0);
    }

    return value;
}

async function pushUpdatesToServer(updates) {
    debugLog('pushUpdatesToServer start, updates count:', updates.length);

    try {
        const url = `${API_BASE_URL}/update_all_traffic/`;
        debugLog('push URL:', url);

        const requestData = {
            updates: updates
        };

        const request = new Request(url);
        request.method = 'POST';
        request.headers = {
            "User-Agent": "Scriptable-Traffic-Monitor-Auto/1.0",
            "Accept": "application/json",
            "Content-Type": "application/json"
        };
        request.body = JSON.stringify(requestData);
        request.timeoutInterval = 8; // 减少超时时间，提高响应速度

        const response = await request.loadJSON();
        debugLog('push response received:', response);

        if (response && response.success) {
            debugLog(`数据推送成功: ${response.message}`);
            return response;
        } else {
            debugLog('数据推送失败:', response);
            throw new Error(response?.error || "数据推送失败");
        }
    } catch (error) {
        debugLog('pushUpdatesToServer error:', error.message);
        throw new Error(`数据推送失败: ${error.message}`);
    }
}

async function fetchTrafficData() {
    const now = Date.now();

    if (cachedData && (now - lastFetchTime) < CACHE_DURATION) {
        debugLog('using cached data');
        return cachedData;
    }

    debugLog('fetching fresh data from:', API_BASE_URL);

    try {
        const url = `${API_BASE_URL}/subscriptions/`;
        debugLog('request URL:', url);

        const request = new Request(url);
        request.headers = {
            "User-Agent": "Scriptable-Traffic-Monitor-Auto/1.0",
            "Accept": "application/json"
        };
        request.timeoutInterval = 8; // 减少超时时间，提高响应速度

        const response = await request.loadJSON();
        debugLog('API response received');

        if (response && response.success && response.data && Array.isArray(response.data)) {
            const validData = response.data.filter(item => item && item.id && item.name);
            cachedData = validData;
            lastFetchTime = now;
            debugLog('fetched data count:', cachedData.length);
            return cachedData;
        } else {
            debugLog('invalid response format:', response);
            throw new Error(response?.error || "数据格式错误或获取失败");
        }
    } catch (error) {
        debugLog('fetch error:', error.message);
        debugLog('error details:', JSON.stringify(error));

        if (cachedData) {
            debugLog('using expired cache due to fetch error');
            return cachedData;
        }

        // 在调试模式下提供测试数据
        if (DEBUG && config.runsInApp) {
            debugLog('DEBUG: using test data for preview');
            return getTestData();
        }

        throw new Error(`API连接失败: ${error.message}\n地址: ${API_BASE_URL}`);
    }
}

function getTestData() {
    return [
        {
            id: 1,
            name: "M78星云",
            traffic: "180.25 GB",
            previous_traffic: "194.93 GB",
            traffic_change: "14.68 GB",
            order: 0
        },
        {
            id: 2,
            name: "智联云",
            traffic: "712.76 GB",
            previous_traffic: "1078.68 GB",
            traffic_change: "0.05 GB",
            order: 1
        },
        {
            id: 3,
            name: "蜂群",
            traffic: "840.72 GB",
            previous_traffic: "860.81 GB",
            traffic_change: "20.09 GB",
            order: 2
        }
    ];
}



function formatDate(date, fmt) {
    const df = new DateFormatter();
    df.dateFormat = fmt;
    return df.string(date);
}

await run();