// Variables used by Scriptable.
// These must be at the very top of the file. Do not edit.
// icon-color: blue; icon-glyph: grin-squint-tears;

const API_URL = "https://s.weibo.com/top/summary?cate=realtimehot";
const WEIBO_ICON_URL = "https://s1.ax1x.com/2022/07/10/jrj056.png";
const MAX_COUNT = 15;
const DEBUG = false;
const SHOW_TAG_LABELS = false;
let weiboIconImage = null;

function debugLog(...args) {
	if (!DEBUG) return;
	try {
		const msg = args.map(a => {
			if (a == null) return String(a);
			if (typeof a === 'string') return a;
			try { return JSON.stringify(a); } catch (_) { return String(a); }
		}).join(' ');
		console.log(`[WB_DEBUG] ${msg}`);
	} catch (_) { }
}

function dynamicColor(lightColor, darkColor) {
	return Color.dynamic(new Color(lightColor), new Color(darkColor));
}

async function run() {
	debugLog('run() start');
	const widget = await renderLarge();
	if (config.runsInApp) {
		await widget.presentLarge();
	}
	Script.setWidget(widget);
	Script.complete();
	debugLog('run() done');
}

async function addHeader(w) {
	w.addSpacer(10);
	const head = w.addStack();
	head.setPadding(0, 12, 0, 12);
	head.centerAlignContent();
	try {
		if (!weiboIconImage) {
			const r = new Request(WEIBO_ICON_URL);
			weiboIconImage = await r.loadImage();
		}
		const img = head.addImage(weiboIconImage);
		img.imageSize = new Size(16, 16);
	} catch (e) {
		const icon = SFSymbol.named('square.grid.2x2');
		const img = head.addImage(icon.image);
		img.tintColor = dynamicColor("#ff4757", "#ff6b81");
		img.imageSize = new Size(16, 16);
	}
	head.addSpacer(6);
	const title = head.addText('微博热搜');
	title.font = Font.boldSystemFont(13);
	title.textColor = dynamicColor("#333333", "#ffffff");
	head.addSpacer();
	const timeStack = head.addStack();
	timeStack.addSpacer(4);
	const updateText = timeStack.addText('更新于 ');
	updateText.font = Font.systemFont(10);
	updateText.textColor = dynamicColor("#666666", "#999999");
	updateText.textOpacity = 0.5;
	const time = timeStack.addText(formatDate(new Date(), "HH:mm"));
	time.font = Font.systemFont(10);
	time.textColor = dynamicColor("#666666", "#999999");
	time.textOpacity = 0.5;
	w.addSpacer(4);
}

async function renderLarge() {
	const widget = new ListWidget();
	widget.backgroundColor = dynamicColor("#f6f6f6", "#1e293b");
	widget.setPadding(8, 8, 8, 8);

	const list = await fetchList();
	const count = Math.min(MAX_COUNT, Array.isArray(list) ? list.length : 0);

	if (!count) {
		const empty = widget.addStack();
		empty.centerAlignContent();
		addText(empty, "暂无数据", 12, { opacity: 0.7, color: dynamicColor("#666666", "#999999") });
		return widget;
	}

	await addHeader(widget);

	const contentStack = widget.addStack();
	contentStack.layoutVertically();
	contentStack.setPadding(0, 12, 0, 12);

	for (let i = 0; i < count; i++) {
		await addItem(contentStack, list[i], i + 1);
		if (i < count - 1) {
			contentStack.addSpacer(6);
		}
	}

	widget.addSpacer(10);

	return widget;
}

async function addItem(parent, item, index) {
	const stack = parent.addStack();
	stack.centerAlignContent();

	const keyword = item.keyword || extractKeyword(item);
	stack.url = buildSearchUrl(keyword);

	const idx = stack.addText(String(index));
	idx.font = Font.boldSystemFont(13);
	idx.textColor = dynamicColor(index <= 3 ? "#fe4f67" : (index <= 6 ? "#f5c94c" : "#9195a3"), index <= 3 ? "#ff4757" : (index <= 6 ? "#ffa502" : "#a4b0be"));
	stack.addSpacer(8);

	const title = item.title || item.word || "";
	const titleText = stack.addText(title);
	titleText.font = Font.systemFont(13);
	titleText.textColor = dynamicColor("#333333", "#ffffff");
	titleText.lineLimit = 1;

	if (item.icon) {
		stack.addSpacer(4);
		const icon = await loadImage(item.icon);
		const iconImg = stack.addImage(icon);
		iconImg.imageSize = new Size(12, 12);
	}

	stack.addSpacer();
	if (item.number) {
		const heatText = stack.addText(String(item.number));
		heatText.font = Font.systemFont(10);
		heatText.textColor = dynamicColor("#999999", "#666666");
		heatText.textOpacity = 0.6;
	}
}

function mapTagToLabel(tag) {
	switch (tag) {
		case "hot": return "热";
		case "new": return "新";
		case "bang": return "爆";
		case "recommend": return "荐";
		case "first": return "首发";
		case "top":
		case "pinned": return "置顶";
		default: return "";
	}
}

function mapTagToColor(tag) {
	switch (tag) {
		case "hot": return "#ff9800";
		case "new": return "#f44336";
		case "bang": return "#f5c94c";
		case "recommend": return "#ff9800";
		case "first": return "#2196f3";
		case "top":
		case "pinned": return "#9e9e9e";
		default: return "#999999";
	}
}

function extractKeyword(item) {
	let keyword = item.word || item.title || "";
	try {
		const href = item.href || "";
		const m = href.match(/q=([^&]+)/);
		if (m) keyword = decodeURIComponent(m[1]);
	} catch (_) { }
	return keyword;
}

function buildSearchUrl(keyword) {
	const container = encodeURIComponent("100103type=1&t=10&q=" + keyword);
	return `https://m.weibo.cn/search?containerid=${container}`;
}

async function fetchList() {
	try {
		debugLog('fetchList start');
		const cookies = await getVisitorCookies();
		debugLog('cookies acquired', cookies);
		const req = new Request(API_URL);
		req.headers = {
			"User-Agent": "Mozilla/5.0 (iPhone; CPU iPhone OS 16_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/16.0 Mobile/15E148 Safari/604.1",
			"Cookie": `SUB=${cookies.SUB}; SUBP=${cookies.SUBP}`,
			"Referer": "https://s.weibo.com/",
		};
		req.timeoutInterval = 10;
		const html = await req.loadString();
		debugLog('fetched html length:', html ? html.length : 0);
		const list = parseSummaryHtml(html);
		debugLog('parsed list length:', list.length, 'first item:', list[0]);
		return list;
	} catch (e) {
		debugLog('fetchList error:', String(e));
	}
	return [];
}

function parseSummaryHtml(html) {
	const list = [];
	if (!html || !html.trim()) {
		debugLog('parseSummaryHtml: empty html');
		return list;
	}
	const re = /<a\s+href="([^"]*?band_rank=(\d+)[^"]*)"[^>]*>([\s\S]*?)<\/a>/g;
	let m; let anchorCount = 0;
	while ((m = re.exec(html)) && list.length < MAX_COUNT) {
		anchorCount++;
		const href = m[1];
		const rank = parseInt(m[2]);
		const inner = m[3];
		const sm = inner.match(/<span>([^<]*?)<em>(\d+)<\/em>/);
		if (!sm) continue;
		const title = decodeHtml(sm[1].trim());
		const heat = parseInt(sm[2]);
		const keywordMatch = href.match(/q=([^&]+)/);
		const keyword = keywordMatch ? decodeURIComponent(keywordMatch[1]) : title;
		const tag = findTag(html, m.index, m[0], inner) || '';

		list.push({ title, number: heat, href, keyword, tag });
		if (DEBUG && list.length <= 10) {
			debugLog(`[tag] #${rank} '${title}' => '${tag}'`);
		}
	}
	debugLog('anchors scanned:', anchorCount, 'items collected:', list.length);

	if (!list.length) {
		const re2 = /<td class="td-01">(\d+)<\/td>[\s\S]*?<td class="td-02">[\s\S]*?<a[^>]*href="([^"]+)"[^>]*>([^<]+)<\/a>[\s\S]*?<td class="td-03">(\d+)<\/td>/g;
		let mm; let tableCount = 0;
		while ((mm = re2.exec(html)) && list.length < MAX_COUNT) {
			tableCount++;
			const rank = parseInt(mm[1]);
			const href = mm[2];
			const title = decodeHtml(mm[3].trim());
			const heat = parseInt(mm[4]);
			const keywordMatch = href.match(/q=([^&]+)/);
			const keyword = keywordMatch ? decodeURIComponent(keywordMatch[1]) : title;
			list.push({ title, number: heat, href, keyword, tag: '' });
		}
		debugLog('table rows scanned:', tableCount, 'items collected:', list.length);
	}

	debugLog('sample items:', list.slice(0, 3));
	return list;
}

function findTag(fullHtml, anchorIndex, anchorHtml, innerHtml) {
	try {
		const liStart = fullHtml.lastIndexOf('<li', anchorIndex);
		const liEnd = fullHtml.indexOf('</li>', anchorIndex);
		const liBlock = (liStart !== -1 && liEnd !== -1) ? fullHtml.slice(liStart, liEnd + 5) : '';
		const trStart = fullHtml.lastIndexOf('<tr', anchorIndex);
		const trEnd = fullHtml.indexOf('</tr>', anchorIndex);
		const trBlock = (trStart !== -1 && trEnd !== -1) ? fullHtml.slice(trStart, trEnd + 5) : '';
		const clsRe = /class=["'][^"']*(icon[^"']*)["']/g;
		const tagRe = /(icon[-_]?)(hot|new|bang|recommend|top|pinned|fire|first)/i;
		const zhRe = />\s*(热|新|爆|荐|置顶|首发)\s*</g;

		let tag = '';
		const scanBlocks = [liBlock, trBlock];
		for (const blk of scanBlocks) {
			if (!blk) continue;
			let cm;
			while ((cm = clsRe.exec(blk))) {
				const cls = cm[1];
				const tm = cls.match(tagRe);
				if (tm) { tag = tm[2].toLowerCase(); break; }
			}
			if (tag) return normalizeTag(tag);
			let zm;
			while ((zm = zhRe.exec(blk))) {
				const zh = zm[1];
				const nt = zhLabelToTag(zh);
				if (nt) { tag = nt; break; }
			}
			if (tag) return normalizeTag(tag);
		}

		const before = Math.max(0, anchorIndex - 1200);
		const after = Math.min(fullHtml.length, anchorIndex + anchorHtml.length + 1200);
		const neighborhood = fullHtml.slice(before, after);
		let nm;
		while ((nm = clsRe.exec(neighborhood))) {
			const cls = nm[1];
			const tm = cls.match(tagRe);
			if (tm) { tag = tm[2].toLowerCase(); break; }
		}
		if (tag) return normalizeTag(tag);
		let zn;
		while ((zn = zhRe.exec(neighborhood))) {
			const zh = zn[1];
			const nt = zhLabelToTag(zh);
			if (nt) { tag = nt; break; }
		}
		if (tag) return normalizeTag(tag);

		const fromAnchor = (anchorHtml.match(tagRe) || innerHtml.match(tagRe));
		if (fromAnchor) return normalizeTag(fromAnchor[2].toLowerCase());
		const zhAnchor = (anchorHtml.match(zhRe) || innerHtml.match(zhRe));
		if (zhAnchor) {
			const zh = zhAnchor[0].replace(/>|</g, '').trim();
			const nt = zhLabelToTag(zh);
			if (nt) return normalizeTag(nt);
		}

		return '';
	} catch (_) {
		return '';
	}
}

function normalizeTag(tag) {
	switch (tag) {
		case 'hot':
		case 'fire': return 'hot';
		case 'new': return 'new';
		case 'bang': return 'bang';
		case 'recommend': return 'recommend';
		case 'first': return 'first';
		case 'top':
		case 'pinned': return 'top';
		default: return '';
	}
}

function zhLabelToTag(zh) {
	switch (zh) {
		case '热': return 'hot';
		case '新': return 'new';
		case '爆': return 'bang';
		case '荐': return 'recommend';
		case '置顶': return 'top';
		case '首发': return 'first';
		default: return '';
	}
}

async function getVisitorCookies() {
	const cachedExists = Keychain.contains("WB_VISITOR");
	const cached = cachedExists ? JSON.parse(Keychain.get("WB_VISITOR")) : null;
	if (cached && cached.expire > Date.now()) {
		debugLog('use cached cookies');
		return cached.cookies;
	}

	const ua = "Mozilla/5.0 (iPhone; CPU iPhone OS 16_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/16.0 Mobile/15E148 Safari/604.1";

	const reqTid = new Request("https://passport.weibo.com/visitor/genvisitor?cb=gen_callback&from=weibo&_rand=" + Math.random());
	reqTid.headers = { "User-Agent": ua };
	reqTid.timeoutInterval = 10;
	const tidText = await reqTid.loadString();
	debugLog('genvisitor length:', tidText ? tidText.length : 0);
	const tidMatch = tidText.match(/\{"retcode":20000000,[\s\S]*?"tid":"([^"]+)"/);
	const tid = tidMatch ? tidMatch[1] : "";
	debugLog('tid:', tid);

	const reqCookie = new Request(`https://passport.weibo.com/visitor/visitor?a=incarnate&t=${tid}&w=3&c=100&cb=cross_domain&from=weibo`);
	reqCookie.headers = { "User-Agent": ua };
	reqCookie.timeoutInterval = 10;
	const cookieText = await reqCookie.loadString();
	debugLog('visitor cookies length:', cookieText ? cookieText.length : 0);
	const subMatch = cookieText.match(/"sub":"([^"]+)"/);
	const subpMatch = cookieText.match(/"subp":"([^"]+)"/);
	const cookies = { SUB: subMatch ? subMatch[1] : "", SUBP: subpMatch ? subpMatch[1] : "" };
	debugLog('cookies parsed:', cookies);

	const expire = Date.now() + 2 * 60 * 60 * 1000;
	Keychain.set("WB_VISITOR", JSON.stringify({ cookies, expire }));
	debugLog('cookies cached for 2h');

	return cookies;
}

async function loadImage(url) {
	try {
		const r = new Request(url);
		r.timeoutInterval = 10;
		return await r.loadImage();
	} catch (e) {
		debugLog('loadImage error:', String(e));
		return SFSymbol.named("photo").image;
	}
}

function addText(stack, text, size, opt = {}) {
	const t = stack.addText(String(text));
	const font = opt.font === "bold" ? Font.boldSystemFont(size) :
		opt.font === "light" ? Font.lightSystemFont(size) : Font.systemFont(size);
	t.font = font;
	if (opt.color) {
		t.textColor = (opt.color instanceof Color) ? opt.color : new Color(opt.color);
	}
	if (opt.opacity != null) t.textOpacity = opt.opacity;
	if (opt.lineLimit != null) t.lineLimit = opt.lineLimit;
	if (opt.align === "right") t.rightAlignText();
	if (opt.align === "center") t.centerAlignText();
	return t;
}

function decodeHtml(s) {
	return s.replace(/&amp;/g, "&").replace(/&lt;/g, "<").replace(/&gt;/g, ">").replace(/&quot;/g, '"').replace(/&#(\d+);/g, (_, n) => String.fromCharCode(parseInt(n)));
}

function formatDate(date, fmt) {
	const df = new DateFormatter();
	df.dateFormat = fmt;
	return df.string(date);
}

await run();
