-- ============================================================================
-- TailwindCSS 自动配置文件创建器 (tailwind-auto-config.lua)
-- 智能检测并自动创建 TailwindCSS 配置文件，提升开发体验
-- ============================================================================
--
-- 功能概述：
-- 1. 只在打开 HTML 文件时触发，避免不必要的配置文件生成
-- 2. 自动查找项目根目录（通过 package.json、.git 等标识文件）
-- 3. 只在项目根目录创建 tailwind.config.js，避免子目录重复生成
-- 4. 创建配置文件后自动重启 TailwindCSS LSP 服务，确保补全功能立即可用
-- 5. TailwindCSS 补全功能通过检测项目根目录的配置文件来工作
--
-- 使用场景：
-- - HTML 项目开发：打开 HTML 文件时自动配置 TailwindCSS 环境
-- - 项目根目录管理：确保配置文件位于正确位置，便于项目管理
-- - 避免配置冗余：防止在每个子目录都生成配置文件
-- ============================================================================

local M = {}

-- 支持的文件类型（只在 HTML 文件时触发）
local supported_filetypes = {
	"html", "htmldjango"
}

-- 最简洁的 TailwindCSS 配置文件内容
local minimal_config = [[
/** @type {import('tailwindcss').Config} */
module.exports = {
  content: [
    "./*.{html,js,ts,jsx,tsx,vue,svelte,astro,php,twig,md}",
    "./**/*.{html,js,ts,jsx,tsx,vue,svelte,astro,php,twig,md}"
  ],
  theme: {
    extend: {},
  },
  plugins: [],
}
]]

-- 检查项目根目录是否已有 TailwindCSS 配置文件
local function has_tailwind_config(dir)
	local config_files = {
		"tailwind.config.js",
		"tailwind.config.ts",
		"tailwind.config.cjs",
		"tailwind.config.mjs"
	}

	for _, config_file in ipairs(config_files) do
		local config_path = dir .. "/" .. config_file
		if vim.fn.filereadable(config_path) == 1 then
			return true
		end
	end
	return false
end

-- 查找项目根目录，与 LSP 配置保持一致
local function find_project_root(current_dir)
	-- 首先检查是否已有 TailwindCSS 配置文件
	local config_files = {
		"tailwind.config.js",
		"tailwind.config.ts",
		"tailwind.config.cjs",
		"tailwind.config.mjs"
	}
	
	local dir = current_dir
	while dir ~= "/" do
		for _, config_file in ipairs(config_files) do
			local config_path = dir .. "/" .. config_file
			if vim.fn.filereadable(config_path) == 1 then
				return dir
			end
		end
		dir = vim.fn.fnamemodify(dir, ":h")
	end
	
	-- 如果没有配置文件，查找项目根目录标识
	local root_markers = {
		"package.json",
		".git",
		"package-lock.json",
		"yarn.lock",
		"pnpm-lock.yaml",
		".gitignore",
		"tsconfig.json",
		"jsconfig.json"
	}
	
	dir = current_dir
	while dir ~= "/" do
		for _, marker in ipairs(root_markers) do
			local marker_path = dir .. "/" .. marker
			if vim.fn.filereadable(marker_path) == 1 or vim.fn.isdirectory(marker_path) == 1 then
				return dir
			end
		end
		dir = vim.fn.fnamemodify(dir, ":h")
	end
	
	-- 智能选择创建位置：优先选择有意义的目录层级
	-- 如果当前目录看起来像项目目录（包含多个文件），就在当前目录创建
	-- 否则尝试在父目录中找到更合适的位置
	dir = current_dir
	local max_attempts = 3  -- 最多向上查找3级目录
	local attempts = 0
	
	while dir ~= "/" and attempts < max_attempts do
		-- 检查当前目录是否适合作为项目根目录
		local files = vim.fn.globpath(dir, "*", false, true)
		local html_files = vim.fn.globpath(dir, "*.html", false, true)
		local css_files = vim.fn.globpath(dir, "*.css", false, true)
		local js_files = vim.fn.globpath(dir, "*.js", false, true)
		
		-- 如果目录包含多个文件，或者包含web相关文件，认为这是合适的位置
		if #files >= 2 or #html_files >= 1 or #css_files >= 1 or #js_files >= 1 then
			return dir
		end
		
		dir = vim.fn.fnamemodify(dir, ":h")
		attempts = attempts + 1
	end
	
	-- 如果没有找到合适的位置，返回当前HTML文件所在目录
	return current_dir
end

-- 重启 TailwindCSS LSP 服务
local function restart_tailwindcss_lsp()
	-- 获取当前缓冲区的客户端
	local clients = vim.lsp.get_clients({ name = "tailwindcss" })
	
	for _, client in ipairs(clients) do
		-- 停止客户端
		client.stop()
	end
	
	-- 延迟重启 LSP，确保配置文件已被识别
	vim.defer_fn(function()
		-- 重新启动 LSP
		vim.cmd("LspRestart tailwindcss")
		vim.notify("[TailwindCSS] LSP 服务已重启，补全功能现在可用", vim.log.levels.INFO)
	end, 500)
end

-- 创建最简洁的 TailwindCSS 配置文件（只在项目根目录创建）
local function create_tailwind_config(root_dir)
	local config_path = root_dir .. "/tailwind.config.js"
	
	-- 检查文件是否已存在
	if vim.fn.filereadable(config_path) == 1 then
		return false
	end
	
	-- 创建配置文件
	local file = io.open(config_path, "w")
	if file then
		file:write(minimal_config)
		file:close()
		vim.notify("[TailwindCSS] 已在项目根目录创建 tailwind.config.js: " .. root_dir, vim.log.levels.INFO)
		
		-- 重启 TailwindCSS LSP 服务以立即启用补全
		restart_tailwindcss_lsp()
		
		return true
	else
		vim.notify("[TailwindCSS] 创建配置文件失败: " .. config_path, vim.log.levels.ERROR)
		return false
	end
end

-- 检查文件类型是否支持
local function is_supported_filetype(filetype)
	for _, ft in ipairs(supported_filetypes) do
		if ft == filetype then
			return true
		end
	end
	return false
end

-- 主要的自动配置函数 - 现在直接在当前目录创建
function M.auto_create_config()
	local current_file = vim.fn.expand("%:p")
	local current_dir = vim.fn.fnamemodify(current_file, ":h")
	local filetype = vim.bo.filetype

	-- 检查是否为支持的文件类型（只在 HTML 文件时触发）
	if not is_supported_filetype(filetype) then
		return
	end

	-- 直接在当前目录检查是否已有配置文件
	if has_tailwind_config(current_dir) then
		return
	end

	-- 直接在当前目录创建配置文件
	create_tailwind_config(current_dir)
end

-- 设置自动命令
function M.setup()
	-- 创建自动命令组
	local augroup = vim.api.nvim_create_augroup("TailwindAutoConfig", { clear = true })

	-- 当打开 HTML 文件时，自动在项目根目录创建配置文件
	vim.api.nvim_create_autocmd({ "BufEnter", "BufNewFile" }, {
		group = augroup,
		pattern = {"*.html", "*.htm"},
		callback = function()
			-- 延迟执行，确保文件类型已正确设置
			vim.defer_fn(function()
				M.auto_create_config()
			end, 100)
		end,
	})
end

-- 手动重启 TailwindCSS LSP（用于调试）
function M.restart_lsp()
	restart_tailwindcss_lsp()
end

-- 手动创建 TailwindCSS 配置文件
function M.create_config_manually()
	local current_file = vim.fn.expand("%:p")
	local current_dir = vim.fn.fnamemodify(current_file, ":h")
	local root_dir = find_project_root(current_dir)
	
	-- 检查项目根目录是否已有配置文件
	if has_tailwind_config(root_dir) then
		vim.notify("[TailwindCSS] 配置文件已存在于: " .. root_dir, vim.log.levels.WARN)
		return false
	end
	
	-- 创建配置文件
	local success = create_tailwind_config(root_dir)
	if success then
		vim.notify("[TailwindCSS] 已手动创建配置文件于: " .. root_dir, vim.log.levels.INFO)
	end
	return success
end

-- 手动诊断 TailwindCSS LSP 状态
function M.diagnose()
	print("=== TailwindCSS LSP 诊断 ===")
	
	-- 检查 LSP 服务器是否可执行
	local server_path = vim.fn.stdpath("data") .. "/mason/bin/tailwindcss-language-server"
	if vim.fn.executable(server_path) == 1 then
		print("✅ TailwindCSS LSP 服务器已安装")
	else
		print("❌ TailwindCSS LSP 服务器未找到:", server_path)
		return
	end
	
	-- 检查当前活动的 LSP 客户端
	local clients = vim.lsp.get_clients()
	local tailwind_client = nil
	for _, client in ipairs(clients) do
		if client.name == "tailwindcss" then
			tailwind_client = client
			break
		end
	end
	
	if tailwind_client then
		print("✅ TailwindCSS LSP 客户端活跃")
		print("   - 根目录:", tailwind_client.config.root_dir)
		print("   - 状态:", tailwind_client.is_stopped() and "已停止" or "运行中")
	else
		print("❌ TailwindCSS LSP 客户端未活跃")
		print("   - 提示: 尝试打开 HTML 文件或运行 :lua require('configs.tailwind-auto-config').restart_lsp()")
	end
	
	-- 检查当前目录和项目根目录
	local current_file = vim.fn.expand("%:p")
	local current_dir = vim.fn.fnamemodify(current_file, ":h")
	local root_dir = find_project_root(current_dir)
	
	print("当前文件:", current_file)
	print("当前目录:", current_dir)
	print("项目根目录:", root_dir)
	print("是否有配置文件:", has_tailwind_config(root_dir) and "是" or "否")
	
	-- 检查文件类型
	local filetype = vim.bo.filetype
	print("当前文件类型:", filetype)
	print("是否支持的文件类型:", is_supported_filetype(filetype) and "是" or "否")
	
	-- 如果没有配置文件，提示手动创建
	if not has_tailwind_config(root_dir) then
		print("💡 手动创建配置文件: :lua require('configs.tailwind-auto-config').create_config_manually()")
	end
end

return M
