-- ============================================================================
-- LSP (Language Server Protocol) 配置文件 (lspconfig.lua)
-- 配置语言服务器，提供代码补全、语法检查、跳转定义等功能
-- ============================================================================

-- ============================================================================
-- 1. 基础配置初始化
-- ============================================================================
-- 加载 NvChad 的默认 LSP 配置
require("nvchad.configs.lspconfig").defaults()

-- ============================================================================
-- 1.1 LSP 诊断信息中文化配置
-- ============================================================================

-- 常见LSP诊断信息的中文翻译映射
local chinese_diagnostics = {
	-- Python/Pyright 相关
	["Expression value is unused"] = "表达式的值未被使用",
	["Argument of type .* is incompatible"] = "参数类型不兼容",
	["Cannot access member"] = "无法访问成员",
	["is not defined"] = "未定义",
	["is not accessed"] = "未被访问",
	["Missing positional argument"] = "缺少位置参数",
	["Too many positional arguments"] = "位置参数过多",
	["Unexpected keyword argument"] = "意外的关键字参数",
	["Variable .* is possibly unbound"] = "变量可能未绑定",
	["Import .* could not be resolved"] = "导入无法解析",
	["Function with declared type of .* must return value"] = "声明类型的函数必须返回值",
	
	-- JavaScript/TypeScript 相关
	["Property .* does not exist"] = "属性不存在",
	["Object is possibly 'null'"] = "对象可能为'null'",
	["Object is possibly 'undefined'"] = "对象可能为'undefined'",
	["Cannot find name"] = "找不到名称",
	["Expected .* arguments, but got .*"] = "期望参数数量不匹配",
	["Type .* is not assignable to type"] = "类型不能赋值给另一类型",
	["Parameter .* implicitly has an 'any' type"] = "参数隐式具有'any'类型",
	
	-- 通用LSP错误
	["Syntax error"] = "语法错误",
	["Unexpected token"] = "意外的标记",
	["Missing semicolon"] = "缺少分号",
	["Unreachable code"] = "无法到达的代码",
	["Unused variable"] = "未使用的变量",
	["Unused parameter"] = "未使用的参数",
	["Unused import"] = "未使用的导入",
}

-- 翻译函数：将英文诊断信息翻译为中文
local function translate_diagnostic(message)
	-- 直接匹配
	if chinese_diagnostics[message] then
		return chinese_diagnostics[message]
	end
	
	-- 模糊匹配（使用模式匹配）
	for pattern, translation in pairs(chinese_diagnostics) do
		if message:match(pattern) then
			return translation
		end
	end
	
	-- 特殊处理一些常见模式
	if message:match("is not defined") then
		local var_name = message:match("(.+) is not defined")
		if var_name then
			return string.format("'%s' 未定义", var_name:gsub("\"", ""):gsub("'", ""))
		end
	end
	
	if message:match("is not accessed") then
		local var_name = message:match("\"(.+)\" is not accessed")
		if var_name then
			return string.format("参数 '%s' 未被使用", var_name)
		end
	end
	
	if message:match("Arguments missing") then
		return "缺少参数"
	end
	
	-- 如果没有找到翻译，返回原文（保持英文）
	return message
end

-- 自定义 LSP 诊断显示处理器
vim.diagnostic.config({
	virtual_text = {
		-- 格式化虚拟文本，添加中文翻译
		format = function(diagnostic)
			local message = diagnostic.message
			local chinese_message = translate_diagnostic(message)
			
			-- 如果有中文翻译且不同于原文，显示中文
			if chinese_message ~= message then
				return string.format("%s", chinese_message)
			else
				-- 否则显示原文
				return message
			end
		end,
	},
	-- 浮动窗口中也使用中文翻译
	float = {
		format = function(diagnostic)
			local message = diagnostic.message
			local chinese_message = translate_diagnostic(message)
			
			if chinese_message ~= message then
				return string.format("%s\n原文: %s", chinese_message, message)
			else
				return message
			end
		end,
	},
})

-- ============================================================================
-- 2. 语言服务器配置
-- ============================================================================

-- 2.1 定义要启用的语言服务器列表
local servers = {
	-- "cssls", -- CSS 语言服务器单独配置，不在这里重复
	-- "lua_ls", -- Lua 语言服务器单独配置，不在这里重复
	"ts_ls", -- TypeScript/JavaScript 语言服务器，已安装 typescript-language-server
	"pyright", -- Python 语言服务器，启用自动签名帮助
	"gopls", -- Go 语言服务器，支持重命名
	-- html 和 emmet_ls 语言服务器暂时禁用，避免启动问题
	-- HTML 使用自定义重命名功能
	-- tailwindcss 使用下面的自定义配置，不在这里启用
}

-- 2.2 启用指定的语言服务器，并为所有服务器设置自定义on_attach
for _, server in ipairs(servers) do
	vim.lsp.config[server] = {
		cmd = vim.lsp.config[server] and vim.lsp.config[server].cmd or nil,
		filetypes = vim.lsp.config[server] and vim.lsp.config[server].filetypes or nil,
		root_markers = vim.lsp.config[server] and vim.lsp.config[server].root_markers or nil,
		capabilities = require("nvchad.configs.lspconfig").capabilities,
		settings = {},
	}
	vim.lsp.enable(server, {
		bufnr = 0,
	})
end

-- ============================================================================
-- 3. 自定义 LSP 服务器配置
-- ============================================================================

-- 3.1 HTML 语言服务器配置 - 使用Mason安装的vscode-html-language-server
vim.lsp.config.html = {
	cmd = { vim.fn.stdpath("data") .. "/mason/bin/vscode-html-language-server", "--stdio" },
	filetypes = { "html", "htm" },
	root_markers = { ".git", "package.json" },
	capabilities = require("nvchad.configs.lspconfig").capabilities,
	settings = {
		html = {
			validate = true,
			hover = {
				documentation = true,
				references = true
			},
			completion = {
				attributeDefaultValue = "doublequotes"
			}
		}
	},
	init_options = {
		configurationSection = { "html", "css", "javascript" },
		embeddedLanguages = {
			css = true,
			javascript = true
		},
		provideFormatter = true
	}
}
vim.lsp.enable("html")

-- 3.2 Emmet 语言服务器配置已禁用
-- 由于安装问题，Emmet语言服务器暂时禁用

-- 3.3 禁用NvChad默认的LSP重命名映射
-- 创建一个自定义的LSP附加处理函数
local original_on_attach = require("nvchad.configs.lspconfig").on_attach
local function custom_on_attach(client, bufnr)
	-- 先调用原始的on_attach函数
	if original_on_attach then
		original_on_attach(client, bufnr)
	end
	
	-- 启用LSP签名帮助功能（仅手动触发，避免阻挡输入）
	if client.server_capabilities.signatureHelpProvider then
		-- 设置签名帮助的快捷键（手动触发）
		vim.keymap.set("n", "<leader>sh", vim.lsp.buf.signature_help, {
			buffer = bufnr,
			desc = "📝 LSP签名帮助"
		})
		
		-- 在插入模式下触发签名帮助
		vim.keymap.set("i", "<C-s>", vim.lsp.buf.signature_help, {
			buffer = bufnr,
			desc = "📝 LSP签名帮助"
		})
		
		-- 在插入模式下使用Ctrl+K也可以触发签名帮助（更方便的快捷键）
		vim.keymap.set("i", "<C-k>", vim.lsp.buf.signature_help, {
			buffer = bufnr,
			desc = "📝 LSP签名帮助"
		})
		
		-- 增强的ESC键，确保能关闭任何LSP弹窗
		vim.keymap.set("i", "<Esc>", function()
			-- 关闭可能的LSP弹窗（包括签名帮助、悬停信息等）
			vim.lsp.handlers["textDocument/signatureHelp"](nil, nil, {})
			pcall(vim.lsp.util.close_preview_window)
			
			-- 执行正常的ESC功能
			return "<Esc>"
		end, { buffer = bufnr, expr = true, desc = "ESC关闭LSP弹窗并退出插入模式" })
	end
	
	-- 获取文件类型
	local filetype = vim.api.nvim_buf_get_option(bufnr, "filetype")
	
	-- 对于HTML和CSS文件，删除LSP的重命名映射
	if vim.tbl_contains({ "html", "htm", "css", "scss", "sass", "less" }, filetype) then
		-- 延迟删除以确保在所有映射设置完成后执行
		vim.schedule(function()
			pcall(vim.keymap.del, "n", "<leader>ra", { buffer = bufnr })
			
			-- 设置自定义重命名功能
			vim.keymap.set("n", "<leader>ra", function()
				if _G.html_rename then
					_G.html_rename()
				else
					vim.notify("❌ HTML重命名功能未加载", vim.log.levels.ERROR)
				end
			end, { 
				buffer = bufnr, 
				desc = "🏷️ " .. string.upper(filetype) .. "重命名 (搜索替换)",
				silent = true,
				noremap = true
			})
		end)
	end
	
	-- 确保不覆盖全局的 <leader>D 映射
	vim.schedule(function()
		-- 删除可能被LSP设置的 <leader>D 映射，保持全局映射生效
		pcall(vim.keymap.del, "n", "<leader>D", { buffer = bufnr })
	end)
end

-- 3.4 Lua 语言服务器配置 - 为 Neovim 配置提供最佳支持
-- 暂时禁用，需要安装 lua-language-server
-- 安装方法: 在Neovim中运行 :Mason 然后搜索安装 lua-language-server
-- require("lspconfig").lua_ls.setup({
-- 	on_attach = custom_on_attach,
-- 	settings = {
-- 		Lua = {
-- 			runtime = {
-- 				version = "LuaJIT",
-- 			},
-- 			diagnostics = {
-- 				globals = { "vim" }, -- 识别 vim 全局变量
-- 			},
-- 			workspace = {
-- 				library = vim.api.nvim_get_runtime_file("", true),
-- 				checkThirdParty = false, -- 禁用第三方检查
-- 			},
-- 			telemetry = {
-- 				enable = false, -- 禁用遥测
-- 			},
-- 		},
-- 	},
-- })

-- 3.4 CSS 语言服务器配置 - 使用Mason安装的vscode-css-language-server
vim.lsp.config.cssls = {
	cmd = { vim.fn.stdpath("data") .. "/mason/bin/vscode-css-language-server", "--stdio" },
	filetypes = { "css", "scss", "less" },
	root_markers = { ".git", "package.json" },
	capabilities = require("nvchad.configs.lspconfig").capabilities,
	settings = {
		css = {
			validate = true,
			lint = {
				unknownAtRules = "ignore",
			},
		},
		scss = {
			validate = true,
			lint = {
				unknownAtRules = "ignore",
			},
		},
		less = {
			validate = true,
			lint = {
				unknownAtRules = "ignore",
			},
		},
	},
}
vim.lsp.enable("cssls")

-- 3.5 Python 语言服务器配置 - 使用Mason安装的pyright，支持Django
vim.lsp.config.pyright = {
	cmd = { vim.fn.stdpath("data") .. "/mason/bin/pyright-langserver", "--stdio" },
	filetypes = { "python" },
	root_markers = {
		'manage.py',
		'settings.py', 
		'wsgi.py',
		'asgi.py',
		'pyproject.toml',
		'setup.py',
		'setup.cfg',
		'requirements.txt',
		'Pipfile',
		'pyrightconfig.json',
		'.git'
	},
	capabilities = require("nvchad.configs.lspconfig").capabilities,
	settings = {
		python = {
			analysis = {
				-- 基础分析设置
				autoSearchPaths = true,
				useLibraryCodeForTypes = true,
				diagnosticMode = "workspace",
				-- 类型检查严格性
				typeCheckingMode = "basic", -- 可选: "off", "basic", "strict"
				-- 自动导入设置
				autoImportCompletions = true,
				-- 包含所有Python文件
				include = { "**/*.py" },
				-- 排除一些不必要的目录
				exclude = {
					"**/node_modules",
					"**/__pycache__",
					".venv",
					"venv",
					"env"
				},
				-- Django支持设置
				extraPaths = {
					-- 自动检测Django项目路径
				},
				-- 存根路径
				stubPath = vim.fn.stdpath("data") .. "/mason/packages/pyright/node_modules/pyright/dist/typeshed-fallback/stubs"
			},
			-- Python解释器路径（可根据项目自动检测虚拟环境）
			pythonPath = vim.fn.exepath("python3") or vim.fn.exepath("python"),
			-- Linting设置
			linting = {
				enabled = true,
			}
		},
		-- Pyright特定设置
		pyright = {
			-- 禁用一些不必要的诊断
			disableOrganizeImports = false,
			-- 使用库代码进行类型检查
			useLibraryCodeForTypes = true
		}
	}
}
vim.lsp.enable("pyright")

-- 3.6 TypeScript/JavaScript 语言服务器配置 - 使用Mason安装的typescript-language-server
vim.lsp.config.ts_ls = {
	cmd = { vim.fn.stdpath("data") .. "/mason/bin/typescript-language-server", "--stdio" },
	filetypes = { "javascript", "javascriptreact", "typescript", "typescriptreact" },
	root_markers = { "package.json", "tsconfig.json", "jsconfig.json", ".git" },
	capabilities = require("nvchad.configs.lspconfig").capabilities,
	settings = {
		typescript = {
			inlayHints = {
				includeInlayParameterNameHints = "all",
				includeInlayParameterNameHintsWhenArgumentMatchesName = false,
				includeInlayFunctionParameterTypeHints = true,
				includeInlayVariableTypeHints = true,
				includeInlayPropertyDeclarationTypeHints = true,
				includeInlayFunctionLikeReturnTypeHints = true,
				includeInlayEnumMemberValueHints = true,
			}
		},
		javascript = {
			inlayHints = {
				includeInlayParameterNameHints = "all",
				includeInlayParameterNameHintsWhenArgumentMatchesName = false,
				includeInlayFunctionParameterTypeHints = true,
				includeInlayVariableTypeHints = true,
				includeInlayPropertyDeclarationTypeHints = true,
				includeInlayFunctionLikeReturnTypeHints = true,
				includeInlayEnumMemberValueHints = true,
			}
		}
	},
	init_options = {
		preferences = {
			-- 启用签名帮助
			provideSignatureHelp = true,
			-- 其他偏好设置
			disableSuggestions = false,
		}
	}
}
vim.lsp.enable("ts_ls")

-- 3.7 TailwindCSS 语言服务器配置 - 使用Mason安装的tailwindcss-language-server
vim.lsp.config.tailwindcss = {
	cmd = { vim.fn.stdpath("data") .. "/mason/bin/tailwindcss-language-server", "--stdio" },
	-- 支持的文件类型 - 扩展支持更多文件类型
	filetypes = {
		"html", "htm", "htmldjango", "css", "scss", "sass", "postcss", "less",
		"vue", "svelte", "astro", "php", "twig", "markdown", "javascript", "typescript",
		"javascriptreact", "typescriptreact", "json", "yaml"
	},
	root_markers = {
		'tailwind.config.js',
		'tailwind.config.ts',
		'tailwind.config.cjs',
		'tailwind.config.mjs',
		'package.json',
		'.git',
		'package-lock.json',
		'yarn.lock',
		'pnpm-lock.yaml',
		'.gitignore',
		'tsconfig.json',
		'jsconfig.json'
	},
	capabilities = require("nvchad.configs.lspconfig").capabilities,
	-- LSP 服务器设置
	settings = {
		tailwindCSS = {
			-- 类属性识别
			classAttributes = {
				"class", "className", "classList", "ngClass",
				"class:list", "[ngClass]", ":class"
			},
			-- 验证设置
			validate = true,
			-- 错误检查配置
			lint = {
				cssConflict = "warning",
				invalidApply = "error",
				invalidConfigPath = "ignore", -- 忽略配置文件路径错误
				invalidScreen = "error",
				invalidTailwindDirective = "error",
				invalidVariant = "error",
				recommendedVariantOrder = "warning"
			},
			-- 基础设置
			rootFontSize = 16,
			-- 实验性功能
			experimental = {
				classRegex = {
					"class\\s*=\\s*['\"]([^'\"]*)['\"]" -- 匹配 class="..." 模式
				}
			},
			-- 内置默认配置，无需外部配置文件
			includeLanguages = {
				html = "html",
				css = "css",
				javascript = "javascript",
				typescript = "typescript",
				javascriptreact = "javascript",
				typescriptreact = "typescript",
				vue = "html",
				svelte = "html",
				php = "html",
				markdown = "html"
			},
			emmetCompletions = true
		}
	}
}
vim.lsp.enable("tailwindcss")

-- ============================================================================
-- 4. LSP 事件处理和自动命令
-- ============================================================================

-- 4.1 设置 LSP 附加处理程序
vim.api.nvim_create_autocmd("LspAttach", {
	group = vim.api.nvim_create_augroup("UserLspConfig", {}),
	callback = function(ev)
		local client = vim.lsp.get_client_by_id(ev.data.client_id)
		local bufnr = ev.buf
		
		-- 调用自定义的 on_attach 函数
		custom_on_attach(client, bufnr)
	end,
})

-- ============================================================================
-- 5. 配置扩展说明和示例
-- ============================================================================

-- 5.1 添加更多语言服务器的方法
-- 如需启用更多语言服务器，可以在 servers 表中添加服务器名称：
-- local servers = { "html", "cssls", "ts_ls", "pyright", "lua_ls", "gopls" }

-- 5.2 自定义语言服务器配置示例
-- 对于需要特殊配置的语言服务器，可以使用 vim.lsp.config.server_name 方法：
--
-- 示例 1：Lua 语言服务器配置
-- vim.lsp.config.lua_ls = {
--   cmd = { "lua-language-server" },
--   filetypes = { "lua" },
--   root_markers = { ".luarc.json", ".luarc.jsonc", ".luacheckrc", ".stylua.toml", "stylua.toml", "selene.toml", "selene.yml", ".git" },
--   settings = {
--     Lua = {
--       runtime = { version = "LuaJIT" },
--       diagnostics = { globals = { "vim" } },
--       workspace = { library = vim.api.nvim_get_runtime_file("", true) },
--       telemetry = { enable = false },
--     }
--   }
-- }
-- vim.lsp.enable("lua_ls")
--
-- 更多配置选项请参考：:h vim.lsp.config 或访问 https://github.com/neovim/nvim-lspconfig
