-- ============================================================================
-- NvimTree 配置模块（简化版，兼容 Neovim 0.11.0）
-- 从原 NvChad 配置迁移的文件树配置，保持 Git 集成功能
-- ============================================================================

local M = {}

-- ============================================================================
-- NvimTree 基础配置（版本兼容性优化）
-- ============================================================================

M.config = {
	disable_netrw = true,
	hijack_netrw = true,
	hijack_cursor = true,
	hijack_unnamed_buffer_when_opening = false,
	sync_root_with_cwd = true,
	respect_buf_cwd = true,
	update_focused_file = {
		enable = true,
		update_root = true,
		ignore_list = {},
	},
	git = {
		enable = true,
		ignore = false,
		timeout = 400,
	},
	actions = {
		open_file = {
			quit_on_open = false,
			resize_window = true,
			window_picker = {
				enable = true,
				picker = "default",
				chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890",
				exclude = {
					filetype = { "notify", "packer", "qf", "diff", "fugitive", "fugitiveblame" },
					buftype = { "nofile", "terminal", "help" },
				},
			},
		},
		remove_file = {
			close_window = true,
		},
	},
	view = {
		width = 30,
		side = "left",
		preserve_window_proportions = false,
		number = false,
		relativenumber = false,
		signcolumn = "yes",
		float = {
			enable = false,
			quit_on_focus_loss = true,
			open_win_config = {
				relative = "editor",
				border = "rounded",
				width = 30,
				height = 30,
				row = 1,
				col = 1,
			},
		},
	},
	renderer = {
		add_trailing = false,
		group_empty = false,
		highlight_git = true,
		full_name = false,
		highlight_opened_files = "none",
		root_folder_label = ":~:s?$?/..",
		indent_width = 2,
		indent_markers = {
			enable = false,
			inline_arrows = true,
			icons = {
				corner = "└",
				edge = "│",
				item = "│",
				bottom = "─",
				none = " ",
			},
		},
		icons = {
			webdev_colors = true,
			git_placement = "before",
			modified_placement = "after",
			padding = " ",
			symlink_arrow = " ➛ ",
			show = {
				file = true,
				folder = true,
				folder_arrow = true,
				git = true,
				modified = true,
			},
			glyphs = {
				default = "",
				symlink = "",
				bookmark = "",
				modified = "●",
				folder = {
					arrow_closed = "",
					arrow_open = "",
					default = "",
					open = "",
					empty = "",
					empty_open = "",
					symlink = "",
					symlink_open = "",
				},
				git = {
					unstaged = "✗",
					staged = "✓",
					unmerged = "",
					renamed = "➜",
					untracked = "★",
					deleted = "",
					ignored = "◌",
				},
			},
		},
		special_files = {
			"Cargo.toml",
			"Makefile",
			"README.md",
			"readme.md",
			"index.js",
			"package.json",
			"yarn.lock",
			"package-lock.json",
			"tailwind.config.js",
			"tailwind.config.ts",
			"vue.config.js",
			"vite.config.js",
			"vite.config.ts",
		},
	},
	filters = {
		dotfiles = false,
		custom = {
			"^.git$",
			"node_modules",
			".cache",
			"__pycache__",
			".vscode",
			".idea",
			"*.pyc",
			"*.pyo",
			"*.pyd",
			".DS_Store",
			"Thumbs.db",
		},
	},
	trash = {
		cmd = "trash",
		require_confirm = true,
	},
	notify = {
		threshold = vim.log.levels.INFO,
	},
	ui = {
		confirm = {
			remove = true,
			trash = true,
		},
	},
}

-- ============================================================================
-- Git 颜色修复和高亮增强
-- ============================================================================

local function setup_git_highlights()
	-- Git 状态颜色
	local highlights = {
		NvimTreeGitDirty = { fg = "#ff9e64", bold = true },
		NvimTreeGitStaged = { fg = "#7aa2f7", bold = true },
		NvimTreeGitMerge = { fg = "#bb9af7", bold = true },
		NvimTreeGitRenamed = { fg = "#e0af68", bold = true },
		NvimTreeGitNew = { fg = "#9ece6a", bold = true },
		NvimTreeGitDeleted = { fg = "#f7768e", bold = true },
		NvimTreeGitIgnored = { fg = "#565f89", italic = true },

		-- 文件类型颜色增强
		NvimTreeFolderName = { fg = "#7aa2f7", bold = true },
		NvimTreeOpenedFolderName = { fg = "#7aa2f7", bold = true, italic = true },
		NvimTreeEmptyFolderName = { fg = "#565f89", italic = true },
		NvimTreeFolderIcon = { fg = "#7aa2f7" },

		-- 文件图标颜色
		NvimTreeFileIcon = { fg = "#c0caf5" },
		NvimTreeExecFile = { fg = "#9ece6a", bold = true },
		NvimTreeSpecialFile = { fg = "#bb9af7", bold = true, underline = true },

		-- 高亮当前文件
		NvimTreeImageFile = { fg = "#e0af68" },
		NvimTreeIndentMarker = { fg = "#414868" },
		NvimTreeSymlink = { fg = "#bb9af7" },
		NvimTreeSymlinkIcon = { fg = "#bb9af7" },

		-- 光标行
		NvimTreeCursorLine = { bg = "#1f2335" },
		NvimTreeCursorColumn = { bg = "#1f2335" },

		-- 边框和线条
		NvimTreeVertSplit = { fg = "#1f2335" },
		NvimTreeEndOfBuffer = { fg = "#1f2335" },

		-- 书签和标记
		NvimTreeBookmark = { fg = "#e0af68", bold = true },
		NvimTreeModifiedIcon = { fg = "#f7768e", bold = true },
	}

	-- 应用高亮
	for hl_name, hl_config in pairs(highlights) do
		vim.api.nvim_set_hl(0, hl_name, hl_config)
	end
end

-- 终极 Git 颜色修复函数
function M.ultimate_git_color_fix()
	local ok, nvim_tree_api = pcall(require, "nvim-tree.api")

	-- 强制刷新 Git 状态
	if ok and nvim_tree_api and nvim_tree_api.git then
		nvim_tree_api.git.reload()
	end

	-- 等待一小段时间后再次刷新
	vim.defer_fn(function()
		if ok and nvim_tree_api and nvim_tree_api.tree then
			nvim_tree_api.tree.reload()
		end

		-- 重新应用高亮
		setup_git_highlights()

		-- 强制重绘
		vim.cmd("redraw!")
	end, 100)
end

-- 强制刷新 NvimTree
function M.force_refresh_nvimtree()
	local ok, nvim_tree_api = pcall(require, "nvim-tree.api")

	if ok and nvim_tree_api and nvim_tree_api.tree then
		nvim_tree_api.tree.reload()
	end

	-- 刷新 Git 状态
	if ok and nvim_tree_api and nvim_tree_api.git then
		nvim_tree_api.git.reload()
	end

	-- 应用颜色修复
	M.ultimate_git_color_fix()
end

-- 设置 NvimTree 高亮
function M.set_nvim_tree_highlights()
	setup_git_highlights()
end

-- ============================================================================
-- 自定义命令和键位映射
-- ============================================================================

local function setup_commands_and_mappings()
	local ok, api = pcall(require, "nvim-tree.api")
	if not ok then
		return
	end

	-- 自定义键位映射
	local function opts(desc)
		return { desc = desc, noremap = true, silent = true, nowait = true, buffer = bufnr }
	end

	-- 在 NvimTree 缓冲区中设置键位映射
	vim.api.nvim_create_autocmd("FileType", {
		pattern = "NvimTree",
		callback = function()
			local bufnr = vim.api.nvim_get_current_buf()

			-- 基础导航
			vim.keymap.set("n", "<CR>", api.node.open.edit, opts("打开文件"))
			vim.keymap.set("n", "o", api.node.open.edit, opts("打开文件"))
			vim.keymap.set("n", "<2-LeftMouse>", api.node.open.edit, opts("打开文件"))
			vim.keymap.set("n", "O", api.node.open.no_window_picker, opts("打开文件（无窗口选择）"))

			-- 编辑和创建
			vim.keymap.set("n", "<C-e>", api.node.open.replace_tree_buffer, opts("打开文件并替换树"))
			vim.keymap.set("n", "<C-t>", api.node.open.tab, opts("在新标签页中打开"))
			vim.keymap.set("n", "<C-v>", api.node.open.vertical, opts("垂直分割打开"))
			vim.keymap.set("n", "<C-s>", api.node.open.horizontal, opts("水平分割打开"))
			vim.keymap.set("n", "t", api.node.open.tab, opts("在新标签页中打开"))
			vim.keymap.set("n", "v", api.node.open.vertical, opts("垂直分割打开"))
			vim.keymap.set("n", "s", api.node.open.horizontal, opts("水平分割打开"))

			-- 文件系统操作
			vim.keymap.set("n", "a", api.fs.create, opts("创建文件/目录"))
			vim.keymap.set("n", "d", api.fs.remove, opts("删除"))
			vim.keymap.set("n", "r", api.fs.rename, opts("重命名"))
			vim.keymap.set("n", "x", api.fs.cut, opts("剪切"))
			vim.keymap.set("n", "c", api.fs.copy.node, opts("复制"))
			vim.keymap.set("n", "p", api.fs.paste, opts("粘贴"))
			vim.keymap.set("n", "y", api.fs.copy.filename, opts("复制文件名"))
			vim.keymap.set("n", "Y", api.fs.copy.relative_path, opts("复制相对路径"))

			-- 树操作
			vim.keymap.set("n", "E", api.tree.expand_all, opts("展开所有"))
			vim.keymap.set("n", "W", api.tree.collapse_all, opts("折叠所有"))
			vim.keymap.set("n", "S", api.tree.search_node, opts("搜索节点"))
			vim.keymap.set("n", "f", api.live_filter.start, opts("实时过滤"))
			vim.keymap.set("n", "F", api.live_filter.clear, opts("清除过滤"))

			-- 书签
			vim.keymap.set("n", "m", api.marks.toggle, opts("切换书签"))
			vim.keymap.set("n", "b", api.marks.toggle, opts("切换书签"))
			vim.keymap.set("n", "gc", api.marks.clear, opts("清除书签"))
			vim.keymap.set("n", "gm", api.marks.list, opts("列出书签"))

			-- 切换和刷新
			vim.keymap.set("n", "R", api.tree.reload, opts("刷新"))
			vim.keymap.set("n", "i", api.node.show_info_popup, opts("显示信息"))
			vim.keymap.set("n", "q", api.tree.close, opts("关闭"))
			vim.keymap.set("n", "<Esc>", api.tree.close, opts("关闭"))
			vim.keymap.set("n", "<C-c>", api.tree.close, opts("关闭"))

			-- 强制覆盖翻页快捷键
			local function set_paging_maps()
				vim.keymap.set("n", "<C-j>", "<C-d>", {
					desc = "向下半页",
					noremap = true,
					silent = true,
					buffer = bufnr,
					nowait = true
				})
				vim.keymap.set("n", "<C-k>", "<C-u>", {
					desc = "向上半页",
					noremap = true,
					silent = true,
					buffer = bufnr,
					nowait = true
				})
			end

			-- 立即设置
			set_paging_maps()

			-- 帮助
			vim.keymap.set("n", "?", api.tree.toggle_help, opts("显示帮助"))

			-- 多次延迟设置以确保覆盖
			vim.defer_fn(set_paging_maps, 50)
			vim.defer_fn(set_paging_maps, 200)
			vim.defer_fn(set_paging_maps, 500)
		end,
	})

	-- 全局键位映射
	vim.keymap.set("n", "<leader>e", api.tree.toggle, opts("切换文件树"))
	vim.keymap.set("n", "<leader>o", api.tree.focus, opts("聚焦文件树"))
end

-- ============================================================================
-- 自动命令设置
-- ============================================================================

local function setup_autocmds()
	local group = vim.api.nvim_create_augroup("NvimTreeCustom", { clear = true })

	-- 打开文件时自动定位
	vim.api.nvim_create_autocmd("BufEnter", {
		group = group,
		callback = function()
			local ok, api = pcall(require, "nvim-tree.api")
			if ok and api.tree.is_visible() then
				api.tree.find_file()
			end
		end,
	})

	-- Git 状态变更时自动刷新
	vim.api.nvim_create_autocmd({ "BufWritePost", "FocusGained", "CursorHold", "CursorHoldI" }, {
		group = group,
		callback = function()
			vim.defer_fn(function()
				M.force_refresh_nvimtree()
			end, 500)
		end,
	})

	-- NvimTree 打开时的特殊处理
	vim.api.nvim_create_autocmd("FileType", {
		group = group,
		pattern = "NvimTree",
		callback = function()
			-- 设置窗口类型
			vim.w.win_type = "sidebar"

			-- 应用自定义高亮
			M.set_nvim_tree_highlights()

			-- 延迟执行颜色修复
			vim.defer_fn(function()
				M.ultimate_git_color_fix()
			end, 50)
		end,
	})
end

-- ============================================================================
-- 主要设置函数
-- ============================================================================

function M.setup()
	-- 设置 NvimTree
	local ok, nvimtree = pcall(require, "nvim-tree")
	if not ok then
		vim.notify("NvimTree 插件未正确加载", vim.log.levels.ERROR)
		return
	end

	nvimtree.setup(M.config)

	-- 设置命令和键位映射
	setup_commands_and_mappings()

	-- 设置自动命令
	setup_autocmds()

	-- 初始高亮设置
	vim.defer_fn(function()
		M.set_nvim_tree_highlights()
		M.ultimate_git_color_fix()
	end, 200)
end

return M