-- ============================================================================
-- LSP 配置 (使用 Neovim 0.11+ 新 API)
-- 从原 NvChad 配置迁移的 LSP 设置，兼容新版 Neovim
-- ============================================================================

local M = {}

-- ============================================================================
-- LSP 诊断信息中文化配置
-- ============================================================================

local chinese_diagnostics = {
	-- Python/Pyright 相关
	["Expression value is unused"] = "表达式的值未被使用",
	["Argument of type .* is incompatible"] = "参数类型不兼容",
	["Cannot access member"] = "无法访问成员",
	["is not defined"] = "未定义",
	["is not accessed"] = "未被访问",
	["Missing positional argument"] = "缺少位置参数",
	["Too many positional arguments"] = "位置参数过多",
	["Unexpected keyword argument"] = "意外的关键字参数",
	["Variable .* is possibly unbound"] = "变量可能未绑定",
	["Import .* could not be resolved"] = "导入无法解析",
	["Function with declared type of .* must return value"] = "声明类型的函数必须返回值",

	-- JavaScript/TypeScript 相关
	["Property .* does not exist"] = "属性不存在",
	["Object is possibly 'null'"] = "对象可能为'null'",
	["Object is possibly 'undefined'"] = "对象可能为'undefined'",
	["Cannot find name"] = "找不到名称",
	["Expected .* arguments, but got .*"] = "期望参数数量不匹配",
	["Type .* is not assignable to type"] = "类型不能赋值给另一类型",
	["Parameter .* implicitly has an 'any' type"] = "参数隐式具有'any'类型",

	-- 通用LSP错误
	["Syntax error"] = "语法错误",
	["Unexpected token"] = "意外的标记",
	["Missing semicolon"] = "缺少分号",
	["Unreachable code"] = "无法到达的代码",
	["Unused variable"] = "未使用的变量",
	["Unused parameter"] = "未使用的参数",
	["Unused import"] = "未使用的导入",
}

-- 翻译函数：将英文诊断信息翻译为中文
local function translate_diagnostic(message)
	for english, chinese in pairs(chinese_diagnostics) do
		if string.match(message, english) then
			return string.gsub(message, english, chinese)
		end
	end
	return message
end

-- ============================================================================
-- LSP 诊断处理
-- ============================================================================

-- 自定义诊断处理器
M.setup_diagnostics = function()
	local orig_signs_handler = vim.diagnostic.handlers.signs

	vim.diagnostic.handlers.signs = {
		show = function(_, bufnr, diagnostics, _)
			local translated_diagnostics = {}
			for _, diagnostic in ipairs(diagnostics) do
				local translated = vim.deepcopy(diagnostic)
				translated.message = translate_diagnostic(diagnostic.message)
				table.insert(translated_diagnostics, translated)
			end
			return orig_signs_handler.show(nil, bufnr, translated_diagnostics, _)
		end,
		hide = orig_signs_handler.hide,
	}

	-- 设置诊断显示格式
	vim.diagnostic.config({
		virtual_text = {
			prefix = "●",
			spacing = 4,
		},
		float = {
			border = "rounded",
			source = "always",
			header = "",
			prefix = "",
		},
		signs = true,
		underline = true,
		update_in_insert = false,
		severity_sort = true,
	})
end

-- ============================================================================
-- 自定义 LSP 键位映射
-- ============================================================================

M.setup_lsp_keymaps = function(bufnr)
	local opts = { buffer = bufnr, silent = true }

	-- 跳转到定义
	vim.keymap.set("n", "gd", function() vim.lsp.buf.definition() end, opts)

	-- 跳转到声明
	vim.keymap.set("n", "gD", function() vim.lsp.buf.declaration() end, opts)

	-- 查看引用
	vim.keymap.set("n", "gr", function() vim.lsp.buf.references() end, opts)

	-- 查看实现
	vim.keymap.set("n", "gi", function() vim.lsp.buf.implementation() end, opts)

	-- 查看类型定义
	vim.keymap.set("n", "gt", function() vim.lsp.buf.type_definition() end, opts)

	-- 悬浮信息
	vim.keymap.set("n", "K", function() vim.lsp.buf.hover() end, opts)

	-- 签名帮助
	vim.keymap.set("i", "<C-k>", function() vim.lsp.buf.signature_help() end, opts)

	-- 重命名
	vim.keymap.set("n", "<leader>rn", function() vim.lsp.buf.rename() end, opts)

	-- 代码操作
	vim.keymap.set("n", "<leader>ca", function() vim.lsp.buf.code_action() end, opts)

	-- 格式化
	vim.keymap.set("n", "<leader>cf", function() vim.lsp.buf.format() end, opts)

	-- 诊断导航
	vim.keymap.set("n", "[d", function() vim.diagnostic.goto_prev() end, opts)
	vim.keymap.set("n", "]d", function() vim.diagnostic.goto_next() end, opts)
	vim.keymap.set("n", "<leader>e", function() vim.diagnostic.open_float() end, opts)

	-- 工作区符号
	vim.keymap.set("n", "<leader>ws", function() vim.lsp.buf.workspace_symbol() end, opts)

	-- 文档符号
	vim.keymap.set("n", "<leader>ds", function() vim.lsp.buf.document_symbol() end, opts)
end

-- ============================================================================
-- LSP 服务器配置 (混合使用新旧 API 以减少警告)
-- ============================================================================

M.setup_lsp = function()
	-- 使用 lspconfig 配置
	local lspconfig = require("lspconfig")
	local capabilities = require("cmp_nvim_lsp").default_capabilities()

	-- Python (pyright)
	lspconfig.pyright.setup({
		capabilities = capabilities,
		on_attach = function(client, bufnr)
			M.setup_lsp_keymaps(bufnr)
		end,
		settings = {
			python = {
				analysis = {
					autoSearchPaths = true,
					diagnosticMode = "workspace",
					useLibraryCodeForTypes = true,
				},
			},
		},
	})

	-- JavaScript/TypeScript (ts_ls)
	lspconfig.ts_ls.setup({
		capabilities = capabilities,
		on_attach = function(client, bufnr)
			M.setup_lsp_keymaps(bufnr)
		end,
		settings = {
			typescript = {
				inlayHints = {
					includeInlayParameterNameHints = "all",
					includeInlayParameterNameHintsWhenArgumentMatchesName = true,
					includeInlayFunctionParameterTypeHints = true,
					includeInlayVariableTypeHints = true,
					includeInlayPropertyDeclarationTypeHints = true,
					includeInlayFunctionLikeReturnTypeHints = true,
					includeInlayEnumMemberValueHints = true,
				},
			},
			javascript = {
				inlayHints = {
					includeInlayParameterNameHints = "all",
					includeInlayParameterNameHintsWhenArgumentMatchesName = true,
					includeInlayFunctionParameterTypeHints = true,
					includeInlayVariableTypeHints = true,
					includeInlayPropertyDeclarationTypeHints = true,
					includeInlayFunctionLikeReturnTypeHints = true,
					includeInlayEnumMemberValueHints = true,
				},
			},
		},
	})

	-- Lua (lua_ls)
	lspconfig.lua_ls.setup({
		capabilities = capabilities,
		on_attach = function(client, bufnr)
			M.setup_lsp_keymaps(bufnr)
		end,
		settings = {
			Lua = {
				runtime = {
					version = "LuaJIT",
				},
				diagnostics = {
					globals = { "vim" },
				},
				workspace = {
					library = vim.api.nvim_get_runtime_file("", true),
				},
				telemetry = {
					enable = false,
				},
			},
		},
	})

	-- HTML/CSS (html, cssls)
	lspconfig.html.setup({
		capabilities = capabilities,
		on_attach = function(client, bufnr)
			M.setup_lsp_keymaps(bufnr)
		end,
	})

	lspconfig.cssls.setup({
		capabilities = capabilities,
		on_attach = function(client, bufnr)
			M.setup_lsp_keymaps(bufnr)
		end,
	})

	-- JSON (jsonls)
	lspconfig.jsonls.setup({
		capabilities = capabilities,
		on_attach = function(client, bufnr)
			M.setup_lsp_keymaps(bufnr)
		end,
	})

	-- Tailwind CSS (tailwindcss)
	lspconfig.tailwindcss.setup({
		capabilities = capabilities,
		on_attach = function(client, bufnr)
			M.setup_lsp_keymaps(bufnr)
		end,
	})
end

-- ============================================================================
-- 初始化函数
-- ============================================================================

M.setup = function()
	-- 设置诊断
	M.setup_diagnostics()

	-- 延迟设置 LSP（等待插件加载完成）
	vim.defer_fn(function()
		M.setup_lsp()
	end, 100)
end

return M