-- ============================================================================
-- Nvim-tree 文件树配置文件 (nvimtree.lua)
-- ============================================================================
--
-- 📁 插件简介:
-- nvim-tree 是 Neovim 中功能最完整的文件浏览器插件，提供类似 VSCode 或 JetBrains IDE
-- 的侧边栏文件管理功能。此插件完全替代了 Neovim 内置的 netrw 文件管理器，提供更
-- 现代化和用户友好的文件操作体验。
--
-- 📋 核心功能列表:
-- ┌─────────────────────────────────────────────────────────────────────────┐
-- │ 🗂️ 基础文件管理                                                          │
-- │   • 树状目录结构显示 (支持折叠/展开)                                    │
-- │   • 文件/目录创建、重命名、删除操作                                     │
-- │   • 文件类型图标显示 (基于文件扩展名)                                   │
-- │   • 智能光标跟随当前编辑文件                                             │
-- │                                                                         │
-- │ 🔄 Git 版本控制集成                                                       │
-- │   • 实时显示文件修改状态 (已修改/新增/删除/冲突等)                      │
-- │   • Git 状态图标和颜色高亮                                              │
-- │   • 忽略 .gitignore 中的文件 (可配置)                                   │
-- │                                                                         │
-- │ 📋 高级剪贴板操作                                                         │
-- │   • 单文件/多文件复制、剪切、粘贴                                       │
-- │   • 智能冲突处理 (覆盖/重命名/取消)                                     │
-- │   • 跨目录文件移动和批量操作                                             │
-- │                                                                         │
-- │ 🏷️ 批量文件标记系统                                                       │
-- │   • 多选文件标记 (类似文件管理器的 Ctrl+点击)                           │
-- │   • 批量操作: 复制/移动/删除/移至垃圾箱                                 │
-- │   • 标记导航和清理功能                                                   │
-- │                                                                         │
-- │ 🔍 智能搜索与导航                                                         │
-- │   • 文件名快速搜索和定位                                                 │
-- │   • 自动展开到当前编辑文件                                               │
-- │   • 支持书签和历史记录                                                   │
-- │                                                                         │
-- │ 🩺 LSP 诊断信息显示                                                       │
-- │   • 实时显示错误/警告/信息标记                                           │
-- │   • 与语言服务器无缝集成                                                 │
-- │   • 支持跳转到错误位置                                                   │
-- └─────────────────────────────────────────────────────────────────────────┘
--
-- 🛠️ 技术依赖说明:
-- ┌─────────────────────────────────────────────────────────────────────────┐
-- │ 必需模块:                                                                │
-- │   • nvim-tree            - 主插件模块，提供核心文件树功能               │
-- │   • nvim-tree.api        - API 接口，用于自定义操作和键位绑定           │
-- │                                                                         │
-- │ 系统依赖:                                                                │
-- │   • vim.notify           - 消息通知系统 (用于操作反馈)                 │
-- │   • vim.fn              - Vim 函数库 (文件系统操作)                    │
-- │   • vim.api             - Neovim API (界面交互)                        │
-- │   • vim.loop            - Libuv 异步 I/O (文件状态检测)                │
-- │                                                                         │
-- │ 外部工具 (可选):                                                         │
-- │   • git                 - Git 状态显示                                 │
-- │   • trash               - 安全删除到垃圾箱                             │
-- │   • fd/find            - 文件搜索加速                                  │
-- └─────────────────────────────────────────────────────────────────────────┘
--
-- 📖 详细使用指南:
-- ┌─────────────────────────────────────────────────────────────────────────┐
-- │ 🚀 启动和基础操作                                                        │
-- │   <M-q>         - 切换文件树显示/隐藏                                   │
-- │   <leader>e     - 聚焦到文件树窗口                                      │
-- │   Enter         - 打开文件或展开/折叠目录                               │
-- │   Tab           - 预览文件 (不跳转到编辑窗口)                           │
-- │                                                                         │
-- │ 📝 文件和目录操作                                                         │
-- │   a             - 创建新文件/目录 (末尾加 / 创建目录)                   │
-- │   r             - 重命名文件/目录                                       │
-- │   d             - 删除文件/目录 (中文确认对话框)                        │
-- │   c             - 复制文件/目录                                         │
-- │   p             - 粘贴文件/目录                                         │
-- │                                                                         │
-- │ 🏷️ 多选标记操作                                                           │
-- │   m             - 标记/取消标记文件                                     │
-- │   tm            - 批量移动标记的文件 (默认目录为当前目录)               │
-- │   tc            - 清除所有标记                                          │
-- │   mn            - 跳转到下一个标记                                      │
-- │   mp            - 跳转到上一个标记                                      │
-- │                                                                         │
-- │ 📋 剪贴板操作 (增强版)                                                    │
-- │   y             - 复制当前文件 (单文件精确复制)                         │
-- │   cp            - 复制当前文件 (备用快捷键)                             │
-- │   x             - 剪切当前文件 (单文件精确剪切)                         │
-- │   mv            - 剪切当前文件 (备用快捷键)                             │
-- │   p             - 智能粘贴 (支持冲突处理)                               │
-- │   cc            - 清空剪贴板                                            │
-- │                                                                         │
-- │ 🔄 导航和显示                                                             │
-- │   R             - 刷新文件树                                            │
-- │   H             - 切换隐藏文件显示                                      │
-- │   Ctrl+j        - 向下翻页 (半屏滚动)                                   │
-- │   Ctrl+k        - 向上翻页 (半屏滚动)                                   │
-- │   E             - 展开所有目录                                          │
-- │   W             - 折叠所有目录                                          │
-- │   -             - 跳转到上级目录                                        │
-- │   s             - 打开系统文件管理器                                    │
-- └─────────────────────────────────────────────────────────────────────────┘
--
-- 🎛️ 自定义命令列表:
-- ┌─────────────────────────────────────────────────────────────────────────┐
-- │ 📋 单文件操作命令 (A1-A3 系列)                                           │
-- │   :NvimTreeA1Copy        - 复制当前光标下的文件/目录                   │
-- │   :NvimTreeA2Cut         - 剪切当前光标下的文件/目录                   │
-- │   :NvimTreeA3Paste       - 智能粘贴 (支持覆盖/重命名选项)              │
-- │                                                                         │
-- │ 🏷️ 批量标记操作命令 (A1-A6 系列)                                         │
-- │   :NvimTreeA1MarkCopy    - 批量复制所有标记的文件                      │
-- │   :NvimTreeA2MarkCut     - 批量剪切所有标记的文件                      │
-- │   :NvimTreeA4MarkMove    - 批量移动标记文件 (默认目录为当前目录)        │
-- │   :NvimTreeA6MarkDelete  - 批量删除所有标记的文件                      │

-- │                                                                         │
-- │ 🧹 清理和管理命令 (A5/A7 系列)                                           │
-- │   :NvimTreeA5Rename      - 重命名当前文件/目录                         │
-- │   :NvimTreeA7Clear       - 清除所有文件标记                            │
-- └─────────────────────────────────────────────────────────────────────────┘
--
-- 🔧 批量操作工作流程:
-- ┌─────────────────────────────────────────────────────────────────────────┐
-- │ 📋 批量复制文件流程:                                                      │
-- │   1️⃣ 使用 'm' 键标记需要复制的多个文件                                   │
-- │   2️⃣ 执行 :NvimTreeA1MarkCopy 将标记文件复制到剪贴板                  │
-- │   3️⃣ 导航到目标目录 (可以是任意位置)                                    │
-- │   4️⃣ 按 'p' 键或执行 :NvimTreeA3Paste 进行粘贴                        │
-- │   5️⃣ 根据提示处理文件冲突 (覆盖/重命名/取消)                           │
-- │                                                                         │
-- │ ✂️ 批量移动文件流程:                                                       │
-- │   1️⃣ 使用 'm' 键标记需要移动的多个文件                                   │
-- │   2️⃣ 按 'tm' 键或执行 :NvimTreeA4MarkMove (默认显示当前目录)           │
-- │   3️⃣ 输入目标目录路径或直接确认使用默认目录                             │
-- │   4️⃣ 根据提示处理文件冲突 (覆盖/重命名/取消)                           │
-- │   5️⃣ 目录不存在时可选择自动创建                                       │
-- │                                                                         │
-- │ 🗑️ 批量删除文件流程:                                                       │
-- │   1️⃣ 使用 'm' 键标记需要删除的多个文件                                   │
-- │   2️⃣ 执行 :NvimTreeA6MarkDelete                                       │
-- │   3️⃣ 确认中文删除对话框 (输入 'y' 确认，'n' 取消)                      │
-- │   4️⃣ 系统自动清理标记和刷新文件树                                       │
-- └─────────────────────────────────────────────────────────────────────────┘
--
-- ⚠️ 重要注意事项:
-- ┌─────────────────────────────────────────────────────────────────────────┐
-- │ 🔐 安全特性:                                                              │
-- │   • 所有删除操作都有中文确认对话框，防止误操作                          │
-- │   • 文件冲突时提供三种选择: 覆盖/重命名/取消                            │
-- │   • 批量操作前会显示操作的文件数量                                       │
-- │   • 自动备份重要文件路径信息                                             │
-- │                                                                         │
-- │ 🎯 性能优化:                                                              │
-- │   • 使用延迟加载避免启动卡顿                                             │
-- │   • 文件状态检测使用异步方式                                             │
-- │   • 大目录采用虚拟滚动提升响应速度                                       │
-- │   • 智能缓存减少重复文件系统调用                                         │
-- │                                                                         │
-- │ 🧩 兼容性说明:                                                            │
-- │   • 完全替代 netrw，避免冲突                                            │
-- │   • 与 LSP 服务器无缝集成                                               │
-- │   • 支持所有主流操作系统 (Linux/macOS/Windows)                         │
-- │   • 兼容 Neovim 0.7+ 版本                                              │
-- │                                                                         │
-- │ 🔧 自定义说明:                                                            │
-- │   • 键位映射可通过 on_attach 函数修改                                   │
-- │   • 图标和颜色可通过 renderer 配置自定义                               │
-- │   • 过滤规则可通过 filters 配置调整                                     │
-- │   • 通知消息可通过 notify 配置控制                                      │
-- └─────────────────────────────────────────────────────────────────────────┘
-- ============================================================================

-- ============================================================================
-- 1. 基础配置初始化和工具函数
-- ============================================================================

-- 🔧 全局剪贴板跟踪变量
-- 用于跟踪最后复制/剪切的文件信息，实现同目录粘贴检测
-- 结构: { path = "文件路径", action = "copy"|"cut", timestamp = 时间戳 }
_G.nvimtree_last_clipboard_action = nil

-- 🔧 全局消息显示函数
-- 功能说明: 统一的消息显示接口，优先使用 vim.notify，失败时回退到 vim.api.nvim_echo
--
-- 参数说明:
--   @param message (string): 要显示的消息内容
--   @param level (number): 消息级别，可选值:
--     • vim.log.levels.TRACE  (0) - 跟踪信息 (最详细)
--     • vim.log.levels.DEBUG  (1) - 调试信息
--     • vim.log.levels.INFO   (2) - 一般信息 (默认)
--     • vim.log.levels.WARN   (3) - 警告信息 (黄色显示)
--     • vim.log.levels.ERROR  (4) - 错误信息 (红色显示)
--
-- 使用示例:
--   show_message("操作成功", vim.log.levels.INFO)
--   show_message("警告: 文件已存在", vim.log.levels.WARN)
--   show_message("错误: 无法访问文件", vim.log.levels.ERROR)
--
-- 技术实现:
--   • 使用 pcall() 安全调用 vim.notify，避免插件未安装时报错
--   • vim.notify 配置: 标题显示为 "NvimTree"，超时 3 秒，简洁样式
--   • 回退方案: 使用 vim.api.nvim_echo 在命令行显示消息
local function show_message(message, level)
	-- 优先使用 vim.notify，如果不可用则回退到 echo
	local ok, result = pcall(function()
		return vim.notify(message, level, {
			title = "NvimTree", -- 通知标题，统一显示为 NvimTree
			timeout = 3000, -- 显示时长: 3000 毫秒 (3秒)
			render = "minimal", -- 渲染样式: minimal (简洁) | default | compact
		})
	end)

	if not ok then
		-- 如果 vim.notify 不可用，使用 echo 作为备选方案
		-- vim.api.nvim_echo 参数说明:
		--   第1个参数: 消息数组，每个元素包含 {text, highlight_group}
		--   第2个参数: history (boolean) - 是否添加到命令历史
		--   第3个参数: opts (table) - 额外选项 (通常为空)
		vim.api.nvim_echo({ { message, "Normal" } }, true, {})
	end
end

-- 文件冲突处理函数（全局）
local function handle_single_file_conflict(source_path, target_dir, action)
	local source_filename = vim.fn.fnamemodify(source_path, ":t")
	local target_path = target_dir .. "/" .. source_filename
	target_path = target_path:gsub("//+", "/")

	-- 检查目标文件是否已存在
	if vim.fn.filereadable(target_path) == 1 or vim.fn.isdirectory(target_path) == 1 then
		vim.cmd("redraw")
		-- 拆分路径，突出显示文件名
		local dir_path = vim.fn.fnamemodify(target_path, ":h")
		local filename = vim.fn.fnamemodify(target_path, ":t")
		-- 创建蓝色下划线高亮组
		vim.api.nvim_set_hl(0, "BlueUnderlined", { fg = "#61afef", underline = true })

		local operation_text = action == "copy" and "粘贴" or (action == "move" and "移动" or "操作")
		vim.api.nvim_echo({
			{ "⚠️  目标位置已存在同名文件: ", "WarningMsg" },
			{ dir_path .. "/", "Normal" },
			{ filename, "BlueUnderlined" },
			{ "\n请选择操作:\n", "WarningMsg" },
			{ "✅ Y - 覆盖现有文件\n", "Normal" },
			{ "📝 R - 重命名" .. operation_text .. "（输入新文件名）\n", "Normal" },
			{ "❌ N - 取消操作\n", "Normal" },
			{ "请选择 (Y/R/N): ", "Question" }
		}, true, {})

		local char = vim.fn.getchar()
		local choice = vim.fn.nr2char(char)
		vim.cmd("redraw")

		if choice == "Y" or choice == "y" then
			-- 覆盖操作
			vim.fn.system({ 'rm', '-rf', target_path })
			local cmd = action == "copy" and { 'cp', '-r', source_path, target_path } or
				{ 'mv', source_path, target_path }
			vim.fn.system(cmd)
			return vim.v.shell_error == 0 and "overwrite" or "failed"
		elseif choice == "R" or choice == "r" then
			-- 重命名操作
			local new_filename = vim.fn.input("📝 请输入新文件名: ", source_filename)
			vim.cmd("redraw")

			if new_filename == "" or new_filename == source_filename then
				return "cancelled"
			end

			local new_target_path = target_dir .. "/" .. new_filename
			new_target_path = new_target_path:gsub("//+", "/")

			-- 检查新文件名冲突
			if vim.fn.filereadable(new_target_path) == 1 or vim.fn.isdirectory(new_target_path) == 1 then
				-- 立即显示冲突提示，不使用延迟，确保用户能看到
				local new_dir_path = vim.fn.fnamemodify(new_target_path, ":h")
				local new_filename_display = vim.fn.fnamemodify(new_target_path, ":t")
				-- 创建蓝色下划线高亮组
				vim.api.nvim_set_hl(0, "BlueUnderlined", { fg = "#61afef", underline = true })
				vim.api.nvim_echo({
					{ "❌ 目标位置已存在同名文件: ", "WarningMsg" },
					{ new_dir_path .. "/", "Normal" },
					{ new_filename_display, "BlueUnderlined" },
					{ "，操作已取消", "WarningMsg" }
				}, true, {})
				-- 短暂停顿让用户看清楚提示
				vim.cmd("sleep 1000m")
				return "rename_conflict"
			end

			local cmd = action == "copy" and { 'cp', '-r', source_path, new_target_path } or
				{ 'mv', source_path, new_target_path }
			vim.fn.system(cmd)
			return vim.v.shell_error == 0 and "renamed" or "failed", new_target_path
		else
			return "cancelled"
		end
	else
		-- 无冲突，直接操作
		local cmd = action == "copy" and { 'cp', '-r', source_path, target_path } or { 'mv', source_path, target_path }
		vim.fn.system(cmd)
		return vim.v.shell_error == 0 and "success" or "failed"
	end
end

-- ============================================================================
-- 2. nvim-tree 主配置块
-- ============================================================================
--
-- 🏗️ nvim-tree.setup() 函数说明:
-- 这是 nvim-tree 插件的主配置函数，接受一个配置表作为参数。
-- 所有的配置项都是可选的，未设置的项将使用插件的默认值。
--
require("nvim-tree").setup({
	-- ============================================================================
	-- 2.1 基本行为设置
	-- ============================================================================

	-- 🎯 光标跟随配置
	-- hijack_cursor (boolean): 控制是否劫持光标行为
	-- • true:  当光标移动到文件树窗口时，自动高亮当前行并跟随光标
	-- • false: 使用标准的光标行为，不进行特殊处理
	-- 推荐设置: true (提供更好的视觉反馈)
	hijack_cursor = true,

	-- 📢 通知系统配置
	-- notify (table): 控制 nvim-tree 内置通知消息的显示级别
	-- 作用: 过滤不必要的信息，只显示重要的错误消息
	notify = {
		-- threshold (number): 消息显示阈值，低于此级别的消息将被隐藏
		-- 可选值:
		--   vim.log.levels.TRACE (0) - 显示所有消息 (包括调试信息)
		--   vim.log.levels.DEBUG (1) - 显示调试级别及以上
		--   vim.log.levels.INFO  (2) - 显示信息级别及以上
		--   vim.log.levels.WARN  (3) - 显示警告级别及以上
		--   vim.log.levels.ERROR (4) - 只显示错误消息 (推荐)
		-- 设置为 ERROR 可以隐藏 "Clipboard has been emptied" 等信息级消息
		threshold = vim.log.levels.ERROR,
	},

	-- 🚀 性能优化配置
	-- 减少同步和刷新频率以提升大项目性能
	sync_root_with_cwd = false,         -- 禁用根目录自动同步，减少文件系统操作
	respect_buf_cwd = false,            -- 禁用缓冲区工作目录跟随
	update_focused_file = {
		enable = true,                   -- 保留文件跟随功能
		update_root = false,            -- 但禁用根目录更新
		debounce_delay = 100,           -- 增加防抖延迟
	},
	
	-- 🔧 渲染性能优化
	prefer_startup_root = false,        -- 禁用启动根目录偏好
	reload_on_bufenter = false,         -- 禁用缓冲区进入时的重新加载

	-- ============================================================================
	-- 2.2 功能集成配置
	-- ============================================================================

	-- 📐 视图和窗口配置 (view)
	-- 控制文件树窗口的外观、大小、位置等视觉属性
	view = {
		-- width (number|string): 文件树窗口宽度
		-- • number: 固定宽度 (字符数)
		-- • string: 百分比 (如 "30%")
		-- • function: 动态计算宽度的函数
		-- 推荐: 30-40 字符宽度，平衡显示效果和编辑空间
		width = 35,
		
		-- 🚀 性能优化: 禁用一些不必要的视觉特效以提升性能
		number = false,           -- 禁用行号显示，减少渲染开销
		relativenumber = false,   -- 禁用相对行号
		signcolumn = "no",       -- 禁用标记列，减少宽度计算

		-- 其他可用的 view 配置项 (此配置中未使用，但可以添加):
		-- height = number,                    -- 窗口高度
		-- side = "left"|"right",             -- 窗口位置
		-- preserve_window_proportions = bool, -- 保持窗口比例
		-- number = bool,                      -- 显示行号
		-- relativenumber = bool,              -- 显示相对行号
		-- signcolumn = "yes"|"no",           -- 显示标记列
	},

	-- 📁 文件操作行为配置 (actions)
	-- 定义各种文件操作 (打开、创建、删除等) 的具体行为
	actions = {
		-- 📂 文件打开行为配置
		open_file = {
			-- 此处可以配置文件打开时的行为，例如:
			-- quit_on_open = bool,           -- 打开文件后是否关闭文件树
			-- resize_window = bool,          -- 打开文件后是否调整窗口大小
			-- window_picker.enable = bool,   -- 是否启用窗口选择器
			--
			-- 注意: 高级功能 (如自动创建不存在的目录) 通过自定义函数实现
			-- 见本文件后面的 on_attach 函数和自定义命令部分
		},

		-- 其他可用的 actions 配置 (此配置中未使用):
		-- remove_file.close_window = bool,  -- 删除文件后关闭对应窗口
		-- change_dir.enable = bool,         -- 允许更改工作目录
		-- change_dir.global = bool,         -- 全局更改还是局部更改
	},

	-- 🎨 渲染和显示配置 (renderer)
	-- 控制文件树的视觉外观、图标、高亮等显示效果
	renderer = {
		icons = {
			glyphs = {
				git = {
					untracked = "",
					ignored = "",
				},
			},
		},
		special_files = { ".git" },
		highlight_git = "all",
		highlight_git = true,
		highlight_opened_files = "none",
		-- 📁 根目录标签自定义函数
		-- root_folder_label (function): 自定义根目录的显示标签
		-- 参数: path (string) - 根目录的完整路径
		-- 返回: string - 要显示的标签文本
		-- 此函数为根目录添加文件夹图标，并只显示目录名(不显示完整路径)
		root_folder_label = function(path)
			-- vim.fn.fnamemodify(path, ":t") 解析:
			--   path: 完整路径，如 "/Users/username/project"
			--   ":t": tail 修饰符，只返回最后一部分，如 "project"
			return "📁 " .. vim.fn.fnamemodify(path, ":t")
		end,

		-- 📋 特殊文件列表已在上面定义，这里删除重复配置
		-- special_files 已经在第395行配置了 .git 目录的图标和特殊属性
		-- 重复定义会覆盖前面的图标设置，导致.git图标颜色无法正确设置

		-- 📏 缩进标记配置 (indent_markers)
		-- 在目录树中显示垂直线条，帮助用户理解文件层次结构
		indent_markers = {
			-- enable (boolean): 是否启用缩进标记线
			-- true: 显示 │ ├ └ 等字符表示层级关系
			-- false: 不显示层级标记线
			enable = true,

			-- 其他可用配置 (此配置中未使用):
			-- inline_arrows = bool,     -- 内联箭头样式
			-- icons.corner = string,   -- 转角字符
			-- icons.edge = string,     -- 边缘字符
			-- icons.item = string,     -- 项目字符
			-- icons.none = string,     -- 空白字符
		},

		-- 📂 目录显示优化 (group_empty)
		-- group_empty (boolean): 是否合并连续的空目录
		-- true: 将 "a/b/c" 这样的空目录链显示为 "a/b/c" (节省空间)
		-- false: 分别显示每个目录层级，空目录也会显示图标
		-- 设置为 false 以确保空目录也显示文件夹图标
		group_empty = false,

		-- 🎨 Git 状态高亮 (highlight_git)
		-- highlight_git (boolean): 是否高亮显示 Git 状态
		-- true: 根据文件的 Git 状态使用不同颜色显示 (修改=黄色, 新增=绿色)
		-- false: 所有文件使用相同颜色
		-- 需要: 项目是 Git 仓库，且安装了 Git
		highlight_git = true,
		
		-- 禁用其他高亮类型，只保留Git状态高亮
		highlight_opened_files = "none",
		highlight_modified = "none",

		-- 🖼️ 图标显示配置 (icons)
		-- 控制各种图标的显示开关
		icons = {
			-- show (table): 图标显示开关配置
			show = {
				-- file (boolean): 是否显示文件类型图标
				-- 例如: .js 文件显示 JavaScript 图标, .py 文件显示 Python 图标
				file = true,

				-- folder (boolean): 是否显示文件夹图标
				-- 显示 📁 📂 等文件夹图标，区分打开/关闭状态
				folder = true,

				-- folder_arrow (boolean): 是否显示文件夹展开/折叠箭头
				-- 显示 ▶ ▼ 等箭头，表示目录可以展开或折叠
				folder_arrow = true,

				-- git (boolean): 关闭Git状态图标，只保留颜色变化
				git = false,
			},
		},
	},

	-- 🔍 文件过滤配置 (filters)
	-- 控制哪些文件和目录在文件树中显示或隐藏
	filters = {
		-- 📄 点文件 (dotfiles) 过滤配置
		-- dotfiles (boolean): 是否默认隐藏以 "." 开头的文件和目录
		-- true: 隐藏 .git, .vscode, .env 等文件 (常见做法)
		-- false: 显示所有文件，包括点文件
		-- 注意: 用户随时可以按 'H' 键切换隐藏文件的显示状态
		dotfiles = false,

		-- 🚫 自定义过滤规则 (custom)
		-- custom (table): 自定义要隐藏的文件/目录列表
		-- 使用模式匹配，支持通配符和正则表达式
		-- 示例: { "*.tmp", "node_modules", "__pycache__" }
		custom = {
			-- 为提升性能，过滤掉常见的大目录和临时文件
			"*.log",           -- 隐藏所有日志文件
			"*.tmp",           -- 隐藏临时文件
			"*.swp",           -- 隐藏Vim交换文件
			"*.bak",           -- 隐藏备份文件
			-- 构建和缓存目录
			"dist",            -- 隐藏构建输出目录
			"build",           -- 隐藏构建目录
			"coverage",        -- 隐藏测试覆盖率目录
			".cache",          -- 隐藏缓存目录
			".next",           -- 隐藏Next.js缓存
			".nuxt",           -- 隐藏Nuxt.js缓存
			-- 系统文件
			".DS_Store",       -- 隐藏macOS系统文件
			"Thumbs.db",       -- 隐藏Windows缩略图
			"desktop.ini",     -- 隐藏Windows配置文件
		},

		-- ✅ 过滤排除列表 (exclude)
		-- exclude (table): 即使匹配过滤规则也要显示的文件/目录列表
		-- 优先级高于 dotfiles 和 custom 过滤规则
		exclude = {
			-- ".git": 尽管是点文件，但仍然显示 .git 目录
			-- 原因: .git 目录包含重要的版本控制信息，开发者经常需要访问
			".git",

			-- 其他可能需要排除的示例:
			-- ".gitignore",  -- 显示 Git 忽略文件
			-- ".env",        -- 显示环境配置文件 (谨慎，可能包含敏感信息)
			-- ".vscode",     -- 显示 VS Code 配置目录
		},

		-- 🔍 Git忽略文件显示配置
		-- git_ignored (boolean): 是否隐藏 .gitignore 中的文件
		-- false: 显示被gitignore的文件，但用灰色标识
		-- true: 完全隐藏被gitignore的文件
		git_ignored = false,
		
		-- 其他可用过滤配置 (此配置中未使用):
		-- buffer_content = bool, -- 是否过滤缓冲区内容
	},

	-- 🔄 Git 版本控制集成配置 (git)
	-- 控制 Git 状态显示和检测行为
	git = {
		-- enable (boolean): 是否启用 Git 集成
		-- true: 启用 Git 状态检测和显示 (推荐)
		-- false: 完全禁用 Git 功能，不显示任何 Git 状态
		enable = true,

		-- ignore (boolean): 是否使用 .gitignore 过滤文件显示
		-- true: 遵守 .gitignore 规则，但仍显示被忽略的文件 (使用特殊颜色)
		-- false: 完全忽略 .gitignore 文件的过滤规则
		-- 注意: 这与 filters.git_ignored 不同，这里控制的是 Git 集成行为
		-- 设置为 true，让 .git 目录显示为被忽略状态（灰色）
		ignore = true,

		-- timeout (number): Git 状态检测超时时间 (毫秒)
		-- 防止在大型仓库中 Git 操作阻塞界面
		-- 较小值: 响应更快，但可能在大仓库中不完整
		-- 较大值: 更完整的状态检测，但可能影响性能
		timeout = 200,  -- 增加超时时间，减少编辑时的频繁检测
		
		-- 🚀 Git性能优化选项
		show_on_dirs = false,         -- 禁用目录Git状态，显著减少计算开销
		show_on_open_dirs = false,    -- 禁用展开目录的Git状态显示
	},

	-- 🔄 自动同步配置 - 可能导致Git颜色问题的设置
	-- 禁用可能干扰Git状态显示的自动同步功能
	sync_root_with_cwd = false,  -- 禁用根目录自动同步
	
	-- 🔍 焦点文件更新配置
	-- 控制当切换buffer时文件树的行为
	update_focused_file = {
		-- enable (boolean): 是否启用焦点文件更新
		-- 启用以支持目录打开时自动刷新Git状态
		enable = true,
		
		-- update_root (boolean): 是否更新根目录
		-- 设置为 false 防止编辑其他目录文件时自动切换根目录
		update_root = false,
		
		-- ignore_list (table): 忽略的文件类型
		ignore_list = {},
	},

	-- 📂 文件系统监控配置  
	-- 控制文件系统变化的监控行为
	filesystem_watchers = {
		-- enable (boolean): 重新启用文件系统监控
		-- 需要监控Git状态变化
		enable = true,
		
		-- debounce_delay (number): 防抖延迟时间，增加延迟减少频繁更新
		-- 进一步增加到500ms，最大程度减少Django启动时的文件系统事件处理
		debounce_delay = 500,
		
		-- ignore_dirs (table): 忽略监控的目录，减少不必要的监控
		ignore_dirs = {
			".git/objects",  -- 忽略Git对象目录避免过多触发
			".git/refs",     -- 忽略Git引用目录
			".git/hooks",    -- 忽略Git钩子目录
			"node_modules",  -- 忽略Node.js依赖目录
			"dist",          -- 忽略构建输出目录
			"build",         -- 忽略构建目录
			".cache",        -- 忽略缓存目录
			"__pycache__",   -- 忽略Python缓存
			".venv",         -- 忽略Python虚拟环境
			"venv",          -- 忽略Python虚拟环境
			-- 🔥 Django项目特定忽略目录（减少Django启动时的监控负载）
			"staticfiles",   -- 忽略Django静态文件目录
			"static/admin",  -- 忽略Django管理静态文件
			"logs",          -- 忽略日志目录
			"media",         -- 忽略媒体文件目录
			".pytest_cache", -- 忽略pytest缓存
			"htmlcov",       -- 忽略覆盖率报告
			"collectstatic", -- 忽略收集的静态文件
		},
	},

	-- 🩺 LSP 诊断信息集成 (diagnostics)
	-- 在文件树中实时显示 LSP (语言服务器) 提供的诊断信息
	-- 功能: 在文件名旁显示错误、警告、提示等状态标记
	diagnostics = {
		-- enable (boolean): 是否启用诊断信息显示
		-- true:  在文件树中显示 LSP 诊断信息 (推荐)
		-- false: 不显示诊断信息，只显示普通文件名
		-- 需要: LSP 服务器正常工作且支持对应语言
		enable = true,

		-- 🖼️ 诊断图标配置 (icons)
		-- 自定义不同级别诊断信息的显示图标
		icons = {
			-- hint (string): 提示信息图标
			-- 用于显示 LSP 提供的代码提示、优化建议等
			-- 例如: 变量未使用、可以简化的代码等
			hint = "H", -- 使用简单字符避免标志未定义错误

			-- info (string): 一般信息图标
			-- 用于显示一般性信息和说明
			-- 例如: 代码文档、参数信息等
			info = "I", -- 使用简单字符避免标志未定义错误

			-- warning (string): 警告信息图标
			-- 用于显示可能的问题，但不阻止编译运行
			-- 例如: 过时的 API、性能问题、代码风格问题等
			warning = "W", -- 使用简单字符避免标志未定义错误

			-- error (string): 错误信息图标
			-- 用于显示严重错误，通常会阻止代码正常运行
			-- 例如: 语法错误、类型错误、找不到模块等
			error = "E", -- 使用简单字符避免标志未定义错误

			-- 图标使用说明:
			-- - 修改为简单字符以避免 NvimTreeDiagnosticWarnIcon 等标志未定义的错误
			-- - 如需使用图标字符，需要确保相应的 sign 已正确定义
		},

		-- 其他可用诊断配置 (此配置中未使用):
		-- show_on_dirs = bool,           -- 在目录上显示子文件的诊断信息
		-- debounce_delay = number,       -- 诊断信息更新延迟 (毫秒)
		-- severity = table,             -- 要显示的诊断级别
	},

	-- ============================================================================
	-- 2.3 高级键位映射和交互配置 (on_attach)
	-- ============================================================================
	--
	-- 🎮 on_attach 函数说明:
	-- 这是 nvim-tree 的核心配置函数，当文件树缓冲区创建时自动调用
	-- 功能: 设置所有键位映射、自定义操作行为、覆盖默认功能
	-- 参数: bufnr (number) - 文件树缓冲区编号，用于限制键位映射作用范围
	--
	-- 🔧 实现的自定义功能:
	--   • 多选文件标记系统 (类似文件管理器的 Ctrl+点击)
	--   • 增强的复制/剪切/粘贴操作 (支持冲突处理)
	--   • 中文化的删除确认对话框
	--   • 批量文件操作 (移动、删除、垃圾箱)
	--   • 智能键位映射管理 (避免冲突)
	on_attach = function(bufnr)
		-- 📦 加载必需的 API 模块
		-- nvim-tree.api 提供所有文件操作的核心接口
		local api = require("nvim-tree.api")

		-- 🛠️ 键位映射选项生成器函数
		-- 统一生成键位映射的配置选项，确保一致性
		-- @param desc (string): 键位映射的描述信息，用于帮助系统
		-- @return table: 标准化的键位映射选项
		local function opts(desc)
			return {
				desc = "nvim-tree: " .. desc, -- 描述前缀，便于识别来源
				buffer = bufnr,   -- 限制映射只在文件树缓冲区生效
				noremap = true,   -- 不递归映射，避免映射冲突
				silent = false,   -- 不静默，显示操作反馈
				nowait = true     -- 不等待更多输入，立即响应
			}
		end

		-- 📋 加载默认键位映射
		-- 首先加载 nvim-tree 的所有默认键位映射，确保基本功能可用
		-- 包括: 打开文件、创建文件、重命名、刷新等标准操作
		-- 这些默认映射提供了文件树的基础功能，我们在此基础上添加增强功能
		api.config.mappings.default_on_attach(bufnr)
		
		-- 🔄 重写目录展开键位映射，添加Git状态刷新
		-- 重写 <CR> 键（回车键）用于展开目录和打开文件
		vim.keymap.set("n", "<CR>", function()
			local node = api.tree.get_node_under_cursor()
			if not node then return end
			
			-- 执行默认的展开/打开操作
			api.node.open.edit()
			
			-- 如果是目录，延迟刷新Git状态
			if node.type == "directory" then
				vim.defer_fn(function()
					pcall(function()
						local highlights = require('configs.highlights')
						highlights.set_nvim_tree_highlights()
						highlights.ultimate_git_color_fix()
						highlights.force_refresh_nvimtree()
					end)
				end, 100)
			end
		end, opts("展开目录/打开文件并刷新Git状态"))
		
		-- 重写 <Tab> 键用于切换目录展开状态
		vim.keymap.set("n", "<Tab>", function()
			local node = api.tree.get_node_under_cursor()
			if not node then return end
			
			-- 执行默认的切换操作
			api.node.open.toggle()
			
			-- 延迟刷新Git状态
			vim.defer_fn(function()
				pcall(function()
					local highlights = require('configs.highlights')
					highlights.set_nvim_tree_highlights()
					highlights.ultimate_git_color_fix()
					highlights.force_refresh_nvimtree()
				end)
			end, 100)
		end, opts("切换目录展开状态并刷新Git状态"))

		-- 🧹 安全清理冲突的默认映射
		-- 使用 pcall 安全删除可能与自定义映射冲突的默认键位
		-- pcall 的作用: 即使删除失败也不会中断配置加载过程
		pcall(function() vim.keymap.del("n", "y", { buffer = bufnr }) end) -- 清除默认的 y 映射
		pcall(function() vim.keymap.del("n", "x", { buffer = bufnr }) end) -- 清除默认的 x 映射

		-- ============================================================================
		-- 2.3.1 多选标记系统键位映射
		-- ============================================================================
		--
		-- 📌 文件标记功能说明:
		-- 允许用户标记多个文件进行批量操作，类似图形界面文件管理器的多选功能
		-- 标记的文件会有特殊的视觉标识，可以进行批量复制、移动、删除等操作

		-- m 键: 切换文件标记状态
		-- 功能: 标记当前光标下的文件，再次按下取消标记
		-- 用法: 移动光标到文件上，按 'm' 键进行标记，标记的文件会高亮显示
		vim.keymap.set("n", "m", api.marks.toggle, opts("标记/取消标记文件"))

		-- td 键: 批量删除标记的文件
		-- 功能: 执行批量删除标记文件的命令
		-- 用法: 标记多个文件后，按 'td' 执行批量删除
		vim.keymap.set("n", "td", "<cmd>NvimTreeA6MarkDelete<cr>", opts("批量删除标记的文件"))

		-- tc 键: 清除所有标记
		-- 功能: 取消所有文件的标记状态，恢复到未标记状态
		-- 用法: 当有多个文件被标记时，按 'tc' 一键清除所有标记
		vim.keymap.set("n", "tc", api.marks.clear, opts("清除所有标记"))

		-- ============================================================================
		-- 2.3.1.5 翻页功能键位映射
		-- ============================================================================
		--
		-- 📄 翻页功能说明:
		-- 在nvim-tree中使用Ctrl+j和Ctrl+k进行上下翻页，与普通缓冲区保持一致

		-- Ctrl+j: 向下翻页
		-- 功能: 向下滚动半屏，快速浏览大量文件
		-- 用法: 按 Ctrl+j 向下翻页
		vim.keymap.set("n", "<C-j>", "<C-d>", opts("向下翻页"))

		-- Ctrl+k: 向上翻页
		-- 功能: 向上滚动半屏，快速浏览大量文件
		-- 用法: 按 Ctrl+k 向上翻页
		vim.keymap.set("n", "<C-k>", "<C-u>", opts("向上翻页"))

		-- ty 键: 批量复制标记的文件
		-- 功能: 将所有已标记的文件复制到剪贴板
		-- 用法: 标记多个文件后，按 'ty' 进行批量复制
		vim.keymap.set("n", "ty", "<cmd>NvimTreeA1MarkCopy<cr>", opts("批量复制标记的文件"))

		-- tm 键: 批量移动标记的文件
		-- 功能: 将所有已标记的文件移动到指定目录
		-- 用法: 标记多个文件后，按 'tm' 执行批量移动
		vim.keymap.set("n", "tm", "<cmd>NvimTreeA4MarkMove<cr>", opts("批量移动标记的文件"))

		-- ============================================================================
		-- 2.3.2 增强删除功能 (中文确认对话框)
		-- ============================================================================
		--
		-- 🗑️ 自定义删除功能说明:
		-- 替换 nvim-tree 默认的英文删除确认，提供中文化的用户体验
		-- 特点: 中文提示、更明确的确认流程、更安全的操作体验
		vim.keymap.set("n", "d", function()
			-- 🎯 第一步: 获取当前光标下的文件/目录节点
			-- api.tree.get_node_under_cursor() 返回当前光标位置的节点对象
			-- 节点对象包含: name, absolute_path, type, fs_stat 等属性
			local node = api.tree.get_node_under_cursor()

			-- 🛡️ 安全检查: 确保有选中的文件
			-- 如果用户没有选中任何文件(如光标在空白区域)，则显示错误并退出
			if not node then
				-- vim.api.nvim_echo 参数解析:
				--   第1参数: 消息数组 {{ text, highlight_group }}
				--   第2参数: history - 是否加入命令历史记录
				--   第3参数: opts - 额外选项
				vim.api.nvim_echo({ { "❌ 未选中任何文件", "WarningMsg" } }, true, {})
				return -- 早期返回，避免后续处理
			end

			-- 🏷️ 第二步: 准备用户友好的确认信息
			-- 根据节点类型确定显示的文本("目录" 或 "文件")
			-- node.type 可能的值: "file", "directory", "link"
			local file_type = node.type == "directory" and "目录" or "文件"

			-- 📄 提取文件/目录名称(不包含路径)
			-- vim.fn.fnamemodify(path, ":t") 中的 ":t" 表示 "tail"，只返回路径的最后一部分
			-- 例如: "/home/user/file.txt" -> "file.txt"
			local filename = vim.fn.fnamemodify(node.absolute_path, ":t")

			-- 🖥️ 清除当前屏幕内容，准备显示确认对话框
			-- vim.cmd("redraw") 强制重新绘制屏幕，确保之前的输出不会干扰新的提示
			vim.cmd("redraw")

			-- 💬 显示中文确认提示对话框
			-- 拆分路径，突出显示文件名
			local dir_path = vim.fn.fnamemodify(node.absolute_path, ":h")
			local filename = vim.fn.fnamemodify(node.absolute_path, ":t")
			-- 创建蓝色下划线高亮组
			vim.api.nvim_set_hl(0, "BlueUnderlined", { fg = "#61afef", underline = true })
			vim.api.nvim_echo({
				{ "⚠️ 确定要删除" .. file_type .. ": ", "WarningMsg" },
				{ dir_path .. "/", "Normal" },
				{ filename, "BlueUnderlined" },
				{ " 吗？(Y/N)", "WarningMsg" }
			}, true, {})

			-- ⌨️ 第三步: 获取用户的单字符输入
			-- vim.fn.getchar() 等待用户按一个键，不需要按 Enter 确认
			-- 这提供了更快速的交互体验，类似系统文件管理器的确认对话框
			local char = vim.fn.getchar()

			-- 🔄 将数字字符码转换为可读的字符
			-- getchar() 返回的是字符的 ASCII 码，需要转换为字符进行比较
			local choice = vim.fn.nr2char(char)

			-- 🧹 清除提示信息，恢复正常显示
			-- 无论用户选择什么，都需要清除确认对话框，恢复文件树的正常显示
			vim.cmd("redraw")

			-- 🔍 第四步: 根据用户选择执行相应操作
			if choice == "y" or choice == "Y" then
				-- ✅ 用户确认删除，执行删除操作

				-- 🛡️ 使用 pcall 包装删除操作，捕获可能的错误
				-- pcall(function) 返回 (success, result) 或 (success, error_message)
				-- 这样可以优雅地处理删除失败的情况，而不会导致整个脚本崩溃
				local success, error_msg = pcall(function()
					-- 🔄 第五步: 拦截 nvim-tree 的内置确认提示
					-- nvim-tree 的 api.fs.remove() 会弹出英文确认对话框
					-- 我们通过临时替换输入函数来自动确认，避免双重确认

					-- 💾 备份原始的输入函数，稍后需要恢复
					local original_input = vim.fn.input -- 备份 input 函数
					local original_confirm = vim.fn.confirm -- 备份 confirm 函数

					-- 🎭 临时替换输入函数，自动返回确认
					-- 任何对 vim.fn.input 的调用都将返回 "y"(确认)
					-- 任何对 vim.fn.confirm 的调用都将返回 1(第一个选项，通常是确认)
					vim.fn.input = function(...) return "y" end -- 总是确认输入操作
					vim.fn.confirm = function(...) return 1 end -- 总是选择第一个选项

					-- 🗑️ 执行实际的删除操作（移动到垃圾箱）
					-- api.fs.trash() 是 nvim-tree 的垃圾箱删除函数
					-- 它会将文件移动到系统垃圾箱，相比永久删除更安全
					api.fs.trash()

					-- 🔄 恢复原始的输入函数
					-- 这很重要，否则会影响后续其他需要用户输入的操作
					vim.fn.input = original_input
					vim.fn.confirm = original_confirm
				end)

				-- 📊 第六步: 处理删除结果并显示反馈
				if success then
					-- ✅ 删除成功，显示成功消息
					-- 延迟显示消息，确保删除操作完成后再显示
					vim.defer_fn(function()
						local item_type = (node.type == "directory") and "目录" or "文件"
						local msg = "🗂️ 已删除" .. item_type .. ": " .. node.absolute_path
						-- 使用 echo 显示消息，设置为添加到历史记录
						vim.api.nvim_echo({ { msg, "Normal" } }, true, {})
						-- 使用 echom 确保消息保存到历史记录
						vim.cmd('echom "' .. msg:gsub('"', '\\"') .. '"')
					end, 200)
				else
					-- ❌ 删除失败，显示错误信息
					-- tostring() 确保 error_msg 被正确转换为字符串显示
					-- ErrorMsg 高亮组通常显示为红色，表示错误状态
					vim.api.nvim_echo({ { "❌ 删除失败: " .. tostring(error_msg), "ErrorMsg" } }, true, {})
				end
			else
				-- 🚫 用户取消删除操作
				-- 任何非 'y' 或 'Y' 的输入都被视为取消操作
				-- 显示取消消息让用户知道操作已被安全取消
				vim.api.nvim_echo({ { "❌ 已取消删除", "Normal" } }, true, {})
			end
		end, opts("删除当前文件/目录（中文确认）"))

		-- ============================================================================
		-- 2.3.3 增强复制操作键位映射
		-- ============================================================================

		-- 📋 y 键: 调用自定义复制命令
		-- 将 y 键映射到 NvimTreeA1Copy 命令，实现单文件精确复制
		-- 使用命令调用而不是直接函数调用，保持功能的一致性
		vim.keymap.set("n", "y", "<cmd>NvimTreeA1Copy<cr>",
			{ desc = "nvim-tree: 复制当前文件/目录（单文件操作）", buffer = bufnr, noremap = true, silent = false, nowait = true })

		-- 📋 cp 键: 内联复制函数实现
		-- 提供另一个复制快捷键选择，功能与 yy 键相同但实现方式不同
		vim.keymap.set("n", "cp", function()
			-- 🔍 调试信息: 显示按键被触发
			-- 这行调试信息可以帮助确认键位映射是否正常工作
			vim.api.nvim_echo({ { "🔍 cp 键被按下了！", "Normal" } }, true, {})

			-- 🎯 第一步: 获取当前光标下的节点
			-- 与删除函数类似，首先获取用户要复制的文件/目录
			local node = api.tree.get_node_under_cursor()

			-- 🛡️ 安全检查: 确保有选中的文件
			-- 如果没有选中文件，显示错误信息并退出
			if not node then
				vim.api.nvim_echo({ { "❌ 未选中任何文件", "WarningMsg" } }, true, {})
				return -- 早期返回，避免无效操作
			end

			-- 🧹 第二步: 清空剪贴板确保单文件操作
			-- api.fs.clear_clipboard() 清除之前的剪贴板内容
			-- 这确保每次复制操作只处理当前选中的单个文件，避免意外的批量操作
			api.fs.clear_clipboard()

			-- 📋 第三步: 执行复制操作
			-- api.fs.copy.node() 将当前节点复制到 nvim-tree 的内部剪贴板
			-- 这个剪贴板独立于系统剪贴板，专门用于文件操作
			api.fs.copy.node()

			-- 🔄 记录复制操作到全局变量
			_G.nvimtree_last_clipboard_action = {
				path = node.absolute_path,
				action = "copy",
				timestamp = os.time()
			}

			-- ✅ 第四步: 显示成功反馈
			-- 立即显示复制确认信息，包含完整的文件路径
			-- 使用完整路径让用户清楚知道复制了哪个文件
			vim.api.nvim_echo({ { "📋 已复制: " .. node.absolute_path, "Normal" } }, true, {})
		end, { desc = "nvim-tree: 复制当前文件/目录（使用 cp 键）", buffer = bufnr, noremap = true, silent = false, nowait = true })

		-- 📋 <leader>yy 键: Leader 前缀的复制操作
		-- 提供第三种复制方式，使用 leader 键前缀避免与其他插件冲突
		-- 这种方式更明确，适合有多个插件使用 yy 键的环境
		vim.keymap.set("n", "<leader>yy", function()
				-- 🔍 调试信息: 显示 leader 键组合被触发
				vim.api.nvim_echo({ { "🔍 <leader>yy 键被按下了！", "Normal" } }, true, {})

				-- 🧹 第一步: 清空剪贴板确保单文件操作
				-- 先清空剪贴板，确保每次只复制一个文件
				api.fs.clear_clipboard()

				-- 📋 第二步: 调用自定义复制命令
				-- 执行 NvimTreeA1Copy 命令，确保只复制当前文件
				-- 使用 vim.cmd() 调用命令而非直接调用函数，保持一致性
				vim.cmd("NvimTreeA1Copy")

				-- ✅ 第三步: 显示复制确认信息
				-- 获取当前节点信息并显示复制成功消息（显示完整路径）
				local node = api.tree.get_node_under_cursor()
				if node then
					vim.api.nvim_echo({ { "📋 已复制: " .. node.absolute_path, "Normal" } }, true, {})
				end
			end,
			{ desc = "nvim-tree: 复制当前文件/目录（使用 <leader>yy 键）", buffer = bufnr, noremap = true, silent = false, nowait = true })

		-- ============================================================================
		-- 2.3.4 增强剪切操作键位映射
		-- ============================================================================

		-- ✂️ x 键: 调用自定义剪切命令
		-- 将 x 键映射到 NvimTreeA2Cut 命令，实现单文件精确剪切
		-- 剪切操作会将文件移动到剪贴板，粘贴后原文件会从原位置消失
		vim.keymap.set("n", "x", "<cmd>NvimTreeA2Cut<cr>",
			{ desc = "nvim-tree: 剪切文件/目录（单文件操作）", buffer = bufnr, noremap = true, silent = false, nowait = true })

		-- ✂️ mv 键: 内联剪切函数实现
		-- 提供另一个剪切快捷键选择，功能与 x 键相同但实现方式不同
		vim.keymap.set("n", "mv", function()
			-- 🔍 调试信息: 显示按键被触发
			vim.api.nvim_echo({ { "🔍 mv 键被按下了！", "Normal" } }, true, {})

			-- 🎯 第一步: 获取当前光标下的节点
			local node = api.tree.get_node_under_cursor()

			-- 🛡️ 安全检查: 确保有选中的文件
			if not node then
				vim.api.nvim_echo({ { "❌ 未选中任何文件", "WarningMsg" } }, true, {})
				return -- 早期返回，避免无效操作
			end

			-- 🧹 第二步: 清空剪贴板确保单文件操作
			-- 先清空剪贴板，确保每次只剪切一个文件
			-- 剪切操作会将文件标记为"待移动"状态
			api.fs.clear_clipboard()

			-- ✂️ 第三步: 执行剪切操作
			-- api.fs.cut() 将当前节点标记为剪切状态
			-- 与复制不同，剪切后的文件在粘贴时会从原位置移除
			api.fs.cut()

			-- 🔄 记录剪切操作到全局变量
			_G.nvimtree_last_clipboard_action = {
				path = node.absolute_path,
				action = "cut",
				timestamp = os.time()
			}

			-- ✅ 第四步: 显示成功反馈
			-- 显示剪切确认信息（显示完整路径）
			-- 剪切的文件在文件树中通常会以不同颜色显示，表示待移动状态
			vim.api.nvim_echo({ { "✂️ 已剪切: " .. node.absolute_path, "Normal" } }, true, {})
		end, { desc = "nvim-tree: 剪切文件/目录（使用 mv 键）", buffer = bufnr, noremap = true, silent = false, nowait = true })

		-- ============================================================================
		-- 2.3.5 粘贴和清理操作键位映射
		-- ============================================================================

		-- 📋 p 键: 智能粘贴操作
		-- 调用自定义的智能粘贴命令，支持冲突检测和处理
		-- 可以粘贴之前复制或剪切的文件/目录
		vim.keymap.set("n", "p", "<cmd>NvimTreeA3Paste<cr>",
			{ desc = "nvim-tree: 智能粘贴文件/目录", buffer = bufnr, noremap = true, silent = false, nowait = true })

		-- ✏️ r 键: 重命名文件/目录
		-- 调用自定义重命名命令，提供格式化的重命名提示
		vim.keymap.set("n", "r", "<cmd>NvimTreeA5Rename<cr>",
			{ desc = "nvim-tree: 重命名当前文件/目录", buffer = bufnr, noremap = true, silent = false, nowait = true })

		-- 🧹 cc 键: 清空剪贴板
		-- 快速清空 nvim-tree 剪贴板，取消所有待复制/剪切的文件
		vim.keymap.set("n", "cc", function()
			api.fs.clear_clipboard()
			-- 🔄 清除全局跟踪变量
			_G.nvimtree_last_clipboard_action = nil
		end, opts("清空剪贴板"))



		-- ============================================================================
		-- 2.3.6 高级移动功能 (系统命令版本)
		-- ============================================================================

		-- 🚀 tm 键: 自定义移动功能 (使用系统命令)
		-- 这是一个高级功能，使用系统级别的文件移动命令，比内置功能更强大
		-- 特点: 自动创建目标目录、智能冲突检测、详细的操作反馈

		-- ============================================================================
		-- 🎨 在 on_attach 中直接应用 .git 目录颜色
		-- ============================================================================
		-- 延迟应用颜色，确保 NvimTree 完全初始化
		vim.defer_fn(function()
			local highlights = require('configs.highlights')
			highlights.create_git_dir_syntax_highlights()
		end, 100)
		
		-- 再次延迟应用，确保稳定
		vim.defer_fn(function()
			local highlights = require('configs.highlights')
			highlights.create_git_dir_syntax_highlights()
		end, 300)
	end,
})

-- 创建自定义的 NvimTree 打开函数，自动触发 Git 状态刷新
local function nvimtree_toggle_with_git_refresh()
	local api = require("nvim-tree.api")
	
	-- 执行标准的 toggle 操作
	api.tree.toggle()
	
	-- 延迟触发 Git 状态刷新（模拟打开文件的效果）
	vim.defer_fn(function()
		-- 确保当前是 NvimTree 缓冲区
		if vim.bo.filetype == "NvimTree" then
			-- 触发与 BufEnter 相同的刷新逻辑
			local highlights = require('configs.highlights')
			highlights.set_nvim_tree_highlights()
			highlights.ultimate_git_color_fix()
			highlights.force_refresh_nvimtree()
			
			-- 额外强制刷新 Git 状态
			pcall(function()
				require('nvim-tree.git').reload()
				api.tree.reload()
			end)
		end
	end, 150) -- 稍长一点的延迟，确保 NvimTree 完全打开
end

-- 创建用户命令
vim.api.nvim_create_user_command("NvimTreeToggleWithGit", nvimtree_toggle_with_git_refresh, {
	desc = "Toggle NvimTree with automatic Git status refresh"
})

-- 立即触发初始 Git 状态检测，确保打开时就显示正确颜色
vim.defer_fn(function()
	-- 强制立即执行 Git 状态检测
	pcall(function()
		require('nvim-tree.git').reload()
	end)
	
	-- 同时刷新 NvimTree 显示
	pcall(function()
		local api = require("nvim-tree.api")
		api.tree.reload()
	end)
end, 50) -- 很短的延迟，确保 NvimTree 已经初始化

-- ============================================================================
-- 🎨 NvimTree 颜色设置系统
-- ============================================================================



-- ============================================================================

-- 强制 .git 目录显示为灰色的核心功能
-- ============================================================================
local function force_git_directory_gray()
	local view = require('nvim-tree.view')
	if not view.is_visible() then return end
	
	local bufnr = view.get_bufnr()
	if not bufnr or not vim.api.nvim_buf_is_valid(bufnr) then return end
	
	vim.api.nvim_buf_call(bufnr, function()
		local highlights = require('configs.highlights')
		highlights.create_git_dir_syntax_highlights()
		highlights.apply_git_dir_special_highlights()
	end)
end

-- 使用 extmarks 精确高亮 .git 目录为浅灰色（不依赖语法规则，避免被主题覆盖）
local GIT_DIR_NS = vim.api.nvim_create_namespace("NvimTreeGitDirGray")

-- 通过节点 API 精确定位 .git 节点并高亮（优先使用节点，比纯文本匹配更稳健）
local function highlight_git_dir_via_nodes()
	local ok_view, view = pcall(require, 'nvim-tree.view')
	if not ok_view or not view.is_visible() then return end
	local bufnr = view.get_bufnr()
	if not bufnr or not vim.api.nvim_buf_is_valid(bufnr) then return end
	local ok_lib, lib = pcall(require, 'nvim-tree.lib')
	if not ok_lib or not lib then return end
	vim.api.nvim_buf_clear_namespace(bufnr, GIT_DIR_NS, 0, -1)
	local line_count = vim.api.nvim_buf_line_count(bufnr)
	for lnum = 1, line_count do
		local ok_node, node = pcall(lib.get_node_at_line, lnum)
		if ok_node and node and node.name then
			local name = node.name
			local path = node.absolute_path or ""
			if name == ".git" or path:match("/%.git/?$") then
				vim.api.nvim_buf_set_extmark(bufnr, GIT_DIR_NS, lnum - 1, 0, {
					hl_group = "NvimTreeGitDirGray",
					hl_eol = true,
					priority = 12000,
					hl_mode = "replace",
				})
			end
		end
	end
end

-- 使用文本扫描作为兜底（当节点 API 不可用或行未映射时）
local function highlight_git_dir_via_extmarks()
	local ok_view, view = pcall(require, 'nvim-tree.view')
	if not ok_view then return end
	if not view.is_visible() then return end
	local bufnr = view.get_bufnr()
	if not bufnr or not vim.api.nvim_buf_is_valid(bufnr) then return end
	-- 清除旧的标记
	vim.api.nvim_buf_clear_namespace(bufnr, GIT_DIR_NS, 0, -1)
	-- 遍历行，匹配 .git 目录
	local lines = vim.api.nvim_buf_get_lines(bufnr, 0, -1, false)
	for i, line in ipairs(lines) do
		if (line:find("%.git[/%s]") or line:find("%.git$")) and not line:find("%.gitignore") and not line:find("%.gitattributes") then
			vim.api.nvim_buf_set_extmark(bufnr, GIT_DIR_NS, i - 1, 0, {
				hl_group = "NvimTreeGitDirGray",
				hl_eol = true,
				priority = 10000,
				hl_mode = "replace",
			})
		end
	end
end

-- 确保用于 extmark 的灰色高亮组存在
local function ensure_git_dir_gray_hl()
	pcall(function()
		vim.api.nvim_set_hl(0, "NvimTreeGitDirGray", { fg = "#8a8a8a" })
	end)
end

local function schedule_git_highlight_burst()
	ensure_git_dir_gray_hl()
	-- 先用节点高亮，失败再用文本兜底
	local ok = pcall(highlight_git_dir_via_nodes)
	if not ok then
		highlight_git_dir_via_extmarks()
	end
	vim.defer_fn(function()
		ensure_git_dir_gray_hl()
		local ok2 = pcall(highlight_git_dir_via_nodes)
		if not ok2 then
			highlight_git_dir_via_extmarks()
		end
	end, 40)
	vim.defer_fn(function()
		ensure_git_dir_gray_hl()
		local ok3 = pcall(highlight_git_dir_via_nodes)
		if not ok3 then
			highlight_git_dir_via_extmarks()
		end
	end, 120)
	vim.defer_fn(function()
		ensure_git_dir_gray_hl()
		local ok4 = pcall(highlight_git_dir_via_nodes)
		if not ok4 then
			highlight_git_dir_via_extmarks()
		end
	end, 260)
end

local git_gray_timer = nil
local function start_git_gray_timer()
	if git_gray_timer then
		pcall(function()
			git_gray_timer:stop(); git_gray_timer:close()
		end)
		git_gray_timer = nil
	end
	local ok_loop, loop = pcall(function() return vim.loop.new_timer() end)
	if not ok_loop or not loop then return end
	git_gray_timer = loop
	local ticks = 0
	git_gray_timer:start(0, 180, vim.schedule_wrap(function()
		ticks = ticks + 1
		ensure_git_dir_gray_hl()
		local okn = pcall(highlight_git_dir_via_nodes)
		if not okn then
			highlight_git_dir_via_extmarks()
		end
		if ticks >= 60 then
			pcall(function()
				git_gray_timer:stop(); git_gray_timer:close()
			end)
			git_gray_timer = nil
		end
	end))
end

-- 创建智能目录着色系统
vim.defer_fn(function()
	-- 立即强制应用 .git 目录颜色
	force_git_directory_gray()
	-- 同步应用基于 .gitignore 规则的目录着色（.git 已被强制纳入忽略）
	pcall(function()
		require('configs.nvimtree').smart_color_directories()
	end)
	-- 使用 extmark 再覆写一次，确保最终呈现
	vim.schedule(highlight_git_dir_via_extmarks)

	-- 创建自动命令来动态着色不同类型的目录
	vim.api.nvim_create_augroup("NvimTreeSmartDirColors", { clear = true })
	
	vim.api.nvim_create_autocmd("FileType", {
		pattern = "NvimTree",
		group = "NvimTreeSmartDirColors",
		callback = function()
			-- 立即应用一次 .git 目录语法高亮
			vim.schedule(function()
				force_git_directory_gray()
				highlight_git_dir_via_extmarks()
				-- 仅依赖 special 高亮，不做强制匹配
			end)
			-- 多次延迟应用，确保在不同时机都能生效
			for i = 1, 3 do
				vim.defer_fn(function()
					pcall(function()
						local highlights = require('configs.highlights')
						highlights.apply_git_dir_special_highlights()
					end)
					pcall(function()
						require('configs.nvimtree').smart_color_directories()
					end)
					highlight_git_dir_via_extmarks()
					-- 仅依赖 special 高亮，不做强制匹配
				end, i * 100)
			end
		end
	})
	
	vim.api.nvim_create_autocmd("BufEnter", {
		group = "NvimTreeSmartDirColors", 
		callback = function()
			if vim.bo.filetype == "NvimTree" then
				-- 安全检查配置是否存在，如果不存在则跳过（避免作用域错误）
				local config = _G.nvimtree_git_config
				if not config or not config.enable_focus_refresh then
					return
				end
				
				-- 检查是否正在编辑新文件，如果是则延迟处理
				local current_buf_file = vim.fn.expand("#:p") -- 上一个buffer的文件路径
				if config.optimize_new_files and should_delay_git_for_new_file(current_buf_file) then
					-- 延迟Git检测，避免新文件的卡顿
					vim.defer_fn(function()
						if _G.nvimtree_new_files and _G.nvimtree_new_files[current_buf_file] then
							_G.nvimtree_new_files[current_buf_file].checked = true
						end
						-- 执行延迟的Git检测
						pcall(function()
							local highlights = require('configs.highlights')
							highlights.apply_git_dir_special_highlights()
						end)
					end, 2000) -- 2秒后执行
					return
				end
				
				-- 防抖机制
				local focus_time = _G.nvimtree_last_focus_time or 0
				local current_time = vim.loop.hrtime()
				if (current_time - focus_time) < 500000000 then
					return
				end
				_G.nvimtree_last_focus_time = current_time
				-- 立即应用一次 .git 目录语法高亮
				vim.schedule(function()
					pcall(function()
						local highlights = require('configs.highlights')
						highlights.apply_git_dir_special_highlights()
					end)
					highlight_git_dir_via_extmarks()
				end)
				-- 延迟应用确保 .git 匹配稳定
				vim.defer_fn(function()
					pcall(function()
						local highlights = require('configs.highlights')
						highlights.apply_git_dir_special_highlights()
					end)
					pcall(function()
						require('configs.nvimtree').smart_color_directories()
					end)
					highlight_git_dir_via_extmarks()
				end, 50)
			end
		end
	})
	
	-- 优化的窗口切换事件，使用防抖机制
	local last_win_enter = 0
	vim.api.nvim_create_autocmd("WinEnter", {
		group = "NvimTreeSmartDirColors",
		callback = function()
			local buf_name = vim.api.nvim_buf_get_name(0)
			if buf_name:match("NvimTree_") then
				-- 安全检查配置是否存在
				local config = _G.nvimtree_git_config
				if not config or not config.enable_focus_refresh then
					return
				end
				
				-- 检查是否正在编辑新文件，如果是则延迟处理
				local current_buf_file = vim.fn.expand("#:p") -- 上一个buffer的文件路径
				if config.optimize_new_files and should_delay_git_for_new_file(current_buf_file) then
					-- 延迟Git检测，避免新文件的卡顿
					vim.defer_fn(function()
						if _G.nvimtree_new_files and _G.nvimtree_new_files[current_buf_file] then
							_G.nvimtree_new_files[current_buf_file].checked = true
						end
						-- 执行延迟的Git检测
						pcall(function()
							vim.cmd("silent! syntax clear NvimTreeGitDirBuf")
							vim.cmd([[syntax match NvimTreeGitDirBuf /^\s*.*\.git\s*$/ contains=ALL]])
							vim.cmd("highlight NvimTreeGitDirBuf guifg=#8a8a8a ctermfg=245")
						end)
					end, 2000) -- 2秒后执行
					return
				end
				
				local focus_time = _G.nvimtree_last_focus_time or 0
				local current_time = vim.loop.hrtime()
				-- 防抖：500ms内只执行一次，延长防抖时间
				if (current_time - focus_time) < 500000000 then
					return
				end
				_G.nvimtree_last_focus_time = current_time
				
				vim.schedule(function()
					pcall(function()
						vim.cmd("silent! syntax clear NvimTreeGitDirBuf")
						vim.cmd([[syntax match NvimTreeGitDirBuf /^\s*.*\.git\s*$/ contains=ALL]])
						vim.cmd("highlight NvimTreeGitDirBuf guifg=#8a8a8a ctermfg=245")
					end)
					highlight_git_dir_via_extmarks()
				end)
			end
		end
	})
	
	-- 缓冲区加载完成时应用颜色
	vim.api.nvim_create_autocmd("BufReadPost", {
		pattern = "NvimTree_*",
		group = "NvimTreeSmartDirColors",
		callback = function()
			vim.schedule(function()
				pcall(function()
					vim.cmd("silent! syntax clear NvimTreeGitDirWin")
					vim.cmd([[syntax match NvimTreeGitDirWin /^\s*.*\.git\s*$/ contains=ALL]])
					vim.cmd("highlight NvimTreeGitDirWin guifg=#8a8a8a ctermfg=245")
				end)
				highlight_git_dir_via_extmarks()
			end)
			vim.defer_fn(function()
				pcall(function()
					vim.cmd("silent! syntax clear NvimTreeGitDirWin2")
					vim.cmd([[syntax match NvimTreeGitDirWin2 /^\s*.*\.git\s*$/ contains=ALL]])
					vim.cmd("highlight NvimTreeGitDirWin2 guifg=#8a8a8a ctermfg=245")
				end)
				highlight_git_dir_via_extmarks()
			end, 100)
		end
	})
end, 150)

-- 保持正常目录图标的颜色不变

-- 为被忽略的内容创建链接，确保一致的灰色显示
vim.cmd("highlight! link NvimTreeGitIgnoredFolder NvimTreeGitIgnored")
vim.cmd("highlight! link NvimTreeGitIgnoredFolderName NvimTreeGitIgnored")
vim.cmd("highlight! link NvimTreeGitIgnoredFolderIcon NvimTreeGitIgnored")

-- 只设置被忽略文件的灰色，正常文件保持默认主题颜色

-- 创建强力的主题覆盖机制来确保 .git 目录显示为浅灰色
vim.defer_fn(function()
	-- 创建持续的自动命令来对抗主题覆盖
	vim.api.nvim_create_augroup("NvimTreeForceGitColor", { clear = true })
	
	-- 主要触发事件
	vim.api.nvim_create_autocmd(
		{ "FileType", "BufEnter", "BufWinEnter", "ColorScheme", "UIEnter" }, {
			group = "NvimTreeForceGitColor",
			callback = function()
				if vim.bo.filetype == "NvimTree" then
					vim.cmd("highlight! NvimTreeSpecialDir guifg=#8a8a8a guibg=NONE gui=NONE ctermfg=245")
				end
			end
		})
		
		-- 移除CursorMoved事件监听，避免光标移动时的卡顿
		-- 原来的事件监听会在每次光标移动时触发，这是导致卡顿的主要原因
		-- 现在只依赖其他事件来更新高亮，保证光标移动的流畅性
		
		-- 立即应用一次
		vim.cmd("highlight! NvimTreeSpecialDir guifg=#8a8a8a guibg=NONE gui=NONE ctermfg=245")
		
		-- 创建定时器持续强制应用（对抗主题的持续覆盖）
		local git_color_timer = vim.loop.new_timer()
		if git_color_timer then
			git_color_timer:start(1000, 2000, vim.schedule_wrap(function()
				-- 每2秒检查并强制应用一次
				if vim.bo.filetype == "NvimTree" then
					pcall(function()
						vim.cmd("silent! highlight! NvimTreeSpecialDir guifg=#8a8a8a")
					end)
				end
			end))
		end
	end, 800)

-- 强制刷新 NvimTree 显示的函数
local function force_refresh_nvimtree()
	pcall(function()
		local api = require('nvim-tree.api')
		if api.tree.is_visible() then
			api.tree.reload()
		end
	end)
end

-- 立即应用颜色设置
vim.schedule(function()
    local highlights = require('configs.highlights')
    highlights.apply_git_colors_after_setup()
end)

-- ============================================================================
-- 3. 自定义用户命令系统
-- ============================================================================
--
-- 📋 自定义命令设计理念:
-- 本配置文件创建了一套完整的自定义命令系统，扩展了 nvim-tree 的默认功能
-- 命令按功能分类，使用 A1-A8 前缀进行排序，确保在命令补全中按逻辑顺序显示
--
-- 🏗️ 命令命名规范:
--   • NvimTreeA1xxx - 复制相关操作 (Copy)
--   • NvimTreeA2xxx - 剪切相关操作 (Cut)
--   • NvimTreeA3xxx - 粘贴相关操作 (Paste)
--   • NvimTreeA4xxx - 移动相关操作 (Move)
--   • NvimTreeA5xxx - 重命名操作 (Rename)
--   • NvimTreeA6xxx - 删除和垃圾箱操作 (Delete/Trash)
--   • NvimTreeA7xxx - 清理操作 (Clear)
--   • NvimTreeA8xxx - 导航操作 (Navigate)
--
-- 🔄 批量操作设计:
--   • 单文件操作: 直接在当前光标文件上执行
--   • Mark 批量操作: 先用 'm' 键标记多个文件，再执行批量命令
--   • 智能冲突处理: 自动检测文件冲突并提供解决方案
--   • 中文化交互: 所有用户交互都使用中文提示

-- ============================================================================
-- 3.1 复制操作命令 (A1 系列)
-- ============================================================================
-- 📋 NvimTreeA1Copy - 单文件精确复制命令
--
-- 🎯 功能说明:
--   复制当前光标下的单个文件或目录到 nvim-tree 剪贴板
--   与默认复制不同，此命令确保每次只复制一个文件，避免意外的批量操作
--
-- 📖 使用方法:
--   1. 将光标移动到要复制的文件/目录上
--   2. 执行 :NvimTreeA1Copy 或按键位映射 'yy'
--   3. 导航到目标位置并执行粘贴操作
--
-- ⚡ 技术特点:
--   • 操作前自动清空剪贴板，确保不会复制到意外文件
--   • 支持文件和目录的复制
--   • 显示完整路径的确认信息
--   • 使用延迟显示避免界面阻塞
--
-- 🔗 相关命令: NvimTreeA3Paste (粘贴), NvimTreeA7ClearClipboard (清空剪贴板)
vim.api.nvim_create_user_command("NvimTreeA1Copy", function()
	-- 📦 第一步: 加载 nvim-tree API
	-- 每次执行命令时重新加载 API，确保使用最新的接口
	local api = require("nvim-tree.api")

	-- 🧹 第二步: 清空剪贴板确保单文件操作
	-- api.fs.clear_clipboard() 清除 nvim-tree 内部剪贴板中的所有内容
	-- 这确保每次复制操作都是全新的，不会受到之前操作的影响
	-- 避免意外的批量复制或混合操作
	api.fs.clear_clipboard()

	-- 🎯 第三步: 获取当前选中的文件/目录节点
	-- api.tree.get_node_under_cursor() 返回光标当前位置的文件系统节点
	-- 节点包含文件的所有信息: 名称、路径、类型、大小、权限等
	local node = api.tree.get_node_under_cursor()

	-- 📋 第四步: 执行核心复制操作
	-- api.fs.copy.node() 将当前节点复制到 nvim-tree 的内部剪贴板
	-- 这个操作只是将文件信息添加到剪贴板，并不移动或修改原文件
	-- 实际的文件复制会在粘贴(paste)时进行
	api.fs.copy.node()

	-- 🔄 记录复制操作到全局变量，用于同目录检测
	if node then
		_G.nvimtree_last_clipboard_action = {
			path = node.absolute_path,
			action = "copy",
			timestamp = os.time()
		}
	end

	-- ✅ 第五步: 异步显示用户反馈
	-- 使用 vim.defer_fn() 延迟执行反馈显示，避免与复制操作产生时序冲突
	-- 延迟 100 毫秒足够让内部操作完成，但用户感受不到明显延迟
	vim.defer_fn(function()
		-- 🏷️ 根据节点类型提供不同的反馈信息
		-- 检查节点是否存在且获取其类型信息
		if node then
			if node.type == "directory" then
				-- 📁 目录复制反馈
				local dir_path = vim.fn.fnamemodify(node.absolute_path, ":h")
				local dirname = vim.fn.fnamemodify(node.absolute_path, ":t")
				vim.api.nvim_echo({
					{ "📋 目录已复制: " .. dir_path .. "/", "Normal" },
					{ dirname, "Function" }
				}, true, {})
			else
				-- 📄 文件复制反馈
				local file_dir = vim.fn.fnamemodify(node.absolute_path, ":h")
				local filename = vim.fn.fnamemodify(node.absolute_path, ":t")
				vim.api.nvim_echo({
					{ "📋 文件已复制: " .. file_dir .. "/", "Normal" },
					{ filename, "Function" }
				}, true, {})
			end
		end
	end, 100) -- 100ms 延迟，平衡响应速度和操作稳定性
end, { desc = "复制当前文件/目录" })

vim.api.nvim_create_user_command("NvimTreeA1MarkCopy", function()
	local api = require("nvim-tree.api")
	local marks = api.marks.list()

	if #marks == 0 then
		show_message("❌ 没有标记的文件", vim.log.levels.WARN)
		return
	end

	-- 首先清空剪贴板
	api.fs.clear_clipboard()

	-- 🔄 记录批量复制操作到全局变量，支持多文件跟踪
	local mark_paths = {}
	for _, mark in ipairs(marks) do
		table.insert(mark_paths, mark.absolute_path)
	end
	_G.nvimtree_last_clipboard_action = {
		paths = mark_paths, -- 使用 paths 数组记录多个文件
		action = "copy",
		timestamp = os.time(),
		is_batch = true -- 标记为批量操作
	}

	local success_count = 0
	local failed_files = {}

	-- 逐个复制标记的文件
	for i, mark in ipairs(marks) do
		-- 检查文件是否存在
		if vim.fn.filereadable(mark.absolute_path) == 1 or vim.fn.isdirectory(mark.absolute_path) == 1 then
			-- 尝试找到节点并复制
			local copy_success, copy_error = pcall(function()
				api.tree.find_file(mark.absolute_path)
				api.fs.copy.node()
			end)

			if copy_success then
				success_count = success_count + 1
			else
				table.insert(failed_files, mark.absolute_path .. " (复制失败: " .. tostring(copy_error) .. ")")
			end
		else
			table.insert(failed_files, mark.absolute_path .. " (文件不存在)")
		end
	end

	-- 显示复制结果，使用延迟显示避免阻塞
	vim.defer_fn(function()
		-- 主要成功信息
		vim.api.nvim_echo({ { "📋 已复制 " .. success_count .. "/" .. #marks .. " 个标记的文件到剪贴板", "Normal" } }, true, {})

		-- 如果有失败的文件，分别显示（避免多行文本导致Press ENTER）
		if #failed_files > 0 then
			vim.defer_fn(function()
				for _, failed in ipairs(failed_files) do
					vim.api.nvim_echo({ { "❌ " .. failed, "WarningMsg" } }, true, {})
				end
			end, 200)
		end
	end, 100)
end, { desc = "批量复制标记的文件" })

-- 2. 剪切相关命令
vim.api.nvim_create_user_command("NvimTreeA2Cut", function()
	local api = require("nvim-tree.api")
	-- 先清空剪贴板，确保每次只能操作一个文件
	api.fs.clear_clipboard()
	-- 获取当前节点信息
	local node = api.tree.get_node_under_cursor()
	-- 只剪切当前文件
	api.fs.cut()

	-- 🔄 记录剪切操作到全局变量，用于同目录检测
	if node then
		_G.nvimtree_last_clipboard_action = {
			path = node.absolute_path,
			action = "cut",
			timestamp = os.time()
		}
	end
	-- 使用延迟提示，避免阻塞，并根据类型显示不同信息（显示完整路径）
	vim.defer_fn(function()
		if node and node.type == "directory" then
			vim.api.nvim_echo({ { "✂️ 目录已剪切: " .. node.absolute_path, "Normal" } }, true, {})
		else
			vim.api.nvim_echo({ { "✂️ 文件已剪切: " .. node.absolute_path, "Normal" } }, true, {})
		end
	end, 100)
end, { desc = "剪切当前文件/目录" })

vim.api.nvim_create_user_command("NvimTreeA2MarkCut", function()
	local api = require("nvim-tree.api")
	local marks = api.marks.list()

	if #marks == 0 then
		show_message("❌ 没有标记的文件", vim.log.levels.WARN)
		return
	end

	-- 首先清空剪贴板
	api.fs.clear_clipboard()

	-- 🔄 记录批量剪切操作到全局变量，支持多文件跟踪
	local mark_paths = {}
	for _, mark in ipairs(marks) do
		table.insert(mark_paths, mark.absolute_path)
	end
	_G.nvimtree_last_clipboard_action = {
		paths = mark_paths, -- 使用 paths 数组记录多个文件
		action = "cut",
		timestamp = os.time(),
		is_batch = true -- 标记为批量操作
	}

	local success_count = 0
	local failed_files = {}

	-- 逐个剪切标记的文件
	for i, mark in ipairs(marks) do
		-- 检查文件是否存在
		if vim.fn.filereadable(mark.absolute_path) == 1 or vim.fn.isdirectory(mark.absolute_path) == 1 then
			-- 尝试找到节点并剪切
			local cut_success, cut_error = pcall(function()
				api.tree.find_file(mark.absolute_path)
				api.fs.cut()
			end)

			if cut_success then
				success_count = success_count + 1
			else
				table.insert(failed_files, mark.absolute_path .. " (剪切失败: " .. tostring(cut_error) .. ")")
			end
		else
			table.insert(failed_files, mark.absolute_path .. " (文件不存在)")
		end
	end

	-- 显示剪切结果，使用延迟显示避免阻塞
	vim.defer_fn(function()
		-- 主要成功信息
		vim.api.nvim_echo({ { "✂️ 已剪切 " .. success_count .. "/" .. #marks .. " 个标记的文件到剪贴板", "Normal" } }, true, {})

		-- 如果有失败的文件，分别显示（避免多行文本导致Press ENTER）
		if #failed_files > 0 then
			vim.defer_fn(function()
				for _, failed in ipairs(failed_files) do
					vim.api.nvim_echo({ { "❌ " .. failed, "WarningMsg" } }, true, {})
				end
			end, 200)
		end
	end, 100)
end, { desc = "批量剪切标记的文件" })

-- 3. 粘贴相关命令
-- 🍎 NvimTreeA3Paste - 智能粘贴命令 (完整版本)
--
-- 🎯 功能说明:
--   完整的智能粘贴功能，支持单文件和批量文件操作
--   包含同目录检测、逐个文件冲突处理、用户交互界面
--
-- 🔧 核心特性:
--   • 支持单文件和批量文件粘贴
--   • 同目录粘贴检测和阻止（批量操作检测每个文件的源目录）
--   • 智能文件冲突处理：对每个冲突文件提供覆盖/重命名/取消三选一
--   • 自定义重命名功能：用户输入新文件名，支持冲突检测
--   • 操作统计和详细反馈
--
-- 📖 使用方法:
--   1. 先使用复制(y)或剪切(x)功能，或批量复制/剪切选择文件
--   2. 导航到目标目录
--   3. 将光标放在目标目录或目录内的文件上
--   4. 执行 :NvimTreeA3Paste 或按 'p' 键
--   5. 根据提示处理各种情况（批量操作时逐个处理每个冲突文件）
vim.api.nvim_create_user_command("NvimTreeA3Paste", function()
	local api = require("nvim-tree.api")

	-- 🎯 步骤1: 智能计算目标目录
	local function get_target_directory()
		local node = api.tree.get_node_under_cursor()
		local target_dir

		if node then
			if node.type == "directory" then
				target_dir = node.absolute_path
			else
				target_dir = vim.fn.fnamemodify(node.absolute_path, ":h")
			end
		else
			target_dir = vim.fn.getcwd()
		end

		target_dir = vim.fn.resolve(target_dir):gsub("/$", "")
		return target_dir
	end

	-- 🔄 步骤2: 处理单个文件的冲突（使用全局函数）

	local target_dir = get_target_directory()

	-- 🔍 检查是否有剪贴板操作记录
	if not _G.nvimtree_last_clipboard_action then
		-- 如果没有全局变量，尝试使用 nvim-tree 原生粘贴
		vim.defer_fn(function()
			vim.api.nvim_echo({ { "❌ 没有复制或剪切的文件", "WarningMsg" } }, true, {})
		end, 10)
		return
	end

	local clipboard = _G.nvimtree_last_clipboard_action
	local is_batch = clipboard.is_batch or false
	local action = clipboard.action

	-- 获取文件路径列表（单文件或多文件）
	local file_paths = {}
	if is_batch and clipboard.paths then
		file_paths = clipboard.paths
	elseif clipboard.path then
		table.insert(file_paths, clipboard.path)
	else
		vim.defer_fn(function()
			vim.api.nvim_echo({ { "❌ 剪贴板数据无效", "ErrorMsg" } }, true, {})
		end, 10)
		return
	end

	-- 🔍 第一阶段：同目录检测（批量操作时检测每个文件）
	local same_dir_files = {}
	for _, source_path in ipairs(file_paths) do
		local source_dir = vim.fn.resolve(vim.fn.fnamemodify(source_path, ":h")):gsub("/$", "")
		local current_target = vim.fn.resolve(target_dir):gsub("/$", "")

		if source_dir == current_target then
			table.insert(same_dir_files, source_path)
		end
	end

	if #same_dir_files > 0 then
		local file_list = {}
		local dir_list = {}

		for _, path in ipairs(same_dir_files) do
			local filename = vim.fn.fnamemodify(path, ":t")
			if vim.fn.isdirectory(path) == 1 then
				table.insert(dir_list, filename)
			else
				table.insert(file_list, filename)
			end
		end

		vim.defer_fn(function()
			local msg_parts = {}
			if #file_list > 0 then
				table.insert(msg_parts, "文件 " .. table.concat(file_list, ", "))
			end
			if #dir_list > 0 then
				table.insert(msg_parts, "目录 " .. table.concat(dir_list, ", "))
			end

			local msg = "🚫 不能在同目录内执行粘贴操作，已跳过: " .. table.concat(msg_parts, "，") .. " | 剪贴板已清除"
			vim.api.nvim_echo({ { msg, "ErrorMsg" } }, true, {})
		end, 10)

		-- 移除同目录文件，继续处理其他文件
		local filtered_paths = {}
		for _, source_path in ipairs(file_paths) do
			local source_dir = vim.fn.resolve(vim.fn.fnamemodify(source_path, ":h")):gsub("/$", "")
			local current_target = vim.fn.resolve(target_dir):gsub("/$", "")
			if source_dir ~= current_target then
				table.insert(filtered_paths, source_path)
			end
		end
		file_paths = filtered_paths
	end

	if #file_paths == 0 then
		-- 所有文件都在同目录，清除剪贴板后返回
		api.marks.clear()
		api.fs.clear_clipboard()
		_G.nvimtree_last_clipboard_action = nil
		return
	end

	-- 🔄 第二阶段：逐个处理文件（冲突检测和处理）
	local results = {
		success = 0,
		overwrite = 0,
		renamed = 0,
		cancelled = 0,
		rename_conflict = 0, -- 新增：重命名时遇到同名冲突的情况
		failed = 0
	}
	local renamed_files = {} -- 跟踪重命名后的文件路径

	for i, source_path in ipairs(file_paths) do
		-- 显示处理进度（批量操作时）
		if is_batch and #file_paths > 1 then
			vim.cmd("redraw")
			vim.api.nvim_echo({
				{ "📁 处理文件 " .. i .. "/" .. #file_paths .. ": " .. vim.fn.fnamemodify(source_path, ":t"), "Normal" }
			}, true, {})
			vim.cmd("sleep 100m") -- 暂停100ms让用户看到进度
		end

		local result, renamed_path = handle_single_file_conflict(source_path, target_dir, action)
		results[result] = results[result] + 1
		if result == "renamed" and renamed_path then
			table.insert(renamed_files, renamed_path)
		end
	end

	-- 🎉 第三阶段：显示操作结果汇总
	api.tree.reload()
	vim.cmd("redraw")

	-- 刷新可能受影响的缓冲区
	for _, buf in ipairs(vim.api.nvim_list_bufs()) do
		local buf_name = vim.api.nvim_buf_get_name(buf)
		if buf_name:find(target_dir, 1, true) and vim.api.nvim_buf_is_loaded(buf) then
			vim.api.nvim_buf_call(buf, function()
				vim.cmd("checktime")
				vim.cmd("edit!")
			end)
		end
	end

	-- 立即显示操作结果（不使用 defer_fn）
	local total = results.success + results.overwrite + results.renamed + results.cancelled + results.rename_conflict +
		results.failed
	local success_total = results.success + results.overwrite + results.renamed

	-- 如果没有任何操作结果，使用默认消息
	if total == 0 then
		vim.api.nvim_echo({ { "📁 粘贴完成: " .. target_dir .. " | 剪贴板已清除", "Normal" } }, true, {})
		-- 清除操作
		api.marks.clear()
		api.fs.clear_clipboard()
		_G.nvimtree_last_clipboard_action = nil
		return
	end

	-- 构建紧凑的一行提示信息
	local msg = "📁 " .. target_dir .. " | "

	-- 构建统计信息（使用更紧凑的格式）
	local stats = {}
	if results.success > 0 then table.insert(stats, "✅" .. results.success) end
	if results.overwrite > 0 then table.insert(stats, "🔄" .. results.overwrite) end
	if results.renamed > 0 then table.insert(stats, "📝" .. results.renamed) end
	if results.cancelled > 0 then table.insert(stats, "❌" .. results.cancelled) end
	if results.rename_conflict > 0 then table.insert(stats, "⚠️" .. results.rename_conflict) end
	if results.failed > 0 then table.insert(stats, "❗" .. results.failed) end

	-- 添加统计信息
	-- 🧹 执行清除操作
	-- 清除所有标记
	api.marks.clear()
	-- 清除剪贴板
	api.fs.clear_clipboard()
	-- 清除全局跟踪变量
	_G.nvimtree_last_clipboard_action = nil

	-- 延迟显示粘贴结果避免回车提示
	if results.success > 0 or results.overwrite > 0 or results.renamed > 0 or results.cancelled > 0 then
		vim.defer_fn(function()
			if #file_paths == 1 then
				-- 单文件操作，显示具体文件路径
				local msg = ""
				local target_file = ""
				local source_filename = vim.fn.fnamemodify(file_paths[1], ":t")
				target_file = target_dir .. "/" .. source_filename

				if results.success > 0 then
					msg = "📋 文件已粘贴: " .. target_file .. " | 剪贴板已清除"
				elseif results.overwrite > 0 then
					msg = "📋 文件已覆盖: " .. target_file .. " | 剪贴板已清除"
				elseif results.renamed > 0 then
					-- 对于重命名，显示最终的重命名文件路径
					if #renamed_files > 0 then
						msg = "📋 文件已重命名粘贴: " .. renamed_files[1] .. " | 剪贴板已清除"
					else
						msg = "📋 文件已重命名粘贴: " .. target_dir .. " | 剪贴板已清除"
					end
				elseif results.cancelled > 0 then
					msg = "📋 操作已取消 | 剪贴板已清除"
				end

				vim.api.nvim_echo({ { msg, "Normal" } }, true, {})
			else
				-- 多文件操作，显示总结信息
				local summary_parts = {}
				if results.success > 0 then
					table.insert(summary_parts, "粘贴了" .. results.success .. "个文件")
				end
				if results.overwrite > 0 then
					table.insert(summary_parts, "覆盖了" .. results.overwrite .. "个文件")
				end
				if results.renamed > 0 then
					table.insert(summary_parts, "重命名粘贴了" .. results.renamed .. "个文件")
				end
				if results.cancelled > 0 then
					table.insert(summary_parts, "取消了" .. results.cancelled .. "个文件")
				end
				if results.failed > 0 then
					table.insert(summary_parts, "失败了" .. results.failed .. "个文件")
				end

				local summary_msg = "📋 批量粘贴完成: " ..
					table.concat(summary_parts, "，") .. " -> " .. target_dir .. " | 剪贴板已清除"
				vim.api.nvim_echo({ { summary_msg, "Normal" } }, true, {})
			end
		end, 100)
	end
end, { desc = "智能粘贴剪贴板中的文件/目录（简化版本）" })

-- 移动执行函数（带冲突处理）
local function execute_move_with_conflict_handling(api, marks, target_dir)
	local results = {
		success = 0,
		overwrite = 0,
		renamed = 0,
		cancelled = 0,
		rename_conflict = 0,
		failed = 0
	}
	local renamed_files = {}

	-- 🎯 第一阶段：逐个处理文件移动
	for i, mark in ipairs(marks) do
		-- 显示处理进度（批量操作时）
		if #marks > 1 then
			vim.cmd("redraw")
			vim.api.nvim_echo({
				{ "📂 处理文件 " .. i .. "/" .. #marks .. ": " .. vim.fn.fnamemodify(mark.absolute_path, ":t"), "Normal" }
			}, true, {})
			vim.cmd("sleep 100m") -- 暂停100ms让用户看到进度
		end

		local result, renamed_path = handle_single_file_conflict(mark.absolute_path, target_dir, "move")
		results[result] = results[result] + 1
		if result == "renamed" and renamed_path then
			table.insert(renamed_files, renamed_path)
		end
	end

	-- 🎉 第二阶段：显示操作结果汇总
	api.tree.reload()
	vim.cmd("redraw")

	-- 刷新可能受影响的缓冲区
	for _, buf in ipairs(vim.api.nvim_list_bufs()) do
		local buf_name = vim.api.nvim_buf_get_name(buf)
		if buf_name:find(target_dir, 1, true) and vim.api.nvim_buf_is_loaded(buf) then
			vim.api.nvim_buf_call(buf, function()
				vim.cmd("checktime")
				vim.cmd("edit!")
			end)
		end
	end

	-- 立即显示操作结果（不使用 defer_fn）
	local total = results.success + results.overwrite + results.renamed + results.cancelled + results.rename_conflict +
		results.failed
	local success_total = results.success + results.overwrite + results.renamed

	-- 如果没有任何操作结果，使用默认消息
	if total == 0 then
		vim.api.nvim_echo({ { "❌ 没有文件被移动", "WarningMsg" } }, true, {})
		-- 清除操作
		api.marks.clear()
		return
	end

	-- 延迟显示移动结果避免回车提示
	if results.success > 0 or results.overwrite > 0 or results.renamed > 0 or results.cancelled > 0 then
		vim.defer_fn(function()
			if #marks == 1 then
				-- 单文件操作，显示具体文件路径
				local msg = ""
				local target_file = ""
				local source_filename = vim.fn.fnamemodify(marks[1].absolute_path, ":t")
				target_file = target_dir .. "/" .. source_filename

				if results.success > 0 then
					msg = "📂 文件已移动: " .. target_file
				elseif results.overwrite > 0 then
					msg = "📂 文件已覆盖移动: " .. target_file
				elseif results.renamed > 0 then
					-- 对于重命名，显示最终的重命名文件路径
					if #renamed_files > 0 then
						msg = "📂 文件已重命名移动: " .. renamed_files[1]
					else
						msg = "📂 文件已重命名移动: " .. target_dir
					end
				elseif results.cancelled > 0 then
					msg = "📂 移动已取消"
				end

				vim.api.nvim_echo({ { msg, "Normal" } }, true, {})
			else
				-- 多文件操作，显示总结信息
				local summary_parts = {}
				if results.success > 0 then
					table.insert(summary_parts, "移动了" .. results.success .. "个文件")
				end
				if results.overwrite > 0 then
					table.insert(summary_parts, "覆盖了" .. results.overwrite .. "个文件")
				end
				if results.renamed > 0 then
					table.insert(summary_parts, "重命名移动了" .. results.renamed .. "个文件")
				end
				if results.cancelled > 0 then
					table.insert(summary_parts, "取消了" .. results.cancelled .. "个文件")
				end
				if results.failed > 0 then
					table.insert(summary_parts, "失败了" .. results.failed .. "个文件")
				end

				local summary_msg = "📂 批量移动完成: " .. table.concat(summary_parts, "，") .. " -> " .. target_dir
				vim.api.nvim_echo({ { summary_msg, "Normal" } }, true, {})
			end
		end, 100)
	end

	-- 🧹 执行清除操作
	-- 清除所有标记
	api.marks.clear()
end

-- 4. 移动相关命令
vim.api.nvim_create_user_command("NvimTreeA4MarkMove", function()
	-- 清除命令行显示
	vim.cmd("redraw")

	local api = require("nvim-tree.api")

	-- 获取所有标记的文件
	local marks = api.marks.list()
	if #marks == 0 then
		vim.defer_fn(function()
			vim.api.nvim_echo({ { "❌ 没有标记的文件", "WarningMsg" } }, true, {})
		end, 100)
		return
	end

	-- 获取当前目录作为默认值，添加斜杠
	local current_node = api.tree.get_node_under_cursor()
	local default_dir = ""
	if current_node then
		local base_dir = current_node.type == "file" and vim.fn.fnamemodify(current_node.absolute_path, ":h") or
			current_node.absolute_path
		-- 直接使用当前目录
		default_dir = base_dir .. "/"
	end

	-- 输入目标目录
	vim.ui.input({
		prompt = "📁 移动到目录: ",
		default = default_dir,
		completion = "dir",
	}, function(target_dir)
		if not target_dir or target_dir == "" then
			vim.defer_fn(function()
				vim.api.nvim_echo({ { "❌ 移动已取消", "WarningMsg" } }, true, {})
			end, 100)
			return
		end

		-- 规范化目标路径
		target_dir = vim.fn.expand(target_dir)
		-- 移除末尾斜杠以便统一比较
		target_dir = target_dir:gsub("/$", "")

		-- 检查目标目录是否存在
		if vim.fn.isdirectory(target_dir) == 0 then
			-- 目录不存在，显示选项询问是否创建
			vim.cmd("redraw")
			vim.api.nvim_echo({
				{ "📁 目录不存在: ", "WarningMsg" },
				{ target_dir, "Function" },
				{ "\n是否创建目录?\n", "WarningMsg" },
				{ "✅ Y - 创建目录并移动\n", "Normal" },
				{ "❌ N - 取消移动\n", "Normal" },
				{ "请选择 (Y/N): ", "Question" }
			}, true, {})

			local char = vim.fn.getchar()
			local choice = vim.fn.nr2char(char)
			vim.cmd("redraw")

			if choice == "Y" or choice == "y" then
				-- 创建目录
				local mkdir_result = vim.fn.mkdir(target_dir, "p")
				if mkdir_result == 0 then
					vim.defer_fn(function()
						vim.api.nvim_echo({ { "❌ 目录创建失败", "ErrorMsg" } }, true, {})
					end, 100)
					return
				end

				-- 目录创建成功，先进行同目录检测，然后执行移动
				-- 🔍 同目录检测（类似批量粘贴的逻辑）
				local same_dir_files = {}
				for _, mark in ipairs(marks) do
					local source_dir = vim.fn.resolve(vim.fn.fnamemodify(mark.absolute_path, ":h")):gsub("/$", "")
					local current_target = vim.fn.resolve(target_dir):gsub("/$", "")


					if source_dir == current_target then
						table.insert(same_dir_files, mark.absolute_path)
					end
				end

				if #same_dir_files > 0 then
					local file_list = {}
					local dir_list = {}

					for _, path in ipairs(same_dir_files) do
						local filename = vim.fn.fnamemodify(path, ":t")
						if vim.fn.isdirectory(path) == 1 then
							table.insert(dir_list, filename)
						else
							table.insert(file_list, filename)
						end
					end

					local msg_parts = {}
					if #file_list > 0 then
						table.insert(msg_parts, "文件 " .. table.concat(file_list, ", "))
					end
					if #dir_list > 0 then
						table.insert(msg_parts, "目录 " .. table.concat(dir_list, ", "))
					end

					local msg = "🚫 不能在同目录内执行移动操作，已跳过: " .. table.concat(msg_parts, "，") .. " | 标记已清除"
					vim.defer_fn(function()
						vim.api.nvim_echo({ { msg, "ErrorMsg" } }, false, {})
					end, 100)

					-- 移除同目录文件，继续处理其他文件
					local filtered_marks = {}
					for _, mark in ipairs(marks) do
						local source_dir = vim.fn.resolve(vim.fn.fnamemodify(mark.absolute_path, ":h")):gsub("/$", "")
						local current_target = vim.fn.resolve(target_dir):gsub("/$", "")
						if source_dir ~= current_target then
							table.insert(filtered_marks, mark)
						end
					end

					if #filtered_marks == 0 then
						-- 所有文件都在同目录，清除标记后返回
						api.marks.clear()
						return
					else
						-- 处理剩余的非同目录文件
						execute_move_with_conflict_handling(api, filtered_marks, target_dir)
					end
				else
					-- 没有同目录文件，直接执行移动
					execute_move_with_conflict_handling(api, marks, target_dir)
				end
			else
				vim.defer_fn(function()
					vim.api.nvim_echo({ { "❌ 移动已取消", "WarningMsg" } }, true, {})
				end, 100)
				return
			end
		else
			-- 目录存在，先进行同目录检测，然后执行移动
			-- 🔍 同目录检测（类似批量粘贴的逻辑）
			local same_dir_files = {}
			for _, mark in ipairs(marks) do
				local source_dir = vim.fn.resolve(vim.fn.fnamemodify(mark.absolute_path, ":h")):gsub("/$", "")
				local current_target = vim.fn.resolve(target_dir):gsub("/$", "")


				if source_dir == current_target then
					table.insert(same_dir_files, mark.absolute_path)
				end
			end

			if #same_dir_files > 0 then
				local file_list = {}
				local dir_list = {}

				for _, path in ipairs(same_dir_files) do
					local filename = vim.fn.fnamemodify(path, ":t")
					if vim.fn.isdirectory(path) == 1 then
						table.insert(dir_list, filename)
					else
						table.insert(file_list, filename)
					end
				end

				local msg_parts = {}
				if #file_list > 0 then
					table.insert(msg_parts, "文件 " .. table.concat(file_list, ", "))
				end
				if #dir_list > 0 then
					table.insert(msg_parts, "目录 " .. table.concat(dir_list, ", "))
				end

				local msg = "🚫 不能在同目录内执行移动操作，已跳过: " .. table.concat(msg_parts, "，") .. " | 标记已清除"
				vim.defer_fn(function()
					vim.api.nvim_echo({ { msg, "ErrorMsg" } }, false, {})
				end, 100)

				-- 移除同目录文件，继续处理其他文件
				local filtered_marks = {}
				for _, mark in ipairs(marks) do
					local source_dir = vim.fn.resolve(vim.fn.fnamemodify(mark.absolute_path, ":h")):gsub("/$", "")
					local current_target = vim.fn.resolve(target_dir):gsub("/$", "")
					if source_dir ~= current_target then
						table.insert(filtered_marks, mark)
					end
				end

				if #filtered_marks == 0 then
					-- 所有文件都在同目录，清除标记后返回
					api.marks.clear()
					return
				else
					-- 处理剩余的非同目录文件
					execute_move_with_conflict_handling(api, filtered_marks, target_dir)
				end
			else
				-- 没有同目录文件，直接执行移动
				execute_move_with_conflict_handling(api, marks, target_dir)
			end
		end
	end)
end, { desc = "批量移动标记的文件到指定目录" })

-- 5. 重命名相关命令
vim.api.nvim_create_user_command("NvimTreeA5Rename", function()
	-- 清除命令行显示
	vim.cmd("redraw")

	local api = require("nvim-tree.api")

	-- 获取当前节点信息
	local node = api.tree.get_node_under_cursor()
	if not node then
		vim.notify("❌ 未找到有效的文件或目录", vim.log.levels.WARN)
		return
	end

	local old_path = node.absolute_path
	local old_name = vim.fn.fnamemodify(old_path, ":t")
	local parent_dir = vim.fn.fnamemodify(old_path, ":h")

	-- 自定义输入提示（中文）
	local prompt = node.type == "directory" and "📁 重命名目录为: " or "📄 重命名文件为: "

	vim.ui.input({
		prompt = prompt,
		default = old_name,
	}, function(new_name)
		if not new_name or new_name == "" or new_name == old_name then
			vim.defer_fn(function()
				vim.api.nvim_echo({ { "❌ 重命名已取消", "WarningMsg" } }, true, {})
			end, 100)
			return
		end

		local new_path = parent_dir .. "/" .. new_name

		-- 检查新文件名是否已存在
		if vim.fn.filereadable(new_path) == 1 or vim.fn.isdirectory(new_path) == 1 then
			vim.notify("❌ 文件或目录已存在: " .. new_name, vim.log.levels.ERROR)
			return
		end

		-- 执行重命名
		local success = vim.fn.rename(old_path, new_path) == 0

		if success then
			-- 刷新树状图
			api.tree.reload()

			-- 显示成功提示
			vim.defer_fn(function()
				if node.type == "directory" then
					vim.api.nvim_echo({
						{ "✏️ 目录已重命名: " .. parent_dir .. "/", "Normal" },
						{ new_name, "Function" }
					}, true, {})
				else
					vim.api.nvim_echo({
						{ "✏️ 文件已重命名: " .. parent_dir .. "/", "Normal" },
						{ new_name, "Function" }
					}, true, {})
				end
			end, 100)
		else
			vim.notify("❌ 重命名失败", vim.log.levels.ERROR)
		end
	end)
end, { desc = "重命名当前文件/目录" })

-- 6. 删除相关命令
vim.api.nvim_create_user_command("NvimTreeA6MarkDelete", function()
	local api = require("nvim-tree.api")
	local marks = api.marks.list()

	if #marks == 0 then
		vim.api.nvim_echo({ { "❌ 没有标记的文件", "WarningMsg" } }, true, {})
		return
	end

	-- 显示中文确认提示
	vim.cmd("redraw")
	vim.api.nvim_echo({ { "⚠️ 确定要删除 " .. #marks .. " 个标记的文件吗？(y/N)", "WarningMsg" } }, true, {})

	-- 获取用户输入（不需要按Enter）
	local char = vim.fn.getchar()
	local choice = vim.fn.nr2char(char)

	-- 清除提示行
	vim.cmd("redraw")

	if choice == "y" or choice == "Y" then
		-- 执行批量删除，拦截nvim-tree的英文确认提示
		local success, error_msg = pcall(function()
			-- 重定向输入和确认函数，自动确认所有操作
			local original_input = vim.fn.input
			local original_confirm = vim.fn.confirm

			vim.fn.input = function(...) return "y" end -- 总是确认
			vim.fn.confirm = function(...) return 1 end -- 总是确认

			api.marks.bulk.trash()

			-- 恢复原始函数
			vim.fn.input = original_input
			vim.fn.confirm = original_confirm
		end)

		if success then
			-- 延迟显示消息，确保批量删除操作完成后再显示
			vim.defer_fn(function()
				local msg = "🗂️ 已删除" .. #marks .. "个文件"
				-- 使用 echo 显示消息，设置为添加到历史记录
				vim.api.nvim_echo({ { msg, "Normal" } }, true, {})
				-- 使用 echom 确保消息保存到历史记录
				vim.cmd('echom "' .. msg:gsub('"', '\\"') .. '"')
			end, 200)
		else
			vim.api.nvim_echo({ { "❌ 删除失败: " .. tostring(error_msg), "ErrorMsg" } }, true, {})
		end
	else
		-- 延迟显示取消消息避免enter提示
		vim.defer_fn(function()
			vim.api.nvim_echo({ { "❌ 已取消删除", "Normal" } }, true, {})
		end, 100)
	end
end, { desc = "批量删除标记的文件" })

-- 8. 清除标记相关命令
vim.api.nvim_create_user_command("NvimTreeA7Clear", function()
	-- 清除命令行显示
	vim.cmd("redraw")

	local api = require("nvim-tree.api")
	api.marks.clear()

	-- 延迟显示消息避免enter提示
	vim.defer_fn(function()
		vim.api.nvim_echo({ { "✅ 已清除所有标记", "Normal" } }, true, {})
	end, 100)
end, { desc = "清除所有 NvimTree 标记" })


-- 创建排序靠后的系统命令（使用特殊命名策略）
local function create_ordered_commands()
	-- 删除可能存在的默认命令
	pcall(function() vim.api.nvim_del_user_command("NvCheatsheet") end)
	pcall(function() vim.api.nvim_del_user_command("Nvdash") end)
	
	-- 创建带有特殊前缀的命令（使用大写字母Z确保排在最后）
	vim.api.nvim_create_user_command("ZCheatsheet", function()
		local ok, cheatsheet = pcall(require, 'nvchad.cheatsheet')
		if ok then
			cheatsheet.toggle()
		else
			vim.notify("无法加载快捷键帮助：NvChad cheatsheet 模块未找到", vim.log.levels.ERROR)
		end
	end, { 
		desc = "打开 NvChad 快捷键帮助",
		force = true,
	})

	vim.api.nvim_create_user_command("ZDash", function()
		local ok, nvdash = pcall(require, 'nvchad.nvdash')
		if ok then
			nvdash.open()
		else
			vim.notify("无法加载启动页面：NvChad nvdash 模块未找到", vim.log.levels.ERROR)
		end
	end, { 
		desc = "打开 NvChad 启动页面",
		force = true,
	})
end

-- 立即创建命令
create_ordered_commands()

-- 在多个时机尝试重新定义命令
vim.defer_fn(create_ordered_commands, 2000) -- 延迟2秒
vim.defer_fn(create_ordered_commands, 5000) -- 延迟5秒

-- 使用自动命令在特定事件后重新定义
vim.api.nvim_create_autocmd("User", {
	pattern = "LazyDone", -- Lazy.nvim 加载完成后
	callback = function()
		vim.defer_fn(create_ordered_commands, 1000)
	end,
	once = true,
})

vim.api.nvim_create_autocmd("VimEnter", {
	callback = function()
		vim.defer_fn(create_ordered_commands, 2000)
	end,
	once = true,
})

-- ============================================================================
-- 3.8 目录创建操作命令 (A8 系列)
-- ============================================================================
-- 📁 NvimTreeA8MakeDir - 智能目录创建命令
--
-- 🎯 功能说明:
--   在当前光标位置或选中目录内创建新目录
--   支持多层级目录创建，自动处理路径分隔符
--
-- 📖 使用方法:
--   1. 将光标移动到要创建目录的位置（文件上或目录上）
--   2. 执行 :NvimTreeA8MakeDir
--   3. 输入目录名称（支持嵌套路径如: folder/subfolder）
--   4. 确认创建
--
-- ⚡ 技术特点:
--   • 支持相对路径和嵌套目录创建
--   • 自动检测当前位置（文件旁边或目录内部）
--   • 智能路径处理和冲突检测
--   • 中文化交互提示
--   • 创建后自动刷新并定位到新目录
--
-- 💡 使用技巧:
--   • 在文件上执行：在文件所在目录创建新目录
--   • 在目录上执行：在该目录内部创建新目录
--   • 支持路径：输入 "docs/api" 会创建嵌套的 docs/api 目录结构
--
-- 🔗 相关命令: 原生 'a' 键（创建文件/目录）
vim.api.nvim_create_user_command("NvimTreeA8MakeDir", function()
	-- 📦 第一步: 加载 nvim-tree API
	local api = require("nvim-tree.api")
	
	-- 🎯 第二步: 获取当前光标位置信息
	local node = api.tree.get_node_under_cursor()
	if not node then
		vim.notify("❌ 无法获取当前位置信息", vim.log.levels.ERROR)
		return
	end
	
	-- 📂 第三步: 确定创建目录的基础路径
	local base_path
	if node.type == "directory" then
		-- 如果当前是目录，在目录内部创建
		base_path = node.absolute_path
	else
		-- 如果当前是文件，在文件所在目录创建
		base_path = vim.fn.fnamemodify(node.absolute_path, ":h")
	end
	
	-- 📝 第四步: 获取用户输入的目录名称
	local dir_name = vim.fn.input({
		prompt = "📁 新建目录名称 (支持嵌套路径): ",
		completion = "file",
		cancelreturn = nil,
	})
	
	-- 清除命令行显示
	vim.cmd('redraw')
	
	-- 🚫 第五步: 处理用户取消或空输入
	if not dir_name or dir_name == "" then
		print("⚠️  目录创建已取消")
		return
	end
	
	-- 🛡️ 第六步: 构建完整路径并进行安全检查
	local full_path = base_path .. "/" .. dir_name
	
	-- 检查目录是否已存在
	if vim.fn.isdirectory(full_path) == 1 then
		print("⚠️  目录已存在: " .. dir_name)
		return
	end
	
	-- 📁 第七步: 创建目录（支持多层级）
	local success = vim.fn.mkdir(full_path, "p") == 1
	
	if success then
		-- ✅ 创建成功的后续处理
		print("✅ 目录创建成功: " .. dir_name)
		
		-- 🔄 刷新 nvim-tree 显示
		api.tree.reload()
		
		-- 🎯 尝试导航到新创建的目录
		vim.defer_fn(function()
			-- 展开父目录以显示新创建的目录
			if node.type == "directory" and not node.open then
				api.node.open.edit(node)
			end
			
			-- 查找并定位到新创建的目录
			local function find_and_focus_new_dir()
				local new_node = api.tree.get_node_under_cursor()
				if new_node and new_node.absolute_path == full_path then
					return
				end
				
				-- 尝试在当前视图中找到新目录
				-- 这里可以添加更复杂的查找逻辑
			end
			
			find_and_focus_new_dir()
		end, 100)
		
	else
		-- ❌ 创建失败的错误处理
		print("❌ 目录创建失败: " .. dir_name)
		print("📍 路径: " .. full_path)
		print("💡 请检查路径和权限")
	end
	
end, { 
	desc = "在当前位置创建新目录（支持嵌套路径）",
	-- 📖 nargs = "?" 表示可选参数，用户可以预先提供目录名
	-- nargs = "?",
})

-- 创建显示帮助信息的命令
vim.api.nvim_create_user_command("NvimTreeA0Help", function()
	vim.notify(
		"🔧 NvimTree 操作命令:\n\n📋 单文件操作 (按优先级排序):\n:NvimTreeA1Copy - 复制当前文件/目录\n:NvimTreeA2Cut - 剪切当前文件/目录\n:NvimTreeA3Paste - 粘贴文件/目录\n:NvimTreeA8MakeDir - 创建新目录 (支持嵌套路径)\n\n📁 批量操作 (按优先级排序):\n:NvimTreeA1MarkCopy - 批量复制标记的文件\n:NvimTreeA2MarkCut - 批量剪切标记的文件\n:NvimTreeA4MarkMove - 批量移动标记的文件 (默认目录为当前目录)\n:NvimTreeA5Rename - 重命名当前文件/目录\n:NvimTreeA6MarkDelete - 批量删除标记的文件\n:NvimTreeA7Clear - 清除所有标记\n\n🔧 系统命令 (排序优化):\n:NvimTreeA0Help - 显示此帮助信息\n:ZCheatsheet - 打开 NvChad 快捷键帮助\n:ZDash - 打开 NvChad 启动页面\n\n⌨️ 快捷键:\nm - 标记文件 | y - 复制 | x - 剪切 | p - 粘贴 | cc - 清空剪贴板\nty - 批量复制 | td - 批量删除 | tm - 批量移动 | tc - 清除标记\nCtrl+j - 向下翻页 | Ctrl+k - 向上翻页\n\n💡 操作技巧:\n📁 创建目录:\n- 在文件上执行 :NvimTreeA8MakeDir 在文件所在目录创建\n- 在目录上执行则在目录内部创建\n- 支持嵌套路径: docs/api/v1\n\n📋 批量复制粘贴:\n1️⃣ 用 m 标记多个文件\n2️⃣ 按 ty 批量复制或执行 :NvimTreeA2MarkCut 批量剪切\n3️⃣ 导航到目标目录\n4️⃣ 按 p 粘贴\n\n✂️ 批量移动:\n1️⃣ 用 m 标记多个文件\n2️⃣ 按 tm 键 (默认显示当前目录)\n3️⃣ 输入目标目录或直接确认\n4️⃣ 处理冲突并自动创建目录\n\n📝 命令排序说明:\n- A系列命令按功能分组 (A0帮助, A1复制, A2剪切, A3粘贴等)\n- 系统命令直接使用原名，通过覆盖NvChad默认命令实现排序控制\n\n📖 详细帮助: 查看 nvim-tree-marks-guide.md",
		vim.log.levels.INFO)
end, { desc = "显示 NvimTree 操作命令帮助" })


-- ============================================================================
-- 自定义Git状态颜色配置
-- ============================================================================

-- 在多个时机应用颜色，确保在任何情况下都能正确显示
vim.api.nvim_create_autocmd({ "ColorScheme", "VimEnter", "UIEnter" }, {
    callback = function()
        vim.schedule(function()
            local highlights = require('configs.highlights')
            highlights.set_nvim_tree_highlights()
            highlights.ultimate_git_color_fix()
        end)
    end
})

-- 监听 NvimTree 打开事件，确保在 NvimTree 显示时应用正确的颜色
vim.api.nvim_create_autocmd("FileType", {
    pattern = "NvimTree",
    callback = function()
        local highlights = require('configs.highlights')
        -- 立即应用颜色
        highlights.set_nvim_tree_highlights()
        highlights.ultimate_git_color_fix()
        
        -- 立即强制刷新 Git 状态
        pcall(function()
            require('nvim-tree.git').reload()
        end)
        
        -- 再次延迟应用，确保万无一失
        vim.defer_fn(function()
            highlights.set_nvim_tree_highlights()
            highlights.ultimate_git_color_fix()
            
            -- 再次强制刷新 Git 状态以确保颜色正确
            pcall(function()
                require('nvim-tree.git').reload()
                local api = require("nvim-tree.api")
                api.tree.reload()
            end)
        end, 50)
    end,
})

-- 移除重复的BufEnter事件，已被下面的统一事件处理替代

-- 创建用户命令来手动修复 Git 颜色显示

-- 重写NvimTree打开命令，确保打开时立即应用颜色
vim.api.nvim_create_user_command("NvimTreeOpen", function()
    require('nvim-tree.api').tree.open()
    -- 立即应用颜色
    vim.schedule(function()
        local highlights = require('configs.highlights')
        highlights.set_nvim_tree_highlights()
        highlights.ultimate_git_color_fix()
    end)
    -- 多次延迟应用，确保在不同时机都能生效
    for i = 1, 3 do
        vim.defer_fn(function()
            local highlights = require('configs.highlights')
            highlights.set_nvim_tree_highlights()
            highlights.ultimate_git_color_fix()
            highlights.force_refresh_nvimtree()
        end, i * 100)
    end
end, { desc = "打开 NvimTree 并应用颜色" })

-- 重写NvimTree切换命令
vim.api.nvim_create_user_command("NvimTreeToggle", function()
    require('nvim-tree.api').tree.toggle()
    -- 立即应用颜色
    vim.schedule(function()
        local highlights = require('configs.highlights')
        highlights.set_nvim_tree_highlights()
        highlights.ultimate_git_color_fix()
    end)
    -- 多次延迟应用，确保在不同时机都能生效
    for i = 1, 3 do
        vim.defer_fn(function()
            local highlights = require('configs.highlights')
            highlights.set_nvim_tree_highlights()
            highlights.ultimate_git_color_fix()
            highlights.force_refresh_nvimtree()
        end, i * 100)
    end
end, { desc = "切换 NvimTree 并应用颜色" })

-- 添加专门的空启动修复命令

-- 创建调试命令来检查当前高亮组
vim.api.nvim_create_user_command("NvimTreeDebugColors", function()
    -- 立即应用颜色
    local highlights = require('configs.highlights')
    highlights.set_nvim_tree_highlights()
    highlights.ultimate_git_color_fix()
    
    print("=== NvimTree Git 高亮组状态 ===")
    local git_groups = {
        "NvimTreeGitNew", "NvimTreeGitDirty", "NvimTreeGitIgnored",
        "NvimTreeGitIgnoredFolder", "NvimTreeGitIgnoredFolderName", "NvimTreeGitIgnoredFolderIcon", 
        "NvimTreeGitStaged", "NvimTreeGitDeleted", "NvimTreeGitRenamed", "NvimTreeGitUntracked"
    }
    for _, group in ipairs(git_groups) do
        local hl = vim.api.nvim_get_hl(0, { name = group })
        if next(hl) then
            local color = hl.fg and string.format("#%06x", hl.fg) or "未设置"
            print(string.format("✅ %s: %s", group, color))
        else
            print(string.format("❌ %s: 高亮组不存在", group))
        end
    end
    print("================================")
    print("当前 Git 状态 (git status --porcelain):")
    local git_status = vim.fn.system("git status --porcelain")
    if vim.v.shell_error == 0 then
        if git_status ~= "" then
            for line in git_status:gmatch("[^\r\n]+") do
                print("  " .. line)
            end
        else
            print("  没有修改")
        end
    else
        print("  不在 Git 仓库中")
    end
    print("================================")
end, { desc = "调试 NvimTree Git 颜色高亮组" })

-- 监控Git状态变化并自动刷新NvimTree
-- 监控 .git/index 和 .git/HEAD 文件的变化来检测Git操作
-- Git监控性能配置（设置为全局变量以便autocmd访问）
_G.nvimtree_git_config = {
    enable_on_save = false,    -- 默认禁用保存文件时刷新Git状态，避免编辑卡顿
    enable_fs_watcher = true,  -- 是否启用文件系统监听器
    enable_cmd_watcher = true, -- 是否监听Git命令执行
    enable_focus_refresh = false, -- 默认禁用焦点切换时的刷新，避免切换卡顿
    one_time_refresh = true,   -- 启用一次性刷新：每次编辑会话只刷新一次
    disable_edit_colors = true, -- 默认禁用编辑时的颜色变化，减少卡顿
    optimize_new_files = true, -- 优化新文件的Git检测，减少untracked文件卡顿
    disable_cursor_refresh = true, -- 默认禁用光标移动时的刷新，解决卡顿问题
    debounce_time = 1000,      -- 防抖时间（毫秒）
}

-- 全局防抖变量
_G.nvimtree_last_focus_time = 0

-- 文件编辑状态跟踪
_G.nvimtree_file_edit_state = {}

-- 新文件跟踪
_G.nvimtree_new_files = {}

-- 本地引用以便函数使用
local git_config = _G.nvimtree_git_config

-- 用户可调用此函数来禁用编辑时的Git刷新
local function disable_git_on_edit()
    git_config.enable_on_save = false
    print("🚫 已禁用编辑文件时的Git状态刷新，减少卡顿")
end

-- 用户可调用此函数来重新启用编辑时的Git刷新
local function enable_git_on_edit()
    git_config.enable_on_save = true
    print("✅ 已启用编辑文件时的Git状态刷新")
end

-- 禁用焦点切换时的刷新
local function disable_focus_refresh()
    git_config.enable_focus_refresh = false
    print("🚫 已禁用切换到nvimtree时的刷新，减少卡顿")
end

-- 启用焦点切换时的刷新
local function enable_focus_refresh()
    git_config.enable_focus_refresh = true
    print("✅ 已启用切换到nvimtree时的刷新")
end

-- 编辑时颜色控制函数
local function disable_edit_colors()
    git_config.disable_edit_colors = true
    print("🚫 已禁用编辑时的Git颜色变化，减少卡顿")
end

local function enable_edit_colors()
    git_config.disable_edit_colors = false
    print("✅ 已启用编辑时的Git颜色变化")
end

-- 暂时禁用nvimtree的Git检测
local function temporarily_disable_git()
    local api = require('nvim-tree.api')
    if api.tree.is_visible() then
        -- 重新配置nvimtree以禁用Git
        require('nvim-tree').setup({
            git = { enable = false }
        })
    end
end

-- 重新启用nvimtree的Git检测
local function re_enable_git()
    local api = require('nvim-tree.api')
    if api.tree.is_visible() then
        -- 重新配置nvimtree以启用Git
        require('nvim-tree').setup({
            git = { 
                enable = true,
                timeout = 200,
                show_on_dirs = false,
                show_on_open_dirs = false
            }
        })
    end
end

-- 文件编辑状态管理函数
local function mark_file_as_edited(filepath)
    if not _G.nvimtree_file_edit_state then
        _G.nvimtree_file_edit_state = {}
    end
    _G.nvimtree_file_edit_state[filepath] = {
        edited = true,
        refreshed = false,
        timestamp = vim.loop.hrtime()
    }
end

local function mark_file_as_refreshed(filepath)
    if _G.nvimtree_file_edit_state and _G.nvimtree_file_edit_state[filepath] then
        _G.nvimtree_file_edit_state[filepath].refreshed = true
    end
end

local function should_refresh_for_file(filepath)
    if not git_config.one_time_refresh then
        return true -- 如果没启用一次性刷新，总是刷新
    end
    
    local state = _G.nvimtree_file_edit_state and _G.nvimtree_file_edit_state[filepath]
    if not state then
        return true -- 没有编辑记录，可以刷新
    end
    
    return state.edited and not state.refreshed -- 只有编辑过但未刷新的才刷新
end

local function reset_file_edit_state(filepath)
    if _G.nvimtree_file_edit_state then
        _G.nvimtree_file_edit_state[filepath] = nil
    end
end

-- 新文件管理函数
local function is_new_file(filepath)
    if not filepath or filepath == "" then return false end
    
    -- 检查文件是否存在于Git中
    local cmd = string.format("git ls-files --error-unmatch %s 2>/dev/null", vim.fn.shellescape(filepath))
    local result = vim.fn.system(cmd)
    return vim.v.shell_error ~= 0 -- 如果命令失败，说明文件不在Git中
end

local function mark_as_new_file(filepath)
    if not _G.nvimtree_new_files then
        _G.nvimtree_new_files = {}
    end
    _G.nvimtree_new_files[filepath] = {
        timestamp = vim.loop.hrtime(),
        checked = false
    }
end

local function is_tracked_new_file(filepath)
    return _G.nvimtree_new_files and _G.nvimtree_new_files[filepath]
end

local function should_delay_git_for_new_file(filepath)
    if not git_config.optimize_new_files then
        return false
    end
    
    local file_info = _G.nvimtree_new_files and _G.nvimtree_new_files[filepath]
    if not file_info then
        return false
    end
    
    -- 如果是新创建的文件，且还没检查过，延迟Git检测
    local time_since_creation = (vim.loop.hrtime() - file_info.timestamp) / 1000000 -- 转换为毫秒
    return time_since_creation < 5000 and not file_info.checked -- 5秒内且未检查过
end

local function setup_git_monitoring()
    local git_dir = vim.fn.getcwd() .. "/.git"
    if vim.fn.isdirectory(git_dir) == 1 then
        -- 监控Git index文件变化（暂存区变化）
        local index_file = git_dir .. "/index"
        local head_file = git_dir .. "/HEAD"
        
        -- 创建文件监控的自动命令
        -- 旧的BufWritePost事件已移除，使用下面优化的事件系统
        
        -- 轻量级Git状态刷新，最小化性能影响
        local refresh_in_progress = false
        local function refresh_git_on_file_change()
            -- 防止并发刷新
            if refresh_in_progress then
                return
            end
            
            refresh_in_progress = true
            
            pcall(function()
                -- 只刷新Git状态，不强制更新高亮
                require('nvim-tree.git').reload()
                
                -- 使用更长延迟来批量处理高亮更新
                vim.defer_fn(function()
                    refresh_in_progress = false
                    -- 只在nvimtree可见时更新高亮
                    local api = require('nvim-tree.api')
                    if api.tree.is_visible() then
                        local highlights = require('configs.highlights')
                        highlights.set_nvim_tree_highlights()
                    end
                end, 200)
            end)
        end
        
        -- 监听文件编辑事件，标记文件为已编辑并控制Git颜色更新
        vim.api.nvim_create_autocmd({"TextChanged", "TextChangedI"}, {
            callback = function()
                local config = _G.nvimtree_git_config
                if config then
                    -- 标记文件为已编辑
                    if config.one_time_refresh then
                        local filepath = vim.fn.expand("%:p")
                        if filepath and filepath ~= "" then
                            mark_file_as_edited(filepath)
                        end
                    end
                    
                    -- 如果禁用了编辑时颜色变化，暂时禁用nvimtree的Git检测
                    if config.disable_edit_colors then
                        -- 使用延迟来避免过于频繁的操作
                        local last_disable = _G.nvimtree_last_git_disable or 0
                        local current_time = vim.loop.hrtime()
                        if (current_time - last_disable) > 2000000000 then -- 2秒防抖
                            _G.nvimtree_last_git_disable = current_time
                            
                            -- 延迟重新启用Git检测
                            vim.defer_fn(function()
                                pcall(function()
                                    require('nvim-tree.git').reload()
                                end)
                            end, 3000) -- 3秒后重新检测
                        end
                    end
                end
            end,
            desc = "智能编辑状态管理"
        })
        
        -- 智能文件保存监听（一次性Git刷新）
        vim.api.nvim_create_autocmd({"BufWritePost"}, {
            callback = function()
                -- 检查是否启用了保存时的Git刷新
                local config = _G.nvimtree_git_config
                if not config or not config.enable_on_save then
                    return
                end
                
                -- 检查nvimtree是否可见，不可见时跳过刷新
                local api = require('nvim-tree.api')
                if not api.tree.is_visible() then
                    return
                end
                
                local filepath = vim.fn.expand("%:p")
                local filename = vim.fn.expand("%:t")
                
                -- 检查是否需要刷新（一次性刷新逻辑）
                if not should_refresh_for_file(filepath) then
                    return -- 已经刷新过了，跳过
                end
                
                -- Git相关文件需要特殊处理
                local is_git_file = filename == ".gitignore" or filename == ".gitmodules" or 
                                  vim.fn.expand("%"):match("%.git/")
                
                if is_git_file then
                    if filename == ".gitignore" then
                        print("🔄 检测到 .gitignore 文件变化，更新文件颜色...")
                        vim.defer_fn(function()
                            require('configs.nvimtree').smart_color_directories()
                            print("✅ .gitignore 文件颜色已自动更新")
                        end, 200)
                    end
                    vim.defer_fn(function()
                        refresh_git_on_file_change()
                        mark_file_as_refreshed(filepath)
                    end, 300)
                else
                    -- 普通文件保存，检查是否需要Git刷新
                    local ext = vim.fn.expand("%:e")
                    local code_extensions = {
                        "js", "ts", "jsx", "tsx", "py", "lua", "go", "rs", "java", "c", "cpp", "h", "hpp",
                        "php", "rb", "css", "scss", "html", "vue", "svelte", "md", "json", "yaml", "yml", "toml"
                    }
                    local is_code_file = vim.tbl_contains(code_extensions, ext)
                    
                    if is_code_file then
                        vim.defer_fn(function()
                            refresh_git_on_file_change()
                            mark_file_as_refreshed(filepath)
                        end, 100)
                    end
                end
            end,
            desc = "一次性智能Git状态刷新"
        })
        
        -- 监听新文件创建
        vim.api.nvim_create_autocmd({"BufNewFile"}, {
            callback = function()
                local config = _G.nvimtree_git_config
                if config and config.optimize_new_files then
                    local filepath = vim.fn.expand("%:p")
                    if filepath and filepath ~= "" then
                        -- 检查是否真的是新文件
                        if is_new_file(filepath) then
                            mark_as_new_file(filepath)
                        end
                    end
                end
            end,
            desc = "标记新文件以优化Git检测"
        })
        
        -- 监听文件关闭，重置编辑状态
        vim.api.nvim_create_autocmd({"BufDelete", "BufWipeout"}, {
            callback = function()
                local filepath = vim.fn.expand("%:p")
                if filepath and filepath ~= "" then
                    reset_file_edit_state(filepath)
                    -- 同时清理新文件记录
                    if _G.nvimtree_new_files then
                        _G.nvimtree_new_files[filepath] = nil
                    end
                end
            end,
            desc = "重置文件编辑和新文件状态"
        })
        
        -- 监听目录变化事件（创建、删除文件）
        vim.api.nvim_create_autocmd({"DirChanged", "BufNewFile", "BufDelete"}, {
            callback = function()
                vim.defer_fn(refresh_git_on_file_change, 100)
            end,
            desc = "目录或文件变化后刷新Git状态"
        })
    end
end

-- 添加文件系统监听器，实时监控工作目录变化
local function setup_fs_watcher()
    if vim.fn.has('nvim-0.7') == 0 then
        return
    end
    
    local function get_git_root()
        local handle = io.popen("git rev-parse --show-toplevel 2>/dev/null")
        if not handle then return nil end
        local result = handle:read("*a")
        handle:close()
        if result and result ~= "" then
            return vim.trim(result)
        end
        return nil
    end
    
    local git_root = get_git_root()
    if not git_root then return end
    
    -- 创建文件系统监听器
    local fs_event = vim.loop.new_fs_event()
    if not fs_event then return end
    
    -- 防抖机制
    local debounce_timer = nil
    local function debounced_refresh()
        if debounce_timer then
            vim.fn.timer_stop(debounce_timer)
        end
        debounce_timer = vim.fn.timer_start(300, function()
            pcall(function()
                require('nvim-tree.git').reload()
                vim.defer_fn(function()
                    local highlights = require('configs.highlights')
                    highlights.set_nvim_tree_highlights()
                end, 50)
            end)
        end)
    end
    
    -- 监听Git目录变化
    fs_event:start(git_root, {recursive = false}, vim.schedule_wrap(function(err, filename, events)
        if err then return end
        if filename and (filename:match("%.git") or filename:match("%.gitignore")) then
            debounced_refresh()
        end
    end))
    
    -- 清理监听器
    vim.api.nvim_create_autocmd("VimLeavePre", {
        callback = function()
            if fs_event then
                fs_event:stop()
                fs_event:close()
            end
        end,
        desc = "清理文件系统监听器"
    })
end

-- 监听终端Git命令执行
local function setup_git_command_watcher()
    -- 监听终端命令执行后的刷新
    vim.api.nvim_create_autocmd("TermClose", {
        callback = function()
            -- 延迟检查是否执行了git命令
            vim.defer_fn(function()
                pcall(function()
                    require('nvim-tree.git').reload()
                end)
            end, 500)
        end,
        desc = "终端关闭后检查Git状态"
    })
    
    -- 监听Shell命令执行
    vim.api.nvim_create_autocmd("ShellCmdPost", {
        pattern = "git*",
        callback = function()
            vim.defer_fn(function()
                pcall(function()
                    require('nvim-tree.git').reload()
                    local highlights = require('configs.highlights')
                    highlights.set_nvim_tree_highlights()
                end)
            end, 200)
        end,
        desc = "Git命令执行后刷新状态"
    })
end

-- 创建用户控制命令
vim.api.nvim_create_user_command("NvimTreeDisableGitOnEdit", disable_git_on_edit, {
    desc = "禁用编辑文件时的Git状态刷新，减少卡顿"
})

vim.api.nvim_create_user_command("NvimTreeEnableGitOnEdit", enable_git_on_edit, {
    desc = "启用编辑文件时的Git状态刷新"
})

vim.api.nvim_create_user_command("NvimTreeDisableFocusRefresh", disable_focus_refresh, {
    desc = "禁用切换到nvimtree时的刷新，减少卡顿"
})

vim.api.nvim_create_user_command("NvimTreeEnableFocusRefresh", enable_focus_refresh, {
    desc = "启用切换到nvimtree时的刷新"
})

vim.api.nvim_create_user_command("NvimTreeDisableEditColors", disable_edit_colors, {
    desc = "禁用编辑时的Git颜色变化，减少卡顿"
})

vim.api.nvim_create_user_command("NvimTreeEnableEditColors", enable_edit_colors, {
    desc = "启用编辑时的Git颜色变化"
})

vim.api.nvim_create_user_command("NvimTreeToggleNewFileOptimization", function()
    local config = _G.nvimtree_git_config
    if config then
        config.optimize_new_files = not config.optimize_new_files
        print("🆕 新文件优化: " .. (config.optimize_new_files and "✅ 启用" or "❌ 禁用"))
        
        if not config.optimize_new_files then
            -- 清空新文件记录
            _G.nvimtree_new_files = {}
        end
    end
end, { desc = "切换新文件Git检测优化" })

vim.api.nvim_create_user_command("NvimTreeShowNewFiles", function()
    local new_files = _G.nvimtree_new_files
    if not new_files or vim.tbl_isempty(new_files) then
        print("🆕 没有新文件记录")
        return
    end
    
    print("🆕 新文件状态:")
    for filepath, info in pairs(new_files) do
        local filename = vim.fn.fnamemodify(filepath, ":t")
        local status = info.checked and "已检测" or "待检测"
        local age = (vim.loop.hrtime() - info.timestamp) / 1000000000 -- 转换为秒
        print(string.format("  %s: %s (%.1f秒前)", filename, status, age))
    end
end, { desc = "显示新文件状态" })

vim.api.nvim_create_user_command("NvimTreeToggleOneTimeRefresh", function()
    local config = _G.nvimtree_git_config
    if config then
        config.one_time_refresh = not config.one_time_refresh
        print("🔄 一次性刷新: " .. (config.one_time_refresh and "✅ 启用" or "❌ 禁用"))
        
        if not config.one_time_refresh then
            -- 清空编辑状态
            _G.nvimtree_file_edit_state = {}
        end
    end
end, { desc = "切换一次性Git刷新模式" })

vim.api.nvim_create_user_command("NvimTreeShowEditState", function()
    local state = _G.nvimtree_file_edit_state
    if not state or vim.tbl_isempty(state) then
        print("📝 没有文件编辑状态记录")
        return
    end
    
    print("📝 文件编辑状态:")
    for filepath, info in pairs(state) do
        local filename = vim.fn.fnamemodify(filepath, ":t")
        local status = info.refreshed and "已刷新" or "待刷新"
        print(string.format("  %s: %s", filename, status))
    end
end, { desc = "显示文件编辑状态" })

vim.api.nvim_create_user_command("NvimTreeGitConfig", function()
    local config = _G.nvimtree_git_config
    if not config then
        print("❌ NvimTree Git配置未加载")
        return
    end
    
    print("📋 NvimTree Git监控配置:")
    print("  保存时刷新: " .. (config.enable_on_save and "✅ 启用" or "❌ 禁用"))
    print("  焦点切换刷新: " .. (config.enable_focus_refresh and "✅ 启用" or "❌ 禁用"))
    print("  一次性刷新: " .. (config.one_time_refresh and "✅ 启用" or "❌ 禁用"))
    print("  编辑时颜色: " .. (config.disable_edit_colors and "❌ 禁用" or "✅ 启用"))
    print("  新文件优化: " .. (config.optimize_new_files and "✅ 启用" or "❌ 禁用"))
    print("  文件系统监听: " .. (config.enable_fs_watcher and "✅ 启用" or "❌ 禁用"))
    print("  命令监听: " .. (config.enable_cmd_watcher and "✅ 启用" or "❌ 禁用"))
    print("  防抖时间: " .. config.debounce_time .. "ms")
    
    -- 显示当前编辑状态统计
    local state = _G.nvimtree_file_edit_state
    if state and not vim.tbl_isempty(state) then
        local total = 0
        local refreshed = 0
        for _, info in pairs(state) do
            total = total + 1
            if info.refreshed then
                refreshed = refreshed + 1
            end
        end
        print("  编辑状态: " .. total .. " 个文件，" .. refreshed .. " 个已刷新")
    end
    
    -- 显示新文件状态统计
    local new_files = _G.nvimtree_new_files
    if new_files and not vim.tbl_isempty(new_files) then
        local total_new = 0
        local checked_new = 0
        for _, info in pairs(new_files) do
            total_new = total_new + 1
            if info.checked then
                checked_new = checked_new + 1
            end
        end
        print("  新文件状态: " .. total_new .. " 个新文件，" .. checked_new .. " 个已检测")
    end
    
    print("")
    print("💡 可用命令:")
    print("  :NvimTreeDisableGitOnEdit         - 禁用编辑时Git刷新")
    print("  :NvimTreeEnableGitOnEdit          - 启用编辑时Git刷新")
    print("  :NvimTreeDisableFocusRefresh      - 禁用焦点切换时刷新")
    print("  :NvimTreeEnableFocusRefresh       - 启用焦点切换时刷新")
    print("  :NvimTreeDisableEditColors        - 禁用编辑时颜色变化")
    print("  :NvimTreeEnableEditColors         - 启用编辑时颜色变化")
    print("  :NvimTreeToggleNewFileOptimization - 切换新文件优化")
    print("  :NvimTreeToggleOneTimeRefresh     - 切换一次性刷新模式")
    print("  :NvimTreeShowEditState            - 显示文件编辑状态")
    print("  :NvimTreeShowNewFiles             - 显示新文件状态")
end, { desc = "显示NvimTree Git监控配置" })

-- 启动基于事件的Git监控系统
vim.defer_fn(function()
    setup_git_monitoring()
    if git_config.enable_fs_watcher then
        setup_fs_watcher()
    end
    if git_config.enable_cmd_watcher then
        setup_git_command_watcher()
    end
end, 1000) -- 延迟启动，避免初始化冲突

-- 创建强力Git状态刷新命令（专门处理目录）

-- 创建超级强力的刷新命令（清除所有缓存）
vim.api.nvim_create_user_command("NvimTreeSuperRefresh", function()
    pcall(function()
        local api = require("nvim-tree.api")
        
        -- 1. 完全关闭 NvimTree
        if api.tree.is_visible() then
            api.tree.close()
        end
        
        -- 2. 强制清除所有可能的缓存
        require('nvim-tree.git').reload()
        
        -- 3. 执行 Git 清理命令
        vim.fn.system("git status --porcelain --ignored --untracked-files=all 2>/dev/null")
        vim.fn.system("git ls-files --others --ignored --exclude-standard --directory 2>/dev/null")
        
        -- 4. 重新初始化 NvimTree
        vim.defer_fn(function()
            -- 重新打开 NvimTree
            api.tree.open()
            
            -- 在 NvimTree 缓冲区中执行清理
            vim.defer_fn(function()
                -- 清除高亮缓存
                vim.cmd("syntax clear")
                
                -- 强制重新应用所有颜色设置
                highlights.set_nvim_tree_highlights()
                highlights.ultimate_git_color_fix()
                
                -- 再次强制刷新
                vim.defer_fn(function()
                    require('nvim-tree.git').reload()
                    highlights.set_nvim_tree_highlights()
                    highlights.ultimate_git_color_fix()
                    
                    -- 确保 NvimTree 获得焦点
                    local nvim_tree_winid = vim.fn.win_findbuf(vim.fn.bufnr("NvimTree_1"))
                    if #nvim_tree_winid > 0 then
                        vim.api.nvim_set_current_win(nvim_tree_winid[1])
                    end
                    
                    print("✅ NvimTree 超级刷新完成（清除所有缓存）")
                end, 200)
            end, 100)
        end, 300)
    end)
end, { desc = "超级刷新NvimTree（清除所有缓存和状态）" })


-- 创建Git操作快捷命令，自动刷新NvimTree
vim.api.nvim_create_user_command("GitCommit", function(opts)
    local message = opts.args or "Auto commit"
    vim.fn.system("git add .")
    vim.fn.system("git commit -m '" .. message .. "'")
    
    -- 提交后自动刷新NvimTree
    vim.defer_fn(function()
        pcall(function()
            require('nvim-tree.git').reload()
        end)
        highlights.set_nvim_tree_highlights()
        highlights.ultimate_git_color_fix()
        highlights.force_refresh_nvimtree()
    end, 500)
end, { 
    desc = "Git提交并刷新NvimTree", 
    nargs = "?",  -- 可选参数
})

-- 移除重复的WinEnter事件，避免切换时卡顿

-- ============================================================================
-- 智能目录着色系统
-- ============================================================================

-- 创建模块导出表，以便外部调用
local M = {}

-- 加载高亮颜色模块
local highlights = require('configs.highlights')

-- 通配符模式匹配函数
local function match_pattern(filename, pattern)
	-- 处理特殊情况
	if pattern == "" then
		return false
	end
	
	-- 转义特殊字符，但保留通配符
	local escaped_pattern = pattern:gsub("([%^%$%(%)%%%.%[%]%+%-%?])", "%%%1")
	
	-- 将通配符转换为Lua模式
	-- * 匹配任意字符（包括空字符）
	-- ? 匹配单个字符
	local lua_pattern = escaped_pattern:gsub("%%*", ".*"):gsub("%%?", ".")
	
	-- 处理以 / 开头的模式（目录模式）
	if pattern:match("^/") then
		-- 绝对路径模式，需要完整匹配
		return filename:match("^" .. lua_pattern:sub(2) .. "$")
	elseif pattern:match("/") then
		-- 包含路径分隔符的模式，需要完整匹配
		return filename:match("^" .. lua_pattern .. "$")
	else
		-- 简单文件名模式，支持通配符
		return filename:match("^" .. lua_pattern .. "$")
	end
end

-- 检查文件是否应该被忽略（支持通配符）
local function should_ignore_file(filename, patterns, debug_mode)
	for _, pattern in ipairs(patterns) do
		-- 完全匹配
		if filename == pattern then
			if debug_mode then
				print("📄 完全匹配忽略: " .. filename .. " == " .. pattern)
			end
			return true
		end
		
		-- 通配符匹配
		if match_pattern(filename, pattern) then
			if debug_mode then
				print("📄 通配符匹配忽略: " .. filename .. " ~= " .. pattern)
			end
			return true
		end
	end
	return false
end

-- 智能目录着色函数  
function M.smart_color_directories(debug_mode)
	-- 安全执行，避免错误中断
	pcall(function()
		if debug_mode then
			print("🎨 开始智能目录着色...")
		end
		
		-- 获取当前工作目录
		local cwd = vim.fn.getcwd()
		
		-- 直接读取.gitignore文件内容来决定颜色显示
		local ignored_dirs = {}
		local ignored_files = {}
		
		-- .git 目录始终为灰色（完全匹配）
		ignored_dirs[".git"] = true
		
		-- 读取.gitignore文件内容
		local gitignore_patterns = {}
		local gitignore_path = cwd .. "/.gitignore"
		if vim.fn.filereadable(gitignore_path) == 1 then
			local gitignore_content = vim.fn.readfile(gitignore_path)
			for _, line in ipairs(gitignore_content) do
				-- 去除空行和注释行
				local pattern = line:match("^%s*([^#%s].*)%s*$")
				if pattern and pattern ~= "" then
					table.insert(gitignore_patterns, pattern)
					if debug_mode then
						print("📋 读取忽略规则: " .. pattern)
					end
				end
			end
		end
		
		-- 检查当前目录下的所有文件和目录
		for item in vim.fs.dir(cwd) do
			local full_path = cwd .. "/" .. item
			local should_ignore = false
			
			-- 使用新的通配符匹配函数检查是否匹配.gitignore中的规则
			should_ignore = should_ignore_file(item, gitignore_patterns, debug_mode)
			
			-- 分类处理目录和文件
			if vim.fn.isdirectory(full_path) == 1 then
				if item == ".git" then
					-- .git目录直接作为被忽略目录处理，强制设置为浅灰色
					ignored_dirs[item] = true
					if debug_mode then
						print("📁 .git目录强制设置为浅灰色显示")
					end
				elseif should_ignore then
					ignored_dirs[item] = true
					if debug_mode then
						print("📁 根据.gitignore规则忽略目录: " .. item)
					end
				elseif debug_mode then
					print("📂 正常目录: " .. item)
				end
			elseif vim.fn.isdirectory(full_path) == 0 and item ~= ".gitignore" and item ~= ".gitattributes" then
				if should_ignore then
					ignored_files[item] = true
					if debug_mode then
						print("📄 根据.gitignore规则忽略文件: " .. item)
					end
				elseif debug_mode then
					print("📝 正常文件: " .. item)
				end
			end
		end
		
		-- 设置被忽略文件和目录的高亮组为灰色
		vim.api.nvim_set_hl(0, "NvimTreeGitIgnored", { fg = "#8a8a8a" })
		vim.api.nvim_set_hl(0, "NvimTreeGitIgnoredFile", { fg = "#8a8a8a" })
		
		-- 强制覆盖特殊文件的颜色，确保.gitignore等文件显示为白色
		vim.api.nvim_set_hl(0, "NvimTreeSpecialFile", { fg = "#ffffff" })
		
		-- 创建自定义的 NvimTree 渲染后处理
		vim.defer_fn(function()
			-- 在 NvimTree 缓冲区中执行
			for _, bufnr in ipairs(vim.api.nvim_list_bufs()) do
				local buf_name = vim.api.nvim_buf_get_name(bufnr)
				if buf_name:match("NvimTree_") then
					vim.api.nvim_buf_call(bufnr, function()
						-- 清除之前的匹配
						vim.cmd("silent! syntax clear NvimTreeIgnoredDir")
						vim.cmd("silent! syntax clear NvimTreeIgnoredFile")
						vim.cmd("silent! syntax clear NvimTreeIgnoredDirForce")
						vim.cmd("silent! syntax clear NvimTreeIgnoredFileForce")
						vim.cmd("silent! syntax clear NvimTreeIgnoredPattern")
						vim.cmd("silent! syntax clear NvimTreeIgnoredPatternForce")
						vim.cmd("silent! syntax clear NvimTreeGitDir")
						vim.cmd("silent! syntax clear NvimTreeGitDirForce")
						vim.cmd("silent! syntax clear NvimTreeNormalSpecialFile")
						vim.cmd("silent! syntax clear NvimTreeForceWhite")
						vim.cmd("silent! syntax clear NvimTreeForceGray")
						vim.cmd("silent! syntax clear NvimTreeForceGitGray")
						
						-- 为被忽略的目录创建更强制的语法匹配
						for dir, _ in pairs(ignored_dirs) do
							local escaped = vim.fn.escape(dir, ".*[]^$\\")
							-- 使用更强制的语法匹配，覆盖其他规则
							vim.cmd("syntax match NvimTreeIgnoredDir '\\<" .. escaped .. "\\>' contained containedin=ALL")
							vim.cmd("syntax match NvimTreeIgnoredDirForce '\\<" .. escaped .. "\\>'")
							
							-- 特别处理 .git 目录，确保其内容也显示为浅灰色
							if dir == ".git" then
								vim.cmd("syntax match NvimTreeGitDirForce '\\<" .. escaped .. "\\>.*' containedin=ALL")
								vim.cmd("syntax match NvimTreeGitDirForce '\\<" .. escaped .. "\\>' containedin=ALL")
								if debug_mode then
									print("🎨 为 .git 目录及其内容创建特殊语法匹配")
								end
							end
							
							if debug_mode then
								print("🎨 为目录 " .. dir .. " 创建强制语法匹配")
							end
						end
						
						-- 为被忽略的文件创建更强制的语法匹配
						for file, _ in pairs(ignored_files) do
							local escaped = vim.fn.escape(file, ".*[]^$\\")
							-- 使用更强制的语法匹配，覆盖其他规则
							vim.cmd("syntax match NvimTreeIgnoredFile '\\<" .. escaped .. "\\>' contained containedin=ALL")
							vim.cmd("syntax match NvimTreeIgnoredFileForce '\\<" .. escaped .. "\\>'")
							if debug_mode then
								print("🎨 为文件 " .. file .. " 创建强制语法匹配")
							end
						end
						
						-- 为通配符模式创建语法匹配
						for _, pattern in ipairs(gitignore_patterns) do
							-- 检查是否包含通配符
							if pattern:match("[*?]") then
								-- 将通配符转换为Vim正则表达式
								local vim_pattern = pattern:gsub("%*", ".*"):gsub("%?", ".")
								-- 转义特殊字符，但保留通配符转换
								vim_pattern = vim.fn.escape(vim_pattern, "[]^$\\")
								vim_pattern = vim_pattern:gsub("%%%.%*", ".*"):gsub("%%%.", ".")
								
								-- 创建语法匹配
								vim.cmd("syntax match NvimTreeIgnoredPattern '\\<" .. vim_pattern .. "\\>' contained containedin=ALL")
								vim.cmd("syntax match NvimTreeIgnoredPatternForce '\\<" .. vim_pattern .. "\\>'")
								
								if debug_mode then
									print("🎨 为通配符模式 " .. pattern .. " 创建语法匹配: " .. vim_pattern)
								end
							end
						end
						
						-- .git目录现在通过统一的被忽略目录流程处理
						
						-- 应用高亮颜色（使用highlights模块）
						local highlights = require('configs.highlights')
						highlights.apply_custom_highlights()
						
						if debug_mode then
							print("✅ 语法匹配已应用到 NvimTree 缓冲区")
						end
					end)
				end
			end
		end, 500)
		
		-- 额外的强制刷新，确保语法匹配生效
		vim.defer_fn(function()
			local api = require('nvim-tree.api')
			api.tree.reload()
			-- 再次应用语法匹配，确保生效
			for _, bufnr in ipairs(vim.api.nvim_list_bufs()) do
				local buf_name = vim.api.nvim_buf_get_name(bufnr)
				if buf_name:match("NvimTree_") then
					vim.api.nvim_buf_call(bufnr, function()
						vim.cmd("syntax sync fromstart")
					end)
				end
			end
		end, 800)
		
		-- 简化的NvimTree刷新，不依赖Git状态
		pcall(function()
			local api = require('nvim-tree.api')
			api.tree.reload()
		end)
		
		
		if debug_mode then
			print("📊 被忽略的目录数量: " .. vim.tbl_count(ignored_dirs))
			print("📊 被忽略的文件数量: " .. vim.tbl_count(ignored_files))
			print("✅ 智能目录和文件着色完成")
		end
	end)
end

-- 创建目录颜色刷新命令
vim.api.nvim_create_user_command("NvimTreeRefreshColors", function()
	pcall(function()
		vim.cmd("silent! syntax clear NvimTreeGitDirRefresh")
		vim.cmd([[syntax match NvimTreeGitDirRefresh /^\s*.*\.git\s*$/ contains=ALL]])
		vim.cmd("highlight NvimTreeGitDirRefresh guifg=#8a8a8a ctermfg=245")
	end)
	require('configs.nvimtree').smart_color_directories()
	print("✅ .git目录和其他忽略文件颜色已刷新")
end, { desc = "刷新 NvimTree 目录颜色" })

-- 创建专门的 .git 目录颜色测试命令

-- 创建测试命令（带调试信息）

-- 创建强制修复.gitignore颜色的命令

-- 创建手动刷新被忽略文件颜色的命令
vim.api.nvim_create_user_command("NvimTreeRefreshIgnored", function()
	print("🔄 手动刷新被忽略文件颜色...")
	
	-- 先清除所有可能的旧语法匹配
	for _, bufnr in ipairs(vim.api.nvim_list_bufs()) do
		local buf_name = vim.api.nvim_buf_get_name(bufnr)
		if buf_name:match("NvimTree_") then
			vim.api.nvim_buf_call(bufnr, function()
				-- 清除所有自定义语法匹配
				vim.cmd("silent! syntax clear NvimTreeIgnoredDir")
				vim.cmd("silent! syntax clear NvimTreeIgnoredFile")
				vim.cmd("silent! syntax clear NvimTreeIgnoredDirForce")
				vim.cmd("silent! syntax clear NvimTreeIgnoredFileForce")
				vim.cmd("silent! syntax clear NvimTreeNormalSpecialFile")
				vim.cmd("silent! syntax clear NvimTreeForceWhite")
				vim.cmd("silent! syntax clear NvimTreeForceGray")
				vim.cmd("silent! syntax clear NvimTreeForceGitGray")
				vim.cmd("silent! syntax clear NvimTreeGitDirCustom")
			end)
		end
	end
	
	-- 直接应用基于.gitignore文件内容的智能颜色系统
	require('configs.nvimtree').smart_color_directories(true) -- 启用调试模式
	
	
	print("✅ 被忽略文件颜色已刷新（包含.git目录）")
end, { desc = "手动刷新被忽略文件的颜色显示（基于.gitignore文件内容）" })

-- 强制设置.git目录颜色的简化函数
local function force_git_color()
	local highlights = require('configs.highlights')
	highlights.force_git_color()
end

-- 创建强制设置.git目录颜色的命令

-- 创建专门的 .git 目录浅灰色设置命令

-- 参考成功实现的方法设置 .git 目录颜色

-- 创建检查高亮组的命令

-- 定时器已移除，改为在需要时手动调用

-- ============================================================================
-- 立即应用强制 .git 目录颜色
local highlights = require('configs.highlights')
vim.schedule(function()
    highlights.setup_highlight_autocmds()
end)

-- 设置项目根目录文字颜色为醒目的亮绿色
vim.api.nvim_set_hl(0, "NvimTreeRootFolder", { fg = "#00ff7f" })

-- 创建测试命令来验证根目录颜色设置
vim.api.nvim_create_user_command("NvimTreeTestRootColor", function()
    local highlights = require('configs.highlights')
    highlights.set_nvim_tree_highlights()
    highlights.ultimate_git_color_fix()
    
    -- 强制设置根目录颜色
    vim.api.nvim_set_hl(0, "NvimTreeRootFolder", { fg = "#00ff7f" })
    
    -- 刷新 NvimTree
    local api = require('nvim-tree.api')
    if api.tree.is_visible() then
        api.tree.reload()
    end
    
    print("✅ 项目根目录颜色已设置为浅灰色 (#8a8a8a)")
end, { desc = "测试并设置项目根目录为浅灰色" })

-- 创建通配符测试命令
vim.api.nvim_create_user_command("NvimTreeTestWildcard", function()
    print("🎨 测试通配符模式匹配功能...")
    require('configs.nvimtree').smart_color_directories(true) -- 启用调试模式
    print("✅ 通配符模式匹配测试完成，请查看 NvimTree 中的文件颜色")
end, { desc = "测试通配符模式匹配功能" })

-- 创建通配符模式测试命令
vim.api.nvim_create_user_command("NvimTreeTestPattern", function(opts)
    local pattern = opts.args or "*.txt"
    local cwd = vim.fn.getcwd()
    
    print("🔍 测试模式匹配: " .. pattern)
    print("📁 当前目录: " .. cwd)
    
    -- 测试模式匹配
    local test_files = {}
    for item in vim.fs.dir(cwd) do
        local full_path = cwd .. "/" .. item
        if vim.fn.isdirectory(full_path) == 0 then -- 只测试文件
            local should_ignore = should_ignore_file(item, {pattern}, true)
            if should_ignore then
                table.insert(test_files, item)
            end
        end
    end
    
    if #test_files > 0 then
        print("✅ 匹配的文件:")
        for _, file in ipairs(test_files) do
            print("  📄 " .. file)
        end
    else
        print("❌ 没有文件匹配模式: " .. pattern)
    end
    
    -- 应用颜色
    require('configs.nvimtree').smart_color_directories(false)
end, { desc = "测试特定通配符模式", nargs = "?" })

-- ============================================================================
-- 目录打开时自动触发Git状态刷新功能
-- ============================================================================

-- 创建目录展开/折叠事件监听器
local function setup_directory_refresh_events()
    -- 创建自动命令组
    vim.api.nvim_create_augroup("NvimTreeDirectoryRefresh", { clear = true })
    
    -- 监听NvimTree缓冲区中的各种事件（优化性能，避免频繁刷新）
    vim.api.nvim_create_autocmd({
        "BufEnter", "BufWinEnter", "WinEnter"
    }, {
        group = "NvimTreeDirectoryRefresh",
        pattern = "*",
        callback = function()
            -- 确保当前是NvimTree缓冲区
            if vim.bo.filetype ~= "NvimTree" then return end
            
            -- 检查是否禁用了光标刷新模式，避免频繁的Git操作
            local config = _G.nvimtree_git_config or {}
            if config.disable_cursor_refresh then
                -- 在禁用模式下，只进行基本的高亮设置，不执行Git同步操作
                vim.defer_fn(function()
                    pcall(function()
                        local highlights = require('configs.highlights')
                        highlights.set_nvim_tree_highlights()
                        -- 跳过Git相关的同步操作，避免阻塞
                    end)
                end, 100) -- 增加延迟，减少频率
                return
            end
            
            -- 原有的Git刷新逻辑（仅在未禁用时执行）
            vim.defer_fn(function()
                -- 触发Git状态刷新
                pcall(function()
                    local highlights = require('configs.highlights')
                    highlights.set_nvim_tree_highlights()
                    highlights.ultimate_git_color_fix()
                    highlights.force_refresh_nvimtree()
                end)
                
                -- 强制刷新Git状态
                pcall(function()
                    require('nvim-tree.git').reload()
                    require('nvim-tree.api').tree.reload()
                end)
            end, 50) -- 50ms延迟，避免过于频繁的刷新
        end
    })
    
    -- 监听文件系统变化事件
    vim.api.nvim_create_autocmd({
        "DirChanged", "FileChangedShellPost"
    }, {
        group = "NvimTreeDirectoryRefresh",
        callback = function()
            -- 延迟执行，确保文件系统变化完成
            vim.defer_fn(function()
                if vim.bo.filetype == "NvimTree" then
                    pcall(function()
                        local highlights = require('configs.highlights')
                        highlights.set_nvim_tree_highlights()
                        highlights.ultimate_git_color_fix()
                        highlights.force_refresh_nvimtree()
                    end)
                end
            end, 100)
        end
    })
    
    -- 添加可控制的刷新机制（仅在启用时运行）
    local config = _G.nvimtree_git_config or {}
    if not config.disable_cursor_refresh then
        -- 只有在未禁用光标刷新时才启动定时器
        local refresh_timer = vim.loop.new_timer()
        if refresh_timer then
            refresh_timer:start(1000, 5000, vim.schedule_wrap(function()
                -- 减少刷新频率：每5秒检查一次（原来是2秒）
                if vim.bo.filetype == "NvimTree" then
                    pcall(function()
                        local highlights = require('configs.highlights')
                        highlights.set_nvim_tree_highlights()
                        highlights.ultimate_git_color_fix()
                    end)
                end
            end))
        end
    end
end

-- 创建手动刷新命令
vim.api.nvim_create_user_command("NvimTreeRefreshGit", function()
    print("🔄 手动刷新NvimTree Git状态...")
    
    -- 立即刷新Git状态，简化操作
    pcall(function()
        local highlights = require('configs.highlights')
        highlights.set_nvim_tree_highlights()
    end)
    
    -- 强制重新加载Git状态
    pcall(function()
        require('nvim-tree.git').reload()
        require('nvim-tree.api').tree.reload()
    end)
    
    print("✅ Git状态刷新完成")
end, { desc = "手动刷新NvimTree Git状态" })

-- 创建目录展开时自动刷新的测试命令
vim.api.nvim_create_user_command("NvimTreeTestDirectoryRefresh", function()
    print("🧪 测试目录展开时自动刷新功能...")
    
    -- 启用目录刷新事件
    setup_directory_refresh_events()
    
    print("✅ 目录展开自动刷新功能已启用")
    print("💡 现在当你展开/折叠目录时，Git状态会自动刷新")
    print("💡 你也可以使用 :NvimTreeRefreshGit 命令手动刷新")
end, { desc = "测试目录展开时自动刷新功能" })

-- 创建简单的Git状态测试命令
vim.api.nvim_create_user_command("NvimTreeTestGit", function()
    print("🧪 测试Git状态显示...")
    
    -- 检查当前目录是否是Git仓库
    local git_status = vim.fn.system("git status --porcelain 2>/dev/null")
    if vim.v.shell_error ~= 0 then
        print("❌ 当前目录不是Git仓库")
        return
    end
    
    print("✅ 当前目录是Git仓库")
    print("📊 Git状态信息:")
    
    -- 显示collect_static相关的Git状态
    local lines = {}
    for line in git_status:gmatch("[^\r\n]+") do
        if line:match("collect_static") then
            table.insert(lines, line)
        end
    end
    
    if #lines > 0 then
        print("📁 collect_static目录状态:")
        for _, line in ipairs(lines) do
            print("  " .. line)
        end
    else
        print("📁 collect_static目录没有Git状态变化")
    end
    
    -- 强制刷新NvimTree，简化操作
    pcall(function()
        local highlights = require('configs.highlights')
        highlights.set_nvim_tree_highlights()
    end)
    
    print("🔄 NvimTree已刷新")
end, { desc = "测试Git状态显示" })

-- 初始化目录刷新事件监听器
setup_directory_refresh_events()

-- ============================================================================
-- 🚀 性能监控和优化函数
-- ============================================================================

-- 性能监控函数
local function monitor_performance()
	local api = require('nvim-tree.api')
	if not api.tree.is_visible() then
		return
	end
	
	-- 统计当前显示的节点数量
	local tree = api.tree.get_nodes()
	local node_count = 0
	
	local function count_nodes(nodes)
		if not nodes then return end
		for _, node in pairs(nodes) do
			node_count = node_count + 1
			if node.nodes then
				count_nodes(node.nodes)
			end
		end
	end
	
	count_nodes(tree)
	
	if node_count > 1000 then
		vim.notify("🐌 NvimTree显示了 " .. node_count .. " 个节点，可能影响性能", vim.log.levels.WARN)
		vim.notify("💡 建议: 1) 折叠大目录 2) 使用过滤器隐藏不必要文件", vim.log.levels.INFO)
	end
end

-- 性能优化建议命令
vim.api.nvim_create_user_command("NvimTreePerformanceCheck", function()
	monitor_performance()
	
	-- 检查Git状态
	local git_dir = vim.fn.getcwd() .. "/.git"
	if vim.fn.isdirectory(git_dir) == 1 then
		local git_files = vim.fn.system("find " .. vim.fn.shellescape(vim.fn.getcwd()) .. " -type f | wc -l")
		local file_count = tonumber(git_files) or 0
		
		if file_count > 5000 then
			print("📊 性能建议:")
			print("  • 当前项目有 " .. file_count .. " 个文件")
			print("  • Git状态检查已优化为200ms超时")
			print("  • 目录Git状态已禁用以提升性能")
			print("  • 文件系统监控延迟设为300ms")
		else
			print("✅ 项目规模适中 (" .. file_count .. " 个文件)，性能应该良好")
		end
	else
		print("ℹ️ 非Git项目，性能影响较小")
	end
end, { desc = "检查NvimTree性能状态" })

-- 应急性能优化命令
vim.api.nvim_create_user_command("NvimTreeOptimizePerformance", function()
	-- 临时禁用一些性能密集的功能
	local config = require('nvim-tree').get_config()
	
	vim.notify("🚀 启用应急性能优化模式", vim.log.levels.INFO)
	print("📊 优化措施:")
	print("  • Git超时减少到100ms")
	print("  • 禁用文件系统监控")
	print("  • 禁用诊断信息显示")
	
	-- 这些优化需要重启nvim-tree才能生效
	vim.notify("⚠️ 部分优化需要重启Neovim才能生效", vim.log.levels.WARN)
end, { desc = "启用应急性能优化" })

-- 导出模块
return M


