-- ============================================================================
-- 核心配置模块 (core.lua)
-- 整合基础选项、自动命令、键位映射和弃用函数修复
-- 统一管理核心功能，减少配置文件分散
-- ============================================================================

local M = {}

-- ============================================================================
-- 1. 基础选项配置
-- ============================================================================
function M.setup_options()
	-- ============================================================================
	-- 1.1 基础配置初始化
	-- ============================================================================
	-- 获取 Neovim 选项对象
	local o = vim.o
	local opt = vim.opt

	-- ============================================================================
	-- 1.2 系统和编码设置
	-- ============================================================================
	o.encoding = "utf-8"                   -- 设置内部编码为 UTF-8
	o.fileencoding = "utf-8"               -- 设置文件编码为 UTF-8
	o.fileencodings = "utf-8,gbk,gb2312,big5" -- 文件编码检测顺序
	o.termguicolors = true                 -- 启用真彩色支持（24位颜色）

	-- ============================================================================
	-- 1.3 显示和界面设置
	-- ============================================================================
	o.number = true      -- 显示行号
	o.relativenumber = true -- 显示相对行号
	o.cursorline = true  -- 高亮当前行
	-- o.signcolumn = "yes" -- 始终显示符号列（用于显示诊断信息、断点等）
	-- o.colorcolumn = "80" -- 在第80列显示参考线（已禁用以避免右侧显示线条）
	o.textwidth = 200 -- 文本最长换行宽度
	o.wrap = true    -- 启用自动换行
	o.scrolloff = 8  -- 光标上下保持8行的滚动边距
	o.sidescrolloff = 8 -- 光标左右保持8列的滚动边距
	o.showmode = false -- 不显示模式信息（由状态栏插件处理）
	o.showcmd = true -- 显示命令信息
	o.cmdheight = 1  -- 命令行高度设为1以避免Press ENTER提示
	-- 设置消息选项以减少 "Press ENTER" 提示
	opt.shortmess:append("c")  -- 不显示补全消息
	opt.shortmess:append("F")  -- 不显示文件信息消息
	o.laststatus = 3 -- 全局状态栏（Neovim 0.7+）
	o.showtabline = 2 -- 始终显示标签页行

	-- GUI 选项设置（隐藏滚动条等界面元素）
	if vim.fn.has('gui_running') == 1 then
		o.guioptions = o.guioptions:gsub('[rLlbh]', '') -- 移除右侧、左侧滚动条和底部滚动条
	end

	-- 禁用vim-airline的scrollbar扩展
	vim.g['airline#extensions#scrollbar#enabled'] = 0

	-- 自定义airline section，移除scrollbar
	vim.g.airline_section_x = '%{airline#util#wrap(airline#parts#filetype(),0)}'

	-- 符号列设置（为书签功能提供固定宽度）
	o.signcolumn = 'yes:1'   -- 固定显示1列符号列（用于书签、诊断信息等）
	o.number = true      -- 显示行号
	o.relativenumber = true -- 显示相对行号
	-- 对于终端中的 Neovim，设置相关选项
	o.mouse = 'a'         -- 启用鼠标支持
	o.mousehide = true    -- 输入时隐藏鼠标

	-- ============================================================================
	-- 1.4 编辑和缩进设置
	-- ============================================================================
	o.expandtab = false -- 使用制表符而不是空格
	o.shiftwidth = 4  -- 自动缩进时使用的空格数
	o.tabstop = 4     -- 制表符显示的空格数
	o.softtabstop = 4 -- 编辑时制表符的宽度
	o.smartindent = true -- 智能缩进
	o.autoindent = true -- 自动缩进
	o.breakindent = true -- 换行时保持缩进

	-- ============================================================================
	-- 1.5 搜索和查找设置
	-- ============================================================================
	o.hlsearch = true -- 高亮搜索匹配
	o.incsearch = true -- 实时显示搜索匹配
	o.ignorecase = true -- 搜索时忽略大小写
	o.smartcase = true -- 智能大小写（包含大写字母时区分大小写）

	-- ============================================================================
	-- 1.6 分屏行为设置
	-- ============================================================================
	o.splitbelow = true -- 水平分屏时新窗口在下方
	o.splitright = true -- 垂直分屏时新窗口在右侧

	-- ============================================================================
	-- 1.7 系统集成和其他设置
	-- ============================================================================
	opt.clipboard:append("unnamedplus") -- 使用系统剪贴板
	o.history = 1000                 -- 命令历史记录数量
	o.autoread = true                -- 文件在外部被修改时自动重新读取
	o.backup = false                 -- 禁用备份文件
	o.writebackup = false            -- 禁用写入时备份
	o.swapfile = false               -- 禁用交换文件
	o.undofile = true                -- 启用持久化撤销
	o.updatetime = 300               -- 更新时间（毫秒）
	o.timeoutlen = 400               -- 键位映射超时时间
end

-- ============================================================================
-- 2. 弃用函数兼容性修复
-- ============================================================================
function M.setup_deprecation_fixes()
	-- ============================================================================
	-- 2.1 表操作函数修复
	-- ============================================================================

	-- 修复 vim.tbl_add_reverse_lookup 弃用警告
	-- 功能：为表添加反向查找，即将值作为键，键作为值
	if vim.tbl_add_reverse_lookup then
		local original = vim.tbl_add_reverse_lookup
		vim.tbl_add_reverse_lookup = function(t)
			return original(t)
		end
	end

	-- 修复 vim.tbl_flatten 弃用警告
	-- 功能：将嵌套表扁平化为一维数组
	if vim.tbl_flatten then
		local original = vim.tbl_flatten
		vim.tbl_flatten = function(t)
			return original(t)
		end
	end

	-- 修复 vim.tbl_islist 弃用警告
	-- 功能：检查表是否为列表（数组）
	if vim.tbl_islist then
		local original = vim.tbl_islist
		vim.tbl_islist = function(t)
			return original(t)
		end
	end

	-- ============================================================================
	-- 2.2 字符串处理函数修复
	-- ============================================================================

	-- 修复 vim.str_utfindex 弃用警告
	-- 功能：获取字符串中指定 UTF-8 字符索引对应的字节索引
	if vim.str_utfindex then
		local original = vim.str_utfindex
		vim.str_utfindex = function(s, encoding, index, strict_indexing)
			if strict_indexing ~= nil then
				return original(s, encoding, index, strict_indexing)
			elseif encoding ~= nil and index ~= nil then
				return original(s, encoding, index)
			else
				return original(s, encoding)
			end
		end
	end
end

-- ============================================================================
-- 3. 统一初始化函数
-- ============================================================================
function M.setup()
	-- 初始化基础选项
	M.setup_options()

	-- 初始化弃用函数修复
	M.setup_deprecation_fixes()
end

return M
