-- ============================================================================
-- NvChad 主配置文件 (chadrc.lua)
-- 这个文件是 NvChad 的核心配置文件，用于自定义主题、UI 和插件设置
-- ============================================================================

-- ============================================================================
-- 1. 基础配置初始化
-- ============================================================================
local M = {}

-- ============================================================================
-- 2. 主题配置 (Base46)
-- Base46 是 NvChad 的主题系统，提供了多种预设主题和自定义选项
-- ============================================================================
M.base46 = {
	theme = "onedark", -- 选择主题名称，可选: onedark, gruvbox, catppuccin 等
	transparency = true, -- 启用透明背景，需要终端支持透明度

	-- 高亮覆盖设置 - 用于自定义特定元素的颜色
	hl_override = {
		-- 当前行背景色设置
		CursorLine = { bg = "#bfbfbf" },
		-- 当前行号的颜色和样式设置
		CursorLineNr = { fg = "#e5c385", bold = true },
		-- 只有当前活跃 buffer 使用白色加粗下划线样式
		TbBufOn = { fg = "#ffffff", bold = true, underline = true },
	},
}

-- ============================================================================
-- 3. 用户界面配置 (UI)
-- 配置 NvChad 的用户界面元素，包括标签栏、文件树等
-- ============================================================================
M.ui = {
	-- CMP 补全菜单配置
	cmp = {
		icons_left = true, -- 保持图标在左侧
		style = "default", -- 可选: default/atom/atom_colored  
	},

	-- Statusline 配置
	statusline = {
		enabled = true,
		theme = "minimal", -- 可选: default/minimal/vscode/vscode_colored
		separator_style = "round", -- 仅default主题可用: default/round/block/arrow
	},

	-- 标签缓冲区行配置
	tabufline = {
		lazyload = false, -- 禁用懒加载，立即加载标签栏
		-- 使用自定义模块实现自适应宽度和智能样式保持
		modules = {
			buffers = function()
				local buffers = {}
				local has_current = false
				
				-- 获取 buffer 列表
				vim.t.bufs = vim.tbl_filter(vim.api.nvim_buf_is_valid, vim.t.bufs or {})
				
				-- 检查当前窗口类型
				local current_buf = vim.api.nvim_get_current_buf()
				local current_buftype = vim.api.nvim_get_option_value('buftype', { buf = current_buf })
				local current_filetype = vim.api.nvim_get_option_value('filetype', { buf = current_buf })
				local is_normal_window = current_buftype == '' and 
					current_filetype ~= 'NvimTree' and 
					current_filetype ~= 'terminal' and
					current_filetype ~= 'help' and
					current_filetype ~= 'alpha'
				
				-- 更新最后活跃的编辑 buffer
				if is_normal_window then
					vim.g.last_active_edit_buf = current_buf
				end
				
				-- 自适应宽度实现与智能样式
				for i, nr in ipairs(vim.t.bufs) do
					local filename = vim.api.nvim_buf_get_name(nr):match("([^/\\]+)[/\\]*$") or "No Name"
					
					-- 智能判断哪个 buffer 应该显示为活跃状态
					local should_be_active
					if is_normal_window then
						-- 在普通编辑窗口中，当前 buffer 为活跃
						should_be_active = (current_buf == nr)
					else
						-- 在特殊窗口中，最后编辑的 buffer 保持活跃样式
						should_be_active = (vim.g.last_active_edit_buf == nr)
					end
					
					-- 计算自适应宽度
					local adaptive_width = math.min(math.max(#filename + 8, 12), 30)
					
					-- 创建自定义的 style_buf 来处理活跃状态
					local function custom_style_buf(bufnr, idx, width)
						local utils = require("nvchad.tabufline.utils")
						local api = vim.api
						local strep = string.rep
						
						-- 获取文件信息
						local buf_name = api.nvim_buf_get_name(bufnr)
						local fname = buf_name:match("([^/\\]+)[/\\]*$") or "No Name"
						
						-- 强制设置高亮组名称
						local tbHlName = should_be_active and "BufOn" or "BufOff"
						
						-- 获取图标（使用透明背景）
						local icon = "󰈚"
						local icon_hl = "%#Tb" .. tbHlName .. "#"
						
						if fname ~= "No Name" then
							local ok, devicons = pcall(require, "nvim-web-devicons")
							if ok then
								local file_icon, devicon_hl = devicons.get_icon(fname)
								if file_icon then
									icon = file_icon
									-- 创建透明背景的图标高亮组
									local icon_fg = api.nvim_get_hl(0, { name = devicon_hl }).fg or 16777215
									local bg_color = api.nvim_get_hl(0, { name = "Tb" .. tbHlName }).bg
									api.nvim_set_hl(0, devicon_hl .. tbHlName, { 
										fg = icon_fg, 
										bg = bg_color  -- 使用与buffer标签相同的背景色
									})
									icon_hl = "%#" .. devicon_hl .. tbHlName .. "#"
								end
							end
						end
						
						-- 计算填充
						local content_len = #icon + 1 + #fname + 1 + 2
						local total_pad = width - content_len
						local left_pad = math.max(math.floor(total_pad / 2), 1)
						local right_pad = math.max(total_pad - left_pad, 1)
						
						-- 构建内容
						local close_btn = " 󰅖 "
						local content = strep(" ", left_pad) .. icon_hl .. icon .. " %#Tb" .. tbHlName .. "#" .. fname .. strep(" ", right_pad - 2) .. close_btn
						
						-- 添加点击功能
						return utils.btn(content, nil, "GoToBuf", bufnr)
					end
					
					local buf_content = custom_style_buf(nr, i, adaptive_width)
					
					has_current = should_be_active or has_current
					table.insert(buffers, buf_content)
					
					-- 空间限制检查
					if #buffers > 8 then
						if has_current then
							break
						else
							table.remove(buffers, 1)
						end
					end
				end
				
				local txt = require("nvchad.tabufline.utils").txt
				return table.concat(buffers) .. txt("%=", "Fill")
			end
		}
	},

	-- nvim-tree 文件树配置
	nvimtree = {
		side = "left", -- 文件树显示在左侧
	},
}

-- ============================================================================
-- 4. 启动页面配置 (NvDash)
-- NvDash 是 NvChad 的启动页面，显示最近文件、快捷操作等
-- ============================================================================
M.nvdash = {
	load_on_startup = true, -- 启动时自动加载 NvDash 页面
}

-- ============================================================================
-- 5. 插件配置
-- ============================================================================
-- 引用外部插件配置文件
M.plugins = "configs.plugins"

-- ============================================================================
-- 6. 配置导出
-- ============================================================================
-- 返回配置表，供 NvChad 使用
return M
