# 🎨 LuaSnip 代码片段自定义指南

## 📍 配置文件位置
主配置文件：`/Users/dyx/.config/nvim/lua/configs/luasnip.lua`

## 🔧 基本语法

## 1. 创建简单片段
```lua
-- 基本格式
s("触发词", {
    t("静态文本"),
    i(1, "第一个占位符"),
    t("更多静态文本"),
    i(2, "第二个占位符"), 
    i(0)  -- 最终光标位置
})
```

### 2. 高级功能

#### 多行文本
```lua
s("multiline", {
    t("第一行"),
    t({"", "第二行"}),  -- {"", "text"} 表示换行+文本
    t({"", "\t缩进的第三行"}),  -- \t 表示制表符
    i(0)
})
```

#### 动态函数节点
```lua
s("date", {
    t("今天是："),
    f(function() return os.date("%Y-%m-%d") end, {}),  -- 插入当前日期
    i(0)
})

-- 根据其他占位符生成内容
s("getter", {
    t("private "),
    i(1, "type"),
    t(" _"),
    i(2, "name"),
    t(";"),
    t({"", "", "public "}),
    f(function(args) return args[1][1] end, {1}),  -- 复制第1个占位符的内容
    t(" get"),
    f(function(args) 
        local name = args[1][1] or "Name"
        return name:sub(1,1):upper() .. name:sub(2)  -- 首字母大写
    end, {2}),
    t("() { return this._"),
    f(function(args) return args[1][1] end, {2}),
    t("; }"),
    i(0)
})
```

## 🎯 实用自定义示例

### JavaScript/TypeScript 自定义

在 `javascript_snippets` 表中添加：

```lua
-- API调用模板
s("api", {
    t("const "),
    i(1, "response"),
    t(" = await fetch('"),
    i(2, "api/endpoint"),
    t("', {"),
    t({"", "\tmethod: '"}),
    i(3, "GET"),
    t("',"),
    t({"", "\theaders: {"}),
    t({"", "\t\t'Content-Type': 'application/json',"}),
    t({"", "\t},"}),
    t({"", "\tbody: JSON.stringify("}),
    i(4, "data"),
    t(")"),
    t({"", "});"}),
    t({"", "const "}),
    i(5, "result"),
    t(" = await "),
    f(function(args) return args[1][1] end, {1}),
    t(".json();"),
    i(0)
}),

-- 测试用例模板  
s("test", {
    t("describe('"),
    i(1, "测试描述"),
    t("', () => {"),
    t({"", "\tit('"}),
    i(2, "具体测试用例"),
    t("', () => {"),
    t({"", "\t\t// Arrange"}),
    t({"", "\t\t"}),
    i(3, "// 准备测试数据"),
    t({"", "", "\t\t// Act"}),
    t({"", "\t\t"}),
    i(4, "// 执行操作"),
    t({"", "", "\t\t// Assert"}),
    t({"", "\t\t"}),
    i(5, "// 验证结果"),
    t({"", "\t});"}),
    t({"", "});"}),
    i(0)
}),

-- localStorage 操作
s("ls", {
    t("localStorage."),
    i(1, "setItem"),
    t("('"),
    i(2, "key"),
    t("', "),
    i(3, "value"),
    t(");"),
    i(0)
}),
```

### Python 自定义

在 `python_snippets` 表中添加：

```lua
-- 装饰器模板
s("deco", {
    t("def "),
    i(1, "decorator_name"),
    t("(func):"),
    t({"", "\tdef wrapper(*args, **kwargs):"}),
    t({"", "\t\t"}),
    i(2, "# 装饰器逻辑"),
    t({"", "\t\tresult = func(*args, **kwargs)"}),
    t({"", "\t\t"}),
    i(3, "# 后处理逻辑"),
    t({"", "\t\treturn result"}),
    t({"", "\treturn wrapper"}),
    i(0)
}),

-- 异常处理模板
s("tryex", {
    t("try:"),
    t({"", "\t"}),
    i(1, "# 可能出错的代码"),
    t({"", "except "}),
    i(2, "Exception"),
    t(" as "),
    i(3, "e"),
    t(":"),
    t({"", "\tprint(f\"错误: {"}),
    f(function(args) return args[1][1] end, {3}),
    t("}\")"),
    t({"", "\t"}),
    i(4, "# 异常处理逻辑"),
    t({"", "finally:"}),
    t({"", "\t"}),
    i(5, "# 清理代码"),
    i(0)
}),
```

### HTML 自定义

在 `html_snippets` 表中添加：

```lua
-- Bootstrap 卡片组件
s("bscard", {
    t("<div class=\"card\" style=\"width: "),
    i(1, "18rem"),
    t(";\">"),
    t({"", "\t<img src=\""}),
    i(2, "image.jpg"),
    t("\" class=\"card-img-top\" alt=\""),
    i(3, "图片描述"),
    t("\">"),
    t({"", "\t<div class=\"card-body\">"}),
    t({"", "\t\t<h5 class=\"card-title\">"}),
    i(4, "卡片标题"),
    t("</h5>"),
    t({"", "\t\t<p class=\"card-text\">"}),
    i(5, "卡片内容描述"),
    t("</p>"),
    t({"", "\t\t<a href=\""}),
    i(6, "#"),
    t("\" class=\"btn btn-primary\">"),
    i(7, "按钮文本"),
    t("</a>"),
    t({"", "\t</div>"}),
    t({"", "</div>"}),
    i(0)
}),
```

## 🚀 如何添加新片段

### 步骤1：编辑配置文件
```bash
nvim ~/.config/nvim/lua/configs/luasnip.lua
```

### 步骤2：找到对应语言的片段表
- JavaScript/TypeScript → `javascript_snippets`
- Python → `python_snippets`  
- HTML → `html_snippets`
- React → `react_snippets`

### 步骤3：添加你的片段
```lua
-- 在对应表中添加
s("你的触发词", {
    -- 你的片段内容
    t("静态文本"),
    i(1, "占位符"),
    i(0)
}),
```

### 步骤4：重启Neovim或重新加载配置
```vim
:source ~/.config/nvim/lua/configs/luasnip.lua
```

## 💡 高级技巧

### 1. 条件片段
```lua
-- 只在特定条件下显示的片段
s("debug", {
    t("console.log('DEBUG: "),
    i(1, "message"),
    t("');"),
    i(0)
}, {
    condition = function()
        -- 只在开发环境下显示
        return vim.fn.expand("%:t"):match("%.dev%.")
    end
})
```

### 2. 选择节点
```lua
local c = ls.choice_node

s("method", {
    t("public "),
    c(1, {t("static "), t("")}),  -- 可选择static或空
    i(2, "void"),
    t(" "),
    i(3, "methodName"),
    t("() {"),
    t({"", "\t"}),
    i(0),
    t({"", "}"}),
})
```

### 3. 正则表达式触发
```lua
s({trig="(%d+)px", regTrig=true}, {
    f(function(args, snip)
        return snip.captures[1] .. " * 0.0625rem" -- px转rem
    end, {})
})
```

## 📝 调试和测试

### 查看当前加载的片段
```vim
:lua print(vim.inspect(require("luasnip").get_snippets()))
```

### 重新加载片段配置
```vim
:lua require("luasnip.loaders.from_lua").load({paths = "~/.config/nvim/lua/configs/luasnip.lua"})
```

## 🎨 快捷键提醒

- **Tab** - 展开片段/跳转到下一个占位符
- **Shift+Tab** - 跳转到上一个占位符  
- **Ctrl+E** - 切换选择项（在choice_node中）

记住：修改配置后需要重启Neovim才能生效！
