-- ============================================================================
-- TailwindCSS 补全配置（独立于 LSP）
-- ============================================================================

local M = {}

-- 常用 TailwindCSS 类名列表（手动维护）
local tailwind_classes = {
  -- 布局
  "block", "inline-block", "inline", "flex", "inline-flex", "table", "inline-table",
  "table-caption", "table-cell", "table-column", "table-column-group", "table-footer-group",
  "table-header-group", "table-row-group", "table-row", "flow-root", "grid", "inline-grid",
  "contents", "list-item", "hidden",
  
  -- Flexbox & Grid
  "flex-1", "flex-auto", "flex-initial", "flex-none", "flex-shrink-0", "flex-shrink",
  "flex-grow-0", "flex-grow", "basis-0", "basis-1", "basis-auto", "basis-px", "basis-0.5",
  "flex-row", "flex-row-reverse", "flex-col", "flex-col-reverse", "flex-wrap", "flex-wrap-reverse", "flex-nowrap",
  "items-start", "items-end", "items-center", "items-baseline", "items-stretch",
  "justify-start", "justify-end", "justify-center", "justify-between", "justify-around", "justify-evenly",
  
  -- 颜色
  "text-black", "text-white", "text-gray-100", "text-gray-200", "text-gray-300", "text-gray-400", "text-gray-500",
  "text-gray-600", "text-gray-700", "text-gray-800", "text-gray-900",
  "text-red-100", "text-red-200", "text-red-300", "text-red-400", "text-red-500", "text-red-600", "text-red-700", "text-red-800", "text-red-900",
  "text-blue-100", "text-blue-200", "text-blue-300", "text-blue-400", "text-blue-500", "text-blue-600", "text-blue-700", "text-blue-800", "text-blue-900",
  "text-green-100", "text-green-200", "text-green-300", "text-green-400", "text-green-500", "text-green-600", "text-green-700", "text-green-800", "text-green-900",
  
  "bg-black", "bg-white", "bg-gray-50", "bg-gray-100", "bg-gray-200", "bg-gray-300", "bg-gray-400", "bg-gray-500",
  "bg-gray-600", "bg-gray-700", "bg-gray-800", "bg-gray-900",
  "bg-red-50", "bg-red-100", "bg-red-200", "bg-red-300", "bg-red-400", "bg-red-500", "bg-red-600", "bg-red-700", "bg-red-800", "bg-red-900",
  "bg-blue-50", "bg-blue-100", "bg-blue-200", "bg-blue-300", "bg-blue-400", "bg-blue-500", "bg-blue-600", "bg-blue-700", "bg-blue-800", "bg-blue-900",
  
  -- 尺寸
  "w-0", "w-px", "w-0.5", "w-1", "w-1.5", "w-2", "w-2.5", "w-3", "w-3.5", "w-4", "w-5", "w-6", "w-7", "w-8", "w-9", "w-10", "w-11", "w-12",
  "w-14", "w-16", "w-20", "w-24", "w-28", "w-32", "w-36", "w-40", "w-44", "w-48", "w-52", "w-56", "w-60", "w-64", "w-72", "w-80", "w-96",
  "w-auto", "w-1/2", "w-1/3", "w-2/3", "w-1/4", "w-2/4", "w-3/4", "w-full", "w-screen", "w-min", "w-max", "w-fit",
  
  "h-0", "h-px", "h-0.5", "h-1", "h-1.5", "h-2", "h-2.5", "h-3", "h-3.5", "h-4", "h-5", "h-6", "h-7", "h-8", "h-9", "h-10", "h-11", "h-12",
  "h-14", "h-16", "h-20", "h-24", "h-28", "h-32", "h-36", "h-40", "h-44", "h-48", "h-52", "h-56", "h-60", "h-64", "h-72", "h-80", "h-96",
  "h-auto", "h-1/2", "h-1/3", "h-2/3", "h-1/4", "h-2/4", "h-3/4", "h-full", "h-screen", "h-min", "h-max", "h-fit",
  
  -- 间距
  "p-0", "p-px", "p-0.5", "p-1", "p-1.5", "p-2", "p-2.5", "p-3", "p-3.5", "p-4", "p-5", "p-6", "p-7", "p-8", "p-9", "p-10", "p-11", "p-12",
  "p-14", "p-16", "p-20", "p-24", "p-28", "p-32", "p-36", "p-40", "p-44", "p-48", "p-52", "p-56", "p-60", "p-64", "p-72", "p-80", "p-96",
  
  "m-0", "m-px", "m-0.5", "m-1", "m-1.5", "m-2", "m-2.5", "m-3", "m-3.5", "m-4", "m-5", "m-6", "m-7", "m-8", "m-9", "m-10", "m-11", "m-12",
  "m-14", "m-16", "m-20", "m-24", "m-28", "m-32", "m-36", "m-40", "m-44", "m-48", "m-52", "m-56", "m-60", "m-64", "m-72", "m-80", "m-96",
  "m-auto", "-m-0.5", "-m-1", "-m-1.5", "-m-2", "-m-2.5", "-m-3", "-m-3.5", "-m-4", "-m-5", "-m-6", "-m-7", "-m-8", "-m-9", "-m-10", "-m-11", "-m-12",
  
  -- 边框
  "border", "border-0", "border-2", "border-4", "border-8", "border-t", "border-r", "border-b", "border-l",
  "border-solid", "border-dashed", "border-dotted", "border-double", "border-none",
  "rounded", "rounded-none", "rounded-sm", "rounded-md", "rounded-lg", "rounded-xl", "rounded-2xl", "rounded-3xl", "rounded-full",
  
  -- 阴影
  "shadow", "shadow-sm", "shadow-md", "shadow-lg", "shadow-xl", "shadow-2xl", "shadow-inner", "shadow-none",
  
  -- 字体
  "text-xs", "text-sm", "text-base", "text-lg", "text-xl", "text-2xl", "text-3xl", "text-4xl", "text-5xl", "text-6xl", "text-7xl", "text-8xl", "text-9xl",
  "font-thin", "font-extralight", "font-light", "font-normal", "font-medium", "font-semibold", "font-bold", "font-extrabold", "font-black",
  "text-left", "text-center", "text-right", "text-justify",
  
  -- 响应式前缀
  "sm:", "md:", "lg:", "xl:", "2xl:",
  "hover:", "focus:", "active:", "visited:", "disabled:",
}

-- 为 nvim-cmp 创建自定义补全源
local function create_tailwind_source()
  local source = {}
  source.new = function()
    return setmetatable({}, { __index = source })
  end

  source.get_trigger_characters = function()
    return { '"', "'", " " }
  end

  source.complete = function(self, request, callback)
    local input = string.sub(request.context.cursor_before_line, request.offset)
    local items = {}
    
    -- 只在 class 属性内提供补全
    local line = request.context.cursor_before_line
    if not (string.match(line, 'class%s*=%s*["\'][^"\']*$') or string.match(line, 'className%s*=%s*["\'][^"\']*$')) then
      callback({ items = {}, isIncomplete = false })
      return
    end
    
    for _, class_name in ipairs(tailwind_classes) do
      if string.find(class_name, input, 1, true) == 1 then
        table.insert(items, {
          label = class_name,
          kind = 21, -- Constant
          detail = "TailwindCSS",
          documentation = "TailwindCSS utility class",
        })
      end
    end
    
    callback({ items = items, isIncomplete = false })
  end

  return source
end

-- 设置函数
function M.setup()
  local cmp = require("cmp")
  
  -- 注册自定义补全源
  cmp.register_source("tailwindcss", create_tailwind_source())
  
  -- 为 HTML 相关文件类型添加补全源
  local filetypes = { "html", "htm", "vue", "jsx", "tsx", "javascriptreact", "typescriptreact" }
  for _, filetype in ipairs(filetypes) do
    cmp.setup.filetype(filetype, {
      sources = cmp.config.sources({
        { name = "tailwindcss" },
        { name = "nvim_lsp" },
        { name = "buffer" },
        { name = "path" },
      })
    })
  end
end

return M