-- ============================================================================
-- GitSigns 配置文件 (configs/gitsigns.lua)
-- 提供 Git 集成功能：显示变更标志、hunk 导航、staging 等
-- ============================================================================

local gitsigns = require("gitsigns")

gitsigns.setup({
	-- ============================================================================
	-- 标志配置
	-- ============================================================================
	signs = {
		add          = { text = '┃' },  -- 新增行标志
		change       = { text = '┃' },  -- 修改行标志
		delete       = { text = '_' },  -- 删除行标志
		topdelete    = { text = '‾' },  -- 顶部删除标志
		changedelete = { text = '~' },  -- 修改删除标志
		untracked    = { text = '┆' },  -- 未跟踪文件标志
	},
	
	-- ============================================================================
	-- 行号列显示标志
	-- ============================================================================
	signs_staged_enable = true,  -- 显示暂存区标志
	signcolumn = true,           -- 在标志列显示标志
	numhl      = false,          -- 高亮行号
	linehl     = false,          -- 高亮整行
	word_diff  = false,          -- 单词级别差异
	
	-- ============================================================================
	-- 监控设置
	-- ============================================================================
	watch_gitdir = {
		follow_files = true
	},
	auto_attach = true,          -- 自动附加到 Git 仓库
	attach_to_untracked = true,  -- 附加到未跟踪文件
	
	-- ============================================================================
	-- 当前行责任信息 (blame)
	-- ============================================================================
	current_line_blame = false,  -- 默认不显示当前行的 blame 信息
	current_line_blame_opts = {
		virt_text = true,        -- 虚拟文本显示
		virt_text_pos = 'eol',   -- 在行末显示
		delay = 1000,            -- 延迟1秒显示
		ignore_whitespace = false,
	},
	current_line_blame_formatter = '<author>, <author_time:%Y-%m-%d> - <summary>',
	
	-- ============================================================================
	-- 预览设置
	-- ============================================================================
	preview_config = {
		border = 'single',       -- 预览窗口边框
		style = 'minimal',       -- 最小化样式
		relative = 'cursor',     -- 相对光标位置
		row = 0,
		col = 1
	},
	
	-- ============================================================================
	-- 自动命令和事件处理
	-- ============================================================================
	on_attach = function(bufnr)
		local gs = package.loaded.gitsigns
		
		-- 定义按键映射的辅助函数
		local function map(mode, l, r, opts)
			opts = opts or {}
			opts.buffer = bufnr
			vim.keymap.set(mode, l, r, opts)
		end
		
		-- ========================================================================
		-- 导航快捷键
		-- ========================================================================
		
		-- 下一个 hunk
		map('n', ']c', function()
			if vim.wo.diff then
				vim.cmd.normal({']c', bang = true})
			else
				gs.nav_hunk('next')
			end
		end, { desc = "下一个 Git hunk" })
		
		-- 上一个 hunk
		map('n', '[c', function()
			if vim.wo.diff then
				vim.cmd.normal({'[c', bang = true})
			else
				gs.nav_hunk('prev')
			end
		end, { desc = "上一个 Git hunk" })
		
		-- ========================================================================
		-- 暂存操作快捷键
		-- ========================================================================
		map('n', '<leader>gs', gs.stage_hunk, { desc = "暂存当前 hunk" })
		map('n', '<leader>gr', gs.reset_hunk, { desc = "重置当前 hunk" })
		map('v', '<leader>gs', function() gs.stage_hunk {vim.fn.line('.'), vim.fn.line('v')} end, { desc = "暂存选中的 hunk" })
		map('v', '<leader>gr', function() gs.reset_hunk {vim.fn.line('.'), vim.fn.line('v')} end, { desc = "重置选中的 hunk" })
		
		-- 暂存/重置整个缓冲区
		map('n', '<leader>gS', gs.stage_buffer, { desc = "暂存整个文件" })
		map('n', '<leader>gR', gs.reset_buffer, { desc = "重置整个文件" })
		
		-- ========================================================================
		-- 撤销和恢复操作
		-- ========================================================================
		map('n', '<leader>gu', gs.undo_stage_hunk, { desc = "撤销暂存 hunk" })
		
		-- ========================================================================
		-- 预览和查看操作
		-- ========================================================================
		map('n', '<leader>gp', gs.preview_hunk, { desc = "预览 hunk" })
		map('n', '<leader>gb', function() gs.blame_line{full=true} end, { desc = "查看行 blame 信息" })
		map('n', '<leader>gB', gs.toggle_current_line_blame, { desc = "切换当前行 blame 显示" })
		map('n', '<leader>gd', gs.diffthis, { desc = "查看文件 diff" })
		map('n', '<leader>gD', function() gs.diffthis('~') end, { desc = "查看与上次提交的 diff" })
		
		-- ========================================================================
		-- 显示切换
		-- ========================================================================
		map('n', '<leader>gt', gs.toggle_deleted, { desc = "切换显示删除的行" })
		map('n', '<leader>gw', gs.toggle_word_diff, { desc = "切换单词级 diff" })
		
		-- ========================================================================
		-- 文本对象 (用于选择 hunk)
		-- ========================================================================
		map({'o', 'x'}, 'ih', ':<C-U>Gitsigns select_hunk<CR>', { desc = "选择 hunk" })
	end
})

-- ============================================================================
-- 高亮组颜色配置
-- ============================================================================
vim.defer_fn(function()
	-- Git 标志颜色
	vim.api.nvim_set_hl(0, 'GitSignsAdd', { fg = '#98be65' })         -- 绿色：新增
	vim.api.nvim_set_hl(0, 'GitSignsChange', { fg = '#ECBE7B' })      -- 橙色：修改
	vim.api.nvim_set_hl(0, 'GitSignsDelete', { fg = '#ec5f67' })      -- 红色：删除
	vim.api.nvim_set_hl(0, 'GitSignsUntracked', { fg = '#5c6370' })   -- 灰色：未跟踪
end, 100)