-- ============================================================================
-- LazyVim 风格的自动命令配置
-- 从原 NvChad 配置迁移的自动命令设置
-- ============================================================================

local api = vim.api
local fn = vim.fn

-- ============================================================================
-- 编辑器增强自动命令
-- ============================================================================

-- 突出显示yanked文本
api.nvim_create_autocmd("TextYankPost", {
	callback = function()
		vim.highlight.on_yank({
			higroup = "IncSearch",
			timeout = 40,
		})
	end,
})

-- 禁用自动注释新行
api.nvim_create_autocmd("BufEnter", {
	callback = function()
		vim.opt.formatoptions:remove({ "c", "r", "o" })
	end,
})

-- ============================================================================
-- 文件类型特定设置
-- ============================================================================

-- 特定文件类型设置
api.nvim_create_autocmd("FileType", {
	pattern = { "gitcommit", "markdown", "text" },
	callback = function()
		vim.opt_local.wrap = true
		vim.opt_local.spell = true
	end,
})

-- ============================================================================
-- 窗口和缓冲区管理
-- ============================================================================

-- 离开时自动切换到英文输入法（macOS）
api.nvim_create_autocmd({ "InsertLeave", "CmdlineLeave" }, {
	callback = function()
		if fn.has("mac") == 1 then
			vim.fn.system("im-select com.apple.keylayout.ABC")
		end
	end,
})

-- 文件恢复设置
api.nvim_create_autocmd("BufReadPost", {
	callback = function()
		local line = vim.fn.line("'\"")
		if line > 1 and line <= vim.fn.line("$") then
			vim.cmd("normal! g'\"")
		end
	end,
})

-- ============================================================================
-- 文件保存时的操作
-- ============================================================================

api.nvim_create_autocmd("BufWritePre", {
	callback = function()
		-- 保存时自动创建不存在的目录
		local dir = fn.expand("%:p:h")
		if fn.isdirectory(dir) == 0 then
			fn.mkdir(dir, "p")
		end

		-- 移除行尾空格
		local save_cursor = api.nvim_win_get_cursor(0)
		vim.cmd([[%s/\s\+$//e]])
		api.nvim_win_set_cursor(0, save_cursor)
	end,
})

-- ============================================================================
-- Quickfix窗口设置
-- ============================================================================

api.nvim_create_autocmd("FileType", {
	pattern = "qf",
	callback = function()
		vim.opt_local.number = true
		vim.opt_local.relativenumber = false
		vim.keymap.set("n", "q", "<cmd>close<cr>", { buffer = true, silent = true })
		vim.keymap.set("n", "<C-n>", "<cmd>cnext<cr>", { buffer = true, silent = true })
		vim.keymap.set("n", "<C-p>", "<cmd>cprev<cr>", { buffer = true, silent = true })
	end,
})

-- ============================================================================
-- 原配置中的特殊自动命令迁移
-- ============================================================================

-- 自动设置文件权限（可执行文件）
api.nvim_create_autocmd("BufWritePost", {
	pattern = { "*.sh", "*.py", "*.pl", "*.rb" },
	callback = function()
		if fn.getfsize(fn.expand("%:p")) > 0 then
			vim.fn.system("chmod +x " .. fn.expand("%:p"))
		end
	end,
})

-- 自动刷新 Git 状态
api.nvim_create_autocmd({ "BufWritePost", "BufEnter", "FocusGained", "CursorHold", "CursorHoldI" }, {
	callback = function()
		-- 检查是否有 git 仓库
		if fn.isdirectory(".git") == 1 then
			-- 刷新 Git 状态相关插件
			pcall(require, "gitsigns")
			if package.loaded["gitsigns"] then
				require("gitsigns").refresh()
			end
		end
	end,
})

-- 自动格式化某些文件类型
api.nvim_create_autocmd("BufWritePre", {
	pattern = { "*.lua", "*.py", "*.js", "*.ts", "*.jsx", "*.tsx", "*.json", "*.yaml", "*.yml" },
	callback = function()
		-- 尝试格式化当前缓冲区
		local ok, conform = pcall(require, "conform")
		if ok then
			conform.format({ buf = 0, async = false })
		end
	end,
})

-- ============================================================================
-- 性能优化自动命令
-- ============================================================================

api.nvim_create_autocmd("BufReadPost", {
	callback = function()
		-- 延迟加载语法高亮以提高大文件性能
		local size = fn.getfsize(fn.expand("%:p"))
		if size > 1024 * 1024 then -- 1MB
			vim.cmd("syntax off")
			vim.defer_fn(function()
				vim.cmd("syntax on")
			end, 100)
		end
	end,
})

-- ============================================================================
-- 终端设置
-- ============================================================================

api.nvim_create_autocmd("TermOpen", {
	callback = function()
		vim.opt_local.number = false
		vim.opt_local.relativenumber = false
		vim.opt_local.cursorline = false
		vim.cmd("startinsert!")
	end,
})

-- 终端关闭时自动切换到普通模式
api.nvim_create_autocmd("BufLeave", {
	callback = function()
		if vim.bo.buftype == "terminal" then
			vim.cmd("stopinsert")
		end
	end,
})

-- ============================================================================
-- 目录切换设置
-- ============================================================================

api.nvim_create_autocmd({ "VimEnter", "BufEnter", "WinEnter" }, {
	callback = function()
		local bufname = api.nvim_buf_get_name(0)
		local dir = bufname:match("^(.*/)")
		if dir and dir ~= vim.fn.getcwd() then
			pcall(vim.cmd, "cd " .. dir)
		end
	end,
})

-- ============================================================================
-- NvimTree 透明背景自动应用
-- ============================================================================

api.nvim_create_autocmd("FileType", {
	pattern = "NvimTree",
	callback = function()
		-- 立即应用透明背景高亮
		api.nvim_set_hl(0, "NvimTreeNormal", { bg = "NONE" })
		api.nvim_set_hl(0, "NvimTreeNormalNC", { bg = "NONE" })

		-- 延迟应用完整的高亮配置以确保生效（减少延迟避免闪烁）
		vim.defer_fn(function()
			local ok, highlights = pcall(require, "config.highlights")
			if ok and highlights and highlights.setup_all_highlights then
				highlights.setup_all_highlights()
			end
		end, 50) -- 减少延迟时间
	end,
})

-- 全局透明背景设置 - 防抖避免重复调用
local transparency_timer = nil
api.nvim_create_autocmd({ "VimEnter", "WinEnter", "BufEnter" }, {
	callback = function()
		-- 防抖机制，避免短时间内重复调用
		if transparency_timer then
			vim.fn.timer_stop(transparency_timer)
		end

		transparency_timer = vim.fn.timer_start(100, function()
			local ok, highlights = pcall(require, "config.highlights")
			if ok and highlights and highlights.setup_all_highlights then
				highlights.setup_all_highlights()
			end
		end)
	end,
})