-- NvimTree 颜色配置
local M = {}

-- 颜色常量定义
local NVIMTREE_COLORS = {
    -- 基础颜色
    BRIGHT_WHITE = "#ffffff",    -- 亮白色：普通文件和目录（未改变）
    LIGHT_GRAY = "#8a8a8a",     -- 浅灰色：被忽略的文件/目录
    ROOT_FOLDER = "#00ff7f",    -- 亮绿色：根目录（醒目颜色）
    
    -- Git 状态颜色（简化版本，只区分commit状态）
    GIT_NEW = "#98c379",        -- 绿色：新增文件和目录
    GIT_MODIFIED = "#e5c07b",   -- 黄色：修改过的文件和目录（统一颜色）
    GIT_UNMODIFIED = "#ffffff", -- 白色：未改变的文件和目录
    GIT_DELETED = "#e06c75",    -- 红色：已删除文件（保留用于特殊情况）
}

-- Git颜色设置函数（目录状态区分版本）
function M.apply_git_colors_after_setup()
    -- ========================================
    -- Git 状态相关颜色（按目录创建状态区分）
    -- ========================================
    
    -- 新建目录和文件：绿色（Git未跟踪的内容）
    vim.api.nvim_set_hl(0, "NvimTreeGitNew", { fg = NVIMTREE_COLORS.GIT_NEW })
    vim.api.nvim_set_hl(0, "NvimTreeGitUntracked", { fg = NVIMTREE_COLORS.GIT_NEW })
    
    -- 已存在目录有变更：黄色（目录之前存在，内部有修改）
    vim.api.nvim_set_hl(0, "NvimTreeGitDirty", { fg = NVIMTREE_COLORS.GIT_MODIFIED })
    vim.api.nvim_set_hl(0, "NvimTreeGitStaged", { fg = NVIMTREE_COLORS.GIT_MODIFIED })
    vim.api.nvim_set_hl(0, "NvimTreeGitMerge", { fg = NVIMTREE_COLORS.GIT_MODIFIED })
    
    -- 修改过的文件：黄色
    vim.api.nvim_set_hl(0, "NvimTreeGitRenamed", { fg = NVIMTREE_COLORS.GIT_MODIFIED })
    
    -- 删除的文件：红色
    vim.api.nvim_set_hl(0, "NvimTreeGitDeleted", { fg = NVIMTREE_COLORS.GIT_DELETED })
    
    -- 被 gitignore 的文件和目录颜色
    vim.api.nvim_set_hl(0, "NvimTreeGitIgnored", { fg = NVIMTREE_COLORS.LIGHT_GRAY })
end

-- 智能目录Git状态检测函数
function M.detect_directory_git_status()
    -- 获取Git状态信息，区分新建目录和已存在目录
    local git_status_output = vim.fn.system("git status --porcelain")
    if vim.v.shell_error ~= 0 then
        return {}
    end
    
    local directory_status = {}
    
    -- 解析Git状态输出
    for line in git_status_output:gmatch("[^\r\n]+") do
        local status = line:sub(1, 2)
        local filepath = line:sub(4)
        
        -- 获取文件的目录路径
        local dir = vim.fn.fnamemodify(filepath, ":h")
        if dir == "." then
            dir = ""
        end
        
        -- 判断目录状态
        if status == "??" then
            -- 未跟踪文件，目录为新建
            directory_status[dir] = "new"
        elseif status:match("[MARC]") or status:match(".[MD]") then
            -- 已跟踪文件有变更，目录为已存在但有修改
            if directory_status[dir] ~= "new" then
                directory_status[dir] = "modified"
            end
        end
        
        -- 递归标记父目录
        local parent_dir = dir
        while parent_dir ~= "" and parent_dir ~= "." do
            parent_dir = vim.fn.fnamemodify(parent_dir, ":h")
            if parent_dir == "." then
                break
            end
            
            if directory_status[parent_dir] ~= "new" then
                directory_status[parent_dir] = "modified"
            end
        end
    end
    
    return directory_status
end

-- 目录状态颜色设置（简化但精确的版本）
function M.apply_enhanced_directory_colors()
    -- 基本Git颜色设置
    M.apply_git_colors_after_setup()
    
    -- 确保普通目录和文件显示为白色
    vim.api.nvim_set_hl(0, "NvimTreeFileName", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })          -- 普通文件
    vim.api.nvim_set_hl(0, "NvimTreeNormal", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })           -- 默认文本
    vim.api.nvim_set_hl(0, "NvimTreeExecFile", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })          -- 可执行文件
    vim.api.nvim_set_hl(0, "NvimTreeSymlink", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })           -- 符号链接
    vim.api.nvim_set_hl(0, "NvimTreeRootFolder", { fg = NVIMTREE_COLORS.ROOT_FOLDER })        -- 根目录
    vim.api.nvim_set_hl(0, "NvimTreeImageFile", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })         -- 图片文件
    vim.api.nvim_set_hl(0, "NvimTreeMarkdownFile", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })      -- Markdown文件
    vim.api.nvim_set_hl(0, "NvimTreeGitIgnoredFolderIcon", { fg = NVIMTREE_COLORS.LIGHT_GRAY })
    
    -- 设置特殊文件（.git目录）为浅灰色，强制覆盖其他颜色
    M.force_git_directory_light_gray()
end

-- 强制.git目录为浅灰色的专门函数
function M.force_git_directory_light_gray()
    -- 仅通过 special 组设置 .git 的颜色（不影响其它文件）
    vim.api.nvim_set_hl(0, "NvimTreeSpecialFile", { fg = NVIMTREE_COLORS.LIGHT_GRAY })
    vim.api.nvim_set_hl(0, "NvimTreeSpecialFileIcon", { fg = NVIMTREE_COLORS.LIGHT_GRAY })
    vim.api.nvim_set_hl(0, "NvimTreeSpecialDir", { fg = NVIMTREE_COLORS.LIGHT_GRAY })
    
    -- 额外的目录图标相关高亮组
    -- vim.api.nvim_set_hl(0, "NvimTreeFolderIcon", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })  -- 普通目录图标保持白色
    -- vim.api.nvim_set_hl(0, "NvimTreeOpenedFolderIcon", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })  -- 展开目录图标保持白色
    
    -- 通过nvim-web-devicons设置.git目录图标颜色
    pcall(function()
        local devicons = require('nvim-web-devicons')
        if devicons then
            devicons.set_icon {
                [".git"] = {
                    icon = " ",
                    color = NVIMTREE_COLORS.LIGHT_GRAY,
                    name = "Git"
                }
            }
        end
    end)
    
    -- 确保.git目录不受任何Git状态影响
    vim.cmd("highlight! NvimTreeGitDirCustom guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " gui=NONE cterm=NONE")
    vim.cmd("highlight! NvimTreeGitDirForce guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " gui=NONE cterm=NONE") 
    vim.cmd("highlight! NvimTreeGitDirUltimate guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245 guibg=NONE")
    
    -- 不再使用额外的语法匹配，以 special 机制为准
    
    -- 设置与.git相关的所有可能的高亮组
    vim.cmd("highlight! link NvimTreeGitFile NvimTreeSpecialFile")
    vim.cmd("highlight! link NvimTreeGitFolder NvimTreeSpecialFile") 
    vim.cmd("highlight! link NvimTreeGitIcon NvimTreeSpecialFileIcon")
    
    -- 立即执行一次刷新，确保设置生效
    vim.schedule(function()
        pcall(function()
            local api = require('nvim-tree.api')
            if api.tree.is_visible() then
                api.tree.reload()
            end
        end)
    end)
end

-- 智能目录着色相关的高亮设置
function M.apply_smart_color_highlights()
    -- 设置被忽略文件和目录的高亮组为灰色
    vim.api.nvim_set_hl(0, "NvimTreeGitIgnored", { fg = NVIMTREE_COLORS.LIGHT_GRAY })
    vim.api.nvim_set_hl(0, "NvimTreeGitIgnoredFile", { fg = NVIMTREE_COLORS.LIGHT_GRAY })
end

-- 应用各种自定义高亮组
function M.apply_custom_highlights()
    -- 被忽略目录和文件的高亮
    vim.cmd("highlight! NvimTreeIgnoredDir guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY)
    vim.cmd("highlight! NvimTreeIgnoredFile guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY)
    vim.cmd("highlight! NvimTreeIgnoredDirForce guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY)
    vim.cmd("highlight! NvimTreeIgnoredFileForce guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY)
    
    -- 通配符模式的高亮
    vim.cmd("highlight! NvimTreeIgnoredPattern guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY)
    vim.cmd("highlight! NvimTreeIgnoredPatternForce guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY)
    
    -- 特别为 .git 目录设置浅灰色
    vim.cmd("highlight! NvimTreeGitDirForce guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " gui=NONE cterm=NONE")
    
end

-- 创建各种即时应用的语法高亮
function M.create_git_dir_syntax_highlights()
    pcall(function()
        vim.cmd([[syntax match NvimTreeGitDirInline /^\s*.*\.git\s*$/ contains=ALL]])
        vim.cmd("highlight NvimTreeGitDirInline guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245")
        
        vim.cmd([[syntax match NvimTreeGitDirInline2 /^\s*.*\.git\s*$/ contains=ALL]])
        vim.cmd("highlight NvimTreeGitDirInline2 guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245")
        
        vim.cmd([[syntax match NvimTreeGitDirImmediate /^\s*\zs\.git$/ contained containedin=ALL]])
        vim.cmd([[highlight! NvimTreeGitDirImmediate guifg=]] .. NVIMTREE_COLORS.LIGHT_GRAY .. [[ ctermfg=245 gui=NONE cterm=NONE]])
        
        vim.cmd([[syntax match NvimTreeGitDirGlobal /^\s*\zs\.git$/ contained containedin=ALL]])
        vim.cmd([[highlight! NvimTreeGitDirGlobal guifg=]] .. NVIMTREE_COLORS.LIGHT_GRAY .. [[ ctermfg=245 gui=NONE cterm=NONE]])
    end)
end

-- 应用各种Git目录特殊高亮
function M.apply_git_dir_special_highlights()
    -- 供 extmark 使用的灰色高亮组
    vim.api.nvim_set_hl(0, "NvimTreeGitDirGray", { fg = NVIMTREE_COLORS.LIGHT_GRAY })
    vim.cmd("highlight! NvimTreeGitDir guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245 guibg=NONE")
    vim.cmd("highlight! NvimTreeGitDirPattern guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245 guibg=NONE")
    vim.cmd("highlight NvimTreeGitDirRefresh guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245")
    
    -- 各种自动化和定时器相关的高亮
    for i = 1, 10 do
        vim.cmd("highlight NvimTreeGitDirAuto" .. i .. " guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245")
    end
    
    vim.cmd("highlight NvimTreeGitDirBuf1 guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245")
    vim.cmd("highlight NvimTreeGitDirBuf2 guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245")
    vim.cmd("highlight NvimTreeGitDirBuf guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245")
    vim.cmd("highlight NvimTreeGitDirWin guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245")
    vim.cmd("highlight NvimTreeGitDirWin2 guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245")
    
    -- 链接相关的高亮设置
    vim.cmd("highlight! link NvimTreeGitIgnoredFolder NvimTreeGitIgnored")
    vim.cmd("highlight! link NvimTreeGitIgnoredFolderName NvimTreeGitIgnored")
    vim.cmd("highlight! link NvimTreeGitIgnoredFolderIcon NvimTreeGitIgnored")
    
    -- SpecialDir相关高亮
    vim.cmd("highlight! NvimTreeSpecialDir guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " guibg=NONE gui=NONE ctermfg=245")
end

-- 使用匹配规则强制将包含 .git 的整行渲染为浅灰色（最高优先级）
local function apply_git_dir_match()
    local bufnr = vim.api.nvim_get_current_buf()
    if vim.bo[bufnr].filetype ~= "NvimTree" then return end
    -- 定义高亮组（再次确保存在）
    vim.api.nvim_set_hl(0, "NvimTreeGitDirGray", { fg = NVIMTREE_COLORS.LIGHT_GRAY })
    -- 清理旧的 match（如果存在）
    pcall(function()
        if vim.b[bufnr].nvimtree_gitdir_matchid then
            vim.fn.matchdelete(vim.b[bufnr].nvimtree_gitdir_matchid)
            vim.b[bufnr].nvimtree_gitdir_matchid = nil
        end
    end)
    -- 匹配包含 .git 的任意行（排除 .gitignore/.gitattributes）
    local pattern = [[\v^.*(\\.git)(/.*)?$]]
    local id = vim.fn.matchadd("NvimTreeGitDirGray", pattern, 100)
    vim.b[bufnr].nvimtree_gitdir_matchid = id
end

-- 创建强制.git目录颜色的自动命令
function M.create_git_color_autocmd()
    vim.api.nvim_create_autocmd({"BufEnter", "BufWinEnter", "ColorScheme"}, {
        pattern = "NvimTree_*",
        callback = function()
            -- 立即应用高亮设置
            vim.api.nvim_set_hl(0, "NvimTreeSpecialFile", { fg = NVIMTREE_COLORS.LIGHT_GRAY })
            vim.api.nvim_set_hl(0, "NvimTreeExecFile", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })
            vim.api.nvim_set_hl(0, "NvimTreeSymlink", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })
            vim.api.nvim_set_hl(0, "NvimTreeRootFolder", { fg = NVIMTREE_COLORS.ROOT_FOLDER })
            vim.api.nvim_set_hl(0, "NvimTreeImageFile", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })
            vim.api.nvim_set_hl(0, "NvimTreeMarkdownFile", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })
            -- vim.api.nvim_set_hl(0, "NvimTreeEmptyFolderName", { fg = NVIMTREE_COLORS.BRIGHT_WHITE })
            vim.api.nvim_set_hl(0, "NvimTreeIgnoredFolderIcon", { fg = NVIMTREE_COLORS.LIGHT_GRAY })
            
            -- 立即应用 .git 目录的语法高亮
            vim.defer_fn(function()
                M.create_git_dir_syntax_highlights()
            end, 1)
        end,
    })
end

-- 设置高亮组
function M.set_nvim_tree_highlights()
    -- 应用增强的目录状态颜色设置
    M.apply_enhanced_directory_colors()
    M.apply_smart_color_highlights()
    M.apply_custom_highlights()
    M.apply_git_dir_special_highlights()
    M.create_git_color_autocmd()
    
    -- 正常文件和目录的颜色（确保展开和未展开目录颜色一致）
    -- vim.api.nvim_command('highlight! NvimTreeFolderName guifg=' .. NVIMTREE_COLORS.BRIGHT_WHITE)  -- 未展开文件夹名称颜色
    -- vim.api.nvim_command('highlight! NvimTreeOpenedFolderName guifg=' .. NVIMTREE_COLORS.BRIGHT_WHITE)  -- 展开文件夹名称颜色
    -- vim.api.nvim_command('highlight! NvimTreeEmptyFolderName guifg=' .. NVIMTREE_COLORS.BRIGHT_WHITE)  -- 空文件夹名称颜色
    -- vim.api.nvim_command('highlight! NvimTreeClosedFolderIcon guifg=' .. NVIMTREE_COLORS.BRIGHT_WHITE)  -- 关闭文件夹图标颜色
    -- vim.api.nvim_command('highlight! NvimTreeOpenedFolderIcon guifg=' .. NVIMTREE_COLORS.BRIGHT_WHITE)  -- 打开文件夹图标颜色

    -- Git ignored 目录的颜色
    vim.api.nvim_command('highlight! NvimTreeGitIgnoredIcon guifg=' .. NVIMTREE_COLORS.LIGHT_GRAY)  -- git ignored 图标颜色
    vim.api.nvim_command('highlight! NvimTreeGitIgnoredFolder guifg=' .. NVIMTREE_COLORS.LIGHT_GRAY)  -- git ignored 文本颜色
    vim.api.nvim_command('highlight! NvimTreeGitIgnoredFolderIcon guifg=' .. NVIMTREE_COLORS.LIGHT_GRAY)  -- git ignored 文件夹图标

    -- .git 目录通过 special 组设置为浅灰色
    vim.api.nvim_command('highlight! NvimTreeSpecialFile guifg=' .. NVIMTREE_COLORS.LIGHT_GRAY)
    vim.api.nvim_command('highlight! NvimTreeSpecialFileIcon guifg=' .. NVIMTREE_COLORS.LIGHT_GRAY)
    vim.api.nvim_command('highlight! NvimTreeSpecialDir guifg=' .. NVIMTREE_COLORS.LIGHT_GRAY)
    
    -- 通过 special 实现，不再额外链接
    
    -- 调用专门的.git目录颜色强制函数
    M.force_git_directory_light_gray()

    -- Git 状态颜色（简化版本，确保展开和未展开目录一致）
    vim.api.nvim_command('highlight! NvimTreeGitDirty guifg=' .. NVIMTREE_COLORS.GIT_MODIFIED)  -- 修改过的文件：黄色
    vim.api.nvim_command('highlight! NvimTreeGitStaged guifg=' .. NVIMTREE_COLORS.GIT_MODIFIED)  -- 修改过的目录：黄色
    vim.api.nvim_command('highlight! NvimTreeGitMerge guifg=' .. NVIMTREE_COLORS.GIT_MODIFIED)  -- 合并冲突：黄色
    vim.api.nvim_command('highlight! NvimTreeGitRenamed guifg=' .. NVIMTREE_COLORS.GIT_MODIFIED)  -- 重命名文件：黄色
    vim.api.nvim_command('highlight! NvimTreeGitNew guifg=' .. NVIMTREE_COLORS.GIT_NEW)  -- 新增文件：绿色
    vim.api.nvim_command('highlight! NvimTreeGitUntracked guifg=' .. NVIMTREE_COLORS.GIT_NEW)  -- 未跟踪文件：绿色
    vim.api.nvim_command('highlight! NvimTreeGitDeleted guifg=' .. NVIMTREE_COLORS.GIT_DELETED)  -- 已删除文件：红色
    
    -- 确保展开目录的Git状态颜色与未展开目录一致
    vim.api.nvim_command('highlight! link NvimTreeOpenedFolderName NvimTreeFolderName')  -- 展开目录名称继承普通目录颜色
    vim.api.nvim_command('highlight! link NvimTreeOpenedFolderIcon NvimTreeClosedFolderIcon')  -- 展开目录图标继承关闭目录颜色

    -- 诊断信息颜色
    vim.api.nvim_command('highlight! NvimTreeDiagnosticError guifg=#cc241d')  -- 错误
    vim.api.nvim_command('highlight! NvimTreeDiagnosticWarning guifg=#d79921')  -- 警告
    vim.api.nvim_command('highlight! NvimTreeDiagnosticInfo guifg=#458588')  -- 信息
    vim.api.nvim_command('highlight! NvimTreeDiagnosticHint guifg=#689d6a')  -- 提示

    -- 其他高亮组设置
    vim.api.nvim_command('highlight! NvimTreeGitIgnored guifg=' .. NVIMTREE_COLORS.LIGHT_GRAY)
    vim.api.nvim_command('highlight! NvimTreeGitIgnoredFile guifg=' .. NVIMTREE_COLORS.LIGHT_GRAY)
    vim.api.nvim_command('highlight! NvimTreeNormalSpecialFile guifg=' .. NVIMTREE_COLORS.BRIGHT_WHITE)
    vim.api.nvim_command('highlight! link NvimTreeSpecialFile NvimTreeNormalSpecialFile')

    -- 不再使用 GitDir* 强制覆盖，避免副作用
end

-- 强制刷新 NvimTree 显示的函数
function M.force_refresh_nvimtree()
    pcall(function()
        local api = require('nvim-tree.api')
        if api.tree.is_visible() then
            api.tree.reload()
        end
    end)
end

-- 强制设置.git目录颜色的简化函数
function M.force_git_color()
    -- 在所有NvimTree缓冲区中应用.git的特殊语法匹配
    for _, bufnr in ipairs(vim.api.nvim_list_bufs()) do
        local buf_name = vim.api.nvim_buf_get_name(bufnr)
        if buf_name:match("NvimTree_") and vim.api.nvim_buf_is_loaded(bufnr) then
            vim.api.nvim_buf_call(bufnr, function()
                -- 清除旧的语法匹配
                vim.cmd("silent! syntax clear NvimTreeGitDirCustom")
                -- 创建专门匹配.git的语法规则，优先级最高
                vim.cmd("syntax match NvimTreeGitDirCustom '\\v^\\s*\\zs\\.git$' containedin=ALL")
                vim.cmd("highlight! NvimTreeGitDirCustom guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " gui=NONE cterm=NONE")
                
                -- 额外设置，确保.git目录及其内容都显示为浅灰色
                vim.cmd("syntax match NvimTreeGitDirCustom '\\v^\\s*\\zs\\.git/.*$' containedin=ALL")
                vim.cmd("highlight! NvimTreeGitDirCustom guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " gui=NONE cterm=NONE")
            end)
        end
    end
end

-- 终极解决方案：强制覆盖所有 .git 目录颜色
function M.ultimate_git_color_fix()
    local view = require('nvim-tree.view')
    if view.is_visible() then
        local bufnr = view.get_bufnr()
        if bufnr and vim.api.nvim_buf_is_valid(bufnr) then
            vim.api.nvim_buf_call(bufnr, function()
                -- 强制语法高亮
                vim.cmd("silent! syntax clear NvimTreeGitDirUltimate")
                vim.cmd([[syntax match NvimTreeGitDirUltimate /^\s*.*\.git\s*$/ contains=ALL]])
                vim.cmd("highlight! NvimTreeGitDirUltimate guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " ctermfg=245 guibg=NONE")
                
                -- 强制覆盖 NvimTree 的内置高亮组
                vim.cmd("highlight! NvimTreeSpecialDir guifg=" .. NVIMTREE_COLORS.LIGHT_GRAY .. " guibg=NONE")
            end)
        end
    end
end

-- 设置自动命令
function M.setup_highlight_autocmds()
    -- 创建超级强力的自动命令组
    vim.api.nvim_create_augroup("UltimateGitColorFix", { clear = true })
    
    -- 在所有可能的事件上强制应用颜色
    vim.api.nvim_create_autocmd({
        "FileType", "BufEnter", "BufWinEnter", "WinEnter", 
        "CursorMoved", "CursorMovedI", "TextChanged", "TextChangedI",
        "ColorScheme", "VimEnter", "UIEnter"
    }, {
        group = "UltimateGitColorFix",
        callback = function()
            if vim.bo.filetype == "NvimTree" or vim.api.nvim_buf_get_name(0):match("NvimTree") then
                vim.defer_fn(M.ultimate_git_color_fix, 1)  -- 几乎立即执行
            end
        end
    })
    
    -- 立即执行一次修复
    M.ultimate_git_color_fix()
    
    -- 延迟执行多次确保生效
    for i = 1, 10 do
        vim.defer_fn(M.ultimate_git_color_fix, i * 20)
    end
end

return M