-- ============================================================================
-- nvim-cmp 补全配置文件 (cmp.lua) 
-- 基于官方推荐配置，提供现代化的补全体验
-- 参考: https://github.com/hrsh7th/nvim-cmp
-- ============================================================================

local cmp = require("cmp")
local luasnip = require("luasnip")

-- 获取默认配置 (但我们会完全覆盖格式化部分)
local default_config = require("nvchad.configs.cmp")

-- ============================================================================
-- LSP Kind 图标配置 (基于 lspkind-nvim 标准)
-- ============================================================================
local kind_icons = {
	Text = "󰉿",
	Method = "󰆧", 
	Function = "󰊕",
	Constructor = "",
	Field = "󰜢",
	Variable = "󰀫",
	Class = "󰠱",
	Interface = "",
	Module = "",
	Property = "󰜢",
	Unit = "󰑭",
	Value = "󰎠",
	Enum = "",
	Keyword = "󰌋",
	Snippet = "",
	Color = "󰏘",
	File = "󰈙",
	Reference = "󰈇",
	Folder = "󰉋",
	EnumMember = "",
	Constant = "󰏿",
	Struct = "󰙅",
	Event = "",
	Operator = "󰆕",
	TypeParameter = "",
}

-- ============================================================================
-- 格式化函数 (图标和类型在补全文本前面)
-- ============================================================================
local function format_completion_item(entry, vim_item)
	-- 启用调试信息
	print("Original vim_item:", vim.inspect(vim_item))
	
	-- 获取图标和类型
	local icon = kind_icons[vim_item.kind] or "🔵"
	local kind_text = vim_item.kind or "Unknown"
	local abbr_text = vim_item.abbr or "missing"
	
	-- 将图标和类型添加到补全文本前面
	vim_item.abbr = string.format('%s %s %s', icon, kind_text, abbr_text)
	
	-- 完全清空其他字段
	vim_item.kind = nil
	vim_item.menu = nil
	
	-- 调试信息：打印处理后数据
	print("Modified vim_item:", vim.inspect(vim_item))

	return vim_item
end

-- ============================================================================
-- 主配置 (基于官方推荐模式)
-- ============================================================================
local config = vim.tbl_deep_extend("force", default_config, {
	-- 片段引擎配置
	snippet = {
		expand = function(args)
			luasnip.lsp_expand(args.body)
		end,
	},

	-- 窗口外观配置
	window = {
		completion = cmp.config.window.bordered(),
		documentation = cmp.config.window.bordered(),
	},

	-- 格式化配置 (只显示abbr字段)
	formatting = {
		fields = { 'abbr' },
		format = format_completion_item,
	},

	-- 补全行为配置
	completion = {
		completeopt = 'menu,menuone,noinsert',
		keyword_length = 1,
	},

	-- 确认行为配置  
	confirmation = {
		behavior = cmp.ConfirmBehavior.Replace,
	},

	-- 实验性功能
	experimental = {
		ghost_text = true,
	},

	-- 键位映射 (官方推荐的 Super-Tab 样式)
	mapping = cmp.mapping.preset.insert({
		-- Ctrl+b/f: 滚动文档
		['<C-b>'] = cmp.mapping.scroll_docs(-4),
		['<C-f>'] = cmp.mapping.scroll_docs(4),
		
		-- Ctrl+Space: 手动触发补全
		['<C-Space>'] = cmp.mapping.complete(),
		
		-- Escape: 关闭补全
		['<C-e>'] = cmp.mapping.abort(),
		
		-- Enter: 确认选择
		['<CR>'] = cmp.mapping.confirm({ select = true }),
		
		-- Super-Tab 映射
		["<Tab>"] = cmp.mapping(function(fallback)
			if cmp.visible() then
				cmp.select_next_item()
			elseif luasnip.expand_or_jumpable() then
				luasnip.expand_or_jump()
			else
				fallback()
			end
		end, { "i", "s" }),

		["<S-Tab>"] = cmp.mapping(function(fallback)
			if cmp.visible() then
				cmp.select_prev_item()
			elseif luasnip.jumpable(-1) then
				luasnip.jump(-1)
			else
				fallback()
			end
		end, { "i", "s" }),
	}),
})

-- 加载彩色图标
require("configs.cmp-icons").setup()
local custom_icons = require("configs.cmp-icons").kind_icons

-- 延迟执行以确保覆盖 NvChad 配置
vim.defer_fn(function()
	cmp.setup({
		window = {
			completion = {
				border = "rounded",
				winhighlight = "Normal:CmpNormal,CursorLine:CmpSel,Search:None",
				scrollbar = false,
			},
			documentation = {
				border = "rounded", 
				winhighlight = "Normal:CmpDoc",
			},
		},
		
		formatting = {
			format = function(entry, item)
				-- 保存原始类型
				local kind = item.kind or ""
				local icon = custom_icons[kind] or ""
				
				-- 应用 icons_left 布局: 图标在左，类型在右
				item.menu = kind  -- 右侧显示类型
				item.kind = string.format(" %s ", icon)  -- 左侧显示图标 + 背景
				item.kind_hl_group = "CmpItemKind" .. kind
				
				return item
			end,
		}
	})
end, 200)

-- ============================================================================
-- 设置自定义高亮组
-- ============================================================================
require("configs.cmp-highlights").setup()

-- ============================================================================
-- 2. 特定文件类型的补全增强配置
-- ============================================================================

-- 2.1 前端开发文件类型的 TailwindCSS 补全优化
-- 为前端相关文件类型提供增强的 TailwindCSS 类名补全和颜色预览
cmp.setup.filetype(
	{ "html", "css", "scss", "javascript", "typescript", "javascriptreact", "typescriptreact", "vue", "svelte" },
	{
		-- 补全源配置（按优先级排序）
		sources = cmp.config.sources({
			{ name = "nvim_lsp" }, -- LSP 补全（最高优先级）
			{ name = "luasnip" }, -- 代码片段补全
			{ name = "tailwind-css", priority = 1000 }, -- TailwindCSS 类名补全（高优先级）
			{ name = "buffer" }, -- 缓冲区内容补全（较低优先级）
		}),
		-- 自动触发配置
		completion = {
			-- 为前端文件类型启用更积极的自动触发策略
			autocomplete = {
				cmp.TriggerEvent.TextChanged, -- 文本改变时触发
				cmp.TriggerEvent.InsertEnter, -- 进入插入模式时触发
			},
		},
		-- 格式化配置 - 使用相同的成功格式
		formatting = {
			fields = { "menu", "abbr" },
			format = function(entry, vim_item)
				local icon = kind_icons[vim_item.kind] or ""
				local kind_text = vim_item.kind
				
				-- 将图标和类型放到 menu 字段（左侧显示）
				vim_item.menu = string.format("%s %s", icon, kind_text)
				
				-- 清空 kind 字段避免重复显示
				vim_item.kind = ""
				
				-- 可选：应用 TailwindCSS 颜色预览
				-- local tailwind_format = require("tailwindcss-colorizer-cmp").formatter
				-- vim_item = tailwind_format(entry, vim_item)
				
				return vim_item
			end,
		},
		-- 窗口样式配置 
		window = {
			completion = cmp.config.window.bordered(),
			documentation = cmp.config.window.bordered(),
		},
	}
)

return config
