-- ============================================================================
-- TailwindCSS 自动配置模块
-- 从原 NvChad 配置迁移的 TailwindCSS 集成功能
-- ============================================================================

local M = {}

-- ============================================================================
-- TailwindCSS 项目检测配置
-- ============================================================================

local tailwind_config_files = {
	"tailwind.config.js",
	"tailwind.config.ts",
	"tailwind.config.mjs",
	"tailwind.config.cjs",
}

local tailwind_package_files = {
	"package.json",
	"postcss.config.js",
	"postcss.config.ts",
	"postcss.config.mjs",
	"postcss.config.cjs",
}

-- ============================================================================
-- 辅助函数
-- ============================================================================

-- 检查文件是否存在
local function file_exists(path)
	return vim.fn.filereadable(path) == 1
end

-- 检查目录是否存在
local function dir_exists(path)
	return vim.fn.isdirectory(path) == 1
end

-- 向上查找文件
local function find_file_upward(filename, start_path)
	start_path = start_path or vim.fn.getcwd()

	local function check_dir(dir)
		local file_path = dir .. "/" .. filename
		if file_exists(file_path) then
			return file_path
		end

		local parent = vim.fn.fnamemodify(dir, ":h")
		if parent == dir then -- 到达根目录
			return nil
		end

		return check_dir(parent)
	end

	return check_dir(start_path)
end

-- 检查是否为 TailwindCSS 项目
local function is_tailwind_project()
	-- 检查 TailwindCSS 配置文件
	for _, config_file in ipairs(tailwind_config_files) do
		if find_file_upward(config_file) then
			return true
		end
	end

	-- 检查 package.json 中的 tailwindcss 依赖
	local package_json = find_file_upward("package.json")
	if package_json then
		local content = vim.fn.readfile(package_json)
		local json_str = table.concat(content, "\n")

		-- 检查 dependencies 和 devDependencies
		if json_str:match('"tailwindcss"') then
			return true
		end
	end

	return false
end

-- ============================================================================
-- LSP 设置增强
-- ============================================================================

local function setup_tailwind_lsp_enhancements()
	local lspconfig = require("lspconfig")
	local capabilities = require("cmp_nvim_lsp").default_capabilities()

	-- TailwindCSS LSP 服务器配置
	lspconfig.tailwindcss.setup({
		capabilities = capabilities,
		on_attach = function(client, bufnr)
			-- 设置键位映射
			require("config.lsp").setup_lsp_keymaps(bufnr)

			-- 禁用格式化功能，因为 TailwindCSS 通常不需要格式化
			client.server_capabilities.documentFormattingProvider = false
		end,
		settings = {
			tailwindCSS = {
				classAttributes = {
					"class",
					"className",
					"ngClass",
					"classList",
				},
				experimental = {
					classRegex = {
						-- Vue 模板中的 class 绑定
						{
							":class\\([^)]*\\)",
							'"([^"]*)"',
						},
						{
							":class\\[([^\\]]*)\\]",
							"'([^']*)'",
						},
						-- React 中的 class name
						{
							"className\\s*=\\s*{([^}]*)}",
							'"([^"]*)"',
						},
						-- 常见的 CSS-in-JS 模式
						{
							"css\\s*`([^`]*)",
							"([^}]*)",
						},
					},
				},
				lint = {
					cssConflict = "warning",
					invalidApply = "error",
					invalidScreen = "error",
					invalidTailwindDirective = "error",
					invalidVariant = "error",
					recommendedVariantOrder = "warning",
				},
				validate = true,
			},
		},
		filetypes = {
			"html",
			"javascript",
			"javascriptreact",
			"typescript",
			"typescriptreact",
			"vue",
			"svelte",
			"php",
			"erb",
			"ejs",
			"pug",
			"twig",
		},
	})
end

-- ============================================================================
-- 补全增强
-- ============================================================================

local function setup_tailwind_cmp()
	local cmp = require("cmp")

	-- TailwindCSS 补全源
	cmp.register_source("tailwindcss", require("cmp_tailwindcss").new())

	-- 设置补全配置
	local cmp_config = cmp.get_config()
	table.insert(cmp_config.sources, {
		name = "tailwindcss",
		priority = 1000,
	})

	cmp.setup(cmp_config)
end

-- ============================================================================
-- 语法高亮增强
-- ============================================================================

local function setup_tailwind_highlights()
	-- 创建自定义高亮组
	vim.api.nvim_set_hl(0, "TailwindClass", { fg = "#60a5fa", bold = true })
	vim.api.nvim_set_hl(0, "TailwindVariant", { fg = "#c084fc", italic = true })
	vim.api.nvim_set_hl(0, "TailwindUtility", { fg = "#34d399" })

	-- 为 HTML 添加 TailwindCSS 类名高亮
	vim.api.nvim_create_autocmd("FileType", {
		pattern = { "html", "typescriptreact", "javascriptreact", "vue", "svelte" },
		callback = function()
			-- 使用 treesitter 高亮
			local parser = require("vim.treesitter.get_parser")
			if parser then
				-- 这里可以添加更复杂的高亮逻辑
			end
		end,
	})
end

-- ============================================================================
-- 自动命令和事件处理
-- ============================================================================

local function setup_autocmds()
	local group = vim.api.nvim_create_augroup("TailwindCSS", { clear = true })

	-- 检测到 TailwindCSS 项目时自动启用相关功能
	vim.api.nvim_create_autocmd({ "BufReadPost", "BufNewFile" }, {
		group = group,
		callback = function()
			local bufnr = vim.api.nvim_get_current_buf()
			local filename = vim.api.nvim_buf_get_name(bufnr)

			-- 只在相关文件类型中激活
			local filetype = vim.bo.filetype
			local supported_filetypes = {
				"html", "css", "scss", "sass", "less",
				"javascript", "javascriptreact",
				"typescript", "typescriptreact",
				"vue", "svelte", "php", "erb", "ejs"
			}

			if vim.tbl_contains(supported_filetypes, filetype) and is_tailwind_project() then
				-- 启用 TailwindCSS 功能
				vim.b.tailwindcss_enabled = true

				-- 设置特定的键位映射
				local opts = { buffer = bufnr, silent = true }
				vim.keymap.set("n", "<leader>tc", function()
					vim.cmd("Telescope tailwindcss classes")
				end, opts)

				vim.keymap.set("n", "<leader>th", function()
					vim.cmd("Telescope tailwindcss history")
				end, opts)
			end
		end,
	})

	-- 项目配置变更时重新检测
	vim.api.nvim_create_autocmd("BufWritePost", {
		group = group,
		pattern = "*tailwind*",
		callback = function()
			-- 重新检测并设置 TailwindCSS
			vim.defer_fn(function()
				M.setup()
			end, 1000)
		end,
	})
end

-- ============================================================================
-- 工具函数
-- ============================================================================

-- 排序 TailwindCSS 类名
function M.sort_tailwind_classes()
	local lines = vim.api.nvim_buf_get_lines(0, 0, -1, false)
	local modified_lines = {}

	for _, line in ipairs(lines) do
		-- 匹配 class 属性中的内容
		local new_line = line

		-- 匹配 class="..." 或 className="..."
		local class_pattern = 'class%w*%s*=%s*"([^"]*)"'
		local start_idx, end_idx, class_content = string.find(line, class_pattern)

		if class_content then
			-- 简单的类名排序逻辑
			local classes = vim.split(class_content, "%s+")
			table.sort(classes)

			-- 过滤空字符串
			local sorted_classes = {}
			for _, class in ipairs(classes) do
				if class ~= "" then
					table.insert(sorted_classes, class)
				end
			end

			-- 重新构建行
			local sorted_content = table.concat(sorted_classes, " ")
			new_line = string.sub(line, 1, start_idx - 1) .. 'class="' .. sorted_content .. '"' .. string.sub(line, end_idx + 1)
		end

		table.insert(modified_lines, new_line)
	end

	vim.api.nvim_buf_set_lines(0, 0, -1, false, modified_lines)
end

-- 检查类名是否有效
function M.validate_tailwind_class(class_name)
	-- 这里可以添加类名验证逻辑
	-- 例如检查是否为有效的 TailwindCSS 类
	return true
end

-- 获取当前光标位置的类名
function M.get_class_under_cursor()
	local line = vim.api.nvim_get_current_line()
	local col = vim.api.nvim_win_get_cursor(0)[2] + 1

	-- 查找 class 属性
	local class_pattern = 'class%w*%s*=%s*"([^"]*)"'
	local start_idx, end_idx, class_content = string.find(line, class_pattern)

	if class_content and col >= start_idx and col <= end_idx then
		-- 获取光标下的具体类名
		local relative_pos = col - start_idx
		local classes = vim.split(class_content, "%s+")

		local current_pos = 0
		for _, class in ipairs(classes) do
			if class ~= "" then
				if relative_pos >= current_pos and relative_pos < current_pos + #class then
					return class
				end
				current_pos = current_pos + #class + 1 -- +1 for space
			end
		end
	end

	return nil
end

-- ============================================================================
-- 主要设置函数
-- ============================================================================

function M.setup()
	-- 检查是否为 TailwindCSS 项目
	if not is_tailwind_project() then
		return
	end

	-- 设置 LSP 增强
	setup_tailwind_lsp_enhancements()

	-- 设置补全
	pcall(setup_tailwind_cmp)

	-- 设置高亮
	setup_tailwind_highlights()

	-- 设置自动命令
	setup_autocmds()

	-- 创建用户命令
	vim.api.nvim_create_user_command("TailwindSort", function()
		M.sort_tailwind_classes()
	end, { desc = "排序 TailwindCSS 类名" })

	vim.api.nvim_create_user_command("TailwindInfo", function()
		local class_name = M.get_class_under_cursor()
		if class_name then
			print("当前类名: " .. class_name)
			-- 这里可以添加更多信息显示
		else
			print("光标下没有找到 TailwindCSS 类名")
		end
	end, { desc = "显示当前 TailwindCSS 类名信息" })
end

return M