-- ============================================================================
-- LuaSnip 代码片段配置 (lua/configs/luasnip.lua)
-- 为各种编程语言定义自定义代码片段，提供快速输入模板
-- ============================================================================

--[[
📖 LuaSnip 语法说明和使用指南

🔧 基本组件：
- s(trigger, nodes) - 创建片段，trigger 是触发词，nodes 是片段内容
- t("text") - 静态文本节点，不可编辑
- i(number, "default") - 插入节点（占位符），可编辑，按数字顺序跳转
- i(0) - 特殊插入节点，表示最终光标位置
- f(function, args) - 函数节点，动态生成内容

📝 高级功能：
- c(number, {choices}) - 选择节点，可在多个选项间切换
- r(number, "default") - 重复节点，重复其他节点的内容
- d(number, function) - 动态节点，根据其他节点动态生成内容

🎯 文本格式：
- t("单行文本") - 简单文本
- t({"", "换行文本"}) - 多行文本，第一个""表示换行
- t({"", "\t缩进文本"}) - 带缩进的文本，\t表示制表符

💡 使用技巧：
1. 占位符编号决定跳转顺序：i(1) → i(2) → i(3) → i(0)
2. i(0) 必须是最后一个，表示片段展开完成后的光标位置
3. 相同编号的占位符会同步编辑：i(1) 和 i(1) 内容会同时改变
4. 函数节点可以引用其他占位符：f(function(args) return args[1][1] end, {1})

🚀 快捷键：
- Tab - 展开片段或跳转到下一个占位符
- Shift+Tab - 跳转到上一个占位符  
- Ctrl+E - 在选择节点中切换选项
- <leader>l - 打开片段管理菜单

📋 自定义步骤：
1. 在对应语言的片段表中添加新片段
2. 使用 <leader>lr 重新加载配置
3. 使用 <leader>lt 在测试文件中验证

⚠️ 注意事项：
- 片段名称（触发词）要简短易记
- 合理使用占位符，避免过多嵌套
- 函数节点要处理空值情况
- 多行文本注意缩进格式
-- ]]

local ls = require("luasnip")
local s = ls.snippet       -- s(trigger, nodes) - 创建代码片段
local t = ls.text_node     -- t("text") - 静态文本，不可编辑
local i = ls.insert_node   -- i(number, "default") - 可编辑占位符，按数字跳转
local f = ls.function_node -- f(function, args) - 动态生成内容的函数节点
local c = ls.choice_node   -- c(number, {choices}) - 可选择的内容节点

-- ============================================================================
-- JavaScript/TypeScript 代码片段
-- ============================================================================
local javascript_snippets = {
	--[[
	片段格式示例说明：
	s("触发词", {
		t("静态文本"),           -- 不可编辑的文本
		i(1, "默认值"),          -- 第1个占位符，可编辑
		t("更多静态文本"),       -- 继续的静态文本
		i(2, "第二个占位符"),    -- 第2个占位符
		i(0)                    -- 最终光标位置（必须）
	}),
	-- ]]
	
	-- console.log 片段 - 触发词 "csl"
	-- 用法：输入 csl 然后按 Tab，光标会定位到括号内
	s("csl", {
		t("console.log("),      -- 静态文本：console.log(
		i(1, ""),               -- 占位符1：括号内的内容
		t(");"),                -- 静态文本：);
		i(0)                    -- 最终光标位置
	}),
	
	-- console.error 片段 - 触发词 "cse"
	s("cse", {
		t("console.error("),
		i(1, ""),
		t(");"),
		i(0)
	}),
	
	-- console.warn 片段 - 触发词 "csw"
	s("csw", {
		t("console.warn("),
		i(1, ""),
		t(");"),
		i(0)
	}),
	
	-- console.info 片段 - 触发词 "csi"
	s("csi", {
		t("console.info("),
		i(1, ""),
		t(");"),
		i(0)
	}),
	
	-- function 声明 - 触发词 "func"
	s("func", {
		t("function "),
		i(1, "name"),
		t("("),
		i(2, ""),
		t(") {"),
		t({"", "\t"}),
		i(3, "// 函数体"),
		t({"", "}"}),
		i(0)
	}),
	
	-- arrow function - 触发词 "af"
	-- 多行片段示例：使用 t({"", "文本"}) 创建换行
	-- 跳转顺序：函数名 → 参数 → 函数体 → 结束位置
	s("af", {
		t("const "),               -- 静态文本：const 
		i(1, "name"),              -- 占位符1：函数名
		t(" = ("),                 -- 静态文本： = (
		i(2, ""),                  -- 占位符2：参数列表
		t(") => {"),               -- 静态文本：) => {
		t({"", "\t"}),             -- 换行 + 制表符缩进
		i(3, "// 函数体"),         -- 占位符3：函数体内容
		t({"", "}"}),              -- 换行 + 右大括号
		i(0)                       -- 最终光标位置
	}),
	
	-- if statement - 触发词 "if"
	s("if", {
		t("if ("),
		i(1, "condition"),
		t(") {"),
		t({"", "\t"}),
		i(2, "// 代码块"),
		t({"", "}"}),
		i(0)
	}),
	
	-- for loop - 触发词 "for" 
	-- 函数节点示例：f(function(args) return ... end, {依赖的占位符编号})
	-- 这里的函数节点会复制第1个占位符的内容，实现变量名同步
	s("for", {
		t("for (let "),                                    -- 静态文本：for (let 
		i(1, "i"),                                         -- 占位符1：循环变量名
		t(" = 0; "),                                       -- 静态文本： = 0; 
		f(function(args) return args[1][1] end, {1}),      -- 函数节点：复制占位符1的内容
		t(" < "),                                          -- 静态文本： < 
		i(2, "length"),                                    -- 占位符2：循环条件
		t("; "),                                           -- 静态文本：; 
		f(function(args) return args[1][1] end, {1}),      -- 函数节点：再次复制占位符1
		t("++) {"),                                        -- 静态文本：++) {
		t({"", "\t"}),                                     -- 换行 + 缩进
		i(3, "// 循环体"),                                 -- 占位符3：循环体内容
		t({"", "}"}),                                      -- 换行 + 右大括号
		i(0)                                               -- 最终光标位置
	}),
	
	-- forEach - 触发词 "fe"
	s("fe", {
		i(1, "array"),
		t(".forEach(("),
		i(2, "item"),
		t(", "),
		i(3, "index"),
		t(") => {"),
		t({"", "\t"}),
		i(4, "// 处理逻辑"),
		t({"", "});"}),
		i(0)
	}),
	
	-- try-catch - 触发词 "try"
	s("try", {
		t("try {"),
		t({"", "\t"}),
		i(1, "// 可能出错的代码"),
		t({"", "} catch ("}),
		i(2, "error"),
		t(") {"),
		t({"", "\t"}),
		i(3, "// 错误处理"),
		t({"", "}"}),
		i(0)
	}),
	
	-- 你的自定义片段示例
	-- setTimeout - 触发词 "st"
	s("st", {
		t("setTimeout(() => {"),
		t({"", "\t"}),
		i(1, "// 延时执行的代码"),
		t({"", "}, "}),
		i(2, "1000"),
		t(");"),
		i(0)
	}),
	
	-- 快速注释块 - 触发词 "comm"
	s("comm", {
		t("/* *"),
		t({"", " * "}),
		i(1, "描述"),
		t({"", " * @param {"}),
		i(2, "type"),
		t("} "),
		i(3, "paramName"),
		t(" - "),
		i(4, "参数描述"),
		t({"", " * @returns {"}),
		i(5, "type"),
		t("} "),
		i(6, "返回值描述"),
		t({"", " */"}),
		i(0)
	}),
	
	-- import 语句 - 触发词 "imp"
	s("imp", {
		t("import "),
		i(1, "{ }"),
		t(" from '"),
		i(2, "module"),
		t("';"),
		i(0)
	}),
	
	-- export 语句 - 触发词 "exp"
	s("exp", {
		t("export "),
		i(1, "const"),
		t(" "),
		i(2, "name"),
		t(" = "),
		i(3, "value"),
		t(";"),
		i(0)
	}),
}

-- React 特定片段
local react_snippets = {
	-- React 函数组件 - 触发词 "rfc"
	-- 复杂片段示例：包含多个占位符和函数节点的综合使用
	s("rfc", {
		t("import React from 'react';"),                      -- 导入语句
		t({"", "", "const "}),                                -- 两个换行 + const 
		i(1, "ComponentName"),                                -- 占位符1：组件名
		t(" = () => {"),                                      -- 函数声明
		t({"", "\treturn ("}),                               -- 换行 + 缩进 + return
		t({"", "\t\t<div>"}),                                -- div 开始标签
		t({"", "\t\t\t"}),                                   -- 内容缩进
		i(2, "内容"),                                         -- 占位符2：组件内容
		t({"", "\t\t</div>"}),                               -- div 结束标签
		t({"", "\t);"}),                                     -- return 结束
		t({"", "};"}),                                       -- 函数结束
		t({"", "", "export default "}),                      -- export 语句
		f(function(args) return args[1][1] end, {1}),        -- 函数节点：复用组件名
		t(";"),                                               -- 分号结束
		i(0)                                                  -- 最终位置
	}),
	
	-- useState hook - 触发词 "us"
	-- 高级函数节点示例：字符串处理，首字母大写
	s("us", {
		t("const ["),                                         -- 解构赋值开始
		i(1, "state"),                                        -- 占位符1：状态名
		t(", set"),                                           -- setter 前缀
		f(function(args)                                      -- 函数节点：生成setter名
			local name = args[1][1] or "State"               -- 获取状态名，默认"State"
			return name:sub(1,1):upper() .. name:sub(2)      -- 首字母大写
		end, {1}),                                           -- 依赖占位符1
		t("] = useState("),                                   -- useState 调用
		i(2, "initialValue"),                                 -- 占位符2：初始值
		t(");"),                                              -- 结束分号
		i(0)                                                  -- 最终位置
	}),
	
	-- useEffect hook - 触发词 "ue"
	s("ue", {
		t("useEffect(() => {"),
		t({"", "\t"}),
		i(1, "// effect 逻辑"),
		t({"", "}, ["}),
		i(2, ""),
		t("]);"),
		i(0)
	}),
}

-- Python 片段
local python_snippets = {
	-- print 语句 - 触发词 "p"
	s("p", {
		t("print("),
		i(1, ""),
		t(")"),
		i(0)
	}),
	
	-- 函数定义 - 触发词 "def"
	s("def", {
		t("def "),
		i(1, "function_name"),
		t("("),
		i(2, ""),
		t("):"),
		t({"", "\t"}),
		i(3, "pass"),
		i(0)
	}),
	
	-- if __name__ == "__main__" - 触发词 "main"
	s("main", {
		t("if __name__ == \"__main__\":"),
		t({"", "\t"}),
		i(1, "pass"),
		i(0)
	}),
	
	-- Python类定义 - 触发词 "class"
	s("class", {
		t("class "),
		i(1, "ClassName"),
		t(":"),
		t({"", "\t\"\"\""}),
		i(2, "类的描述"),
		t({"\"\"\"", "", "\tdef __init__(self"}),
		i(3, ""),
		t("):"),
		t({"", "\t\t"}),
		i(4, "pass"),
		i(0)
	}),
	
	-- Django视图 - 触发词 "djview"
	s("djview", {
		t("def "),
		i(1, "view_name"),
		t("(request):"),
		t({"", "\t\"\"\""}),
		i(2, "视图描述"),
		t({"\"\"\"", "\tif request.method == 'GET':"}),
		t({"", "\t\t"}),
		i(3, "# GET处理逻辑"),
		t({"", "\t\treturn render(request, '"}),
		i(4, "template.html"),
		t("')"),
		t({"", "\t", "\telif request.method == 'POST':"}),
		t({"", "\t\t"}),
		i(5, "# POST处理逻辑"),
		t({"", "\t\treturn redirect('"}),
		i(6, "redirect_url"),
		t("')"),
		i(0)
	}),
}

-- HTML 片段
local html_snippets = {
	-- HTML5 基础模板 - 触发词 "html5"
	s("html5", {
		t("<!DOCTYPE html>"),
		t({"", "<html lang=\"zh-CN\">"}),
		t({"", "<head>"}),
		t({"", "\t<meta charset=\"UTF-8\">"}),
		t({"", "\t<meta name=\"viewport\" content=\"width=device-width, initial-scale=1.0\">"}),
		t({"", "\t<title>"}),
		i(1, "页面标题"),
		t("</title>"),
		t({"", "</head>"}),
		t({"", "<body>"}),
		t({"", "\t"}),
		i(2, "<!-- 页面内容 -->"),
		t({"", "</body>"}),
		t({"", "</html>"}),
		i(0)
	}),
	
	-- div 标签 - 触发词 "div"
	s("div", {
		t("<div"),
		i(1, ""),
		t(">"),
		i(2, ""),
		t("</div>"),
		i(0)
	}),
}

-- ============================================================================
-- 应用片段配置到对应的文件类型
-- ============================================================================
ls.add_snippets("javascript", javascript_snippets)                                -- .js 文件
ls.add_snippets("typescript", javascript_snippets)                                -- .ts 文件
ls.add_snippets("javascriptreact", vim.tbl_extend("force", javascript_snippets, react_snippets))  -- .jsx 文件
ls.add_snippets("typescriptreact", vim.tbl_extend("force", javascript_snippets, react_snippets))  -- .tsx 文件
ls.add_snippets("python", python_snippets)                                        -- .py 文件
ls.add_snippets("html", html_snippets)                                             -- .html 文件

--[[
============================================================================
📝 自定义代码片段实际使用示例

🎯 添加新的 JavaScript 片段：

-- 在 javascript_snippets 表中添加：
s("mysnippet", {
	t("// 我的自定义片段"),           -- 注释
	t({"", "function "}),            -- 换行 + function
	i(1, "myFunction"),              -- 函数名占位符
	t("("),                          -- 左括号
	i(2, "param"),                   -- 参数占位符
	t(") {"),                        -- 右括号 + 左大括号
	t({"", "\t"}),                   -- 换行 + 缩进
	i(3, "// 函数体"),               -- 函数体占位符
	t({"", "\treturn "}),            -- 换行 + return
	i(4, "result"),                  -- 返回值占位符
	t(";"),                          -- 分号
	t({"", "}"}),                    -- 换行 + 右大括号
	i(0)                             -- 最终光标位置
}),

🎯 使用选择节点的示例：

s("method", {
	t("public "),
	c(1, {                           -- 选择节点：可以选择不同选项
		t("static "),                -- 选项1：static
		t(""),                       -- 选项2：空（无static）
		t("async ")                  -- 选项3：async
	}),
	i(2, "void"),                    -- 返回类型
	t(" "),
	i(3, "methodName"),              -- 方法名
	t("() {"),
	t({"", "\t"}),
	i(0),                            -- 方法体
	t({"", "}"})
}),

💡 使用技巧：
1. 输入触发词后按 Tab 展开
2. Tab / Shift+Tab 在占位符间跳转
3. 在选择节点上按 Ctrl+E 切换选项
4. 使用 <leader>lr 重新加载配置
5. 使用 <leader>ll 查看当前文件可用片段

⚠️ 常见问题：
1. 忘记添加 i(0) - 必须有最终光标位置
2. 占位符编号跳跃 - 应该连续：1,2,3...
3. 多行文本格式错误 - 使用 t({"", "文本"})
4. 函数节点参数错误 - 检查依赖的占位符编号
============================================================================
-- ]]

-- 启用历史记录，可以回到之前跳转的位置
require("luasnip.loaders.from_vscode").lazy_load()

return {}
