-- ============================================================================
-- LazyVim 风格的基础选项配置
-- 从原 NvChad 配置迁移的编辑器选项设置
-- ============================================================================

-- 注意：由于版本限制，使用手动配置而不是 LazyVim 核心
local opt = vim.opt

-- ============================================================================
-- 基础编辑器设置
-- ============================================================================
opt.mouse = "a"                     -- 启用鼠标支持
opt.clipboard = "unnamedplus"       -- 使用系统剪贴板
opt.completeopt = "menu,menuone,noselect"

-- ============================================================================
-- 缩进和格式设置
-- ============================================================================
opt.tabstop = 4                     -- Tab 宽度
opt.shiftwidth = 4                  -- 自动缩进宽度
opt.softtabstop = 4                 -- 软 Tab 宽度
opt.expandtab = true                -- 使用空格替代 Tab
opt.smartindent = true              -- 智能缩进
opt.autoindent = true               -- 自动缩进

-- ============================================================================
-- 搜索设置
-- ============================================================================
opt.ignorecase = true               -- 忽略大小写搜索
opt.smartcase = true                -- 智能大小写
opt.hlsearch = true                 -- 高亮搜索结果
opt.incsearch = true                 -- 增量搜索

-- ============================================================================
-- 显示设置
-- ============================================================================
opt.number = true                   -- 显示行号
opt.relativenumber = true           -- 显示相对行号
opt.cursorline = true               -- 高亮当前行
opt.cursorcolumn = false            -- 不高亮当前列
opt.colorcolumn = "100"             -- 100 列标线
opt.signcolumn = "yes"              -- 始终显示符号列
opt.wrap = false                    -- 不自动换行

-- ============================================================================
-- 分割窗口设置
-- ============================================================================
opt.splitbelow = true               -- 水平分割在下方
opt.splitright = true               -- 垂直分割在右侧

-- ============================================================================
-- 其他重要设置
-- ============================================================================
opt.undofile = true                 -- 启用撤销文件
opt.backup = false                  -- 不创建备份文件
opt.writebackup = false             -- 写入时不创建备份
opt.swapfile = false                -- 不创建交换文件
opt.updatetime = 300                -- 更新时间
opt.timeoutlen = 500                -- 超时时间
opt.ttimeoutlen = 10                -- 键码超时时间

-- ============================================================================
-- 填充字符设置（修复版本兼容性问题）
-- ============================================================================
opt.fillchars = {
	fold = " ",
	diff = "╱",
	eob = " ",
}

-- ============================================================================
-- 短设置列表
-- ============================================================================
opt.shortmess:append({ I = true, W = true, c = true, C = true })

-- ============================================================================
-- 折叠设置
-- ============================================================================
opt.foldmethod = "expr"
opt.foldexpr = "nvim_treesitter#foldexpr()"
opt.foldlevel = 99
opt.foldlevelstart = 99
opt.foldenable = true

-- ============================================================================
-- 终端设置
-- ============================================================================
local terminal_settings = {
	shell = vim.o.shell,
	pumheight = 10,
	pumblend = 10,
	wildmode = "longest:full,full",
	wildoptions = "pum",
	showmode = false,
	ruler = false,
	laststatus = 3,
	cmdheight = 0,
	scrolloff = 8,
	sidescrolloff = 8,
}

-- 应用终端设置
for key, value in pairs(terminal_settings) do
	opt[key] = value
end

-- ============================================================================
-- 匹配设置
-- ============================================================================
opt.matchpairs:append("<:>")
opt.matchtime = 2

-- 跳转菜单设置
opt.complete:remove("t")
opt.complete:append("k")

-- 设置编码
opt.encoding = "utf-8"
opt.fileencoding = "utf-8"

-- 设置 iskeyword（用于单词识别）
opt.iskeyword:append("-")
opt.iskeyword:append("_")

-- 列表字符设置
opt.listchars = {
	tab = "│ ",
	trail = "·",
	extends = "⟩",
	precedes = "⟨",
	space = " ",
}

-- 格式化选项
opt.formatoptions = "jcroqlnt"

-- 会话设置（简化版本）
opt.sessionoptions = "blank,buffers,curdir,folds,help,tabpages,winsize,terminal"

-- 视图设置可能引起问题，暂时禁用
-- opt.viewoptions = "folds,cursorline"

-- 撤销目录设置
local undo_dir = vim.fn.stdpath("data") .. "/undo"
vim.fn.mkdir(undo_dir, "p")
opt.undodir = undo_dir