-- ============================================================================
-- LazyVim 键位映射配置
-- 从原 NvChad 配置迁移的所有键位绑定
-- ============================================================================

local map = vim.keymap.set
local opts = { noremap = true, silent = true }

-- ============================================================================
-- 保存时自动格式化命令（保持原功能）
-- ============================================================================

-- 创建 :W 命令，保存并格式化
vim.api.nvim_create_user_command("W", function()
	vim.cmd("write")
	-- 保存后自动格式化
	local ok, conform = pcall(require, "conform")
	if ok then
		conform.format({ async = true, lsp_fallback = true })
	end
end, { desc = "Write and format file" })

-- 创建 :Wq 命令，保存、格式化并退出
vim.api.nvim_create_user_command("Wq", function()
	vim.cmd("writequit")
	-- 保存后自动格式化
	local ok, conform = pcall(require, "conform")
	if ok then
		conform.format({ async = true, lsp_fallback = true })
	end
end, { desc = "Write, format and quit" })

-- ============================================================================
-- 基础功能键位映射
-- ============================================================================

-- 清除搜索高亮
map("n", "<Esc>", "<cmd>noh<cr>", opts)

-- 快速保存
map("n", "<leader>w", "<cmd>w<cr>", opts)

-- 快速退出
map("n", "<leader>q", "<cmd>q<cr>", opts)

-- ============================================================================
-- 导航移动键位映射
-- ============================================================================

-- 行首行尾快速移动
map({ "n", "v" }, "H", "^", opts)
map({ "n", "v" }, "L", "$", opts)

-- 屏幕滚动（居中显示）
map("n", "<C-u>", "<C-u>zz", opts)
map("n", "<C-d>", "<C-d>zz", opts)
map("n", "<C-f>", "<C-f>zz", opts)
map("n", "<C-b>", "<C-b>zz", opts)

-- 缓冲区切换
map("n", "<Tab>", "<cmd>bnext<cr>", opts)
map("n", "<S-Tab>", "<cmd>bprev<cr>", opts)

-- 窗口导航
map("n", "<C-h>", "<C-w>h", opts)
-- map("n", "<C-j>", "<C-w>j", opts)  -- 暂时注释，与 NvimTree 翻页冲突
-- map("n", "<C-k>", "<C-w>k", opts)  -- 暂时注释，与 NvimTree 翻页冲突
map("n", "<C-l>", "<C-w>l", opts)

-- 窗口调整大小
map("n", "<C-Up>", "<cmd>resize +2<cr>", opts)
map("n", "<C-Down>", "<cmd>resize -2<cr>", opts)
map("n", "<C-Left>", "<cmd>vertical resize -2<cr>", opts)
map("n", "<C-Right>", "<cmd>vertical resize +2<cr>", opts)

-- ============================================================================
-- 编辑增强键位映射
-- ============================================================================

-- 快速删除
map("n", "x", '"_x', opts)
map("n", "X", '"_X', opts)

-- 复制到系统剪贴板
map({ "n", "v" }, "<leader>y", '"+y', opts)
map("n", "<leader>yy", '"+yy', opts)

-- 从系统剪贴板粘贴
map({ "n", "v" }, "<leader>p", '"+p', opts)

-- 删除不复制
map({ "n", "v" }, "<leader>d", '"_d', opts)

-- 增强撤销/重做
map("n", "U", "<C-r>", opts)

-- ============================================================================
-- Alt 键修饰的快捷键（保持原配置风格）
-- ============================================================================

-- Alt + hjkl 快速导航
map("n", "<M-h>", "<C-w>h", opts)
map("n", "<M-j>", "<C-w>j", opts)
map("n", "<M-k>", "<C-w>k", opts)
map("n", "<M-l>", "<C-w>l", opts)

-- Alt + 方向键窗口导航
map("n", "<M-Left>", "<C-w>h", opts)
map("n", "<M-Down>", "<C-w>j", opts)
map("n", "<M-Up>", "<C-w>k", opts)
map("n", "<M-Right>", "<C-w>l", opts)

-- Alt + 数字快速跳转
map("n", "<M-1>", "1gt", opts)
map("n", "<M-2>", "2gt", opts)
map("n", "<M-3>", "3gt", opts)
map("n", "<M-4>", "4gt", opts)
map("n", "<M-5>", "5gt", opts)
map("n", "<M-6>", "6gt", opts)
map("n", "<M-7>", "7gt", opts)
map("n", "<M-8>", "8gt", opts)
map("n", "<M-9>", "9gt", opts)

-- ============================================================================
-- 文本对象和可视模式增强
-- ============================================================================

-- 引号内文本对象
map({ "o", "x" }, "a'", "a<'>", opts)
map({ "o", "x" }, "i'", "i<'>", opts)
map({ "o", "x" }, 'a"', 'a<">', opts)
map({ "o", "x" }, 'i"', 'i<">', opts)
map({ "o", "x" }, "a`", "a<`>", opts)
map({ "o", "x" }, "i`", "i<`>", opts)

-- 可视模式增强
map("v", "<", "<gv", opts)
map("v", ">", ">gv", opts)

-- ============================================================================
-- 搜索增强
-- ============================================================================

-- 增强搜索功能
map("n", "*", "*zz", opts)
map("n", "#", "#zz", opts)
map("n", "n", "nzz", opts)
map("n", "N", "Nzz", opts)

-- 清除搜索高亮
map("n", "<leader>ch", "<cmd>noh<cr>", opts)

-- ============================================================================
-- 快速移动和跳跃
-- ============================================================================

-- 快速跳转到行首/行尾
map({ "n", "o", "x" }, "0", "^", opts)
map({ "n", "o", "x" }, "^", "0", opts)

-- 逗号和分号快速跳转
map("n", "<leader>;", ";", opts)
map("n", "<leader>;", ",", opts)

-- ============================================================================
-- 折叠键位映射
-- ============================================================================

-- 折叠操作
map("n", "za", "za", opts)
map("n", "zc", "zc", opts)
map("n", "zo", "zo", opts)
map("n", "zm", "zm", opts)
map("n", "zr", "zr", opts)
map("n", "zM", "zM", opts)
map("n", "zR", "zR", opts)

-- ============================================================================
-- 标签页管理
-- ============================================================================

-- 标签页操作
map("n", "<leader>tn", "<cmd>tabnew<cr>", opts)
map("n", "<leader>tc", "<cmd>tabclose<cr>", opts)
map("n", "<leader>to", "<cmd>tabonly<cr>", opts)

-- 标签页导航
map("n", "gt", "gt", opts)
map("n", "gT", "gT", opts)

-- ============================================================================
-- 宏和录制增强
-- ============================================================================

-- 快速执行宏
map("n", "Q", "@q", opts)
map("n", "<leader>Q", "@@", opts)

-- ============================================================================
-- 终端操作
-- ============================================================================

-- 快速打开终端
map("n", "<leader>t", "<cmd>terminal<cr>", opts)
map("t", "<Esc>", "<C-\\><C-n>", opts)

-- 终端模式下的导航
map("t", "<C-h>", "<C-\\><C-n><C-w>h", opts)
map("t", "<C-j>", "<C-\\><C-n><C-w>j", opts)
map("t", "<C-k>", "<C-\\><C-n><C-w>k", opts)
map("t", "<C-l>", "<C-\\><C-n><C-w>l", opts)

-- ============================================================================
-- 快捷键帮助和文档
-- ============================================================================

-- 显示键位映射
map("n", "<leader>hk", "<cmd>map <leader><cr>", opts)

-- ============================================================================
-- 主题切换快捷键
-- ============================================================================

-- 切换到特定主题
map("n", "<leader>tm", "<cmd>lua require('config.theme').switch_theme('mocha')<cr>", opts)
map("n", "<leader>tf", "<cmd>lua require('config.theme').switch_theme('frappe')<cr>", opts)
map("n", "<leader>tmc", "<cmd>lua require('config.theme').switch_theme('macchiato')<cr>", opts)
map("n", "<leader>tl", "<cmd>lua require('config.theme').switch_theme('latte')<cr>", opts)

-- 循环切换主题
map("n", "<leader>tc", "<cmd>lua require('config.theme').cycle_theme()<cr>", opts)

-- 显示当前主题信息
map("n", "<leader>ti", "<cmd>lua require('config.theme').show_current_theme()<cr>", opts)

-- 显示命令帮助
map("n", "<leader>hc", "<cmd>command<cr>", opts)

-- ============================================================================
-- 文件操作增强
-- ============================================================================

-- 重新加载文件
map("n", "<leader>e", "<cmd>e!<cr>", opts)

-- 重新加载配置
map("n", "<leader>rc", "<cmd>source $MYVIMRC<cr>", opts)

-- ============================================================================
-- 特殊符号输入（保持原配置的便利性）
-- ============================================================================

-- 在插入模式下快速输入特殊符号
local special_chars = {
	["<C-]>"] = "→",
	["<C-[>"] = "←",
	["<C-'>"] = "↑",
	["<C-;>"] = "↓",
	["<C-/>"] = "√",
	["<C-\\>"] = "×",
}

for key, char in pairs(special_chars) do
	map("i", key, char, opts)
end

-- ============================================================================
-- 搜索和替换增强
-- ============================================================================

-- 全局替换
map("n", "<leader>rr", ":%s/", opts)

-- 缓冲区内替换
map("n", "<leader>rs", ":s/", opts)

-- ============================================================================
-- 撤销树（如果存在）
-- ============================================================================

-- 如果安装了 undotree 插件
map("n", "<leader>u", "<cmd>UndotreeToggle<cr>", opts)

-- ============================================================================
-- Git 操作键位映射
-- ============================================================================

-- Git 状态
map("n", "<leader>gs", "<cmd>Git<cr>", opts)

-- Git 提交
map("n", "<leader>gc", "<cmd>Git commit<cr>", opts)

-- Git 推送
map("n", "<leader>gp", "<cmd>Git push<cr>", opts)

-- Git 拉取
map("n", "<leader>gP", "<cmd>Git pull<cr>", opts)

-- Git 分支
map("n", "<leader>gb", "<cmd>Git branch<cr>", opts)

-- Git 差异
map("n", "<leader>gd", "<cmd>Gdiffsplit<cr>", opts)

-- ============================================================================
-- LSP 相关键位映射（将在 LSP 配置中进一步定义）
-- ============================================================================

-- 这里只定义基础键位，具体 LSP 键位将在 lsp.lua 中定义
map("n", "gd", "<cmd>lua vim.lsp.buf.definition()<cr>", opts)
map("n", "gr", "<cmd>lua vim.lsp.buf.references()<cr>", opts)
map("n", "gD", "<cmd>lua vim.lsp.buf.declaration()<cr>", opts)
map("n", "gi", "<cmd>lua vim.lsp.buf.implementation()<cr>", opts)

-- 诊断信息
map("n", "[d", "<cmd>lua vim.diagnostic.goto_prev()<cr>", opts)
map("n", "]d", "<cmd>lua vim.diagnostic.goto_next()<cr>", opts)
map("n", "<leader>e", "<cmd>lua vim.diagnostic.open_float()<cr>", opts)

-- ============================================================================
-- 注意事项
-- ============================================================================

-- 1. 这些键位映射与原 NvChad 配置保持一致
-- 2. Leader 键设置为逗号（,）
-- 3. 空格键映射为冒号（:）用于快速命令输入
-- 4. Alt 键用作主要的修饰键，避免与系统快捷键冲突
-- 5. 插件特定的键位映射将在各自的插件配置文件中定义