-- ============================================================================
-- Neovim 主配置文件 (init.lua)
-- 这是 Neovim 启动时首先加载的配置文件
-- 基于 NvChad 框架，提供现代化的 Neovim 配置
-- ============================================================================

-- ============================================================================
-- 1. 基础全局设置
-- ============================================================================
-- 设置 Leader 键为逗号（用于自定义快捷键前缀）
vim.g.mapleader = ","

-- 设置 Base46 主题缓存目录（必须在插件加载前设置）
vim.g.base46_cache = vim.fn.stdpath("data") .. "/base46/"

-- 设置按下空格为按下:（便捷的命令行访问）
vim.api.nvim_set_keymap("n", "<Space>", ":", { noremap = true, silent = false })

-- ============================================================================
-- 2. Lazy.nvim 插件管理器初始化
-- ============================================================================
-- 设置 Lazy.nvim 插件管理器的安装路径
local lazypath = vim.fn.stdpath("data") .. "/lazy/lazy.nvim"

-- 如果 Lazy.nvim 未安装，则自动克隆安装
if not vim.uv.fs_stat(lazypath) then
	local repo = "https://github.com/folke/lazy.nvim.git"
	vim.fn.system({ "git", "clone", "--filter=blob:none", repo, "--branch=stable", lazypath })
end

-- 将 Lazy.nvim 添加到 Neovim 的运行时路径
vim.opt.rtp:prepend(lazypath)

-- ============================================================================
-- 3. 插件配置和加载
-- ============================================================================
-- 加载 Lazy.nvim 配置选项
local lazy_config = require("configs.lazy")

-- 设置并加载所有插件
require("lazy").setup({
	-- NvChad 核心插件（提供基础框架）
	{
		"NvChad/NvChad",
		lazy = false,        -- 立即加载，不延迟
		branch = "v2.5",     -- 使用稳定的 v2.5 分支
		import = "nvchad.plugins", -- 导入 NvChad 默认插件配置
	},

	-- 用户自定义插件（从 plugins/ 目录导入）
	{ import = "plugins" },
}, lazy_config)

-- ============================================================================
-- 4. 主题和界面初始化
-- ============================================================================
-- 加载默认主题配置（必须在插件加载后）
dofile(vim.g.base46_cache .. "defaults")
-- 加载状态栏主题配置
dofile(vim.g.base46_cache .. "statusline")

-- ============================================================================
-- 5. 核心配置模块加载
-- ============================================================================
-- 加载核心配置统一模块（基础选项和弃用函数修复）
require("configs.core").setup()

-- 加载自动命令配置（事件响应设置）
require("configs.autocmds")

-- 加载 Lazy.nvim 界面颜色配置
require("configs.lazy-colors").setup()

-- 延迟加载键位映射（提高启动速度，在界面准备好后加载）
vim.schedule(function()
	require("configs.mappings")
end)

-- 加载 TailwindCSS 自动配置功能
require("configs.tailwind-auto-config").setup()

-- 加载 NvimTree 配置（包含自定义命令）
require("configs.nvimtree")

-- ============================================================================
-- 6. 自定义功能和增强配置
-- ============================================================================
-- 代码格式化策略（摘要）：
-- - HTML 采用"Prettier 主格式化 + html_noblank 轻量后处理"的组合；
-- 目标：保留自动格式化的一致性，同时移除 <head>/<body> 标签上下多余空行；
-- 相关配置：.prettierrc.js 控制通用风格；lua/configs/conform.lua 定义格式化链路与后处理器。
-- 弃用函数兼容性修复已整合到 core.lua 模块中

-- 6.1 文件类型识别增强
-- 使用 vim.filetype.add 添加自定义文件名模式识别
vim.filetype.add({
	pattern = {
		-- 用正则表达式匹配文件名包含 "docker" 的文件，设置为 dockerfile 类型
		-- 这样可以为 Dockerfile、docker-compose.yml 等文件提供正确的语法高亮
		[".*[dD][oO][cC][kK][eE][rR].*"] = "dockerfile",
	},
})

-- 6.2 文件树自动打开和焦点管理
-- 定义 Vim 启动时的文件树处理函数
local function on_vim_enter(data)
	-- 安全地尝试加载 nvim-tree API
	local ok, nvim_tree_api = pcall(require, "nvim-tree.api")
	if not ok then
		-- 如果 nvim-tree 未正确加载，直接返回
		return
	end

	-- 判断启动参数：目录/文件/无参数
	local opened_path = data.file or ""
	local is_directory = vim.fn.isdirectory(opened_path) == 1
	local has_file = opened_path ~= "" and not is_directory

	if is_directory or not has_file then
		-- vi 或 vi <dir>：聚焦目录树
		if is_directory then
			vim.cmd.cd(opened_path)
		end
		nvim_tree_api.tree.open({ focus = true })
	else
		-- vi <file>：打开目录树但焦点留在文件
		local original_win = vim.api.nvim_get_current_win()
		nvim_tree_api.tree.open({ focus = false })
		-- 定位到该文件对应的节点（可能会改变焦点，稍后恢复）
		pcall(function()
			nvim_tree_api.tree.find_file(vim.fn.expand('%:p'))
		end)
		-- 恢复焦点到原文件窗口
		pcall(function()
			if vim.api.nvim_win_is_valid(original_win) then
				vim.api.nvim_set_current_win(original_win)
			end
		end)
	end

	-- 延迟触发 Git 状态刷新，确保 NvimTree 完全初始化后显示正确的 Git 颜色
	vim.defer_fn(function()
		-- 手动触发 BufEnter 事件，模拟"打开文件时的Git颜色刷新机制"
		-- 这会触发 nvimtree.lua 第2595行的 BufEnter 自动命令
		vim.api.nvim_exec_autocmds("BufEnter", {
			pattern = "*",
			modeline = false,
		})
		
		-- 额外的强制刷新作为备份
		pcall(function()
			require('nvim-tree.git').reload()
			nvim_tree_api.tree.reload()
		end)
		
		-- 触发与手动打开文件时相同的颜色刷新逻辑
		pcall(function()
			local highlights = require('configs.highlights')
			highlights.set_nvim_tree_highlights()
			highlights.ultimate_git_color_fix()
			highlights.force_refresh_nvimtree()
		end)
	end, 300) -- 稍长的延迟，确保 NvimTree 和所有自动命令都已注册

end

-- 6.3 启动时自动命令注册
-- 在 Vim 启动时自动触发文件树打开逻辑
vim.api.nvim_create_autocmd({ "VimEnter" }, {
	callback = on_vim_enter,
	desc = "Auto open nvim-tree on startup", -- 自动命令描述
})

-- 6.4 自定义编辑器命令
-- 创建 :Ht 命令来显示 NvimTree 帮助文档（类似 :h 命令的实现）
vim.api.nvim_create_user_command('Ht', function()
	-- 创建一个新的帮助窗口，类似于 :help 命令的行为
	vim.cmd('botright new')
	vim.cmd('resize 15')
	-- 读取帮助文档内容
	local help_file = vim.fn.stdpath('config') .. '/nvim-tree-marks-guide.md'
	-- 直接打开文件而不是读取到当前缓冲区
	vim.cmd('edit ' .. help_file)
	-- 设置缓冲区选项，使其表现像帮助窗口
	vim.bo.buftype = 'help'
	vim.bo.buflisted = false
	vim.bo.modifiable = false
	vim.bo.readonly = true
	vim.bo.filetype = 'help'
	-- 清除修改标记
	vim.bo.modified = false
	-- 设置窗口选项
	vim.wo.number = false
	vim.wo.relativenumber = false
	vim.wo.signcolumn = 'no'
	-- 添加关闭快捷键
	vim.keymap.set('n', 'q', '<cmd>close<cr>', { buffer = true, silent = true })
	vim.keymap.set('n', '<Esc>', '<cmd>close<cr>', { buffer = true, silent = true })
end, {
	desc = '显示 NvimTree 快捷键帮助文档'
})

-- 创建命令缩写，允许用户输入 :ht 来执行 :Ht
vim.cmd('cabbrev ht Ht')
