-- ============================================================================
-- 简单书签系统 (simple-bookmarks.lua)
-- 实现轻量级的书签功能，支持行号显示和书签标记
-- ============================================================================

local M = {}

-- 书签存储 - 按文件路径存储书签行号
-- 数据结构: bookmarks[file_path] = { [line_num] = true, ... }
M.bookmarks = {}

-- 书签标记符号配置
M.config = {
    sign_name = "SimpleBookmark",
    sign_text = "📌",  -- 书签符号（图钉更明显）
    sign_texthl = "DiagnosticSignWarn",  -- 高亮组
    save_file = vim.fn.expand("~/bookmarks.json"),  -- 保存文件路径
}

-- 确保符号列始终显示固定宽度
local function ensure_signcolumn()
    -- 始终显示符号列，保持界面一致性
    vim.wo.signcolumn = 'yes:1'
end

-- 初始化书签符号定义
local function setup_signs()
    -- 创建自定义高亮组，使用更明显的颜色
    vim.cmd([[
        highlight SimpleBookmarkSign guifg=#ff6b6b guibg=NONE ctermfg=196 ctermbg=NONE gui=bold cterm=bold
        highlight SimpleBookmarkLine guibg=#3a3a3a guifg=NONE ctermbg=237 ctermfg=NONE
        highlight SimpleBookmarkNum guifg=#ff6b6b guibg=NONE ctermfg=196 ctermbg=NONE gui=bold cterm=bold
    ]])
    
    -- 尝试多种符号选择
    local sign_options = {"📌", "🔖", "►", "▶", ">>", "*", "•"}
    local chosen_sign = M.config.sign_text
    
    -- 如果用户没有指定，尝试使用最适合的符号
    for _, sign in ipairs(sign_options) do
        if vim.fn.strdisplaywidth(sign) > 0 then
            chosen_sign = sign
            break
        end
    end
    
    vim.fn.sign_define(M.config.sign_name, {
        text = chosen_sign,
        texthl = "SimpleBookmarkSign",
        linehl = "",  -- 不使用整行高亮，避免干扰
        numhl = "SimpleBookmarkNum",   -- 行号高亮
    })
    
    -- 静默设置符号，不显示提示
end

-- 获取当前文件的书签
local function get_file_bookmarks(file_path)
    file_path = file_path or vim.api.nvim_buf_get_name(vim.api.nvim_get_current_buf())
    if file_path == "" then
        return {}  -- 无名缓冲区不保存书签
    end
    if not M.bookmarks[file_path] then
        M.bookmarks[file_path] = {}
    end
    return M.bookmarks[file_path]
end

-- 检查当前行是否有书签
function M.has_bookmark(file_path, line)
    file_path = file_path or vim.api.nvim_buf_get_name(vim.api.nvim_get_current_buf())
    line = line or vim.api.nvim_win_get_cursor(0)[1]
    local file_bookmarks = get_file_bookmarks(file_path)
    return file_bookmarks[line] == true
end

-- 添加书签标记到指定行
local function place_sign(bufnr, line)
    -- 使用特定的标记ID以便更好地管理
    local sign_id = bufnr * 1000 + line
    vim.fn.sign_place(sign_id, "SimpleBookmarkGroup", M.config.sign_name, bufnr, {
        lnum = line,
        priority = 10
    })
end

-- 移除指定行的书签标记
local function remove_sign(bufnr, line)
    local sign_id = bufnr * 1000 + line
    vim.fn.sign_unplace("SimpleBookmarkGroup", {
        buffer = bufnr,
        id = sign_id
    })
end

-- 确保signcolumn可见（现在全局固定宽度，通常不需要调整）
local function ensure_signcolumn()
    -- 符号列已在core.lua中全局设置为yes:1，通常不需要调整
end

-- 切换当前行的书签状态
function M.toggle_bookmark()
    local bufnr = vim.api.nvim_get_current_buf()
    local line = vim.api.nvim_win_get_cursor(0)[1]
    local file_path = vim.api.nvim_buf_get_name(bufnr)
    
    -- 无名缓冲区不支持持久化书签
    if file_path == "" then
        vim.notify("无名缓冲区不支持持久化书签", vim.log.levels.WARN)
        return
    end
    
    local file_bookmarks = get_file_bookmarks(file_path)
    
    -- 确保标记系统已初始化
    setup_signs()
    
    -- 确保signcolumn可见
    ensure_signcolumn()
    
    if file_bookmarks[line] then
        -- 移除书签
        file_bookmarks[line] = nil
        remove_sign(bufnr, line)
        vim.notify("已移除第 " .. line .. " 行的书签", vim.log.levels.INFO)
        
        -- 如果当前文件没有书签了，清理空条目
        if vim.tbl_isempty(file_bookmarks) then
            M.bookmarks[file_path] = nil
        end
    else
        -- 添加书签
        file_bookmarks[line] = true
        place_sign(bufnr, line)
        vim.notify("已在第 " .. line .. " 行添加书签", vim.log.levels.INFO)
    end
    
    -- 自动保存书签（不显示提示）
    M.save_bookmarks(false)
end

-- 跳转到下一个书签
function M.next_bookmark()
    local bufnr = vim.api.nvim_get_current_buf()
    local current_line = vim.api.nvim_win_get_cursor(0)[1]
    local file_path = vim.api.nvim_buf_get_name(bufnr)
    local file_bookmarks = get_file_bookmarks(file_path)
    
    -- 获取所有书签行号并排序
    local bookmark_lines = {}
    for line, _ in pairs(file_bookmarks) do
        table.insert(bookmark_lines, line)
    end
    
    if #bookmark_lines == 0 then
        vim.notify("当前文件没有书签", vim.log.levels.WARN)
        return
    end
    
    table.sort(bookmark_lines)
    
    -- 找到下一个书签
    local next_line = nil
    for _, line in ipairs(bookmark_lines) do
        if line > current_line then
            next_line = line
            break
        end
    end
    
    -- 如果没找到，跳转到第一个书签
    if not next_line then
        next_line = bookmark_lines[1]
    end
    
    -- 跳转到书签行
    vim.api.nvim_win_set_cursor(0, {next_line, 0})
    vim.notify("跳转到第 " .. next_line .. " 行书签", vim.log.levels.INFO)
end

-- 跳转到上一个书签
function M.prev_bookmark()
    local bufnr = vim.api.nvim_get_current_buf()
    local current_line = vim.api.nvim_win_get_cursor(0)[1]
    local file_path = vim.api.nvim_buf_get_name(bufnr)
    local file_bookmarks = get_file_bookmarks(file_path)
    
    -- 获取所有书签行号并排序
    local bookmark_lines = {}
    for line, _ in pairs(file_bookmarks) do
        table.insert(bookmark_lines, line)
    end
    
    if #bookmark_lines == 0 then
        vim.notify("当前文件没有书签", vim.log.levels.WARN)
        return
    end
    
    table.sort(bookmark_lines, function(a, b) return a > b end) -- 逆序排列
    
    -- 找到上一个书签
    local prev_line = nil
    for _, line in ipairs(bookmark_lines) do
        if line < current_line then
            prev_line = line
            break
        end
    end
    
    -- 如果没找到，跳转到最后一个书签
    if not prev_line then
        prev_line = bookmark_lines[1] -- 逆序后第一个就是最大的
    end
    
    -- 跳转到书签行
    vim.api.nvim_win_set_cursor(0, {prev_line, 0})
    vim.notify("跳转到第 " .. prev_line .. " 行书签", vim.log.levels.INFO)
end

-- 列出所有文件的书签
function M.list_bookmarks()
    -- 检查是否有书签
    if vim.tbl_isempty(M.bookmarks) then
        vim.notify("没有任何书签", vim.log.levels.INFO)
        return
    end
    
    -- 收集所有书签信息
    local qf_items = {}
    local total_count = 0
    
    -- 按文件路径排序
    local sorted_files = {}
    for file_path, _ in pairs(M.bookmarks) do
        table.insert(sorted_files, file_path)
    end
    table.sort(sorted_files)
    
    -- 为每个文件收集书签
    for _, file_path in ipairs(sorted_files) do
        local file_bookmarks = M.bookmarks[file_path]
        
        -- 获取文件的书签行号并排序
        local bookmark_lines = {}
        for line, _ in pairs(file_bookmarks) do
            table.insert(bookmark_lines, line)
        end
        table.sort(bookmark_lines)
        
        -- 为每个书签创建quickfix条目
        for _, line in ipairs(bookmark_lines) do
            local line_content = ""
            
            -- 尝试读取文件内容以显示预览
            if vim.fn.filereadable(file_path) == 1 then
                local file = io.open(file_path, "r")
                if file then
                    local current_line = 1
                    for content_line in file:lines() do
                        if current_line == line then
                            line_content = content_line:gsub("^%s+", "") -- 去除前导空格
                            break
                        end
                        current_line = current_line + 1
                    end
                    file:close()
                end
            else
                line_content = "[文件不存在或无法读取]"
            end
            
            -- 分离目录路径和文件名用于显示
            local dir_path = vim.fn.fnamemodify(file_path, ":h")  -- 获取目录路径
            local filename = vim.fn.fnamemodify(file_path, ":t")  -- 获取文件名
            
            table.insert(qf_items, {
                filename = dir_path .. "/",  -- 左边显示目录路径
                lnum = line,
                text = string.format("%s  %s", filename, line_content), -- 文件名 + 内容
            })
            
            total_count = total_count + 1
        end
    end
    
    -- 创建文件路径映射表（用于跳转修正）
    M.qf_file_mapping = {}
    local qf_index = 1
    
    -- 按照qf_items的构建顺序创建映射
    for _, file_path in ipairs(sorted_files) do
        local file_bookmarks = M.bookmarks[file_path]
        local bookmark_lines = {}
        
        for line, _ in pairs(file_bookmarks) do
            table.insert(bookmark_lines, line)
        end
        table.sort(bookmark_lines)
        
        -- 为这个文件的每个书签创建映射
        for _, line in ipairs(bookmark_lines) do
            M.qf_file_mapping[qf_index] = file_path
            qf_index = qf_index + 1
        end
    end
    
    -- 设置quickfix列表并打开
    vim.fn.setqflist({}, "r", { title = "书签列表", items = qf_items })
    vim.cmd("copen")
    
    -- 设置quickfix窗口的跳转修正
    vim.schedule(function()
        -- 等待quickfix窗口完全加载后设置键位映射
        vim.cmd([[
            augroup BookmarkQuickfixJump
                autocmd!
                autocmd FileType qf lua require('configs.simple-bookmarks').setup_qf_jump()
            augroup END
        ]])
        
        -- 使用Lua API设置BufEnter自动命令
        vim.api.nvim_create_autocmd("BufEnter", {
            group = "BookmarkQuickfixJump",
            callback = function()
                if vim.bo.filetype == 'qf' then
                    M.setup_qf_jump()
                end
            end,
            desc = "设置书签quickfix跳转键位"
        })
        
        -- 立即为当前quickfix窗口设置映射
        if vim.bo.filetype == 'qf' then
            M.setup_qf_jump()
        end
    end)
    
    vim.notify("已显示 " .. #sorted_files .. " 个文件中的 " .. total_count .. " 个书签", vim.log.levels.INFO)
end

-- 设置quickfix跳转修正
function M.setup_qf_jump()
    -- 确保在quickfix窗口中设置键位映射
    if vim.bo.filetype == 'qf' then
        -- 重新映射回车键以实现正确跳转
        vim.api.nvim_buf_set_keymap(0, 'n', '<CR>', 
            ':lua require("configs.simple-bookmarks").qf_jump()<CR>', 
            { noremap = true, silent = true })
        -- 键位映射已设置
    end
end

-- quickfix跳转函数
function M.qf_jump()
    local line_nr = vim.api.nvim_win_get_cursor(0)[1]
    local qf_list = vim.fn.getqflist()
    
    if line_nr <= #qf_list and M.qf_file_mapping and M.qf_file_mapping[line_nr] then
        local file_path = M.qf_file_mapping[line_nr]
        local lnum = qf_list[line_nr].lnum
        
        -- 检查文件是否存在
        if vim.fn.filereadable(file_path) == 0 then
            vim.notify("文件不存在: " .. file_path, vim.log.levels.ERROR)
            return
        end
        
        -- 立即执行跳转，不关闭quickfix窗口
        -- 找到主编辑窗口（最大的非特殊窗口）
        local main_win = nil
        local max_height = 0
        
        for _, win in ipairs(vim.api.nvim_list_wins()) do
            local buf = vim.api.nvim_win_get_buf(win)
            local filetype = vim.api.nvim_buf_get_option(buf, 'filetype')
            local height = vim.api.nvim_win_get_height(win)
            
            if filetype ~= 'NvimTree' and filetype ~= 'qf' and filetype ~= 'quickfix' and height > max_height then
                main_win = win
                max_height = height
            end
        end
        
        if main_win then
            -- 切换到主编辑窗口
            vim.api.nvim_set_current_win(main_win)
            
            -- 强制打开文件
            local escaped_path = vim.fn.fnameescape(file_path)
            vim.cmd("edit " .. escaped_path)
            
            -- 跳转到指定行
            if lnum and lnum > 0 then
                vim.api.nvim_win_set_cursor(0, {lnum, 0})
            end
            
            vim.notify("已跳转到 " .. file_path .. " 第 " .. lnum .. " 行")
        else
            -- 如果没有找到合适的窗口，在quickfix上方创建新窗口
            vim.cmd("wincmd k")  -- 移动到上方
            if vim.bo.filetype == 'qf' or vim.bo.filetype == 'quickfix' then
                vim.cmd("new")  -- 创建新的水平分割窗口
                local escaped_path = vim.fn.fnameescape(file_path)
                vim.cmd("edit " .. escaped_path)
                if lnum and lnum > 0 then
                    vim.api.nvim_win_set_cursor(0, {lnum, 0})
                end
                vim.notify("已跳转到 " .. file_path .. " 第 " .. lnum .. " 行")
            end
        end
        
    else
        -- 回退到默认行为
        local ok, _ = pcall(function()
            vim.cmd("normal! \\<CR>")
        end)
        if not ok then
            vim.notify("跳转失败", vim.log.levels.WARN)
        end
    end
end

-- 清除当前缓冲区所有书签
function M.clear_bookmarks()
    local bufnr = vim.api.nvim_get_current_buf()
    local file_path = vim.api.nvim_buf_get_name(bufnr)
    local file_bookmarks = get_file_bookmarks(file_path)
    
    local count = 0
    for line, _ in pairs(file_bookmarks) do
        remove_sign(bufnr, line)
        count = count + 1
    end
    
    M.bookmarks[file_path] = nil
    
    -- 保持符号列显示
    ensure_signcolumn()
    
    -- 保存书签到文件
    M.save_bookmarks(false)
    
    vim.notify("已清除当前文件的 " .. count .. " 个书签", vim.log.levels.INFO)
end

-- 清除所有文件的书签
function M.clear_all_bookmarks()
    local total_count = 0
    
    -- 先清除所有当前打开缓冲区的标记
    local buffers = vim.api.nvim_list_bufs()
    for _, bufnr in ipairs(buffers) do
        if vim.api.nvim_buf_is_valid(bufnr) then
            local file_path = vim.api.nvim_buf_get_name(bufnr)
            if file_path ~= "" and M.bookmarks[file_path] then
                for line, _ in pairs(M.bookmarks[file_path]) do
                    remove_sign(bufnr, line)
                    total_count = total_count + 1
                end
            end
        end
    end
    
    -- 统计所有书签（包括未打开的文件）
    for file_path, file_bookmarks in pairs(M.bookmarks) do
        for _, _ in pairs(file_bookmarks) do
            total_count = total_count + 1
        end
    end
    
    M.bookmarks = {}
    
    -- 保持符号列显示
    ensure_signcolumn()
    
    -- 保存书签到文件
    M.save_bookmarks(false)
    
    vim.notify("已清除所有 " .. total_count .. " 个书签", vim.log.levels.INFO)
end

-- 保存书签到文件
function M.save_bookmarks(show_notification)
    local save_file = M.config.save_file
    
    -- 创建保存目录
    local save_dir = vim.fn.fnamemodify(save_file, ":h")
    if vim.fn.isdirectory(save_dir) == 0 then
        vim.fn.mkdir(save_dir, "p")
    end
    
    -- 清理空的书签条目
    for file_path, file_bookmarks in pairs(M.bookmarks) do
        if vim.tbl_isempty(file_bookmarks) then
            M.bookmarks[file_path] = nil
        end
    end
    
    -- 将书签数据序列化为JSON
    -- 确保每个文件的书签数据都是正确的对象格式
    local clean_bookmarks = {}
    for file_path, file_bookmarks in pairs(M.bookmarks) do
        if not vim.tbl_isempty(file_bookmarks) then
            -- 将行号转换为字符串以确保生成对象而不是数组
            local string_keyed = {}
            for line, value in pairs(file_bookmarks) do
                string_keyed[tostring(line)] = value
            end
            clean_bookmarks[file_path] = string_keyed
        end
    end
    
    -- 如果没有书签数据，写入空对象而不是空数组
    local json_data = vim.json.encode(clean_bookmarks)
    
    -- 写入文件
    local file = io.open(save_file, "w")
    if file then
        file:write(json_data)
        file:close()
        -- 只在手动保存时显示提示
        if show_notification then
            local bookmark_count = 0
            for _, file_bookmarks in pairs(clean_bookmarks) do
                for _ in pairs(file_bookmarks) do
                    bookmark_count = bookmark_count + 1
                end
            end
            if bookmark_count > 0 then
                vim.notify("书签已保存到 " .. save_file, vim.log.levels.INFO)
            else
                vim.notify("书签文件已清空", vim.log.levels.INFO)
            end
        end
    else
        vim.notify("保存书签失败: 无法写入文件 " .. save_file, vim.log.levels.ERROR)
    end
end

-- 加载书签从文件
function M.load_bookmarks()
    local save_file = M.config.save_file
    
    -- 检查文件是否存在，不存在则创建
    if vim.fn.filereadable(save_file) == 0 then
        -- 创建空的书签文件
        local file = io.open(save_file, "w")
        if file then
            file:write("{}")
            file:close()
        else
            vim.notify("无法创建书签文件: " .. save_file, vim.log.levels.ERROR)
            return
        end
    end
    
    -- 读取文件
    local file = io.open(save_file, "r")
    if not file then
        vim.notify("无法读取书签文件: " .. save_file, vim.log.levels.ERROR)
        return
    end
    
    local content = file:read("*all")
    file:close()
    
    if content == "" then
        vim.notify("书签文件为空", vim.log.levels.DEBUG)
        return
    end
    
    -- 解析JSON数据
    local ok, bookmarks_data = pcall(vim.json.decode, content)
    if not ok then
        vim.notify("书签文件格式错误: " .. tostring(bookmarks_data), vim.log.levels.ERROR)
        return
    end
    
    -- 合并加载的书签数据，将字符串键转换回数字
    for file_path, file_bookmarks in pairs(bookmarks_data) do
        local numeric_keyed = {}
        for line_str, value in pairs(file_bookmarks) do
            local line_num = tonumber(line_str)
            if line_num then
                numeric_keyed[line_num] = value
            end
        end
        M.bookmarks[file_path] = numeric_keyed
    end
    
    local count = 0
    for _, file_bookmarks in pairs(M.bookmarks) do
        for _ in pairs(file_bookmarks) do
            count = count + 1
        end
    end
    
    -- 静默加载，不显示成功提示
end

-- 恢复文件的书签标记
local function restore_file_bookmarks(bufnr)
    local file_path = vim.api.nvim_buf_get_name(bufnr)
    if file_path == "" then
        return  -- 无名缓冲区跳过
    end
    
    local file_bookmarks = get_file_bookmarks(file_path)
    if vim.tbl_isempty(file_bookmarks) then
        return  -- 没有书签跳过
    end
    
    -- 确保标记系统已初始化
    setup_signs()
    
    -- 确保符号列始终显示
    ensure_signcolumn()
    
    -- 恢复所有书签标记
    for line, _ in pairs(file_bookmarks) do
        place_sign(bufnr, line)
    end
    
    -- 静默恢复书签，不显示提示
end

-- 自动命令：文件打开和保存时的书签管理
local function setup_autocmds()
    vim.api.nvim_create_augroup("SimpleBookmarks", { clear = true })
    
    -- 当文件被读取时，恢复书签
    vim.api.nvim_create_autocmd({ "BufReadPost", "BufNewFile" }, {
        group = "SimpleBookmarks",
        callback = function(args)
            -- 延迟执行以确保文件完全加载
            vim.defer_fn(function()
                if vim.api.nvim_buf_is_valid(args.buf) then
                    restore_file_bookmarks(args.buf)
                end
            end, 100)
        end,
        desc = "恢复文件书签"
    })
    
    -- 当Neovim退出时，保存所有书签
    vim.api.nvim_create_autocmd("VimLeavePre", {
        group = "SimpleBookmarks",
        callback = function()
            M.save_bookmarks(false) -- 退出时静默保存
        end,
        desc = "退出时保存书签"
    })
    
    -- 定期保存书签（每5分钟）
    local timer = vim.loop.new_timer()
    timer:start(300000, 300000, vim.schedule_wrap(function()
        if not vim.tbl_isempty(M.bookmarks) then
            M.save_bookmarks(false) -- 定期静默保存
        end
    end))
end

-- 初始化模块
function M.setup(opts)
    -- 默认配置
    local default_opts = {
        fixed_signcolumn = true,   -- 固定符号列宽度
        auto_save = true,  -- 自动保存书签
        auto_load = true   -- 自动加载书签
    }
    
    -- 合并用户配置
    if opts then
        M.config = vim.tbl_deep_extend("force", M.config, opts)
        default_opts = vim.tbl_deep_extend("force", default_opts, opts)
    end
    
    -- 设置标记和自动命令
    setup_signs()
    setup_autocmds()
    
    -- 确保符号列始终显示固定宽度
    ensure_signcolumn()
    
    -- 自动加载书签
    if default_opts.auto_load then
        M.load_bookmarks()
    end
    
    -- 静默初始化，不显示提示
end

-- 获取书签统计信息
function M.get_stats()
    local total_bookmarks = 0
    local file_count = 0
    
    for file_path, file_bookmarks in pairs(M.bookmarks) do
        if not vim.tbl_isempty(file_bookmarks) then
            file_count = file_count + 1
            for _, _ in pairs(file_bookmarks) do
                total_bookmarks = total_bookmarks + 1
            end
        end
    end
    
    return {
        total_bookmarks = total_bookmarks,
        file_count = file_count
    }
end


return M