-- ============================================================================
-- 插件列表配置文件 (plugins.lua)
-- 定义要安装和配置的插件列表，使用 Lazy.nvim 插件管理器
-- ============================================================================

-- ============================================================================
-- 1. 基础配置初始化
-- ============================================================================
---@type LazySpec[]  -- 类型注解：Lazy 插件规范数组
local plugins = {}

-- ============================================================================
-- 2. 核心功能插件
-- ============================================================================

-- 2.1 语法高亮和解析插件
table.insert(plugins, {
	"nvim-treesitter/nvim-treesitter",
	build = ":TSUpdate", -- 安装后自动更新 Tree-sitter 解析器
	-- Tree-sitter 提供现代化的语法高亮、代码折叠和文本对象功能
})

-- ============================================================================
-- 3. 扩展插件配置区域
-- ============================================================================
--
-- 插件管理指南：
-- 1. 所有自定义插件都应在此文件中配置，保持配置的集中管理
-- 2. 避免在 chadrc.lua 中直接修改 M.plugins，维护配置的模块化
-- 3. 每个插件配置应包含清晰的功能说明和配置目的
-- 4. 使用 table.insert() 方法添加插件，保持代码的一致性
--
-- 插件配置结构：
-- table.insert(plugins, {
--   "作者/插件名称",           -- 插件的 GitHub 仓库地址
--   config = function() ... end, -- 插件配置函数
--   event = "事件名称",         -- 懒加载触发事件（可选）
--   dependencies = { ... },     -- 插件依赖（可选）
--   opts = { ... },            -- 插件选项（可选）
-- })
-- ============================================================================

-- 3.1 TailwindCSS 补全源插件（基于nvim-cmp）
table.insert(plugins, {
	"roobert/tailwindcss-colorizer-cmp.nvim",
	config = function()
		require("tailwindcss-colorizer-cmp").setup({
			color_square_width = 2,
		})
	end,
	-- 为 TailwindCSS 类名提供颜色预览和补全
})

-- 注意：TailwindCSS LSP 配置已移至 lspconfig.lua 文件中统一管理
-- 这里不再重复配置，避免配置冲突和重复

-- 3.2 HTML标签自动同步插件
table.insert(plugins, {
	"windwp/nvim-ts-autotag",
	config = function()
		require("nvim-ts-autotag").setup({
			-- 启用自动标签同步
			autotag = {
				enable = true,
				-- 支持的文件类型
				filetypes = {
					"html",
					"xml",
					"javascript",
					"typescript",
					"javascriptreact",
					"typescriptreact",
					"svelte",
					"vue",
					"tsx",
					"jsx",
					"rescript",
					"php",
					"markdown",
					"glimmer",
					"handlebars",
					"hbs",
				},
				-- 启用自动重命名标签
				enable_rename = true,
				-- 启用自动关闭标签
				enable_close = true,
				-- 启用自动关闭空标签
				enable_close_on_slash = true,
			},
		})
	end,
	-- 在文件打开时加载
	event = "BufReadPost",
	-- 依赖 Tree-sitter
	dependencies = { "nvim-treesitter/nvim-treesitter" },
	-- 为HTML标签提供自动同步功能，修改开始标签时自动更新结束标签
})

-- 3.3 其他插件可以在这里添加

-- 3.2.1 自动重命名闭合标签插件（补充方案）
table.insert(plugins, {
	"AndrewRadev/tagalong.vim",
	event = { "BufReadPost", "BufNewFile" },
	config = function()
		vim.g.tagalong_filetypes = {
			"html",
			"xml",
			"javascriptreact",
			"typescriptreact",
			"svelte",
			"vue",
		}
	end,
})

-- 3.3 文件模板系统
-- 提供文件模板的创建、管理和生成功能
-- 注意：模板系统作为本地模块直接加载，不需要插件配置

--
-- 示例插件配置：
-- table.insert(plugins, {
--   "插件作者/插件名称",
--   config = function()
--     -- 插件配置代码
--   end,
--   event = "事件名称",  -- 懒加载触发事件
-- })

-- ============================================================================
-- 4. 配置导出
-- ============================================================================
-- 返回插件列表供 Lazy.nvim 使用
return plugins
