-- ============================================================================
-- HTML 实时预览系统 (html-preview.lua)
-- 提供HTML文件的实时预览功能，基于live-server实现
-- ============================================================================

local M = {}

M.config = {
	port = 8080,
	auto_open = true,
	server_pid = nil,
	current_file = nil,
	server_running = false
}

-- 检查live-server是否已安装
local function check_live_server()
	local handle = io.popen('command -v live-server')
	if handle then
		local result = handle:read('*a')
		handle:close()
		return result and result:len() > 0
	end
	return false
end

-- 安装live-server
local function install_live_server()
	print("正在安装 live-server...")
	local cmd = 'npm install -g live-server'
	local handle = io.popen(cmd)
	if handle then
		local result = handle:read('*a')
		handle:close()
	end
	return check_live_server()
end

-- 停止当前运行的live-server进程
local function kill_server()
	if M.config.server_pid then
		local kill_cmd = string.format('kill %d 2>/dev/null || true', M.config.server_pid)
		os.execute(kill_cmd)
		M.config.server_pid = nil
	end
	-- 确保没有遗留的live-server进程
	os.execute('pkill -f "live-server" 2>/dev/null || true')
end

function M.preview_html()
	-- 检查是否是HTML文件
	local current_file = vim.fn.expand('%:p')
	if not current_file:match('%.html$') then
		print("不是HTML文件")
		return
	end

	-- 确保live-server已安装
	if not check_live_server() then
		if not install_live_server() then
			print("live-server安装失败，请确保已安装npm并重试")
			return
		end
	end

	-- 停止可能运行的服务器
	kill_server()

	-- 获取HTML文件所在目录
	local html_dir = vim.fn.fnamemodify(current_file, ':h')
	local html_file = vim.fn.fnamemodify(current_file, ':t')

	-- 启动live-server
	local cmd = string.format(
		'cd "%s" && live-server --port=%d --no-browser --host=localhost --quiet &',
		html_dir,
		M.config.port
	)

	-- 在后台运行live-server
	print("正在启动预览服务器...")

	-- 先确保没有其他live-server实例在运行
	os.execute('pkill -f "live-server" 2>/dev/null || true')

	-- 启动新的live-server实例
	local result = os.execute(cmd)
	if result == 0 or result == true then
		M.config.server_running = true
		M.config.current_file = current_file

		-- 等待服务器启动
		vim.defer_fn(function()
			-- 在指定的Google Chrome Beta中打开HTML文件
			local url = string.format('http://localhost:%d/%s', M.config.port, html_file)
			if vim.fn.has('mac') == 1 then
				vim.fn.system('open -a "/Applications/Google Chrome Beta.app" "' .. url .. '"')
			elseif vim.fn.has('unix') == 1 then
				vim.fn.system('xdg-open "' .. url .. '"')
			elseif vim.fn.has('win32') == 1 then
				vim.fn.system('start "" "' .. url .. '"')
			end
			print(string.format("预览服务器已启动：%s", url))

			-- 刷新Neovim界面
			vim.schedule(function()
				-- 刷新nvim-tree如果存在（添加错误处理）
				if package.loaded['nvim-tree.api'] then
					pcall(function()
						require('nvim-tree.api').tree.reload()
					end)
				end

				-- 刷新当前缓冲区
				vim.cmd('checktime')
				vim.cmd('redraw!')
			end)
		end, 2000)
	else
		print("启动预览服务器失败")
	end
end

function M.stop_preview()
	kill_server()
	M.config.server_running = false
	M.config.current_file = nil
	print("预览服务器已停止")
end

function M.restart_preview()
	local current_file = M.config.current_file
	M.stop_preview()
	if current_file then
		vim.defer_fn(function()
			M.preview_html()
		end, 500)
	end
end

function M.get_status()
	return {
		running = M.config.server_running,
		port = M.config.port,
		file = M.config.current_file,
		url = M.config.server_running and string.format("http://localhost:%d", M.config.port) or nil
	}
end

-- 设置自动关闭服务器
local function setup_auto_close()
	-- 文件关闭时自动关闭服务器
	vim.api.nvim_create_autocmd({ "BufDelete", "BufWipeout" }, {
		pattern = "*.html",
		callback = function()
			if M.config.server_running then
				M.stop_preview()
			end
		end,
		desc = "HTML文件关闭时自动停止预览服务器"
	})

	-- Neovim退出时自动关闭服务器
	vim.api.nvim_create_autocmd("VimLeavePre", {
		callback = function()
			if M.config.server_running then
				M.stop_preview()
			end
		end,
		desc = "Neovim退出时自动停止预览服务器"
	})
end

function M.setup(opts)
	M.config = vim.tbl_deep_extend("force", M.config, opts or {})
	setup_auto_close()
end

return M
