// Package {{PACKAGE_NAME}} {{DESCRIPTION}}
//
// Author: {{AUTHOR}} <{{EMAIL}}>
// Created: {{DATE}}
// Version: 1.0.0
package {{PACKAGE_NAME}}

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"log"
	"net/http"
	"os"
	"os/signal"
	"sync"
	"syscall"
	"time"
)

// 常量定义
const (
	// DefaultTimeout 默认超时时间
	DefaultTimeout = 30 * time.Second
	// DefaultPort 默认端口
	DefaultPort = 8080
	// MaxRetries 最大重试次数
	MaxRetries = 3
)

// 错误定义
var (
	ErrInvalidInput    = errors.New("invalid input")
	ErrNotFound        = errors.New("not found")
	ErrAlreadyExists   = errors.New("already exists")
	ErrInternalError   = errors.New("internal error")
	ErrTimeout         = errors.New("timeout")
	ErrUnauthorized    = errors.New("unauthorized")
)

// Config 配置结构体
type Config struct {
	Port     int           `json:"port" yaml:"port"`
	Host     string        `json:"host" yaml:"host"`
	Timeout  time.Duration `json:"timeout" yaml:"timeout"`
	Debug    bool          `json:"debug" yaml:"debug"`
	Database DatabaseConfig `json:"database" yaml:"database"`
	Redis    RedisConfig    `json:"redis" yaml:"redis"`
}

// DatabaseConfig 数据库配置
type DatabaseConfig struct {
	Host     string `json:"host" yaml:"host"`
	Port     int    `json:"port" yaml:"port"`
	User     string `json:"user" yaml:"user"`
	Password string `json:"password" yaml:"password"`
	DBName   string `json:"dbname" yaml:"dbname"`
	SSLMode  string `json:"sslmode" yaml:"sslmode"`
}

// RedisConfig Redis配置
type RedisConfig struct {
	Host     string `json:"host" yaml:"host"`
	Port     int    `json:"port" yaml:"port"`
	Password string `json:"password" yaml:"password"`
	DB       int    `json:"db" yaml:"db"`
}

// User 用户结构体
type User struct {
	ID        int64     `json:"id" db:"id"`
	Username  string    `json:"username" db:"username"`
	Email     string    `json:"email" db:"email"`
	Password  string    `json:"-" db:"password"` // 不在JSON中显示
	CreatedAt time.Time `json:"created_at" db:"created_at"`
	UpdatedAt time.Time `json:"updated_at" db:"updated_at"`
}

// Validate 验证用户数据
func (u *User) Validate() error {
	if u.Username == "" {
		return fmt.Errorf("username is required")
	}
	if u.Email == "" {
		return fmt.Errorf("email is required")
	}
	if len(u.Password) < 6 {
		return fmt.Errorf("password must be at least 6 characters")
	}
	return nil
}

// String 实现Stringer接口
func (u *User) String() string {
	return fmt.Sprintf("User{ID: %d, Username: %s, Email: %s}", u.ID, u.Username, u.Email)
}

// Response API响应结构体
type Response struct {
	Success bool        `json:"success"`
	Data    interface{} `json:"data,omitempty"`
	Message string      `json:"message,omitempty"`
	Error   string      `json:"error,omitempty"`
}

// NewSuccessResponse 创建成功响应
func NewSuccessResponse(data interface{}, message string) *Response {
	return &Response{
		Success: true,
		Data:    data,
		Message: message,
	}
}

// NewErrorResponse 创建错误响应
func NewErrorResponse(err error, message string) *Response {
	return &Response{
		Success: false,
		Error:   err.Error(),
		Message: message,
	}
}

// Logger 日志接口
type Logger interface {
	Debug(msg string, fields ...interface{})
	Info(msg string, fields ...interface{})
	Warn(msg string, fields ...interface{})
	Error(msg string, fields ...interface{})
}

// SimpleLogger 简单日志实现
type SimpleLogger struct {
	debug bool
}

// NewSimpleLogger 创建简单日志器
func NewSimpleLogger(debug bool) *SimpleLogger {
	return &SimpleLogger{debug: debug}
}

// Debug 调试日志
func (l *SimpleLogger) Debug(msg string, fields ...interface{}) {
	if l.debug {
		log.Printf("[DEBUG] "+msg, fields...)
	}
}

// Info 信息日志
func (l *SimpleLogger) Info(msg string, fields ...interface{}) {
	log.Printf("[INFO] "+msg, fields...)
}

// Warn 警告日志
func (l *SimpleLogger) Warn(msg string, fields ...interface{}) {
	log.Printf("[WARN] "+msg, fields...)
}

// Error 错误日志
func (l *SimpleLogger) Error(msg string, fields ...interface{}) {
	log.Printf("[ERROR] "+msg, fields...)
}

// Service 服务接口
type Service interface {
	GetUser(ctx context.Context, id int64) (*User, error)
	CreateUser(ctx context.Context, user *User) error
	UpdateUser(ctx context.Context, user *User) error
	DeleteUser(ctx context.Context, id int64) error
	ListUsers(ctx context.Context, limit, offset int) ([]*User, error)
}

// UserService 用户服务实现
type UserService struct {
	logger Logger
	users  map[int64]*User // 简单的内存存储
	mu     sync.RWMutex
	nextID int64
}

// NewUserService 创建用户服务
func NewUserService(logger Logger) *UserService {
	return &UserService{
		logger: logger,
		users:  make(map[int64]*User),
		nextID: 1,
	}
}

// GetUser 获取用户
func (s *UserService) GetUser(ctx context.Context, id int64) (*User, error) {
	s.mu.RLock()
	defer s.mu.RUnlock()

	user, exists := s.users[id]
	if !exists {
		s.logger.Warn("User not found", "id", id)
		return nil, ErrNotFound
	}

	s.logger.Debug("User retrieved", "id", id, "username", user.Username)
	return user, nil
}

// CreateUser 创建用户
func (s *UserService) CreateUser(ctx context.Context, user *User) error {
	if err := user.Validate(); err != nil {
		s.logger.Error("User validation failed", "error", err)
		return fmt.Errorf("validation failed: %w", err)
	}

	s.mu.Lock()
	defer s.mu.Unlock()

	// 检查用户名是否已存在
	for _, existingUser := range s.users {
		if existingUser.Username == user.Username {
			s.logger.Warn("Username already exists", "username", user.Username)
			return ErrAlreadyExists
		}
	}

	user.ID = s.nextID
	s.nextID++
	user.CreatedAt = time.Now()
	user.UpdatedAt = time.Now()

	s.users[user.ID] = user
	s.logger.Info("User created", "id", user.ID, "username", user.Username)
	return nil
}

// UpdateUser 更新用户
func (s *UserService) UpdateUser(ctx context.Context, user *User) error {
	if err := user.Validate(); err != nil {
		return fmt.Errorf("validation failed: %w", err)
	}

	s.mu.Lock()
	defer s.mu.Unlock()

	if _, exists := s.users[user.ID]; !exists {
		s.logger.Warn("User not found for update", "id", user.ID)
		return ErrNotFound
	}

	user.UpdatedAt = time.Now()
	s.users[user.ID] = user
	s.logger.Info("User updated", "id", user.ID, "username", user.Username)
	return nil
}

// DeleteUser 删除用户
func (s *UserService) DeleteUser(ctx context.Context, id int64) error {
	s.mu.Lock()
	defer s.mu.Unlock()

	if _, exists := s.users[id]; !exists {
		s.logger.Warn("User not found for deletion", "id", id)
		return ErrNotFound
	}

	delete(s.users, id)
	s.logger.Info("User deleted", "id", id)
	return nil
}

// ListUsers 列出用户
func (s *UserService) ListUsers(ctx context.Context, limit, offset int) ([]*User, error) {
	s.mu.RLock()
	defer s.mu.RUnlock()

	users := make([]*User, 0, len(s.users))
	for _, user := range s.users {
		users = append(users, user)
	}

	// 简单的分页逻辑
	start := offset
	if start > len(users) {
		start = len(users)
	}

	end := start + limit
	if end > len(users) {
		end = len(users)
	}

	s.logger.Debug("Users listed", "total", len(users), "returned", end-start)
	return users[start:end], nil
}

// HTTPHandler HTTP处理器
type HTTPHandler struct {
	service Service
	logger  Logger
}

// NewHTTPHandler 创建HTTP处理器
func NewHTTPHandler(service Service, logger Logger) *HTTPHandler {
	return &HTTPHandler{
		service: service,
		logger:  logger,
	}
}

// GetUser 获取用户HTTP处理器
func (h *HTTPHandler) GetUser(w http.ResponseWriter, r *http.Request) {
	// 这里应该从URL路径中提取用户ID
	// 为了简化，我们假设ID为1
	var userID int64 = 1

	user, err := h.service.GetUser(r.Context(), userID)
	if err != nil {
		if errors.Is(err, ErrNotFound) {
			h.writeErrorResponse(w, http.StatusNotFound, err, "User not found")
		} else {
			h.writeErrorResponse(w, http.StatusInternalServerError, err, "Failed to get user")
		}
		return
	}

	h.writeSuccessResponse(w, http.StatusOK, user, "User retrieved successfully")
}

// CreateUser 创建用户HTTP处理器
func (h *HTTPHandler) CreateUser(w http.ResponseWriter, r *http.Request) {
	var user User
	if err := json.NewDecoder(r.Body).Decode(&user); err != nil {
		h.writeErrorResponse(w, http.StatusBadRequest, err, "Invalid JSON")
		return
	}

	if err := h.service.CreateUser(r.Context(), &user); err != nil {
		if errors.Is(err, ErrAlreadyExists) {
			h.writeErrorResponse(w, http.StatusConflict, err, "User already exists")
		} else {
			h.writeErrorResponse(w, http.StatusInternalServerError, err, "Failed to create user")
		}
		return
	}

	h.writeSuccessResponse(w, http.StatusCreated, &user, "User created successfully")
}

// writeSuccessResponse 写入成功响应
func (h *HTTPHandler) writeSuccessResponse(w http.ResponseWriter, status int, data interface{}, message string) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)

	response := NewSuccessResponse(data, message)
	if err := json.NewEncoder(w).Encode(response); err != nil {
		h.logger.Error("Failed to encode response", "error", err)
	}
}

// writeErrorResponse 写入错误响应
func (h *HTTPHandler) writeErrorResponse(w http.ResponseWriter, status int, err error, message string) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)

	response := NewErrorResponse(err, message)
	if encodeErr := json.NewEncoder(w).Encode(response); encodeErr != nil {
		h.logger.Error("Failed to encode error response", "error", encodeErr)
	}
}

// Server HTTP服务器
type Server struct {
	config  *Config
	logger  Logger
	service Service
	handler *HTTPHandler
	server  *http.Server
}

// NewServer 创建服务器
func NewServer(config *Config, logger Logger) *Server {
	service := NewUserService(logger)
	handler := NewHTTPHandler(service, logger)

	mux := http.NewServeMux()
	mux.HandleFunc("/users", func(w http.ResponseWriter, r *http.Request) {
		switch r.Method {
		case http.MethodGet:
			handler.GetUser(w, r)
		case http.MethodPost:
			handler.CreateUser(w, r)
		default:
			http.Error(w, "Method not allowed", http.StatusMethodNotAllowed)
		}
	})

	server := &http.Server{
		Addr:         fmt.Sprintf("%s:%d", config.Host, config.Port),
		Handler:      mux,
		ReadTimeout:  config.Timeout,
		WriteTimeout: config.Timeout,
	}

	return &Server{
		config:  config,
		logger:  logger,
		service: service,
		handler: handler,
		server:  server,
	}
}

// Start 启动服务器
func (s *Server) Start(ctx context.Context) error {
	s.logger.Info("Starting server", "address", s.server.Addr)

	// 在goroutine中启动服务器
	go func() {
		if err := s.server.ListenAndServe(); err != nil && !errors.Is(err, http.ErrServerClosed) {
			s.logger.Error("Server failed to start", "error", err)
		}
	}()

	// 等待上下文取消或中断信号
	<-ctx.Done()

	// 优雅关闭
	s.logger.Info("Shutting down server...")
	shutdownCtx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()

	return s.server.Shutdown(shutdownCtx)
}

// LoadConfig 加载配置
func LoadConfig() *Config {
	return &Config{
		Port:    DefaultPort,
		Host:    "localhost",
		Timeout: DefaultTimeout,
		Debug:   true,
		Database: DatabaseConfig{
			Host:    "localhost",
			Port:    5432,
			User:    "postgres",
			DBName:  "{{PACKAGE_NAME}}_db",
			SSLMode: "disable",
		},
		Redis: RedisConfig{
			Host: "localhost",
			Port: 6379,
			DB:   0,
		},
	}
}

// main 主函数（如果这是主包）
func main() {
	// 加载配置
	config := LoadConfig()

	// 创建日志器
	logger := NewSimpleLogger(config.Debug)

	// 创建服务器
	server := NewServer(config, logger)

	// 创建上下文，监听中断信号
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	// 监听系统信号
	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM)

	go func() {
		sig := <-sigChan
		logger.Info("Received signal", "signal", sig)
		cancel()
	}()

	// 启动服务器
	if err := server.Start(ctx); err != nil {
		logger.Error("Server error", "error", err)
		os.Exit(1)
	}

	logger.Info("Server stopped gracefully")
}