-- ============================================================================
-- 插件配置文件 (plugins/init.lua)
-- 定义和配置 Neovim 插件，使用 Lazy.nvim 插件管理器
-- 这里配置的插件会覆盖或扩展 NvChad 的默认插件设置
-- ============================================================================

return {
	-- ============================================================================
	-- 代码格式化插件 - 覆盖默认插件配置
	-- ============================================================================
	{
		"stevearc/conform.nvim",
		event = "BufWritePre", -- 保存文件时触发格式化
		-- 说明：采用"Prettier + html_noblank"的格式化链路；保留自动格式化，同时移除 <head>/<body> 周围多余空行
		-- 详情：.prettierrc.js 负责通用风格；html_noblank 仅做轻量后处理（定义见 lua/configs/conform.lua）
		-- 使用外部配置文件，包含统一的缩进设置
		opts = require "configs.conform",
		-- 快捷键配置
		keys = {
			{
				"<leader>fm", -- 手动格式化快捷键
				function()
					require("conform").format({ async = true, lsp_fallback = true })
				end,
				desc = "Format file", -- 快捷键描述
				mode = { "n", "v" }, -- 支持普通模式和可视模式
			},
		},
		-- 配置函数：设置 conform.nvim
		config = function(_, opts)
			require("conform").setup(opts)
			
			-- 简化的保存时格式化：使用 conform 的内置 format_on_save 功能
			require("conform").setup(vim.tbl_extend("force", opts, {
				format_on_save = function(bufnr)
					-- 获取文件类型
					local filetype = vim.api.nvim_buf_get_option(bufnr, "filetype")
					
					-- 跳过HTML文件，避免与 nvim-ts-autotag 冲突
					if vim.tbl_contains({ "html", "htm" }, filetype) then
						return nil
					end
					
					-- 为支持的文件类型启用格式化
					local supported_types = { 
						"python", "javascript", "typescript", "javascriptreact", "typescriptreact",
						"lua", "css", "scss", "json", "yaml", "go", "vue", "eruby"
					}
					
					if vim.tbl_contains(supported_types, filetype) then
						-- 对于Python文件，先检查语法错误
						if filetype == "python" then
							-- 检查是否有明显的语法错误
							local lines = vim.api.nvim_buf_get_lines(bufnr, 0, -1, false)
							local content = table.concat(lines, "\n")
							
							-- 简单的语法检查：未闭合的括号、引号等
							local open_parens = 0
							local open_brackets = 0
							local open_braces = 0
							local in_string = false
							local string_char = nil
							
							for i = 1, #content do
								local char = content:sub(i, i)
								local prev_char = i > 1 and content:sub(i-1, i-1) or ""
								
								-- 处理字符串
								if (char == '"' or char == "'") and prev_char ~= "\\" then
									if not in_string then
										in_string = true
										string_char = char
									elseif char == string_char then
										in_string = false
										string_char = nil
									end
								elseif not in_string then
									-- 计算括号
									if char == "(" then
										open_parens = open_parens + 1
									elseif char == ")" then
										open_parens = open_parens - 1
									elseif char == "[" then
										open_brackets = open_brackets + 1
									elseif char == "]" then
										open_brackets = open_brackets - 1
									elseif char == "{" then
										open_braces = open_braces + 1
									elseif char == "}" then
										open_braces = open_braces - 1
									end
								end
							end
							
							-- 如果有未闭合的括号或字符串，跳过格式化
							if open_parens ~= 0 or open_brackets ~= 0 or open_braces ~= 0 or in_string then
								return nil  -- 跳过格式化，避免错误
							end
						end
						
						return {
							timeout_ms = 3000,
							lsp_fallback = true,
							quiet = true, -- 减少错误提示噪音
						}
					end
					
					return nil
				end,
			}))
		end,
	},

	-- ============================================================================
	-- 修复CSS文件图标显示问题 + Docker文件图标配置
	-- ============================================================================
	{
		"nvim-tree/nvim-web-devicons",
		opts = function()
			return {
				override = {
					css = { icon = "󰌜", color = "#1572b6", name = "CSS" }
				},
				override_by_filename = {
					-- Docker相关文件（精确匹配常见文件名）
					["dockerfile"] = { icon = "󰡨", color = "#2496ed", name = "Dockerfile" },
					["Dockerfile"] = { icon = "󰡨", color = "#2496ed", name = "Dockerfile" },
					["docker-compose.yml"] = { icon = "󰡨", color = "#2496ed", name = "DockerCompose" },
					["docker-compose.yaml"] = { icon = "󰡨", color = "#2496ed", name = "DockerCompose" },
					["Dockerfile.dev"] = { icon = "󰡨", color = "#2496ed", name = "Dockerfile" },
					["Dockerfile.prod"] = { icon = "󰡨", color = "#2496ed", name = "Dockerfile" },
					["Dockerfile.test"] = { icon = "󰡨", color = "#2496ed", name = "Dockerfile" },
				},
				override_by_extension = {
					-- 通过扩展名匹配（用于 .dockerfile 扩展名）
					["dockerfile"] = { icon = "󰡨", color = "#2496ed", name = "Dockerfile" },
				}
			}
		end,
		config = function(_, opts)
			local devicons = require("nvim-web-devicons")
			devicons.setup(opts)
			
			-- 自定义函数：为包含"docker"的文件名设置图标（不区分大小写）
			local original_get_icon = devicons.get_icon
			devicons.get_icon = function(name, extension, opts_inner)
				-- 首先尝试原始函数
				local icon, hl = original_get_icon(name, extension, opts_inner)
				
				-- 如果文件名包含"docker"（不区分大小写），使用Docker图标
				if name and string.lower(name):match("docker") then
					return "󰡨", "DevIconDockerfile"
				end
				
				return icon, hl
			end
			
			-- 设置Docker图标的高亮组
			vim.api.nvim_set_hl(0, "DevIconDockerfile", { fg = "#2496ed" })
		end
	},

	-- ============================================================================
	-- Tree-sitter 配置（确保安装 HTML 解析器，并启用 autotag 模块）
	-- ============================================================================
	{
		"nvim-treesitter/nvim-treesitter",
		-- 让 NvChad 自带的 treesitter 配置基础上，强制确保安装并启用 autotag
		config = function(_, _)
			local ok, configs = pcall(require, "nvim-treesitter.configs")
			if not ok then return end
			configs.setup({
				highlight = { enable = true },
				incremental_selection = { enable = true },
				auto_install = true,
				ensure_installed = {
					"html", "xml", "javascript", "typescript", "tsx",
					"css", "scss", "json", "lua", "vue", "svelte",
					"php", "markdown", "markdown_inline"
				},
			})
		end,
	},

	-- ============================================================================
	-- 自动同步/重命名 HTML 标签（依赖 Tree-sitter）
	-- ============================================================================
	{
		"windwp/nvim-ts-autotag",
		event = { "BufReadPost", "BufNewFile" },
		dependencies = { "nvim-treesitter/nvim-treesitter" },
		config = function()
			require("nvim-ts-autotag").setup({
				autotag = {
					enable = true,
					filetypes = {
						"html", "xml", "javascript", "typescript",
						"javascriptreact", "typescriptreact", "svelte",
						"vue", "tsx", "jsx", "rescript", "php",
						"markdown", "glimmer", "handlebars", "hbs",
					},
					enable_rename = true,
					enable_close = true,
					enable_close_on_slash = true,
				},
			})
		end,
	},

	-- ============================================================================
	-- LSP (Language Server Protocol) 配置
	-- ============================================================================
	{
		"neovim/nvim-lspconfig",
		config = function()
			require("configs.lspconfig") -- 加载 LSP 配置文件
		end,
		-- nvim-lspconfig 提供各种编程语言的 LSP 客户端配置
		-- 支持代码补全、错误检查、跳转定义等功能
	},

	-- ============================================================================
	-- 快速移动插件
	-- ============================================================================
	{
		"easymotion/vim-easymotion",
		lazy = false, -- 立即加载插件
		config = function()
			-- EasyMotion 核心配置：确保选择目标后直接跳转
			vim.g.EasyMotion_do_mapping = 0  -- 禁用默认映射
			vim.g.EasyMotion_smartcase = 1   -- 智能大小写匹配
			vim.g.EasyMotion_use_smartsign = 1 -- 智能符号匹配
			
			-- 关键配置：这些设置确保不需要按 Enter 确认
			vim.g.EasyMotion_enter_jump_first = 1  -- Enter 键直接跳转到第一个匹配
			vim.g.EasyMotion_space_jump_first = 1  -- 空格键直接跳转到第一个匹配
			
			-- 提示字符设置（按手指舒适度排序）
			-- 优先使用食指和中指位置的字符，然后是其他容易按到的字符
			vim.g.EasyMotion_keys = 'fjdkslahgqwertyuiopzxcvbnm'
			
			-- 启用提示信息，让用户知道当前状态
			vim.g.EasyMotion_prompt = 'EasyMotion: 输入一个字符进行搜索 > '
			vim.g.EasyMotion_confirm_key = ''
			
			-- 设置更短的超时时间，减少等待
			vim.g.EasyMotion_timeout = 3000
			
			-- 跳转后立即清除高亮
			vim.g.EasyMotion_off_screen_search = 0
			
			-- 完全禁用所有可能的高亮效果
			vim.g.EasyMotion_cursor_highlight = 0
			vim.g.EasyMotion_disable_two_key_combo = 1
			
			-- 视觉效果配置：完全简化，去除所有多余的高亮效果
			vim.g.EasyMotion_inc_highlight = 0  -- 禁用增量高亮（去除蓝色效果）
			vim.g.EasyMotion_move_highlight = 0 -- 禁用移动高亮（去除跳转后的红色残留）
			vim.g.EasyMotion_landing_highlight = 0 -- 禁用着陆高亮（去除跳转后的效果）
			
			-- 设置高亮颜色（在插件加载后应用）
			vim.defer_fn(function()
				-- EasyMotion 目标字符高亮：使用红色粗体，跳转后自动清除
				vim.cmd("highlight EasyMotionTarget guifg=#ff0000 gui=bold")
				-- EasyMotion 阴影字符高亮：使用灰色，降低干扰
				vim.cmd("highlight EasyMotionShade guifg=#666666")
				-- 清除所有可能的残留高亮和蓝色效果
				vim.cmd("highlight EasyMotionMoveHL NONE")
				vim.cmd("highlight EasyMotionIncSearch NONE")
				vim.cmd("highlight EasyMotionIncCursor NONE")
				vim.cmd("highlight EasyMotionTarget2First NONE")
				vim.cmd("highlight EasyMotionTarget2Second NONE")
				-- 清除可能的搜索高亮
				vim.cmd("highlight Search NONE")
				vim.cmd("highlight IncSearch NONE")
				-- 清除所有可能的 EasyMotion 相关高亮
				vim.cmd("highlight link EasyMotionIncSearch NONE")
				vim.cmd("highlight link EasyMotionIncCursor NONE")
			end, 100)
			
			-- 添加自动命令来清除 EasyMotion 后的搜索高亮
			vim.api.nvim_create_autocmd("User", {
				pattern = "EasyMotionPromptEnd",
				callback = function()
					vim.cmd("nohlsearch")
				end,
				desc = "Clear search highlight after EasyMotion"
			})
		end,
	},

	-- ============================================================================
	-- 已注释的插件（可根据需要启用）
	-- ============================================================================

	-- 文件树插件（已注释，因为 NvChad 默认使用 nvim-tree）
	-- 如果你坚持使用 NERDTree 而不是 nvim-tree
	-- {
	--   "scrooloose/nerdtree",
	--   cmd = { "NERDTreeToggle", "NERDTree", "NERDTreeFind" },
	--   config = function()
	--     vim.g.NERDTreeShowHidden = 1    -- 显示隐藏文件
	--     vim.g.NERDTreeMinimalUI = 1     -- 使用简洁界面
	--   end,
	-- },

	-- 状态栏插件（已注释）
	-- 注意：NvChad 已经内置了更好的状态栏 (statusline)
	-- 不建议使用 vim-airline，但如果你坚持：
	-- {
	--   "vim-airline/vim-airline",
	--   lazy = false,
	--   config = function()
	--     vim.g['airline#extensions#tabline#enabled'] = 1      -- 启用标签栏
	--     vim.g['airline#extensions#tabline#left_sep'] = ' '   -- 左分隔符
	--     vim.g['airline#extensions#tabline#left_alt_sep'] = '|' -- 左替代分隔符
	--     vim.g['airline#extensions#tabline#formatter'] = 'default' -- 格式化器
	--   end,
	-- },
	-- {
	--   "vim-airline/vim-airline-themes",
	--   lazy = false,
	--   config = function()
	--     vim.g.airline_theme = 'solarized' -- 设置主题
	--   end,
	-- },

	-- ============================================================================
	-- 缩进线显示插件
	-- ============================================================================
	-- 使用 NvChad 自带的 indent-blankline.nvim 插件
	{
		"lukas-reineke/indent-blankline.nvim",
		main = "ibl",
		lazy = false,
		event = { "BufReadPost", "BufNewFile" },
		config = function()
			local ok, ibl = pcall(require, "ibl")
			if not ok then
				vim.notify("indent-blankline.nvim not found", vim.log.levels.ERROR)
				return
			end
			
			-- 先设置默认的缩进线颜色
			vim.api.nvim_set_hl(0, "IblIndent", { fg = "#888888" })
			
			-- 配置缩进参考线，使用默认设置
			ibl.setup({
				indent = {
					char = "¦",  -- 使用虚线字符
					highlight = "IblIndent",
				},
				scope = {
					enabled = false,  -- 禁用scope功能
				},
				exclude = {
					filetypes = { "help", "alpha", "dashboard", "neo-tree", "Trouble", "lazy", "mason" },
				},
			})
			
		end,
	},

	-- ============================================================================
	-- Which-Key 按键提示配置
	-- ============================================================================
	{
		"folke/which-key.nvim",
		event = "VeryLazy",
		config = function()
			local wk = require("which-key")
			wk.setup({
				-- 使用新版本 API 配置
				preset = "modern",
				delay = 500,
				expand = 1,
				notify = true,
				plugins = {
					marks = true,
					registers = true,
					spelling = {
						enabled = true,
						suggestions = 20,
					},
				},
				win = {
					border = "rounded",
					padding = { 1, 2 },
					title = true,
					title_pos = "center",
					zindex = 1000,
				},
				layout = {
					width = { min = 20, max = 50 },
					spacing = 3,
				},
			})

			-- 使用新版本 API 注册按键组
			wk.add({
				{ "<leader>g", group = "🌳 Git 操作" },
				{ "<leader>gs", desc = "暂存当前 hunk" },
				{ "<leader>gr", desc = "重置当前 hunk" },
				{ "<leader>gS", desc = "暂存整个文件" },
				{ "<leader>gR", desc = "重置整个文件" },
				{ "<leader>gu", desc = "撤销暂存 hunk" },
				{ "<leader>gp", desc = "预览 hunk" },
				{ "<leader>gb", desc = "查看行 blame 信息" },
				{ "<leader>gB", desc = "切换当前行 blame 显示" },
				{ "<leader>gd", desc = "查看文件 diff" },
				{ "<leader>gD", desc = "查看与上次提交的 diff" },
				{ "<leader>gt", desc = "切换显示删除的行" },
				{ "<leader>gw", desc = "切换单词级 diff" },
				{ "<leader>r", group = "🔧 重构和运行" },
				{ "<leader>rn", desc = "🔢 切换相对行号" },
				{ "<leader>ra", desc = "🏷️ 重命名符号" },
				{ "<leader>rr", desc = "🚀 运行代码" },
				{ "<leader>re", desc = "📝 编辑运行配置" },
				{ "<leader>rc", desc = "📄 创建运行配置" },
				{ "<leader>rd", desc = "⚙️ 编辑默认配置" },
				{ "<leader>t", group = "🛠️ 模板和终端" },
				{ "<leader>tn", desc = "使用模板创建新文件" },
				{ "<leader>ta", desc = "为当前文件应用模板" },
				{ "<leader>tl", desc = "列出所有可用模板" },
				{ "<leader>th", desc = "主题切换" },
				{ "<leader>tx", desc = "新建水平终端" },
				{ "<leader>ty", desc = "新建垂直终端" },
				{ "<leader>s", group = "📜 未命名" },
				{ "<leader>f", group = "🔍 搜索和查找" },
				{ "<leader>fh", desc = "搜索帮助" },
				{ "<leader>ff", desc = "搜索文件" },
				{ "<leader>fg", desc = "全局搜索" },
				{ "<leader>fb", desc = "搜索缓冲区" },
				{ "<leader>fo", desc = "搜索最近文件" },
				{ "<leader>fm", desc = "搜索标记" },
				{ "<leader>fz", desc = "当前缓冲区模糊搜索" },
				{ "<leader>fc", desc = "搜索Git提交" },
				{ "<leader>fs", desc = "搜索Git状态" },
				{ "<leader>fa", desc = "搜索所有文件" },
				{ "<leader>ft", desc = "选择隐藏终端" },
				{ "<leader>fr", desc = "搜索引用" },
				{ "<leader>fd", desc = "搜索定义" },
				{ "<leader>fi", desc = "搜索实现" },
				{ "<leader>fy", desc = "搜索类型定义" },
				{ "<leader>fw", desc = "搜索工作区符号" },
				{ "<leader>fl", desc = "搜索文档符号" },
				{ "<leader>p", group = "🌐 HTML预览" },
				{ "<leader>ps", desc = "启动预览服务器" },
				{ "<leader>pq", desc = "停止预览服务器" },
				{ "<leader>pr", desc = "重启预览服务器" },
				{ "<leader>pi", desc = "查看服务器状态" },
				{ "<leader>w", group = "🖼️ 窗口管理" },
				{ "<leader>wK", hidden = true },
				{ "<leader>wq", desc = "🔍 查询按键映射" },
				{ "<leader>wh", desc = "⬅️ 切换到左侧窗口" },
				{ "<leader>wj", desc = "⬇️ 切换到下方窗口" },
				{ "<leader>wk", desc = "⬆️ 切换到上方窗口" },
				{ "<leader>l", group = "📝 代码片段管理" },
				{ "<leader>ll", desc = "📋 列出当前文件片段" },
				{ "<leader>lr", desc = "🔄 重新加载片段配置" },
				{ "<leader>le", desc = "✏️ 编辑片段配置文件" },
				{ "<leader>lg", desc = "📖 查看自定义指南" },
				{ "<leader>lt", desc = "🧪 打开测试文件" },
				{ "<leader>lh", desc = "❓ 显示片段帮助" },
				{ "<leader>ln", desc = "➕ 创建新片段模板" },
				{ "<leader>m", group = "🔖 书签管理" },
				{ "<leader>ml", desc = "📋 列出所有书签" },
				{ "<leader>mc", desc = "🗑️ 清除当前文件书签" },
				{ "<leader>mC", desc = "🗑️ 清除所有书签" },
				{ "<leader>ms", desc = "📊 书签统计" },
				{ "<leader>wl", desc = "➡️ 切换到右侧窗口" },
				{ "<leader>ws", desc = "➖ 水平分割窗口" },
				{ "<leader>wv", desc = "➕ 垂直分割窗口" },
				{ "<leader>wc", desc = "❌ 关闭当前窗口" },
				{ "<leader>wo", desc = "🗑️ 关闭其他窗口" },
				{ "<leader>w=", desc = "⚖️ 智能平衡窗口" },
				{ "<leader>w+", desc = "📈 增加窗口高度" },
				{ "<leader>w-", desc = "📉 减少窗口高度" },
				{ "<leader>w>", desc = "↗️ 增加窗口宽度" },
				{ "<leader>w<", desc = "↙️ 减少窗口宽度" },
				{ "<leader>wx", desc = "🔄 交换窗口位置" },
				{ "<leader>wr", desc = "🔃 向右旋转窗口" },
				{ "<leader>wR", desc = "🔄 向左旋转窗口" },
				{ "<leader>ww", desc = "🔄 切换到下一个窗口" },
				{ "<leader>wp", desc = "⏮️ 切换到上一个窗口" },
				{ "<leader>w[", desc = "◀️ 大幅减少宽度" },
				{ "<leader>w]", desc = "▶️ 大幅增加宽度" },
				{ "<leader>w{", desc = "🔻 大幅减少高度" },
				{ "<leader>w}", desc = "🔺 大幅增加高度" },
				{ "<leader>c", group = "📖 快捷键帮助" },
				{ "<leader>ch", desc = "📋 显示快捷键列表" },
				{ "<leader>ca", desc = "⚡ 代码操作" },
				{ "<A-h>", desc = "💻 切换水平终端" },
				{ "<A-v>", desc = "💻 切换垂直终端" },
				{ "<A-i>", desc = "💻 切换浮动终端" },
				{ "<leader><leader>", hidden = true },
				{ "<leader>v", hidden = true },
				{ "<leader>h", hidden = true },
				{ "<leader>n", hidden = true }, -- 隐藏 toggle line number
				{ "<leader>/", hidden = true }, -- 隐藏 toggle comment
				{ "<leader>e", hidden = true }, -- 隐藏 Focus file tree
			})
		end,
	},

	-- ============================================================================
	-- 旧书签功能插件 - 已禁用，使用新的simple-bookmarks替代
	-- ============================================================================
	-- {
	--	"MattesGroeger/vim-bookmarks",
	--	lazy = false, -- 立即加载插件以确保配置生效
	--	config = function()
	--		-- 书签配置
	--		vim.g.bookmark_sign = "🔖"          -- 书签图标
	--		vim.g.bookmark_highlight_lines = 1   -- 高亮整行
	--		vim.g.bookmark_show_warning = 0      -- 关闭警告
	--		vim.g.bookmark_auto_save = 1         -- 自动保存书签
	--		vim.g.bookmark_auto_save_file = vim.fn.expand("~/.vim-bookmarks") -- 保存位置
	--		vim.g.bookmark_manage_per_buffer = 0 -- 全局书签
	--		vim.g.bookmark_save_per_working_dir = 0 -- 不按工作目录保存
	--	end,
	--	-- 提供带行号指示的书签功能
	-- },

	-- ============================================================================
	-- LuaSnip 代码片段插件
	-- ============================================================================
	{
		"L3MON4D3/LuaSnip",
		event = { "BufReadPost", "BufNewFile" },
		dependencies = {
			"rafamadriz/friendly-snippets", -- 预定义的代码片段集合
		},
		config = function()
			require("configs.luasnip")
		end,
	},

	-- ============================================================================
	-- Git 集成插件 - gitsigns.nvim
	-- ============================================================================
	{
		"lewis6991/gitsigns.nvim",
		event = { "BufReadPre", "BufNewFile" },
		dependencies = { "nvim-lua/plenary.nvim" },
		config = function()
			require("configs.gitsigns")
		end,
	},
}
