-- ============================================================================
-- 自定义键位映射配置文件 (mappings.lua)
-- 定义个性化的键位绑定，提升编辑效率和用户体验
-- ============================================================================
--
-- 映射设计理念：
-- 1. 保持与 NvChad 默认映射的兼容性
-- 2. 使用 Alt 键作为主要修饰键，避免与系统快捷键冲突
-- 3. 优化常用操作的便捷性（导航、编辑、选择）
-- 4. 增强文本对象和可视模式的功能
-- 5. 集成插件功能的快速访问
--
-- 键位分类：
-- - 基础功能：搜索、清除高亮
-- - 导航移动：行首行尾、屏幕滚动、缓冲区切换
-- - 编辑增强：快速删除、修改、撤销重做
-- - 文本对象：自定义引号选择、可视模式切换
-- - 插件集成：文件树、快速移动
-- ============================================================================

-- ============================================================================
-- 1. 基础配置初始化
-- ============================================================================
-- 加载 NvChad 的默认键位映射作为基础
require("nvchad.mappings")

-- 获取键位映射设置函数
local map = vim.keymap.set

-- ============================================================================
-- 0. 保存时自动格式化
-- ============================================================================
-- 重写默认的保存命令，添加自动格式化功能
vim.api.nvim_create_user_command("W", function()
	vim.cmd("write")
	-- 保存后自动格式化
	local ok, conform = pcall(require, "conform")
	if ok then
		conform.format({ async = true, lsp_fallback = true })
	end
end, { desc = "Write and format file" })

-- 重写默认的保存命令
vim.api.nvim_create_user_command("Wq", function()
	vim.cmd("writequit")
	-- 保存后自动格式化
	local ok, conform = pcall(require, "conform")
	if ok then
		conform.format({ async = true, lsp_fallback = true })
	end
end, { desc = "Write, quit and format file" })

-- ============================================================================
-- 2. 基础功能映射
-- ============================================================================
-- 注意：空格键已经被设置为 leader 键，下面的映射会产生冲突，因此被注释掉
-- 如果需要使用空格作为命令行快捷键，需要重新考虑 leader 键的设置
-- map('n', '<space>', ':', { noremap = true })

-- 快速取消搜索高亮显示
-- 在普通模式下按 'nh' 来清除搜索高亮
map("n", "nh", ":noh<CR>", { desc = "Clear search highlights" })

-- ============================================================================
-- 3. 导航和移动映射
-- ============================================================================

-- 3.1 行内快速移动
-- 使用 Alt 键组合来快速移动到行首和行尾，提高编辑效率
-- Alt+h: 移动到行首（非空白字符处）
map("n", "<M-h>", "^", { desc = "Move to line start" })
-- Alt+l: 移动到行尾
map("n", "<M-l>", "$", { desc = "Move to line end" })

-- 结合删除操作的行首行尾移动
map("n", "d<M-h>", "d^", { desc = "Delete to line start" })
map("n", "d<M-l>", "d$", { desc = "Delete to line end" })

-- 结合修改操作的行首行尾移动
map("n", "c<M-h>", "c^", { desc = "Change to line start" })
map("n", "c<M-l>", "c$", { desc = "Change to line end" })

-- 结合可视选择的行首行尾移动
map("n", "v<M-h>", "v^", { desc = "Visual select to line start" })
map("n", "v<M-l>", "v$", { desc = "Visual select to line end" })

-- 3.2 缓冲区导航
-- 使用 Alt 键快速在不同缓冲区之间切换
-- Alt + - : 切换到上一个缓冲区
map("n", "<M-->", ":bp<CR>", { desc = "Previous buffer" })
-- Alt + = : 切换到下一个缓冲区
map("n", "<M-=>", ":bn<CR>", { desc = "Next buffer" })

-- 3.3 屏幕滚动控制
-- 重新映射滚动快捷键，提供更舒适的滚动体验
-- Ctrl + k : 向上滚动半屏
map("n", "<C-k>", "<C-u>", { desc = "Scroll up" })
-- Ctrl + j : 向下滚动半屏
map("n", "<C-j>", "<C-d>", { desc = "Scroll down" })

-- Alt + z : 跳转到文件末尾并居中（normal模式）
map("n", "<A-z>", "Gzz", { desc = "Go to last line and center" })

-- 跳转到文件末尾并居中
-- Alt + x : 跳转到文件最后一行并居中显示
map({ "n", "i" }, "<A-x>", "<Esc>Gzzi", { desc = "Go to last line" })
map("n", "<A-x>", "Gzz", { desc = "Go to last line" })

-- ============================================================================
-- 4. 文本对象和选择增强
-- ============================================================================

-- 4.1 自定义文本对象
-- 创建自定义文本对象，可与 d, c, v, y 等操作符配合使用，提高编辑效率

-- 引号内文本对象映射
-- 'iq' => 选择双引号内的内容（不包括引号本身）
map({ "o", "x" }, "iq", 'i"', { desc = "Inside double quotes" })
-- 'ie' => 选择单引号内的内容（不包括引号本身）
map({ "o", "x" }, "ie", "i'", { desc = "Inside single quotes" })

-- 包含引号的文本对象映射
-- 'aq' => 选择双引号及其内容（包括引号本身）
map({ "o", "x" }, "aq", 'a"', { desc = "Around double quotes" })
-- 'ae' => 选择单引号及其内容（包括引号本身）
map({ "o", "x" }, "ae", "a'", { desc = "Around single quotes" })

-- 移动到引号位置的文本对象
-- 'tq' => 移动到下一个双引号位置
map({ "o", "x" }, "tq", 't"', { desc = "To double quotes" })
-- 'te' => 移动到下一个单引号位置
map({ "o", "x" }, "te", "t'", { desc = "To single quotes" })

-- 4.2 模式切换增强
-- Alt + v : 进入可视块模式（替代默认的 Ctrl+v）
-- 这样可以避免与系统粘贴快捷键冲突
map("n", "<A-v>", "<C-v>", { desc = "Visual block mode" })

-- ============================================================================
-- 注意事项：关于可视模式下的移动映射
-- ============================================================================
-- 原始的 v<A-h> 等映射方式在 Neovim Lua 中不推荐使用
-- 因为 'v<A-h>' 不是一个有效的模式前缀
-- 更好的方式是在可视模式（'x'）下直接映射，或使用多模式映射：
-- map({ 'n', 'v' }, '<M-h>', '^', { desc = "Move to line start" })
-- map({ 'n', 'v' }, '<M-l>', '$', { desc = "Move to line end" })
--
-- 关于操作符映射的说明：
-- 上面的 d<M-h> 等映射是有效的，因为它们组合了操作符(d/c/v)和动作(motion)
-- 这样可以实现 "删除到行首" 等复合操作

-- ============================================================================
-- 5. 编辑增强功能
-- 在不同模式下提供更便捷的编辑操作
-- ============================================================================

-- 5.1 插入模式下的撤销和重做操作
-- Ctrl + z : 在插入模式下执行撤销操作
map("i", "<C-z>", "<Esc>ui", { desc = "Undo in insert mode" })
-- Ctrl + r : 在插入模式下执行重做操作
map("i", "<C-r>", "<Esc><C-r>i", { desc = "Redo in insert mode" })

-- 5.2 快速编辑操作
-- Alt + d : 在插入模式下重新编辑当前行
-- 退出插入模式，删除当前行内容并重新进入插入模式
map("i", "<A-d>", "<esc>cc", { desc = "Change entire line" })

-- 5.3 全选功能
-- Leader + a : 全选文档内容（normal模式下）
map("n", "<leader>a", "gg0vG$", { desc = "📋 全选" })
-- 注意：gg0vG$会跳到文档开头第一个字符并选择到最后一个字符

-- 5.4 数字操作（已注释）
-- 用于快速增加或减少数字，需要配合 BetterTouchTool 等工具使用
-- Ctrl + Shift + k : 增加光标下的数字
-- map('n', '<C-S-k>', '<C-a>', { desc = "Increment number" })
-- Ctrl + Shift + j : 减少光标下的数字
-- map('n', '<C-S-j>', '<C-x>', { desc = "Decrement number" })

-- ============================================================================
-- 6. 插件相关映射
-- ============================================================================

-- 6.0 全局搜索定义功能 - 适用于所有文件类型
map("n", "<leader>D", function()
	-- 获取光标下的单词
	local word = vim.fn.expand("<cword>")
	if word == "" then
		vim.notify("💡 请将光标置于要搜索的符号上", vim.log.levels.INFO)
		return
	end
	
	-- 使用Telescope在工作区中搜索相关引用或定义
	require("telescope.builtin").live_grep({
		default_text = word,
		prompt_title = "搜索 " .. word .. " 的引用或定义"
	})
end, { desc = "🔍 搜索引用或定义" })

-- 覆盖NvChad默认的buffer new描述为中文
map("n", "<leader>b", "<cmd>enew<CR>", { desc = "📝 新建缓冲区" })

-- 重新映射诊断信息列表为<leader>d（替换原来的<leader>ds）
map("n", "<leader>d", vim.diagnostic.setloclist, { desc = "🩺 诊断信息列表" })

-- 删除NvChad默认的<leader>ds映射，避免冲突
pcall(vim.keymap.del, "n", "<leader>ds")

-- 覆盖NvChad默认的buffer close描述为中文
map("n", "<leader>x", function()
	require("nvchad.tabufline").close_buffer()
end, { desc = "❌ 关闭缓冲区" })

-- ============================================================================

-- 6.1 代码运行功能 (Code Runner)
-- Leader + r + r : 运行代码
map("n", "<leader>rr", function()
	require("configs.code-runner").select_and_run()
end, { desc = "🚀 运行代码" })

-- Leader + r + e : 编辑运行配置
map("n", "<leader>re", function()
	require("configs.code-runner").edit_configs()
end, { desc = "📝 编辑运行配置" })

-- Leader + r + c : 创建运行配置文件 (在当前目录创建模板文件)
map("n", "<leader>rc", function()
	require("configs.code-runner").create_config_file()
end, { desc = "📄 创建运行配置" })

-- Leader + r + d : 编辑默认运行配置
map("n", "<leader>rd", function()
	require("configs.code-runner").edit_default_configs()
end, { desc = "⚙️ 编辑默认配置" })

-- Leader + r + j : 快速启动Django项目
map("n", "<leader>rj", function()
	require("configs.code-runner").start_django_project()
end, { desc = "🐍 启动Django项目" })

-- ============================================================================

-- 6.2 文件树操作 (nvim-tree)
-- 使用 NvChad 内置的 nvim-tree 进行文件管理
-- Alt + q : 切换文件树显示/隐藏（带自动 Git 状态刷新）
map("n", "<M-q>", "<cmd>NvimTreeToggleWithGit<CR>", { desc = "Toggle file tree with Git refresh" })
-- Leader + e : 聚焦到文件树窗口
map("n", "<leader>e", "<cmd>NvimTreeFocus<CR>", { desc = "Focus file tree" })

-- 6.2 EasyMotion 快速移动插件
-- 提供基于字符的快速跳转功能
-- Leader + f : EasyMotion 双向字符查找
map("n", "<Leader>f", "<Plug>(easymotion-bd-f)", { desc = "EasyMotion find" })
-- s : EasyMotion 两字符搜索（覆盖默认的替换字符功能）
map("n", "s", "<Plug>(easymotion-overwin-f2)", { desc = "EasyMotion 2-char search" })
-- 通过 Command+i 打开 EasyMotion 单字符搜索（GUI 或支持 <D-...> 的终端生效）
map("n", "<D-i>", "<Plug>(easymotion-overwin-f)", { desc = "EasyMotion 1-char search" })
-- 兼容方案：终端下普遍可用的 Alt+i 触发（多数终端将 Alt 传递为 Meta）
map("n", "<A-i>", "<Plug>(easymotion-overwin-f)", { desc = "EasyMotion 1-char search (Alt+i)" })
-- 如果你使用 Neovide，并希望使用 Command 作为修饰键，请在 init.lua 中设置：
--   vim.g.neovide_input_use_logo = true
-- 这样 <D-i> 将作为 "⌘+i" 生效

-- ============================================================================
-- 6.3 代码行移动功能 (Command+Shift+J/K)
-- ============================================================================

-- 6.3.1 在普通模式下移动当前行
-- Cmd+Shift+J : 向下移动当前行
map("n", "<D-S-j>", ":m .+1<CR>==", { desc = "向下移动当前行" })
-- Cmd+Shift+K : 向上移动当前行
map("n", "<D-S-k>", ":m .-2<CR>==", { desc = "向上移动当前行" })

-- 6.3.2 在插入模式下移动当前行
-- Cmd+Shift+J : 向下移动当前行（保持插入模式）
map("i", "<D-S-j>", "<Esc>:m .+1<CR>==gi", { desc = "向下移动当前行（插入模式）" })
-- Cmd+Shift+K : 向上移动当前行（保持插入模式）
map("i", "<D-S-k>", "<Esc>:m .-2<CR>==gi", { desc = "向上移动当前行（插入模式）" })

-- 6.3.3 在可视模式下移动选中的行块
-- Cmd+Shift+J : 向下移动选中的行块
map("v", "<D-S-j>", ":m '>+1<CR>gv=gv", { desc = "向下移动选中行块" })
-- Cmd+Shift+K : 向上移动选中的行块
map("v", "<D-S-k>", ":m '<-2<CR>gv=gv", { desc = "向上移动选中行块" })

-- 6.3.4 兼容方案：Alt+Shift+J/K (适用于不支持Cmd键的终端)
-- Alt+Shift+J : 向下移动当前行
map("n", "<A-S-j>", ":m .+1<CR>==", { desc = "向下移动当前行 (Alt)" })
-- Alt+Shift+K : 向上移动当前行
map("n", "<A-S-k>", ":m .-2<CR>==", { desc = "向上移动当前行 (Alt)" })

-- 在插入模式下的 Alt+Shift+J/K
map("i", "<A-S-j>", "<Esc>:m .+1<CR>==gi", { desc = "向下移动当前行（插入模式，Alt）" })
map("i", "<A-S-k>", "<Esc>:m .-2<CR>==gi", { desc = "向上移动当前行（插入模式，Alt）" })

-- 在可视模式下的 Alt+Shift+J/K
map("v", "<A-S-j>", ":m '>+1<CR>gv=gv", { desc = "向下移动选中行块 (Alt)" })
map("v", "<A-S-k>", ":m '<-2<CR>gv=gv", { desc = "向上移动选中行块 (Alt)" })

-- ============================================================================
-- 7. 文件模板系统映射
-- ============================================================================

-- 7.1 模板管理快捷键
-- Leader + t + n : 使用模板创建新文件
map("n", "<leader>tn", "<cmd>Template<cr>", { desc = "使用模板创建新文件" })
-- Leader + t + a : 为当前文件应用模板
map("n", "<leader>ta", "<cmd>TemplateApply<cr>", { desc = "为当前文件应用模板" })
-- Leader + t + l : 列出所有可用模板
map("n", "<leader>tl", "<cmd>TemplateList<cr>", { desc = "列出所有可用模板" })

-- 7.2 主题切换功能
-- Leader + t + h : 主题切换
map("n", "<leader>th", function()
	require("nvchad.themes").open()
end, { desc = "🎨 主题切换" })

-- ============================================================================
-- 8. 主题切换功能
-- ============================================================================

-- 8.1 背景模式切换函数（保留原生功能）
local function toggle_background()
	-- 获取当前背景设置
	local current_bg = vim.o.background

	-- 切换背景（深色/浅色）
	if current_bg == "dark" then
		vim.o.background = "light"
		vim.notify("背景已切换为浅色模式", vim.log.levels.INFO)
	else
		vim.o.background = "dark"
		vim.notify("背景已切换为深色模式", vim.log.levels.INFO)
	end
end

-- 8.2 背景切换快捷键（如果需要的话）
-- map("n", "<leader>bg", toggle_background, { desc = "切换背景（深色/浅色）" })

-- ============================================================================
-- 9. HTML 实时预览功能
-- ============================================================================

-- 9.1 HTML预览功能快捷键
-- Leader + p : 🌐 HTML预览相关操作

-- 基础映射
map("n", "<leader>p", function()
end, { desc = "🌐 HTML预览" })

-- Leader + p 🌐+ s : 启动HTML预览服务器
map("n", "<leader>ps", function()
	require("configs.html-preview").preview_html()
end, { desc = "🌐 启动服务器" })

-- Leader + p 🌐+ q : 停止HTML预览服务器
map("n", "<leader>pq", function()
	require("configs.html-preview").stop_preview()
end, { desc = "🌐 停止服务器" })

-- Leader + p 🌐+ r : 重启HTML预览服务器
map("n", "<leader>pr", function()
	require("configs.html-preview").restart_preview()
end, { desc = "🌐 重启服务器" })

-- Leader + p 🌐+ i : 查看HTML预览状态
map("n", "<leader>pi", function()
	local status = require("configs.html-preview").get_status()
	if status.running then
		vim.notify(string.format("预览服务器运行中\n端口: %d\n文件: %s\nURL: %s",
			status.port, status.file or "未知", status.url), vim.log.levels.INFO)
	else
		vim.notify("预览服务器未运行", vim.log.levels.INFO)
	end
end, { desc = "🌐 服务器状态" })

-- ============================================================================
-- 10. 终端管理功能
-- ============================================================================

-- 10.1 新建终端快捷键
-- Leader + t + x : 新建水平终端 (右侧分割)
map("n", "<leader>tx", function()
	require("nvchad.term").new { pos = "vsp" }
end, { desc = "💻 新建水平终端" })

-- Leader + t + y : 新建垂直终端 (下方分割)
map("n", "<leader>ty", function()
	require("nvchad.term").new { pos = "sp" }
end, { desc = "💻 新建垂直终端" })

-- ============================================================================
-- 11. Telescope 搜索功能 - 统一在f组
-- ============================================================================

-- 11.1 搜索相关快捷键 (所有telescope功能集中在f组)
-- Leader + f + h : 搜索帮助
map("n", "<leader>fh", "<cmd>Telescope help_tags<CR>", { desc = "🔍 搜索帮助" })

-- Leader + f + f : 搜索文件
map("n", "<leader>ff", "<cmd>Telescope find_files<CR>", { desc = "🔍 搜索文件" })

-- Leader + f + g : 全局搜索
map("n", "<leader>fg", "<cmd>Telescope live_grep<CR>", { desc = "🔍 全局搜索" })

-- Leader + f + b : 搜索缓冲区
map("n", "<leader>fb", "<cmd>Telescope buffers<CR>", { desc = "🔍 搜索缓冲区" })

-- Leader + f + t : 选择隐藏终端
map("n", "<leader>ft", "<cmd>Telescope terms<CR>", { desc = "🔍 选择隐藏终端" })

-- Leader + f + o : 搜索最近文件
map("n", "<leader>fo", "<cmd>Telescope oldfiles<CR>", { desc = "🔍 搜索最近文件" })

-- Leader + f + m : 搜索标记
map("n", "<leader>fm", "<cmd>Telescope marks<CR>", { desc = "🔍 搜索标记" })

-- Leader + f + z : 当前缓冲区模糊搜索
map("n", "<leader>fz", "<cmd>Telescope current_buffer_fuzzy_find<CR>", { desc = "🔍 当前缓冲区模糊搜索" })

-- Leader + f + c : 搜索Git提交 (仅在Git目录下可用)
map("n", "<leader>fc", function()
	local is_git = vim.fn.isdirectory(".git") == 1 or vim.fn.system("git rev-parse --git-dir 2>/dev/null"):match("^%.git") ~= nil
	if is_git then
		vim.cmd("Telescope git_commits")
	else
		vim.notify("当前不在Git目录中", vim.log.levels.WARN)
	end
end, { desc = "🔍 搜索Git提交" })

-- Leader + f + s : 搜索Git状态 (仅在Git目录下可用)
map("n", "<leader>fs", function()
	local is_git = vim.fn.isdirectory(".git") == 1 or vim.fn.system("git rev-parse --git-dir 2>/dev/null"):match("^%.git") ~= nil
	if is_git then
		vim.cmd("Telescope git_status")
	else
		vim.notify("当前不在Git目录中", vim.log.levels.WARN)
	end
end, { desc = "🔍 搜索Git状态" })

-- Leader + f + a : 搜索所有文件
map("n", "<leader>fa", "<cmd>Telescope find_files follow=true no_ignore=true hidden=true<CR>", { desc = "🔍 搜索所有文件" })

-- 11.2 LSP相关的telescope搜索功能
-- Leader + f + r : 搜索引用 (LSP references)
map("n", "<leader>fr", "<cmd>Telescope lsp_references<CR>", { desc = "🔍 搜索引用" })

-- Leader + f + d : 搜索定义 (LSP definitions)  
map("n", "<leader>fd", "<cmd>Telescope lsp_definitions<CR>", { desc = "🔍 搜索定义" })

-- Leader + f + i : 搜索实现 (LSP implementations)
map("n", "<leader>fi", "<cmd>Telescope lsp_implementations<CR>", { desc = "🔍 搜索实现" })

-- Leader + f + y : 搜索类型定义 (LSP type definitions)
map("n", "<leader>fy", "<cmd>Telescope lsp_type_definitions<CR>", { desc = "🔍 搜索类型定义" })

-- Leader + f + w : 搜索工作区符号 (LSP workspace symbols)
map("n", "<leader>fw", "<cmd>Telescope lsp_workspace_symbols<CR>", { desc = "🔍 搜索工作区符号" })

-- Leader + f + l : 搜索文档符号 (LSP document symbols)  
map("n", "<leader>fl", "<cmd>Telescope lsp_document_symbols<CR>", { desc = "🔍 搜索文档符号" })

-- ============================================================================
-- 12. 简单书签功能 (Simple Bookmarks)
-- ============================================================================

-- 初始化简单书签系统
local simple_bookmarks = require("configs.simple-bookmarks")
simple_bookmarks.setup()

-- 12.1 主要书签快捷键
-- m : 切换当前行的书签状态（单键快速操作）
map("n", "m", function()
    simple_bookmarks.toggle_bookmark()
end, { desc = "🔖 切换书签" })

-- 12.2 书签导航快捷键
-- ] m : 跳转到下一个书签
map("n", "]m", function()
    simple_bookmarks.next_bookmark()
end, { desc = "🔖 下一个书签" })

-- [ m : 跳转到上一个书签
map("n", "[m", function()
    simple_bookmarks.prev_bookmark()
end, { desc = "🔖 上一个书签" })

-- 12.3 书签管理快捷键（Leader组合）
-- Leader + m + l : 列出所有书签
map("n", "<leader>ml", function()
    simple_bookmarks.list_bookmarks()
end, { desc = "🔖 列出所有书签" })

-- Leader + m + c : 清除当前文件的所有书签
map("n", "<leader>mc", function()
    simple_bookmarks.clear_bookmarks()
end, { desc = "🔖 清除当前文件书签" })

-- Leader + m + C : 清除所有书签
map("n", "<leader>mC", function()
    simple_bookmarks.clear_all_bookmarks()
end, { desc = "🔖 清除所有书签" })

-- Leader + m + s : 查看书签统计
map("n", "<leader>ms", function()
    local stats = simple_bookmarks.get_stats()
    vim.notify(string.format("书签统计：%d 个文件中共有 %d 个书签", 
        stats.file_count, stats.total_bookmarks), vim.log.levels.INFO)
end, { desc = "📊 书签统计" })




-- ============================================================================

-- 12.1 禁用默认的分散telescope映射 (已统一迁移到f组)
local function disable_default_mappings()
	-- 使用pcall避免映射不存在时的错误
	pcall(vim.keymap.del, "n", "<leader>cm") -- git commits → fc
	pcall(vim.keymap.del, "n", "<leader>gt") -- git status → fs  
	pcall(vim.keymap.del, "n", "<leader>pt") -- pick terms → ft
	pcall(vim.keymap.del, "n", "<leader>ma") -- marks → fm
	pcall(vim.keymap.del, "n", "<leader>wK") -- whichkey all keymaps → 删除
end

-- 延迟执行避免加载时序问题
vim.schedule(disable_default_mappings)

-- 额外确保删除wK映射 - 使用多种方式确保删除
vim.defer_fn(function()
	pcall(vim.keymap.del, "n", "<leader>wK")
	-- 如果删除失败，则用空操作覆盖
	pcall(vim.keymap.set, "n", "<leader>wK", "<nop>", { desc = "" })
end, 100)

-- 立即覆盖wK映射
pcall(vim.keymap.set, "n", "<leader>wK", "<nop>", { desc = "" })

-- 12.2 设置leader+ch为cheatsheet快捷键（类似which-key菜单显示）
map("n", "<leader>ch", function()
	-- 创建类似which-key的浮动菜单
	local function create_cheatsheet_menu()
		-- 计算窗口尺寸（匹配内容尺寸并居中）
		local width = 70
		local height = 26
		local row = math.floor((vim.o.lines - height) / 2)
		local col = math.floor((vim.o.columns - width) / 2) + 3
		
		-- 创建缓冲区
		local buf = vim.api.nvim_create_buf(false, true)
		
		-- 设置浮动窗口选项（类似which-key的样式）
		local opts = {
			relative = 'editor',
			width = width,
			height = height,
			row = row,
			col = col,
			style = 'minimal',
			border = 'rounded',
			zindex = 1000,
		}
		
		-- 创建浮动窗口
		local win = vim.api.nvim_open_win(buf, true, opts)
		
		-- 设置窗口属性（完全透明背景）
		vim.api.nvim_win_set_option(win, 'winhl', 'Normal:NormalFloat,FloatBorder:FloatBorder')
		vim.api.nvim_win_set_option(win, 'winblend', 0)
		
		-- 设置缓冲区属性
		vim.api.nvim_buf_set_option(buf, 'bufhidden', 'wipe')
		vim.api.nvim_buf_set_option(buf, 'buftype', 'nofile')
		vim.api.nvim_buf_set_option(buf, 'swapfile', false)
		
		-- 保存当前设置以便恢复
		local original_cursorline = vim.wo.cursorline
		local original_cursorcolumn = vim.wo.cursorcolumn
		
		-- 临时禁用当前行/列高亮
		vim.wo.cursorline = false
		vim.wo.cursorcolumn = false
		
		-- 快捷键内容（整体居中，左右文字左对齐）
		local function center_title(text, width)
			local text_len = vim.fn.strdisplaywidth(text)
			if text_len >= width then return text end
			local padding = math.floor((width - text_len) / 2)
			return string.rep(" ", padding) .. text
		end
		
		local function left_title(text, width)
			-- 与两列内容的左列对齐
			local left_start = 8
			return string.rep(" ", left_start) .. text
		end
		
		local function format_two_column(left, right, width)
			-- 左列起始位置（向右移动更多）
			local left_start = 8
			-- 右列起始位置（固定位置，确保所有右列内容对齐）
			local right_start = 40
			
			-- 构建左列内容
			local left_content = string.rep(" ", left_start) .. left
			
			-- 计算左列内容占用的实际宽度
			local left_width = vim.fn.strdisplaywidth(left_content)
			
			-- 计算右列需要的空格数，确保右列从固定位置开始
			local right_padding = right_start - left_width
			if right_padding < 1 then right_padding = 1 end
			
			-- 构建最终结果
			local result = left_content .. string.rep(" ", right_padding) .. right
			
			return result
		end
		
		local box_width = 70
		local content = {
			center_title("──────────────────────── 📖 快捷键帮助 ────────────────────────", box_width),
			"",
			center_title("🔍 搜索和查找 (<leader>f)", box_width),
			format_two_column("f f → 🔍 搜索文件", "f g → 🔍 全局搜索", box_width),
			format_two_column("f b → 🔍 搜索缓冲区", "f o → 🔍 搜索最近文件", box_width),
			format_two_column("f m → 🔍 搜索标记", "f z → 🔍 当前缓冲区模糊搜索", box_width),
			format_two_column("f c → 🔍 搜索Git提交", "f s → 🔍 搜索Git状态", box_width),
			format_two_column("f a → 🔍 搜索所有文件", "f t → 🔍 选择隐藏终端", box_width),
			format_two_column("f h → 🔍 搜索帮助", "f r → 🔍 搜索引用", box_width),
			format_two_column("f d → 🔍 搜索定义", "f i → 🔍 搜索实现", box_width),
			"",
			center_title("🖼️ 窗口管理 (<leader>w)", box_width),
			format_two_column("w h → ⬅️ 切换到左侧窗口", "w l → ➡️ 切换到右侧窗口", box_width),
			format_two_column("w j → ⬇️ 切换到下方窗口", "w k → ⬆️ 切换到上方窗口", box_width),
			format_two_column("w s → ➖ 水平分割窗口", "w v → ➕ 垂直分割窗口", box_width),
			format_two_column("w c → ❌ 关闭当前窗口", "w o → 🗑️ 关闭其他窗口", box_width),
			format_two_column("w = → ⚖️ 智能平衡窗口", "w x → 🔄 交换窗口位置", box_width),
			format_two_column("w q → 🔍 查询按键映射", "", box_width),
			"",
			center_title("🛠️ 模板和终端 (<leader>t)", box_width),
			format_two_column("t n → 🛠️ 使用模板创建新文件", "t a → 🛠️ 为当前文件应用模板", box_width),
			format_two_column("t l → 🛠️ 列出所有可用模板", "t h → 🎨 主题切换", box_width),
			format_two_column("t x → 💻 新建水平终端", "t y → 💻 新建垂直终端", box_width),
			"",
			center_title("🔧 重构和运行 (<leader>r)", box_width),
			format_two_column("r r → 🚀 运行代码", "r e → 📝 编辑运行配置", box_width),
			format_two_column("r c → 📄 创建运行配置", "r d → ⚙️ 编辑默认配置", box_width),
			format_two_column("r j → 🐍 启动Django项目", "", box_width),
			"",
			center_title("🌐 HTML预览 (<leader>p)", box_width),
			format_two_column("p s → 🌐 启动预览服务器", "p q → 🌐 停止预览服务器", box_width),
			format_two_column("p r → 🌐 重启预览服务器", "p i → 🌐 查看服务器状态", box_width),
			"",
			center_title("🎯 导航移动", box_width),
			format_two_column("Alt+h → 移动到行首", "Alt+l → 移动到行尾", box_width),
			format_two_column("Alt+- → 上一个缓冲区", "Alt+= → 下一个缓冲区", box_width),
			format_two_column("Ctrl+k → 向上滚动半屏", "Ctrl+j → 向下滚动半屏", box_width),
			"",
			center_title("🌳 文件管理", box_width),
			format_two_column("Alt+q → 切换文件树", "<leader>e → 聚焦文件树", box_width),
			"",
			center_title("💻 终端管理", box_width),
			format_two_column("Alt+h → 切换水平终端", "Alt+v → 切换垂直终端", box_width),
			center_title("Alt+i → 切换浮动终端", box_width),
			"",
			center_title("📖 其他", box_width),
			format_two_column("<leader>ch → 📋 显示快捷键列表", "<leader>ca → ⚡ 代码操作", box_width),
			"",
			center_title("📝 代码片段管理 (<leader>l)", box_width),
			format_two_column("l l → 📋 列出当前文件片段", "l r → 🔄 重新加载片段配置", box_width),
			format_two_column("l e → ✏️ 编辑片段配置文件", "l g → 📖 查看自定义指南", box_width),
			format_two_column("l t → 🧪 打开测试文件", "l h → ❓ 显示片段帮助", box_width),
			format_two_column("l n → ➕ 创建新片段模板", "", box_width),
			"",
			center_title("🔖 书签功能", box_width),
			format_two_column("m → 🔖 切换书签", "]m → 🔖 下一个书签", box_width),
			format_two_column("[m → 🔖 上一个书签", "<leader>ml → 🔖 列出所有书签", box_width),
			format_two_column("<leader>mc → 🔖 清除当前文件书签", "<leader>mC → 🔖 清除所有书签", box_width),
			format_two_column("<leader>ms → 🔖 书签统计", "", box_width),
			"",
			center_title("────────────────────── 按 q 或 Esc 关闭 ──────────────────────", box_width)
		}
		
		-- 设置内容
		vim.api.nvim_buf_set_lines(buf, 0, -1, false, content)
		vim.api.nvim_buf_set_option(buf, 'modifiable', false)
		
		-- 设置退出快捷键（恢复原设置）
		local function close_menu()
			-- 恢复原来的光标行设置
			vim.wo.cursorline = original_cursorline
			vim.wo.cursorcolumn = original_cursorcolumn
			
			-- 关闭窗口
			if vim.api.nvim_win_is_valid(win) then
				vim.api.nvim_win_close(win, true)
			end
		end
		
		vim.keymap.set('n', 'q', close_menu, { buffer = buf, silent = true })
		vim.keymap.set('n', '<Esc>', close_menu, { buffer = buf, silent = true })
		vim.keymap.set('n', '<CR>', close_menu, { buffer = buf, silent = true })
		
		return win, buf
	end
	
	-- 创建并显示菜单
	create_cheatsheet_menu()
end, { desc = "📋 显示快捷键列表" })


-- 12.3 LSP Code Actions 设置到独立快捷键
map("n", "<leader>ca", vim.lsp.buf.code_action, { desc = "⚡ 代码操作" })

-- 12.4 HTML文件特殊重命名功能 (用于不支持LSP重命名的情况)
local function html_rename()
	-- 获取当前光标下的单词
	local current_word = vim.fn.expand("<cword>")
	if current_word == "" then
		vim.notify("❌ 光标下没有单词可以重命名", vim.log.levels.WARN)
		return
	end
	
	-- 显示当前文件类型信息
	local filetype = vim.bo.filetype
	vim.notify("🔧 " .. string.upper(filetype) .. " 文件重命名模式", vim.log.levels.INFO)
	
	-- 询问新名称
	local new_name = vim.fn.input("🏷️  重命名 '" .. current_word .. "' 为: ")
	if new_name == "" or new_name == current_word then
		vim.notify("❌ 已取消重命名", vim.log.levels.INFO)
		return
	end
	
	-- 在当前缓冲区中进行搜索替换
	local pattern = "\\<" .. vim.fn.escape(current_word, ".*[]^$~\\") .. "\\>"
	local replace = vim.fn.escape(new_name, "&~")
	local command = "%s/" .. pattern .. "/" .. replace .. "/gc"
	
	-- 执行替换命令
	pcall(function()
		vim.cmd(command)
	end)
	
	vim.notify("✅ 在当前文件中重命名 '" .. current_word .. "' 为 '" .. new_name .. "'", vim.log.levels.INFO)
end

-- 导出函数以供其他模块使用
local M = {}
M.html_rename = html_rename
_G.html_rename = html_rename -- 全局导出

-- 为HTML和CSS文件重新映射重命名键
vim.api.nvim_create_autocmd({"FileType", "LspAttach"}, {
	pattern = { "html", "htm", "css", "scss", "sass", "less" },
	callback = function(event)
		-- 延迟执行以确保覆盖LSP的映射
		vim.schedule(function()
			-- 先删除可能存在的LSP重命名映射
			pcall(vim.keymap.del, "n", "<leader>ra", { buffer = event.buf })
			
			-- 在HTML/CSS文件中，使用自定义重命名功能
			local filetype = vim.api.nvim_buf_get_option(event.buf, "filetype")
			local desc = "🏷️ " .. string.upper(filetype) .. "重命名 (搜索替换)"
			vim.keymap.set("n", "<leader>ra", html_rename, { 
				buffer = event.buf, 
				desc = desc,
				silent = true,
				noremap = true
			})
			
			-- 仅在FileType事件时显示通知，避免重复
			if event.event == "FileType" then
				-- vim.notify("📝 已为" .. filetype .. "文件启用自定义重命名功能", vim.log.levels.INFO)
			end
		end)
	end,
	desc = "为HTML/CSS文件设置特殊重命名功能"
})

-- ============================================================================
-- 13. 窗口管理功能 (Window Management)
-- ============================================================================

-- 13.1 窗口切换 (保持 Ctrl 键的便捷性，同时提供 leader 方式)
-- Leader + w + h/j/k/l : 切换到不同方向的窗口 (使用标准vim方向键)
map("n", "<leader>wh", "<C-w>h", { desc = "切换到左侧窗口" })
map("n", "<leader>wj", "<C-w>j", { desc = "切换到下方窗口" })  
map("n", "<leader>wk", "<C-w>k", { desc = "切换到上方窗口" })
map("n", "<leader>wl", "<C-w>l", { desc = "切换到右侧窗口" })

-- 13.1.0 快捷窗口切换 (更便捷的窗口导航)
-- Leader + j : 直接切换到下方窗口
map("n", "<leader>j", "<C-w>j", { desc = "⬇️ 切换到下方窗口" })
-- Leader + k : 直接切换到上方窗口
map("n", "<leader>k", "<C-w>k", { desc = "⬆️ 切换到上方窗口" })

-- 13.1.1 查询按键映射功能 (原wk功能移至wq避免冲突)
map("n", "<leader>wq", function()
  vim.cmd("WhichKey " .. vim.fn.input "WhichKey: ")
end, { desc = "查询按键映射" })

-- 13.2 窗口分割
-- Leader + w + s : 水平分割当前窗口
map("n", "<leader>ws", "<C-w>s", { desc = "水平分割窗口" })
-- Leader + w + v : 垂直分割当前窗口  
map("n", "<leader>wv", "<C-w>v", { desc = "垂直分割窗口" })

-- 13.3 窗口关闭
-- Leader + w + c : 关闭当前窗口
map("n", "<leader>wc", "<C-w>c", { desc = "关闭当前窗口" })
-- Leader + w + o : 关闭其他所有窗口
map("n", "<leader>wo", "<C-w>o", { desc = "关闭其他窗口" })

-- 13.4 窗口大小调整
-- Leader + w + = : 智能平衡两个窗口（仅在2窗口时生效）
map("n", "<leader>w=", function()
	local win_count = vim.fn.winnr('$')
	if win_count == 2 then
		-- 获取当前窗口和另一个窗口的位置信息
		local current_win = vim.fn.winnr()
		local other_win = current_win == 1 and 2 or 1
		
		-- 获取两个窗口的位置
		local current_pos = vim.fn.getwininfo(vim.fn.win_getid(current_win))[1]
		local other_pos = vim.fn.getwininfo(vim.fn.win_getid(other_win))[1]
		
		-- 判断是水平分割还是垂直分割
		if current_pos.wincol == other_pos.wincol then
			-- 垂直分割（上下窗口），平衡高度
			local total_height = vim.o.lines - vim.o.cmdheight - 1 -- 减去命令行和状态栏
			local each_height = math.floor(total_height / 2)
			vim.cmd(current_win .. "wincmd w")
			vim.cmd("resize " .. each_height)
		else
			-- 水平分割（左右窗口），平衡宽度
			local total_width = vim.o.columns
			local each_width = math.floor(total_width / 2)
			vim.cmd(current_win .. "wincmd w")
			vim.cmd("vertical resize " .. each_width)
		end
		vim.notify("已平衡两个窗口大小", vim.log.levels.INFO)
	else
		-- 超过2个窗口时使用默认平衡
		vim.cmd("wincmd =")
		vim.notify("已平衡所有窗口大小", vim.log.levels.INFO)
	end
end, { desc = "智能平衡窗口" })
-- Leader + w + + : 增加窗口高度
map("n", "<leader>w+", "<C-w>+", { desc = "增加窗口高度" })
-- Leader + w + - : 减少窗口高度  
map("n", "<leader>w-", "<C-w>-", { desc = "减少窗口高度" })
-- Leader + w + > : 增加窗口宽度
map("n", "<leader>w>", "<C-w>>", { desc = "增加窗口宽度" })
-- Leader + w + < : 减少窗口宽度
map("n", "<leader>w<", "<C-w><", { desc = "减少窗口宽度" })

-- 13.4.1 更精确的窗口宽度调整
-- Leader + w + [ : 大幅减少宽度
map("n", "<leader>w[", "5<C-w><", { desc = "大幅减少宽度" })
-- Leader + w + ] : 大幅增加宽度  
map("n", "<leader>w]", "5<C-w>>", { desc = "大幅增加宽度" })
-- Leader + w + { : 大幅减少高度
map("n", "<leader>w{", "5<C-w>-", { desc = "大幅减少高度" })
-- Leader + w + } : 大幅增加高度
map("n", "<leader>w}", "5<C-w>+", { desc = "大幅增加高度" })

-- 13.5 窗口移动和重排
-- Leader + w + x : 交换窗口位置
map("n", "<leader>wx", "<C-w>x", { desc = "交换窗口位置" })
-- Leader + w + r : 向右/下旋转窗口
map("n", "<leader>wr", "<C-w>r", { desc = "向右旋转窗口" })
-- Leader + w + R : 向左/上旋转窗口  
map("n", "<leader>wR", "<C-w>R", { desc = "向左旋转窗口" })

-- 13.6 窗口导航增强
-- Leader + w + w : 循环切换到下一个窗口
map("n", "<leader>ww", "<C-w>w", { desc = "切换到下一个窗口" })
-- Leader + w + p : 切换到上一个窗口
map("n", "<leader>wp", "<C-w>p", { desc = "切换到上一个窗口" })

-- 注：切换终端快捷键由NvChad默认提供：
-- Alt + h : 切换水平终端
-- Alt + v : 切换垂直终端  
-- Alt + i : 切换浮动终端

-- ============================================================================
-- 14. 禁用NvChad默认映射 (避免冲突)
-- ============================================================================

-- 14.1 禁用默认的分散telescope映射 (已统一迁移到f组)
local function disable_default_mappings()
	-- 使用pcall避免映射不存在时的错误
	pcall(vim.keymap.del, "n", "<leader>cm") -- git commits → fc
	pcall(vim.keymap.del, "n", "<leader>gt") -- git status → fs  
	pcall(vim.keymap.del, "n", "<leader>pt") -- pick terms → ft
	pcall(vim.keymap.del, "n", "<leader>ma") -- marks → fm
	pcall(vim.keymap.del, "n", "<leader>wK") -- whichkey all keymaps → 删除
end

-- 延迟执行避免加载时序问题
vim.schedule(disable_default_mappings)

-- 额外确保删除wK映射 - 使用多种方式确保删除
vim.defer_fn(function()
	pcall(vim.keymap.del, "n", "<leader>wK")
	-- 如果删除失败，则用空操作覆盖
	pcall(vim.keymap.set, "n", "<leader>wK", "<nop>", { desc = "" })
end, 100)

-- 立即覆盖wK映射
pcall(vim.keymap.set, "n", "<leader>wK", "<nop>", { desc = "" })

-- ============================================================================
-- 15. LuaSnip 代码片段快捷键
-- ============================================================================

-- 展开代码片段或跳转到下一个位置
map({"i", "s"}, "<C-l>", function()
	local luasnip = require("luasnip")
	if luasnip.expand_or_jumpable() then
		luasnip.expand_or_jump()
	end
end, { desc = "展开片段或跳转到下一个位置" })

-- 跳转到上一个位置
map({"i", "s"}, "<C-h>", function()
	local luasnip = require("luasnip")
	if luasnip.jumpable(-1) then
		luasnip.jump(-1)
	end
end, { desc = "跳转到上一个位置" })

-- 在选择模式下切换选项
map("s", "<C-e>", function()
	local luasnip = require("luasnip")
	if luasnip.choice_active() then
		luasnip.change_choice(1)
	end
end, { desc = "切换代码片段选项" })

-- ============================================================================
-- 15.1 LuaSnip 管理菜单 (<leader>l)
-- ============================================================================

-- 主菜单：代码片段管理
map("n", "<leader>l", "<cmd>WhichKey <leader>l<CR>", { desc = "📝 代码片段管理" })

-- 列出当前文件类型的所有可用片段
map("n", "<leader>ll", function()
	local luasnip = require("luasnip")
	local filetype = vim.bo.filetype
	local snippets = luasnip.get_snippets(filetype)
	
	if not snippets or #snippets == 0 then
		vim.notify("当前文件类型 (" .. filetype .. ") 没有可用的代码片段", vim.log.levels.WARN)
		return
	end
	
	local lines = {"📝 " .. filetype:upper() .. " 代码片段列表:", ""}
	for _, snippet in pairs(snippets) do
		local trigger = snippet.trigger or "未知"
		local desc = snippet.name or snippet.dscr or "无描述"
		table.insert(lines, "• " .. trigger .. " → " .. desc)
	end
	
	-- 创建浮动窗口显示片段列表
	local buf = vim.api.nvim_create_buf(false, true)
	local width = 60
	local height = math.min(#lines + 2, 20)
	local row = math.floor((vim.o.lines - height) / 2)
	local col = math.floor((vim.o.columns - width) / 2)
	
	local win = vim.api.nvim_open_win(buf, true, {
		relative = 'editor',
		width = width,
		height = height,
		row = row,
		col = col,
		style = 'minimal',
		border = 'rounded',
		title = " 代码片段列表 ",
		title_pos = "center"
	})
	
	vim.api.nvim_buf_set_lines(buf, 0, -1, false, lines)
	vim.api.nvim_buf_set_option(buf, 'modifiable', false)
	vim.api.nvim_buf_set_option(buf, 'bufhidden', 'wipe')
	
	-- 设置退出快捷键
	local close_fn = function()
		vim.api.nvim_win_close(win, true)
	end
	vim.keymap.set('n', 'q', close_fn, { buffer = buf, silent = true })
	vim.keymap.set('n', '<Esc>', close_fn, { buffer = buf, silent = true })
	vim.keymap.set('n', '<CR>', close_fn, { buffer = buf, silent = true })
end, { desc = "📋 列出当前文件类型的片段" })

-- 重新加载片段配置
map("n", "<leader>lr", function()
	-- 重新加载片段配置
	local config_path = vim.fn.stdpath("config") .. "/lua/configs/luasnip.lua"
	
	-- 检查配置文件是否存在
	if vim.fn.filereadable(config_path) == 0 then
		vim.notify("❌ 配置文件不存在: " .. config_path, vim.log.levels.ERROR)
		return
	end
	
	-- 清除旧的片段缓存
	local luasnip = require("luasnip")
	luasnip.cleanup()
	
	-- 重新加载配置
	local success, err = pcall(function()
		-- 清除模块缓存
		package.loaded["configs.luasnip"] = nil
		require("configs.luasnip")
	end)
	
	if success then
		vim.notify("✅ 代码片段配置已重新加载", vim.log.levels.INFO)
	else
		vim.notify("❌ 重新加载失败: " .. tostring(err), vim.log.levels.ERROR)
	end
end, { desc = "🔄 重新加载片段配置" })

-- 编辑片段配置文件
map("n", "<leader>le", function()
	local config_path = vim.fn.stdpath("config") .. "/lua/configs/luasnip.lua"
	vim.cmd("edit " .. config_path)
end, { desc = "✏️ 编辑片段配置文件" })

-- 查看片段自定义指南
map("n", "<leader>lg", function()
	local guide_path = vim.fn.stdpath("config") .. "/snippet-customization-guide.md"
	if vim.fn.filereadable(guide_path) == 1 then
		vim.cmd("edit " .. guide_path)
	else
		vim.notify("❌ 指南文件不存在: " .. guide_path, vim.log.levels.ERROR)
	end
end, { desc = "📖 查看自定义指南" })

-- 测试片段功能
map("n", "<leader>lt", function()
	local test_file = vim.fn.stdpath("config") .. "/test-snippets.js"
	vim.cmd("edit " .. test_file)
	vim.notify("📝 已打开测试文件，尝试输入 'csl' 然后按 Tab 键测试", vim.log.levels.INFO)
end, { desc = "🧪 打开测试文件" })

-- 显示片段帮助
map("n", "<leader>lh", function()
	local help_content = {
		"🎯 代码片段快速使用指南",
		"",
		"📌 基本操作:",
		"• 在插入模式下输入触发词，然后按 Tab 展开",
		"• Tab - 跳转到下一个占位符",
		"• Shift+Tab - 跳转到上一个占位符",
		"• Ctrl+E - 切换选择项（如果有）",
		"",
		"🚀 JavaScript/TypeScript 片段:",
		"• csl → console.log()",
		"• cse → console.error()",
		"• func → 函数声明",
		"• af → 箭头函数",
		"• if → if语句",
		"• for → for循环",
		"• fe → forEach循环",
		"• try → try-catch块",
		"• st → setTimeout模板",
		"• imp → import语句",
		"• exp → export语句",
		"• comm → JSDoc注释",
		"",
		"🐍 Python 片段:",
		"• p → print()",
		"• def → 函数定义",
		"• class → 类定义",
		"• main → if __name__ == '__main__'",
		"• djview → Django视图模板",
		"",
		"🌐 HTML 片段:",
		"• html5 → HTML5基础模板",
		"• div → div标签",
		"",
		"⚡ React 片段:",
		"• rfc → React函数组件",
		"• us → useState hook",
		"• ue → useEffect hook",
		"",
		"📚 更多信息:",
		"• <leader>le - 编辑配置文件",
		"• <leader>lg - 查看详细指南",
		"• <leader>ll - 查看当前文件可用片段",
		"",
		"按 q、Esc 或 Enter 关闭此窗口"
	}
	
	local buf = vim.api.nvim_create_buf(false, true)
	local width = 50
	local height = math.min(#help_content + 2, vim.o.lines - 4)
	local row = math.floor((vim.o.lines - height) / 2)
	local col = math.floor((vim.o.columns - width) / 2)
	
	local win = vim.api.nvim_open_win(buf, true, {
		relative = 'editor',
		width = width,
		height = height,
		row = row,
		col = col,
		style = 'minimal',
		border = 'rounded',
		title = " 代码片段帮助 ",
		title_pos = "center"
	})
	
	vim.api.nvim_buf_set_lines(buf, 0, -1, false, help_content)
	vim.api.nvim_buf_set_option(buf, 'modifiable', false)
	vim.api.nvim_buf_set_option(buf, 'bufhidden', 'wipe')
	
	-- 设置退出快捷键
	local close_fn = function()
		vim.api.nvim_win_close(win, true)
	end
	vim.keymap.set('n', 'q', close_fn, { buffer = buf, silent = true })
	vim.keymap.set('n', '<Esc>', close_fn, { buffer = buf, silent = true })
	vim.keymap.set('n', '<CR>', close_fn, { buffer = buf, silent = true })
end, { desc = "❓ 显示片段帮助" })

-- 创建新的片段模板
map("n", "<leader>ln", function()
	local filetype = vim.fn.input("输入文件类型 (js/py/html/react): ")
	if filetype == "" then
		vim.notify("❌ 已取消创建", vim.log.levels.INFO)
		return
	end
	
	local trigger = vim.fn.input("输入触发词: ")
	if trigger == "" then
		vim.notify("❌ 已取消创建", vim.log.levels.INFO)
		return
	end
	
	local description = vim.fn.input("输入片段描述: ")
	
	-- 生成片段模板
	local template = string.format([[
	-- %s - 触发词 "%s"
	s("%s", {
		t(""),
		i(1, "占位符1"),
		t(""),
		i(0)
	}),]], description ~= "" and description or "新片段", trigger, trigger)
	
	-- 复制到系统剪贴板
	vim.fn.setreg("+", template)
	vim.notify("✅ 片段模板已复制到剪贴板！\n使用 <leader>le 打开配置文件并粘贴", vim.log.levels.INFO)
end, { desc = "➕ 创建新片段模板" })

-- 返回模块
return M
