-- ============================================================================
-- 自动命令配置文件 (autocmds.lua)
-- 定义各种自动触发的命令，用于响应特定事件和改善编辑体验
-- ============================================================================

-- ============================================================================
-- 1. 基础配置初始化
-- ============================================================================
-- 首先加载 NvChad 的默认自动命令
require("nvchad.autocmds")

-- 获取自动命令和自动命令组的创建函数
local autocmd = vim.api.nvim_create_autocmd -- 创建自动命令
local augroup = vim.api.nvim_create_augroup -- 创建自动命令组

-- ============================================================================
-- 2. 界面和视觉增强
-- ============================================================================

-- 2.1 窗口焦点高亮功能（已禁用）
-- 原本的窗口焦点切换光标行高亮功能会在文件保存时造成干扰
-- 现在改为始终启用光标行高亮，避免保存文件时背景颜色变化
-- 如需重新启用窗口焦点高亮，请取消下面代码的注释

-- 创建背景高亮自动命令组，clear = true 表示清除同名的旧组
-- augroup("BgHighlight", { clear = true })
--
-- -- 当进入窗口时启用光标行高亮
-- autocmd("WinEnter", {
-- 	callback = function()
-- 		-- 检查当前窗口是否为普通缓冲区，避免在格式化等操作时误触发
-- 		if vim.bo.buftype == "" and vim.fn.mode() == "n" then
-- 			vim.wo.cursorline = true
-- 		end
-- 	end,
-- 	group = "BgHighlight",
-- 	desc = "Enable cursorline in active window",
-- })
--
-- -- 当离开窗口时禁用光标行高亮
-- autocmd("WinLeave", {
-- 	callback = function()
-- 		-- 检查当前窗口是否为普通缓冲区
-- 		if vim.bo.buftype == "" then
-- 			vim.wo.cursorline = false
-- 		end
-- 	end,
-- 	group = "BgHighlight",
-- 	desc = "Disable cursorline in inactive window",
-- })

-- ============================================================================
-- 3. 系统集成和终端增强
-- ============================================================================

-- 3.1 Screen 终端集成设置
-- 当在 GNU Screen 环境中使用 Neovim 时，自动更新窗口标题
-- 检查是否在 Screen 终端环境中运行
if vim.env.TERM and vim.env.TERM:match("screen") then
	-- 当进入缓冲区时更新 Screen 窗口标题
	autocmd("BufEnter", {
		callback = function()
			-- 检查当前缓冲区不是特殊协议的缓冲区（如 http://、ftp:// 等）
			if not vim.fn.bufname(""):match("^[A-Za-z0-9?]*://") then
				-- 设置 Screen 窗口标题为：[主机名:当前目录/当前文件名]
				vim.cmd('silent! !echo -n "\\ek[`hostname`:`basename $PWD`/`basename %`]\\e\\\\"')
			end
		end,
		desc = "Update screen window title with current file info",
	})

	-- 当退出 Vim 时重置 Screen 窗口标题
	autocmd("VimLeave", {
		callback = function()
			-- 重置标题为：[主机名:当前目录]
			vim.cmd('silent! !echo -n "\\ek[`hostname`:`basename $PWD`]\\e\\\\"')
		end,
		desc = "Reset screen window title on exit",
	})
end

-- ============================================================================
-- 4. 样式和主题定制
-- ============================================================================

-- 4.1 注释颜色自定义设置
-- 在 Vim 启动和切换主题时自动应用自定义的注释颜色样式
vim.api.nvim_create_autocmd({ "VimEnter", "ColorScheme" }, {
	callback = function()
		-- 延迟执行，确保主题完全加载后再应用自定义设置
		vim.defer_fn(function()
			-- ============================================================================
			-- Lazy.nvim 界面颜色优化
			-- ============================================================================
			-- 设置 Lazy 界面的高亮组，提高可见性
			vim.cmd("highlight LazyNormal guifg=#ffffff guibg=#1a1a1a gui=NONE")
			vim.cmd("highlight LazyComment guifg=#888888 gui=NONE")
			vim.cmd("highlight LazyCommit guifg=#ffff00 gui=NONE")
			vim.cmd("highlight LazyCommitIssue guifg=#ff6666 gui=NONE")
			vim.cmd("highlight LazyCommitScope guifg=#66ff66 gui=NONE")
			vim.cmd("highlight LazyCommitType guifg=#6666ff gui=NONE")
			vim.cmd("highlight LazyDimmed guifg=#666666 gui=NONE")
			vim.cmd("highlight LazyDir guifg=#66ffff gui=NONE")
			vim.cmd("highlight LazyH1 guifg=#ffffff guibg=#333333 gui=bold")
			vim.cmd("highlight LazyH2 guifg=#ffff00 gui=bold")
			vim.cmd("highlight LazyLocal guifg=#ff9900 gui=NONE")
			vim.cmd("highlight LazyProgressDone guifg=#00ff00 gui=NONE")
			vim.cmd("highlight LazyProgressTodo guifg=#666666 gui=NONE")
			vim.cmd("highlight LazyProp guifg=#99ccff gui=NONE")
			vim.cmd("highlight LazyReasonCmd guifg=#ffcc99 gui=NONE")
			vim.cmd("highlight LazyReasonEvent guifg=#ccffcc gui=NONE")
			vim.cmd("highlight LazyReasonFt guifg=#ffccff gui=NONE")
			vim.cmd("highlight LazyReasonImport guifg=#ccccff gui=NONE")
			vim.cmd("highlight LazyReasonKeys guifg=#ffffcc gui=NONE")
			vim.cmd("highlight LazyReasonPlugin guifg=#ffcccc gui=NONE")
			vim.cmd("highlight LazyReasonRuntime guifg=#ccffff gui=NONE")
			vim.cmd("highlight LazyReasonSource guifg=#ffffff gui=NONE")
			vim.cmd("highlight LazyReasonStart guifg=#00ffff gui=NONE")
			vim.cmd("highlight LazySpecial guifg=#ff00ff gui=NONE")
			vim.cmd("highlight LazyTaskError guifg=#ff0000 gui=bold")
			vim.cmd("highlight LazyTaskOutput guifg=#cccccc gui=NONE")
			vim.cmd("highlight LazyUrl guifg=#0099ff gui=underline")
			vim.cmd("highlight LazyValue guifg=#99ff99 gui=NONE")
			-- 设置 Lazy 浮动窗口背景
			vim.cmd("highlight LazyBackdrop guibg=#000000 gui=NONE")
			vim.cmd("highlight LazyButton guifg=#ffffff guibg=#444444 gui=NONE")
			vim.cmd("highlight LazyButtonActive guifg=#000000 guibg=#ffff00 gui=bold")
			-- ============================================================================
			-- 注释颜色设置（已禁用，保持主题默认颜色）
			-- 设置主要注释高亮组为浅灰色，无特殊样式（不加粗、不倾斜）
			-- Comment: 传统的注释高亮组
			-- vim.cmd("highlight Comment guifg=#8f8f8f gui=NONE")
			-- @comment: Tree-sitter 的注释高亮组（现代语法高亮）
			-- vim.cmd("highlight @comment guifg=#8f8f8f gui=NONE")

			-- 设置其他相关的注释高亮组
			-- CommentURL: 注释中的 URL 链接，保持下划线样式
			-- vim.cmd("highlight CommentURL guifg=#8f8f8f gui=underline")
			-- CommentNote: 注释中的特殊标记（如 TODO、NOTE 等）
			-- vim.cmd("highlight CommentNote guifg=#8f8f8f gui=NONE")

			-- 强制设置光标行背景为固定浅灰色，使用 force 确保不被覆盖
			-- CursorLine: 光标所在行的背景高亮
			vim.cmd("highlight! CursorLine guibg=#3a3a3a ctermbg=237 gui=NONE")
			-- 强制启用光标行高亮
			vim.wo.cursorline = true

			-- 设置行号颜色，让非当前行的行号更亮一些
			-- LineNr: 非当前行的行号颜色
			vim.cmd("highlight LineNr guifg=#888888 gui=NONE")
			-- CursorLineNr: 当前行的行号颜色（保持默认或稍微突出）
			vim.cmd("highlight CursorLineNr guifg=#ffffff gui=bold")

			-- 保持背景透明度设置，防止主题切换后背景变不透明
			-- Normal: 普通文本背景
			vim.cmd("highlight Normal guibg=NONE ctermbg=NONE")
			-- NonText: 非文本区域背景
			vim.cmd("highlight NonText guibg=NONE ctermbg=NONE")
			-- SignColumn: 符号列背景
			vim.cmd("highlight SignColumn guibg=NONE ctermbg=NONE")
			-- EndOfBuffer: 缓冲区末尾背景
			vim.cmd("highlight EndOfBuffer guibg=NONE ctermbg=NONE")

			-- 设置窗口分隔符颜色（编辑器和目录树之间的分界线）
			-- VertSplit: 垂直分隔符（旧版本Neovim）
			vim.cmd("highlight! VertSplit guifg=#ffffff guibg=NONE gui=NONE")
			-- WinSeparator: 窗口分隔符（新版本Neovim 0.8+）
			vim.cmd("highlight! WinSeparator guifg=#ffffff guibg=NONE gui=NONE")
			-- 设置当前活跃buffer的标签样式：白色加粗文字和下划线
			vim.cmd("highlight! TbBufOn guifg=#ffffff gui=bold,underline")
			-- 缩进线颜色由插件配置管理，这里不重复设置
		end, 50) -- 延迟 50 毫秒执行，确保主题切换完成
	end,
	desc = "设置统一的浅灰色注释样式和固定的光标行背景，不随主题变化",
})

-- 4.1.1 光标行颜色强制保护
-- 创建专门的自动命令来强制保持光标行颜色不变
vim.api.nvim_create_autocmd({ "ColorScheme", "BufEnter", "WinEnter", "WinLeave", "FocusGained", "FocusLost", "CmdlineLeave" }, {
	callback = function()
		-- 立即强制设置光标行颜色，不延迟
		vim.cmd("highlight! CursorLine guibg=#3a3a3a ctermbg=237 gui=NONE")
		vim.wo.cursorline = true
		-- 强制设置窗口分隔符颜色
		vim.cmd("highlight! VertSplit guifg=#ffffff guibg=NONE gui=NONE")
		vim.cmd("highlight! WinSeparator guifg=#ffffff guibg=NONE gui=NONE")
		-- 缩进线颜色由插件配置的自动命令管理
		-- 强制设置 Lazy 界面颜色
		vim.cmd("highlight! LazyNormal guifg=#ffffff guibg=#1a1a1a gui=NONE")
		vim.cmd("highlight! LazyH1 guifg=#ffffff guibg=#333333 gui=bold")
		vim.cmd("highlight! LazyH2 guifg=#ffff00 gui=bold")
		vim.cmd("highlight! LazyButtonActive guifg=#66ccff guibg=#333333 gui=bold")
		vim.cmd("highlight! LazyKey guifg=#66ccff gui=bold")
		vim.cmd("highlight! LazyTaskError guifg=#ff0000 gui=bold")
		-- 强制设置当前活跃buffer的标签样式
		vim.cmd("highlight! TbBufOn guifg=#ffffff gui=bold,underline")
		-- 再次延迟执行，确保覆盖任何后续的主题设置
		vim.defer_fn(function()
			vim.cmd("highlight! CursorLine guibg=#3a3a3a ctermbg=237 gui=NONE")
			vim.wo.cursorline = true
			-- 再次强制设置窗口分隔符颜色
			vim.cmd("highlight! VertSplit guifg=#ffffff guibg=NONE gui=NONE")
			vim.cmd("highlight! WinSeparator guifg=#ffffff guibg=NONE gui=NONE")
			-- 再次强制设置当前活跃buffer的标签样式
			vim.cmd("highlight! TbBufOn guifg=#ffffff gui=bold,underline")
		end, 10)
	end,
	desc = "强制保持光标行浅灰色背景，防止被主题覆盖",
})


-- 4.2 格式化后颜色修复（已禁用注释颜色重置）
-- 专门处理格式化操作后的颜色重置问题，但不再强制修改注释颜色
vim.api.nvim_create_autocmd({ "BufWritePost" }, {
	callback = function()
		-- 延迟执行，确保格式化完全完成后再应用设置
		-- 增加延迟时间到 600 毫秒，避免与 conform.nvim 的格式化过程冲突
		vim.defer_fn(function()
			-- 注释：已禁用注释颜色重置，保持原有注释颜色不变
			-- vim.cmd("highlight Comment guifg=#8f8f8f gui=NONE")
			-- vim.cmd("highlight @comment guifg=#8f8f8f gui=NONE")
			-- vim.cmd("highlight CommentURL guifg=#8f8f8f gui=underline")
			-- vim.cmd("highlight CommentNote guifg=#8f8f8f gui=NONE")
			-- 强制重新应用光标行背景色（固定浅灰色）
			vim.cmd("highlight! CursorLine guibg=#3a3a3a ctermbg=237 gui=NONE")
			-- 强制启用光标行高亮
			vim.wo.cursorline = true
			-- 重新应用行号颜色
			vim.cmd("highlight LineNr guifg=#888888 gui=NONE")
			vim.cmd("highlight CursorLineNr guifg=#ffffff gui=bold")
			-- 重新应用透明度设置
			vim.cmd("highlight Normal guibg=NONE ctermbg=NONE")
			vim.cmd("highlight NonText guibg=NONE ctermbg=NONE")
			vim.cmd("highlight SignColumn guibg=NONE ctermbg=NONE")
			vim.cmd("highlight EndOfBuffer guibg=NONE ctermbg=NONE")
			-- 重新应用窗口分隔符颜色
			vim.cmd("highlight! VertSplit guifg=#ffffff guibg=NONE gui=NONE")
			vim.cmd("highlight! WinSeparator guifg=#ffffff guibg=NONE gui=NONE")
			-- 重新应用当前活跃buffer的标签样式
			vim.cmd("highlight! TbBufOn guifg=#ffffff gui=bold,underline")
		end, 600) -- 延迟 600 毫秒执行，避免与格式化冲突
	end,
	desc = "格式化后延迟重新应用光标行和行号颜色设置（不修改注释颜色）",
})

-- ============================================================================
-- 7. 代码格式化自动命令
-- ============================================================================

-- 7.1 保存时自动格式化配置
-- 注释掉这里的自动格式化配置，避免与 plugins/init.lua 中的配置冲突
-- 现在使用 plugins/init.lua 中的 format_on_save 配置
--[[
-- 在 Neovim 完全启动后设置自动格式化
vim.api.nvim_create_autocmd("VimEnter", {
	callback = function()
		-- 延迟执行，确保所有插件都已加载
		vim.defer_fn(function()
			-- 检查 conform.nvim 是否可用
			local ok, conform = pcall(require, "conform")
			if ok then
				-- 创建保存时自动格式化命令
				vim.api.nvim_create_autocmd("BufWritePre", {
					pattern = "*",
					callback = function()
						conform.format({ async = false, lsp_fallback = true })
					end,
					desc = "Format file on save using conform.nvim",
				})				
			else
				vim.notify("Conform.nvim not available for auto-format", vim.log.levels.WARN)
			end
		end, 100) -- 延迟 100ms
	end,
	desc = "Setup auto-format on save",
})
--]]

-- ============================================================================
-- 8. 文件模板系统初始化
-- ============================================================================

-- 在 Neovim 完全启动后初始化文件模板系统和HTML预览系统
autocmd("VimEnter", {
	callback = function()
		-- 延迟加载模板系统，避免循环依赖
		vim.defer_fn(function()
			-- 初始化文件模板系统
			local ok, templates = pcall(require, "configs.templates")
			if ok and templates.setup then
				templates.setup({
					-- 自定义作者信息
					author = "Your Name",
					email = "your.email@example.com",
					-- 生成模板后自动进入插入模式
					auto_insert = true,
					-- 显示模板选择菜单
					show_menu = true,
				})
			else
				vim.notify("模板模块加载失败", vim.log.levels.WARN)
			end

			-- 初始化HTML预览系统
			local ok_preview, html_preview = pcall(require, "configs.html-preview")
			if ok_preview and html_preview.setup then
				html_preview.setup({
					-- 默认端口
					port = 8080,
					-- 自动打开浏览器
					auto_open = true,
				})
			else
				vim.notify("HTML预览模块加载失败", vim.log.levels.WARN)
			end


		end, 100) -- 延迟 100ms
	end,
	desc = "Initialize file template system and HTML preview system",
})

-- ============================================================================
-- 9. 文件类型和语法高亮修复
-- ============================================================================

-- 9.1 修复 .gitignore 和 .dockerignore 文件的语法高亮显示
-- 问题：.gitignore 和 .dockerignore 文件在某些配色主题下显示为红色文字
-- 解决：强制禁用语法高亮，使用普通文本显示
autocmd({ "FileType", "BufRead", "BufNewFile", "BufEnter" }, {
	pattern = { "*gitignore*", ".gitignore", "*dockerignore*", ".dockerignore" },
	callback = function()
		-- 获取当前缓冲区
		local bufnr = vim.api.nvim_get_current_buf()
		local filename = vim.api.nvim_buf_get_name(bufnr)
		
		-- 处理 .gitignore 和 .dockerignore 文件
		if filename:match("%.gitignore$") or filename:match("/%.gitignore$") or 
		   filename:match("%.dockerignore$") or filename:match("/%.dockerignore$") then
			-- 方法1：完全禁用语法高亮，使用普通文本颜色
			vim.bo.syntax = "off"
			vim.bo.filetype = ""
			
			-- 方法2：强制重置所有可能的高亮组为正常颜色
			vim.defer_fn(function()
				-- 清除所有语法高亮
				vim.cmd("syntax clear")
				
				-- 强制设置所有文本为正常颜色
				vim.cmd("highlight! default Normal guifg=#ffffff guibg=NONE")
				vim.cmd("syntax match IgnoreText '.*'")
				vim.cmd("highlight! link IgnoreText Normal")
				
				-- 重新应用到当前缓冲区
				vim.cmd("redraw!")
			end, 50)
		end
	end,
	desc = "强制修复 .gitignore 和 .dockerignore 文件的红色文字显示问题",
})

-- 9.2 特殊文件类型颜色修复
-- 为特定文件类型设置正确的语法高亮，避免颜色显示异常
autocmd("FileType", {
	pattern = { "gitcommit", "gitconfig", "gitrebase" },
	callback = function()
		-- Git 相关文件的颜色修复
		vim.cmd("highlight! link GitCommitSummary Normal")
		vim.cmd("highlight! link GitCommitComment Comment") 
	end,
	desc = "修复 Git 相关文件的语法高亮颜色",
})

-- ============================================================================
-- 结束
-- ============================================================================
