-- ============================================================================
-- 主题切换配置
-- 提供便捷的主题切换功能
-- ============================================================================

local M = {}

-- 主题切换函数
function M.switch_theme(flavour)
    flavour = flavour or "mocha"

    -- 验证主题名称
    local valid_themes = {"mocha", "frappe", "macchiato", "latte"}
    local is_valid = false
    for _, theme in ipairs(valid_themes) do
        if theme == flavour then
            is_valid = true
            break
        end
    end

    if not is_valid then
        vim.notify("无效的主题: " .. flavour .. "\n可用主题: mocha, frappe, macchiato, latte", vim.log.levels.ERROR)
        return
    end

    -- 设置新主题
    require('catppuccin').setup({
        flavour = flavour,
        background = {
            light = "latte",
            dark = "mocha",
        },
        transparent_background = false,
        show_end_of_buffer = false,
        term_colors = false,
        dim_inactive = {
            enabled = false,
            shade = "dark",
            percentage = 0.15,
        },
        styles = {
            comments = { "italic" },
            conditionals = { "italic" },
            loops = {},
            functions = {},
            keywords = {},
            strings = {},
            variables = {},
            numbers = {},
            booleans = {},
            properties = {},
            types = {},
            operators = {},
        },
        color_overrides = {},
        custom_highlights = {},
        integrations = {
            cmp = true,
            gitsigns = true,
            nvimtree = true,
            treesitter = true,
            notify = false,
            mini = false,
            telescope = true,
        },
    })

    -- 应用主题
    vim.cmd('colorscheme catppuccin')

    -- 重新应用透明背景设置
    vim.defer_fn(function()
        local ok, highlights = pcall(require, 'config.highlights')
        if ok and highlights.setup_all_highlights then
            highlights.setup_all_highlights()
        end
    end, 100)

    -- 保存当前主题到全局变量
    vim.g.current_catppuccin_flavour = flavour

    print("主题已切换到: " .. flavour)
end

-- 循环切换主题
function M.cycle_theme()
    local themes = {"mocha", "frappe", "macchiato", "latte"}
    local current_flavour = vim.g.current_catppuccin_flavour or "mocha"

    local next_index = 1
    for i, theme in ipairs(themes) do
        if theme == current_flavour then
            next_index = (i % #themes) + 1
            break
        end
    end

    local next_theme = themes[next_index]
    M.switch_theme(next_theme)
end

-- 显示当前主题信息
function M.show_current_theme()
    local current_flavour = vim.g.current_catppuccin_flavour or "mocha"
    local theme_info = {
        mocha = "🌙 Mocha - 深邃的深色主题",
        frappe = "☕ Frappe - 明亮的深色主题",
        macchiato = "🎨 Macchiato - 柔和的深色主题",
        latte = "☀️ Latte - 清新的浅色主题"
    }

    local info = theme_info[current_flavour] or "未知主题"
    vim.notify("当前主题: " .. info, vim.log.levels.INFO)
end

-- 可用的主题列表
M.themes = {
    "mocha",
    "frappe",
    "macchiato",
    "latte"
}

return M