#!/bin/bash

# {{FILENAME}}
# {{DESCRIPTION}}
#
# Author: {{AUTHOR}} <{{EMAIL}}>
# Created: {{DATE}}
# Version: 1.0.0
#
# Usage: ./{{FILENAME}} [options] [arguments]
# Example: ./{{FILENAME}} --help

set -euo pipefail  # 严格模式：遇到错误退出，未定义变量报错，管道失败时退出

# =============================================================================
# 全局变量
# =============================================================================

readonly SCRIPT_NAME="$(basename "$0")"
readonly SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
readonly SCRIPT_PATH="${SCRIPT_DIR}/${SCRIPT_NAME}"
readonly PID_FILE="/tmp/${SCRIPT_NAME}.pid"
readonly LOG_FILE="/tmp/${SCRIPT_NAME}.log"

# 颜色定义
readonly RED='\033[0;31m'
readonly GREEN='\033[0;32m'
readonly YELLOW='\033[1;33m'
readonly BLUE='\033[0;34m'
readonly PURPLE='\033[0;35m'
readonly CYAN='\033[0;36m'
readonly WHITE='\033[1;37m'
readonly NC='\033[0m' # No Color

# 默认配置
DEBUG=${DEBUG:-false}
VERBOSE=${VERBOSE:-false}
DRY_RUN=${DRY_RUN:-false}

# =============================================================================
# 工具函数
# =============================================================================

# 日志函数
log() {
    local level="$1"
    shift
    local message="$*"
    local timestamp=$(date '+%Y-%m-%d %H:%M:%S')
    
    case "$level" in
        "ERROR")
            echo -e "${RED}[${timestamp}] [ERROR] ${message}${NC}" >&2
            echo "[${timestamp}] [ERROR] ${message}" >> "$LOG_FILE"
            ;;
        "WARN")
            echo -e "${YELLOW}[${timestamp}] [WARN] ${message}${NC}" >&2
            echo "[${timestamp}] [WARN] ${message}" >> "$LOG_FILE"
            ;;
        "INFO")
            echo -e "${GREEN}[${timestamp}] [INFO] ${message}${NC}"
            echo "[${timestamp}] [INFO] ${message}" >> "$LOG_FILE"
            ;;
        "DEBUG")
            if [[ "$DEBUG" == "true" ]]; then
                echo -e "${BLUE}[${timestamp}] [DEBUG] ${message}${NC}"
                echo "[${timestamp}] [DEBUG] ${message}" >> "$LOG_FILE"
            fi
            ;;
        *)
            echo -e "${WHITE}[${timestamp}] ${message}${NC}"
            echo "[${timestamp}] ${message}" >> "$LOG_FILE"
            ;;
    esac
}

# 错误处理函数
error_exit() {
    log "ERROR" "$1"
    cleanup
    exit "${2:-1}"
}

# 清理函数
cleanup() {
    log "DEBUG" "执行清理操作..."
    
    # 删除PID文件
    if [[ -f "$PID_FILE" ]]; then
        rm -f "$PID_FILE"
    fi
    
    # 其他清理操作
    # ...
}

# 检查依赖
check_dependencies() {
    local deps=("curl" "jq" "git")
    
    for dep in "${deps[@]}"; do
        if ! command -v "$dep" &> /dev/null; then
            error_exit "依赖 '$dep' 未安装，请先安装后再运行此脚本"
        fi
    done
    
    log "DEBUG" "所有依赖检查通过"
}

# 检查权限
check_permissions() {
    if [[ $EUID -eq 0 ]]; then
        log "WARN" "正在以root用户运行脚本"
    fi
    
    # 检查文件权限
    if [[ ! -w "$(dirname "$LOG_FILE")" ]]; then
        error_exit "无法写入日志文件: $LOG_FILE"
    fi
}

# 创建PID文件
create_pid_file() {
    if [[ -f "$PID_FILE" ]]; then
        local old_pid
        old_pid=$(cat "$PID_FILE")
        if kill -0 "$old_pid" 2>/dev/null; then
            error_exit "脚本已在运行 (PID: $old_pid)"
        else
            log "WARN" "发现过期的PID文件，正在清理..."
            rm -f "$PID_FILE"
        fi
    fi
    
    echo $$ > "$PID_FILE"
    log "DEBUG" "创建PID文件: $PID_FILE (PID: $$)"
}

# 显示帮助信息
show_help() {
    cat << EOF
使用方法: $SCRIPT_NAME [选项] [参数]

{{DESCRIPTION}}

选项:
    -h, --help          显示此帮助信息
    -v, --version       显示版本信息
    -d, --debug         启用调试模式
    -V, --verbose       启用详细输出
    -n, --dry-run       模拟运行（不执行实际操作）
    -c, --config FILE   指定配置文件
    -o, --output DIR    指定输出目录
    -f, --force         强制执行
    -q, --quiet         静默模式

参数:
    INPUT_FILE          输入文件路径
    OUTPUT_FILE         输出文件路径

示例:
    $SCRIPT_NAME --debug input.txt output.txt
    $SCRIPT_NAME --config config.ini --output /tmp
    $SCRIPT_NAME --dry-run --verbose

作者: {{AUTHOR}} <{{EMAIL}}>
版本: 1.0.0
EOF
}

# 显示版本信息
show_version() {
    echo "$SCRIPT_NAME 1.0.0"
    echo "作者: {{AUTHOR}} <{{EMAIL}}>"
    echo "创建时间: {{DATE}}"
}

# 验证输入参数
validate_input() {
    local input_file="$1"
    
    if [[ -z "$input_file" ]]; then
        error_exit "请提供输入文件"
    fi
    
    if [[ ! -f "$input_file" ]]; then
        error_exit "输入文件不存在: $input_file"
    fi
    
    if [[ ! -r "$input_file" ]]; then
        error_exit "无法读取输入文件: $input_file"
    fi
    
    log "DEBUG" "输入验证通过: $input_file"
}

# =============================================================================
# 主要功能函数
# =============================================================================

# 处理文件
process_file() {
    local input_file="$1"
    local output_file="$2"
    
    log "INFO" "开始处理文件: $input_file"
    
    if [[ "$DRY_RUN" == "true" ]]; then
        log "INFO" "[模拟] 处理文件: $input_file -> $output_file"
        return 0
    fi
    
    # 实际处理逻辑
    if cp "$input_file" "$output_file"; then
        log "INFO" "文件处理完成: $output_file"
    else
        error_exit "文件处理失败"
    fi
}

# 执行主要任务
execute_main_task() {
    local input_file="$1"
    local output_file="$2"
    
    log "INFO" "开始执行主要任务..."
    
    # 验证输入
    validate_input "$input_file"
    
    # 处理文件
    process_file "$input_file" "$output_file"
    
    log "INFO" "主要任务执行完成"
}

# =============================================================================
# 信号处理
# =============================================================================

# 信号处理函数
signal_handler() {
    local signal="$1"
    log "WARN" "接收到信号: $signal"
    log "INFO" "正在优雅退出..."
    cleanup
    exit 130
}

# 注册信号处理
trap 'signal_handler SIGINT' SIGINT
trap 'signal_handler SIGTERM' SIGTERM
trap 'cleanup' EXIT

# =============================================================================
# 主函数
# =============================================================================

main() {
    local input_file=""
    local output_file=""
    local config_file=""
    local output_dir=""
    local force=false
    local quiet=false
    
    # 解析命令行参数
    while [[ $# -gt 0 ]]; do
        case $1 in
            -h|--help)
                show_help
                exit 0
                ;;
            -v|--version)
                show_version
                exit 0
                ;;
            -d|--debug)
                DEBUG=true
                shift
                ;;
            -V|--verbose)
                VERBOSE=true
                shift
                ;;
            -n|--dry-run)
                DRY_RUN=true
                shift
                ;;
            -c|--config)
                config_file="$2"
                shift 2
                ;;
            -o|--output)
                output_dir="$2"
                shift 2
                ;;
            -f|--force)
                force=true
                shift
                ;;
            -q|--quiet)
                quiet=true
                shift
                ;;
            -*)
                error_exit "未知选项: $1"
                ;;
            *)
                if [[ -z "$input_file" ]]; then
                    input_file="$1"
                elif [[ -z "$output_file" ]]; then
                    output_file="$1"
                else
                    error_exit "过多的参数: $1"
                fi
                shift
                ;;
        esac
    done
    
    # 静默模式处理
    if [[ "$quiet" == "true" ]]; then
        exec 1>/dev/null
    fi
    
    # 初始化
    log "INFO" "脚本开始执行: $SCRIPT_NAME"
    log "DEBUG" "脚本路径: $SCRIPT_PATH"
    log "DEBUG" "工作目录: $(pwd)"
    
    # 检查依赖和权限
    check_dependencies
    check_permissions
    
    # 创建PID文件
    create_pid_file
    
    # 设置默认值
    if [[ -z "$output_file" && -n "$input_file" ]]; then
        output_file="${input_file}.processed"
    fi
    
    if [[ -n "$output_dir" && -n "$output_file" ]]; then
        output_file="${output_dir}/$(basename "$output_file")"
    fi
    
    # 执行主要任务
    if [[ -n "$input_file" ]]; then
        execute_main_task "$input_file" "$output_file"
    else
        log "WARN" "未提供输入文件，显示帮助信息"
        show_help
        exit 1
    fi
    
    log "INFO" "脚本执行完成"
}

# =============================================================================
# 脚本入口
# =============================================================================

# 只有在直接执行脚本时才运行main函数
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    main "$@"
fi