-- ============================================================================
-- 自定义高亮配置
-- 从原 NvChad 配置迁移的高亮设置
-- ============================================================================

local M = {}

-- ============================================================================
-- Git 相关高亮增强
-- ============================================================================

function M.set_nvim_tree_highlights()
	-- Git 状态颜色
	local highlights = {
		NvimTreeGitDirty = { fg = "#ff9e64", bold = true },
		NvimTreeGitStaged = { fg = "#7aa2f7", bold = true },
		NvimTreeGitMerge = { fg = "#bb9af7", bold = true },
		NvimTreeGitRenamed = { fg = "#e0af68", bold = true },
		NvimTreeGitNew = { fg = "#9ece6a", bold = true },
		NvimTreeGitDeleted = { fg = "#f7768e", bold = true },
		NvimTreeGitIgnored = { fg = "#565f89", italic = true },

		-- 文件类型颜色增强
		NvimTreeFolderName = { fg = "#7aa2f7", bold = true },
		NvimTreeOpenedFolderName = { fg = "#7aa2f7", bold = true, italic = true },
		NvimTreeEmptyFolderName = { fg = "#565f89", italic = true },
		NvimTreeFolderIcon = { fg = "#7aa2f7" },

		-- 文件图标颜色
		NvimTreeFileIcon = { fg = "#c0caf5" },
		NvimTreeExecFile = { fg = "#9ece6a", bold = true },
		NvimTreeSpecialFile = { fg = "#bb9af7", bold = true, underline = true },

		-- 高亮当前文件
		NvimTreeImageFile = { fg = "#e0af68" },
		NvimTreeIndentMarker = { fg = "#414868" },
		NvimTreeSymlink = { fg = "#bb9af7" },
		NvimTreeSymlinkIcon = { fg = "#bb9af7" },

		-- 光标行（透明背景）
		NvimTreeCursorLine = { bg = "NONE", reverse = true },
		NvimTreeCursorColumn = { bg = "NONE" },

		-- 边框和线条（透明背景）
		NvimTreeVertSplit = { fg = "#414868", bg = "NONE" },
		NvimTreeEndOfBuffer = { fg = "#414868", bg = "NONE" },

		-- 书签和标记
		NvimTreeBookmark = { fg = "#e0af68", bold = true },
		NvimTreeModifiedIcon = { fg = "#f7768e", bold = true },
	}

	-- 应用高亮
	for hl_name, hl_config in pairs(highlights) do
		vim.api.nvim_set_hl(0, hl_name, hl_config)
	end
end

-- 终极 Git 颜色修复函数
function M.ultimate_git_color_fix()
	local nvim_tree_api = require("nvim-tree.api")

	-- 强制刷新 Git 状态
	if nvim_tree_api and nvim_tree_api.git then
		nvim_tree_api.git.reload()
	end

	-- 等待一小段时间后再次刷新
	vim.defer_fn(function()
		if nvim_tree_api and nvim_tree_api.tree then
			nvim_tree_api.tree.reload()
		end

		-- 重新应用高亮
		M.set_nvim_tree_highlights()

		-- 强制重绘
		vim.cmd("redraw!")
	end, 100)
end

-- 强制刷新 NvimTree
function M.force_refresh_nvimtree()
	local nvim_tree_api = require("nvim-tree.api")

	if nvim_tree_api and nvim_tree_api.tree then
		nvim_tree_api.tree.reload()
	end

	-- 刷新 Git 状态
	if nvim_tree_api and nvim_tree_api.git then
		nvim_tree_api.git.reload()
	end

	-- 应用颜色修复
	M.ultimate_git_color_fix()
end

-- ============================================================================
-- 全局透明背景设置
-- ============================================================================

function M.setup_transparency_background()
	-- 完全透明的全局设置
	local transparent_highlights = {
		-- 基础编辑器背景
		"Normal", "NormalNC", "NormalFloat",
		"StatusLine", "StatusLineNC",
		"TabLine", "TabLineFill", "TabLineSel",
		"WinSeparator", "VertSplit",
		"SignColumn", "FoldColumn",
		"LineNr", "LineNrAbove", "LineNrBelow",
		"CursorLine", "CursorLineNr",

		-- 侧边栏和文件树
		"NvimTreeNormal", "NvimTreeNormalNC",
		"NvimTreeCursorLine", "NvimTreeCursorColumn",
		"NvimTreeVertSplit", "NvimTreeEndOfBuffer",
		"SidebarNvimNormal",

		-- 浮动窗口和弹出菜单
		"FloatBorder", "FloatTitle",
		"Pmenu", "PmenuSel", "PmenuSbar", "PmenuThumb",
		"DiagnosticFloat", "DiagnosticInfo",

		-- 终端
		"NormalFloat",

		-- 其他UI元素
		"EndOfBuffer", "NonText",
		"Whitespace", "SpaceChar", "SpaceChar2",
		"TabLineFill", "TabLine",
		"WinBar", "WinBarNC",
	}

	-- 应用所有透明高亮
	for _, hl in ipairs(transparent_highlights) do
		vim.api.nvim_set_hl(0, hl, { bg = "NONE" })
	end

	-- 特殊处理：确保文字颜色正确
	local text_highlights = {
		{ "Normal", { fg = "#c0caf5" } },
		{ "NormalNC", { fg = "#c0caf5" } },
		{ "NormalFloat", { fg = "#c0caf5", bg = "#1f2335" } },
		{ "StatusLine", { fg = "#c0caf5" } },
		{ "StatusLineNC", { fg = "#565f89" } },
		{ "LineNr", { fg = "#565f89" } },
		{ "CursorLineNr", { fg = "#7aa2f7" } },
		{ "WinSeparator", { fg = "#414868" } },
		{ "SignColumn", { fg = "#414868" } },
		{ "FoldColumn", { fg = "#414868" } },
		{ "NvimTreeNormal", { fg = "#c0caf5" } },
		{ "NvimTreeNormalNC", { fg = "#c0caf5" } },
		{ "NvimTreeVertSplit", { fg = "#414868" } },
		{ "NvimTreeEndOfBuffer", { fg = "#414868" } },
	}

	-- 应用文字颜色设置
	for _, hl_config in ipairs(text_highlights) do
		local hl_name, config = hl_config[1], hl_config[2]
		local current = vim.api.nvim_get_hl(0, { name = hl_name })
		vim.api.nvim_set_hl(0, hl_name, vim.tbl_extend("force", current, config))
	end

	-- 设置内联提示透明背景
	vim.api.nvim_set_hl(0, "LspInlayHint", { bg = "NONE", fg = "#565f89", italic = true })

	-- 设置补全菜单半透明（不是完全透明，为了可读性）
	vim.api.nvim_set_hl(0, "Pmenu", { bg = "#1f2335", fg = "#c0caf5" })
	vim.api.nvim_set_hl(0, "PmenuSel", { bg = "#33467c", fg = "#c0caf5" })
	vim.api.nvim_set_hl(0, "PmenuSbar", { bg = "#1f2335" })
	vim.api.nvim_set_hl(0, "PmenuThumb", { bg = "#414868" })

	-- 设置诊断信息半透明
	vim.api.nvim_set_hl(0, "DiagnosticFloat", { bg = "#1f2335", fg = "#c0caf5" })
end

-- ============================================================================
-- 其他自定义高亮
-- ============================================================================

function M.setup_custom_highlights()
	-- 搜索高亮
	vim.api.nvim_set_hl(0, "IncSearch", { fg = "#1d202f", bg = "#e0af68", bold = true })
	vim.api.nvim_set_hl(0, "Search", { fg = "#1d202f", bg = "#7aa2f7", bold = true })

	-- 可视模式选择
	vim.api.nvim_set_hl(0, "Visual", { bg = "#33467c" })

	-- 当前行高亮
	vim.api.nvim_set_hl(0, "CursorLine", { bg = "#292e42" })
	vim.api.nvim_set_hl(0, "CursorLineNr", { fg = "#7aa2f7", bold = true })

	-- 行号高亮
	vim.api.nvim_set_hl(0, "LineNr", { fg = "#565f89" })
	vim.api.nvim_set_hl(0, "LineNrAbove", { fg = "#414868" })
	vim.api.nvim_set_hl(0, "LineNrBelow", { fg = "#414868" })

	-- 折叠高亮
	vim.api.nvim_set_hl(0, "Folded", { fg = "#565f89", bg = "#1f2335", italic = true })
	vim.api.nvim_set_hl(0, "FoldColumn", { fg = "#414868", bg = "#1d202f" })

	-- 标记符号高亮
	vim.api.nvim_set_hl(0, "SignColumn", { fg = "#414868", bg = "#1d202f" })

	-- 状态栏高亮
	vim.api.nvim_set_hl(0, "StatusLine", { fg = "#c0caf5", bg = "#1f2335" })
	vim.api.nvim_set_hl(0, "StatusLineNC", { fg = "#565f89", bg = "#1f2335" })

	-- 诊断高亮增强
	vim.api.nvim_set_hl(0, "DiagnosticError", { fg = "#f7768e", bold = true })
	vim.api.nvim_set_hl(0, "DiagnosticWarn", { fg = "#e0af68", bold = true })
	vim.api.nvim_set_hl(0, "DiagnosticInfo", { fg = "#7aa2f7", bold = true })
	vim.api.nvim_set_hl(0, "DiagnosticHint", { fg = "#9ece6a", bold = true })

	-- 浮动窗口高亮
	vim.api.nvim_set_hl(0, "NormalFloat", { fg = "#c0caf5", bg = "#1f2335" })
	vim.api.nvim_set_hl(0, "FloatBorder", { fg = "#414868", bg = "#1f2335" })

	-- 补全菜单高亮
	vim.api.nvim_set_hl(0, "CmpItemAbbr", { fg = "#c0caf5" })
	vim.api.nvim_set_hl(0, "CmpItemAbbrMatch", { fg = "#7aa2f7", bold = true })
	vim.api.nvim_set_hl(0, "CmpItemAbbrMatchFuzzy", { fg = "#7aa2f7", bold = true })

	-- 补全菜单分类高亮
	vim.api.nvim_set_hl(0, "CmpItemKind", { fg = "#bb9af7" })
	vim.api.nvim_set_hl(0, "CmpItemKindFunction", { fg = "#7aa2f7" })
	vim.api.nvim_set_hl(0, "CmpItemKindVariable", { fg = "#9ece6a" })
	vim.api.nvim_set_hl(0, "CmpItemKindClass", { fg = "#e0af68" })
	vim.api.nvim_set_hl(0, "CmpItemKindMethod", { fg = "#7aa2f7" })
	vim.api.nvim_set_hl(0, "CmpItemKindProperty", { fg = "#9ece6a" })
	vim.api.nvim_set_hl(0, "CmpItemKindUnit", { fg = "#f7768e" })
	vim.api.nvim_set_hl(0, "CmpItemKindKeyword", { fg = "#bb9af7" })
	vim.api.nvim_set_hl(0, "CmpItemKindSnippet", { fg = "#e0af68" })
	vim.api.nvim_set_hl(0, "CmpItemKindColor", { fg = "#f7768e" })
	vim.api.nvim_set_hl(0, "CmpItemKindFile", { fg = "#7aa2f7" })
	vim.api.nvim_set_hl(0, "CmpItemKindReference", { fg = "#bb9af7" })
	vim.api.nvim_set_hl(0, "CmpItemKindFolder", { fg = "#7aa2f7" })
	vim.api.nvim_set_hl(0, "CmpItemKindEnumMember", { fg = "#9ece6a" })
	vim.api.nvim_set_hl(0, "CmpItemKindConstant", { fg = "#f7768e" })
	vim.api.nvim_set_hl(0, "CmpItemKindStruct", { fg = "#e0af68" })
	vim.api.nvim_set_hl(0, "CmpItemKindEvent", { fg = "#bb9af7" })
	vim.api.nvim_set_hl(0, "CmpItemKindOperator", { fg = "#f7768e" })
	vim.api.nvim_set_hl(0, "CmpItemKindTypeParameter", { fg = "#e0af68" })

	-- 语法高亮增强
	vim.api.nvim_set_hl(0, "Comment", { fg = "#565f89", italic = true })
	vim.api.nvim_set_hl(0, "String", { fg = "#9ece6a" })
	vim.api.nvim_set_hl(0, "Number", { fg = "#f7768e" })
	vim.api.nvim_set_hl(0, "Boolean", { fg = "#f7768e", bold = true })
	vim.api.nvim_set_hl(0, "Identifier", { fg = "#c0caf5" })
	vim.api.nvim_set_hl(0, "Function", { fg = "#7aa2f7" })
	vim.api.nvim_set_hl(0, "Statement", { fg = "#bb9af7" })
	vim.api.nvim_set_hl(0, "Keyword", { fg = "#bb9af7", bold = true })
	vim.api.nvim_set_hl(0, "Conditional", { fg = "#bb9af7" })
	vim.api.nvim_set_hl(0, "Repeat", { fg = "#bb9af7" })
	vim.api.nvim_set_hl(0, "Operator", { fg = "#89ddff" })
	vim.api.nvim_set_hl(0, "PreProc", { fg = "#e0af68" })
	vim.api.nvim_set_hl(0, "Type", { fg = "#e0af68" })
	vim.api.nvim_set_hl(0, "Special", { fg = "#f7768e" })
	vim.api.nvim_set_hl(0, "Error", { fg = "#f7768e", bold = true, undercurl = true })
	vim.api.nvim_set_hl(0, "Todo", { fg = "#e0af68", bold = true, italic = true })

	-- NvimTree 透明背景设置 (参考 LazyVim)
	vim.api.nvim_set_hl(0, "NvimTreeNormal", { bg = "NONE", fg = "#c0caf5" })
	vim.api.nvim_set_hl(0, "NvimTreeNormalNC", { bg = "NONE", fg = "#c0caf5" })
	vim.api.nvim_set_hl(0, "NvimTreeRootFolder", { fg = "#7aa2f7", bg = "NONE", bold = true })
	vim.api.nvim_set_hl(0, "NvimTreeFolderIcon", { fg = "#7aa2f7", bg = "NONE" })
	vim.api.nvim_set_hl(0, "NvimTreeFileIcon", { fg = "#c0caf5", bg = "NONE" })

	-- 其他 NvimTree 元素的透明背景
	vim.api.nvim_set_hl(0, "NvimTreeOpenedFolderName", { fg = "#7aa2f7", bg = "NONE", bold = true })
	vim.api.nvim_set_hl(0, "NvimTreeEmptyFolderName", { fg = "#565f89", bg = "NONE", italic = true })
	vim.api.nvim_set_hl(0, "NvimTreeImageFile", { fg = "#e0af68", bg = "NONE" })
	vim.api.nvim_set_hl(0, "NvimTreeSymlink", { fg = "#bb9af7", bg = "NONE" })
	vim.api.nvim_set_hl(0, "NvimTreeSymlinkIcon", { fg = "#bb9af7", bg = "NONE" })
end

-- ============================================================================
-- 主设置函数
-- ============================================================================

-- 统一的高亮设置函数，避免重复调用导致的闪烁
function M.setup_all_highlights()
	-- 1. 首先设置全局透明背景
	M.setup_transparency_background()

	-- 2. 然后设置自定义高亮（会覆盖一些透明设置）
	M.setup_custom_highlights()

	-- 3. 最后设置 NvimTree 特定高亮
	M.set_nvim_tree_highlights()

	-- 4. 应用 Git 颜色修复
	M.ultimate_git_color_fix()
end

function M.setup()
	-- 优化的单次高亮设置
	vim.defer_fn(function()
		-- 合并所有高亮设置到一个函数中，避免闪烁
		M.setup_all_highlights()
	end, 100)

	-- 创建自动命令来重新应用高亮
	vim.api.nvim_create_augroup("CustomHighlights", { clear = true })

	-- 颜色主题切换时重新应用高亮
	vim.api.nvim_create_autocmd("ColorScheme", {
		group = "CustomHighlights",
		callback = function()
			vim.defer_fn(function()
				M.setup_all_highlights()
			end, 50)
		end,
	})

	-- NvimTree 打开时重新应用高亮（减少延迟避免闪烁）
	vim.api.nvim_create_autocmd("FileType", {
		group = "CustomHighlights",
		pattern = "NvimTree",
		callback = function()
			vim.defer_fn(function()
				M.setup_all_highlights()
			end, 25) -- 减少延迟时间
		end,
	})
end

return M