-- ============================================================================
-- LazyVim 插件配置入口
-- 从原 NvChad 配置迁移的插件管理设置
-- ============================================================================

-- 设置插件加载的优先级和配置
local lazypath = vim.fn.stdpath("data") .. "/lazy/lazy.nvim"

return {
	-- =============================================================================
	-- LazyVim 插件（不使用核心，只使用插件）
	-- =============================================================================
	-- 注意：由于 Neovim 版本限制，我们将手动配置 LazyVim 风格的插件设置
	-- 而不直接依赖 LazyVim 核心框架

	-- =============================================================================
	-- 颜色主题
	-- =============================================================================
	{
		"catppuccin/nvim",
		name = "catppuccin",
		priority = 1000,
		lazy = false,
		config = function()
			require("catppuccin").setup({
				flavour = "mocha", -- latte, frappe, macchiato, mocha
				background = {
					light = "latte",
					dark = "mocha",
				},
				transparent_background = false,
				show_end_of_buffer = false,
				term_colors = false,
				dim_inactive = {
					enabled = false,
					shade = "dark",
					percentage = 0.15,
				},
				no_italic = false,
				no_bold = false,
				no_underline = false,
				styles = {
					comments = { "italic" },
					conditionals = { "italic" },
					loops = {},
					functions = {},
					keywords = {},
					strings = {},
					variables = {},
					numbers = {},
					booleans = {},
					properties = {},
					types = {},
					operators = {},
				},
				color_overrides = {},
				custom_highlights = {},
				integrations = {
					cmp = true,
					gitsigns = true,
					nvimtree = true,
					treesitter = true,
					notify = false,
					mini = false,
					telescope = true,
				},
			})

			-- 加载主题
			vim.cmd("colorscheme catppuccin")
		end,
	},

	-- =============================================================================
	-- Treesitter - 语法高亮
	-- =============================================================================
	{
		"nvim-treesitter/nvim-treesitter",
		version = false,
		build = ":TSUpdate",
		event = { "BufReadPost", "BufNewFile" },
		dependencies = {
			"nvim-treesitter/nvim-treesitter-textobjects",
		},
		config = function()
			require("nvim-treesitter.configs").setup({
				ensure_installed = {
					"bash",
					"c",
					"cpp",
					"css",
					"html",
					"javascript",
					"json",
					"lua",
					"luadoc",
					"markdown",
					"markdown_inline",
					"python",
					"query",
					"regex",
					"tsx",
					"typescript",
					"vim",
					"vimdoc",
					"yaml",
					"vue",
					"svelte",
					"scss",
					"php",
					"dockerfile",
					"gitignore",
					"prisma",
					"rust",
					"go",
				},
				auto_install = true,
				highlight = {
					enable = true,
					additional_vim_regex_highlighting = false,
				},
				indent = {
					enable = true,
				},
				incremental_selection = {
					enable = true,
					keymaps = {
						init_selection = "<C-space>",
						node_incremental = "<C-space>",
						scope_incremental = false,
						node_decremental = "<bs>",
					},
				},
				textobjects = {
					select = {
						enable = true,
						lookahead = true,
						keymaps = {
							["aa"] = "@parameter.outer",
							["ia"] = "@parameter.inner",
							["af"] = "@function.outer",
							["if"] = "@function.inner",
							["ac"] = "@class.outer",
							["ic"] = "@class.inner",
						},
					},
					move = {
						enable = true,
						set_jumps = true,
						goto_next_start = {
							["]f"] = "@function.outer",
							["]c"] = "@class.outer",
						},
						goto_next_end = {
							["]F"] = "@function.outer",
							["]C"] = "@class.outer",
						},
						goto_previous_start = {
							["[f"] = "@function.outer",
							["[c"] = "@class.outer",
						},
						goto_previous_end = {
							["[F"] = "@function.outer",
							["[C"] = "@class.outer",
						},
					},
					swap = {
						enable = true,
						swap_next = {
							["<leader>a"] = "@parameter.inner",
						},
						swap_previous = {
							["<leader>A"] = "@parameter.inner",
						},
					},
				},
			})
		end,
	},

	-- =============================================================================
	-- 文件管理器 - NvimTree
	-- =============================================================================
	{
		"nvim-tree/nvim-tree.lua",
		lazy = false,
		dependencies = {
			"nvim-tree/nvim-web-devicons",
		},
		config = function()
			require("config.nvimtree").setup()
		end,
	},

	-- =============================================================================
	-- LSP 配置 (保持 lspconfig 依赖以兼容其他插件)
	-- =============================================================================
	{
		"neovim/nvim-lspconfig",
		event = { "BufReadPre", "BufNewFile" },
		dependencies = {
			{ "folke/neodev.nvim", config = true },
			"mason.nvim",
			"williamboman/mason-lspconfig.nvim",
			"hrsh7th/cmp-nvim-lsp",
		},
		config = function()
			-- LSP 配置在 config/lsp.lua 中设置
			vim.defer_fn(function()
				require("config.lsp").setup()
			end, 100)
		end,
	},

	-- =============================================================================
	-- 代码补全
	-- =============================================================================
	{
		"hrsh7th/nvim-cmp",
		version = false,
		event = "InsertEnter",
		dependencies = {
			"hrsh7th/cmp-nvim-lsp",
			"hrsh7th/cmp-buffer",
			"hrsh7th/cmp-path",
			"hrsh7th/cmp-cmdline",
			"saadparwaiz1/cmp_luasnip",
			"L3MON4D3/LuaSnip",
		},
		config = function()
			local cmp = require("cmp")
			local luasnip = require("luasnip")

			cmp.setup({
				snippet = {
					expand = function(args)
						luasnip.lsp_expand(args.body)
					end,
				},
				mapping = cmp.mapping.preset.insert({
					["<C-n>"] = cmp.mapping.select_next_item({ behavior = cmp.SelectBehavior.Insert }),
					["<C-p>"] = cmp.mapping.select_prev_item({ behavior = cmp.SelectBehavior.Insert }),
					["<C-b>"] = cmp.mapping.scroll_docs(-4),
					["<C-f>"] = cmp.mapping.scroll_docs(4),
					["<C-Space>"] = cmp.mapping.complete(),
					["<C-e>"] = cmp.mapping.abort(),
					["<CR>"] = cmp.mapping.confirm({ select = true }),
					["<S-CR>"] = cmp.mapping.confirm({
						behavior = cmp.ConfirmBehavior.Replace,
						select = true,
					}),
					["<C-CR>"] = function(fallback)
						cmp.abort()
						fallback()
					end,
				}),
				sources = cmp.config.sources({
					{ name = "nvim_lsp" },
					{ name = "luasnip" },
					{ name = "buffer" },
					{ name = "path" },
				}),
				experimental = {
					ghost_text = {
						hl_group = "CmpGhostText",
					},
				},
			})
		end,
	},

	-- =============================================================================
	-- 代码格式化
	-- =============================================================================
	{
		"stevearc/conform.nvim",
		lazy = false,
		keys = {
			{
				"<leader>f",
				function()
					require("conform").format({ async = true, lsp_fallback = true })
				end,
				mode = "",
				desc = "格式化缓冲区",
			},
		},
		opts = {
			notify_on_error = false,
			format_on_save = function(bufnr)
				-- 对特定文件类型禁用自动格式化
				local disable_filetypes = { "markdown", "text" }
				if vim.tbl_contains(disable_filetypes, vim.bo[bufnr].filetype) then
					return nil
				end

				return {
					timeout_ms = 500,
					lsp_fallback = true,
				}
			end,
			formatters_by_ft = {
				lua = { "stylua" },
				python = { "black" },
				javascript = { "prettier" },
				typescript = { "prettier" },
				javascriptreact = { "prettier" },
				typescriptreact = { "prettier" },
				vue = { "prettier" },
				css = { "prettier" },
				scss = { "prettier" },
				sass = { "prettier" },
				html = { "prettier" },
				json = { "prettier" },
				yaml = { "prettier" },
				markdown = { "prettier" },
				graphql = { "prettier" },
				liquid = { "prettier" },
			},
		},
	},

	-- =============================================================================
	-- Git 集成
	-- =============================================================================
	{
		"lewis6991/gitsigns.nvim",
		event = { "BufReadPre", "BufNewFile" },
		opts = {
			signs = {
				add = { text = "┃" },
				change = { text = "┃" },
				delete = { text = "_" },
				topdelete = { text = "‾" },
				changedelete = { text = "~" },
				untracked = { text = "┆" },
			},
			signcolumn = true,
			numhl = false,
			linehl = false,
			word_diff = false,
			watch_gitdir = {
				follow_files = true,
			},
			auto_attach = true,
			attach_to_untracked = false,
			current_line_blame = false,
			current_line_blame_opts = {
				virt_text = true,
				virt_text_pos = "eol",
				delay = 1000,
			},
			current_line_blame_formatter = "<author>, <author_time> - <summary>",
			sign_priority = 6,
			update_debounce = 100,
			status_formatter = nil,
			max_file_length = 40000,
			preview_config = {
				border = "single",
				style = "minimal",
				relative = "cursor",
				row = 0,
				col = 1,
			},
		},
	},

	-- =============================================================================
	-- 模糊查找器
	-- =============================================================================
	{
		"nvim-telescope/telescope.nvim",
		tag = "0.1.8",
		dependencies = {
			"nvim-lua/plenary.nvim",
			{ "nvim-telescope/telescope-fzf-native.nvim", build = "make" },
			"nvim-tree/nvim-web-devicons",
		},
		config = function()
			require("telescope").setup({
				defaults = {
					prompt_prefix = "",
					selection_caret = " ",
					path_display = { "truncate" },
					selection_strategy = "reset",
					sorting_strategy = "ascending",
					layout_strategy = "horizontal",
					layout_config = {
						horizontal = {
							prompt_position = "top",
							preview_width = 0.55,
							results_width = 0.8,
						},
						vertical = {
							mirror = false,
						},
					},
					file_sorter = require("telescope.sorters").get_fuzzy_file,
					generic_sorter = require("telescope.sorters").get_generic_fuzzy_sorter,
					file_ignore_patterns = { "node_modules", ".git/", "target/", "build/" },
					extensions = {
						fzf = {
							fuzzy = true,
							override_generic_sorter = true,
							override_file_sorter = true,
							case_mode = "smart_case",
						},
					},
				},
			})

			-- 加载扩展
			require("telescope").load_extension("fzf")
		end,
		keys = {
			{
				"<leader>f",
				function()
					require("telescope.builtin").find_files()
				end,
				desc = "查找文件",
			},
			{
				"<leader>g",
				function()
					require("telescope.builtin").live_grep()
				end,
				desc = "实时搜索",
			},
			{
				"<leader>b",
				function()
					require("telescope.builtin").buffers()
				end,
				desc = "缓冲区列表",
			},
			{
				"<leader>h",
				function()
					require("telescope.builtin").help_tags()
				end,
				desc = "帮助标签",
			},
		},
	},

	-- =============================================================================
	-- Mason - LSP 服务器管理
	-- =============================================================================
	{
		"williamboman/mason.nvim",
		cmd = "Mason",
		keys = { { "<leader>cm", "<cmd>Mason<cr>", desc = "Mason" } },
		opts = {
			ensure_installed = {
				"stylua",
				"shfmt",
				"black",
				"prettier",
				"pyright",
				"ts_ls",
				"html",
				"cssls",
				"tailwindcss",
				"lua_ls",
				"jsonls",
				"vimls",
				"yamlls",
				"dockerls",
				"eslint_d",
				"markdownlint",
			},
		},
		config = function()
			require("mason").setup()
			require("mason-lspconfig").setup()
		end,
	},

	-- =============================================================================
	-- 界面增强
	-- =============================================================================
	{
		"stevearc/dressing.nvim",
		lazy = true,
		opts = {},
	},

	{
		"folke/noice.nvim",
		event = "VeryLazy",
		opts = {
			lsp = {
				override = {
					["vim.lsp.util.convert_input_to_markdown_lines"] = true,
					["vim.lsp.util.stylize_markdown"] = true,
					["cmp.entry.get_documentation"] = true,
				},
			},
			presets = {
				bottom_search = true,
				command_palette = true,
				long_message_to_split = true,
				inc_rename = false,
				lsp_doc_border = false,
			},
		},
		dependencies = {
			"MunifTanjim/nui.nvim",
			"rcarriga/nvim-notify",
		},
	},

	-- =============================================================================
	-- 状态栏
	-- =============================================================================
	{
		"nvim-lualine/lualine.nvim",
		event = "VeryLazy",
		opts = {
			options = {
				icons_enabled = true,
				theme = "auto",
				component_separators = { left = "", right = "" },
				section_separators = { left = "", right = "" },
				disabled_filetypes = {
					statusline = { "dashboard", "alpha", "starter" },
				},
				globalstatus = true,
			},
			sections = {
				lualine_a = { "mode" },
				lualine_b = { "branch", "diff", "diagnostics" },
				lualine_c = { "filename" },
				lualine_x = { "encoding", "fileformat", "filetype" },
				lualine_y = { "progress" },
				lualine_z = { "location" },
			},
		},
	},

	-- =============================================================================
	-- 启动屏幕
	-- =============================================================================
	{
		"nvimdev/dashboard-nvim",
		lazy = false,
		opts = function()
			local logo = [[
			███╗   ██╗██╗   ██╗██╗███╗   ███╗
			████╗  ██║██║   ██║██║████╗ ████║
			██╔██╗ ██║██║   ██║██║██╔████╔██║
			██║╚██╗██║╚██╗ ██╔╝██║██║╚██╔╝██║
			██║ ╚████║ ╚████╔╝ ██║██║ ╚═╝ ██║
			╚═╝  ╚═══╝  ╚═══╝  ╚═╝╚═╝     ╚═╝
			]]

			logo = string.rep("\n", 8) .. logo .. "\n\n"

			return {
				theme = "doom",
				hide = {
					statusline = true,
					tabline = true,
					winbar = true,
				},
				config = {
					header = vim.split(logo, "\n"),
					center = {
						{ action = "Telescope find_files", desc = " 查找文件", icon = "", key = "f" },
						{ action = "ene | startinsert", desc = " 新建文件", icon = "", key = "n" },
						{ action = "Telescope oldfiles", desc = " 最近文件", icon = "", key = "r" },
						{ action = "Telescope live_grep", desc = " 搜索文本", icon = "", key = "g" },
						{ action = "lua require('persistence').load()", desc = " 恢复会话", icon = "", key = "s" },
						{ action = "Lazy", desc = " Lazy 插件", icon = "", key = "l" },
						{ action = "qa", desc = " 退出", icon = "", key = "q" },
					},
					footer = function()
						local stats = require("lazy").stats()
						local ms = (math.floor(stats.startuptime * 100 + 0.5) / 100)
						return { "⚡ Neovim 加载 " .. stats.count .. " 个插件，用时 " .. ms .. "ms" }
					end,
				},
			}
		end,
	},
}