-- ============================================================================
-- Neovim 主配置文件 (init.lua) - 基于 LazyVim
-- 从 NvChad 迁移到 LazyVim，保持所有原有功能和键位绑定
-- ============================================================================
-- 设置 Leader 键为逗号（与原配置保持一致）
vim.g.mapleader = ","
vim.g.maplocalleader = ","

-- 设置按下空格为按下:（便捷的命令行访问）
vim.api.nvim_set_keymap("n", "<Space>", ":", { noremap = true, silent = false })

-- 抑制 lspconfig 弃用警告
local original_notify = vim.notify
vim.notify = function(msg, level, opts)
	if type(msg) == "string" and msg:match("deprecated.*lspconfig") then
		return -- 忽略 lspconfig 弃用警告
	end
	return original_notify(msg, level, opts)
end

-- ============================================================================
-- Lazy.nvim 插件管理器初始化
-- ============================================================================
local lazypath = vim.fn.stdpath("data") .. "/lazy/lazy.nvim"
if not vim.uv.fs_stat(lazypath) then
	local repo = "https://github.com/folke/lazy.nvim.git"
	vim.fn.system({ "git", "clone", "--filter=blob:none", repo, "--branch=stable", lazypath })
end
vim.opt.rtp:prepend(lazypath)

-- ============================================================================
-- LazyVim 配置
-- ============================================================================
require("lazy").setup({
	-- 导入所有插件配置
	{ import = "plugins" },
}, {
	-- Lazy.nvim 配置
	change_detection = {
		notify = false, -- 禁用配置变更通知
	},
	performance = {
		rtp = {
			disabled_plugins = {
				-- 禁用一些不需要的插件以提高性能
				"gzip",
				"matchit",
				"matchparen",
				"netrwPlugin",
			},
		},
	},
})

-- ============================================================================
-- 基础配置加载
-- ============================================================================
-- 加载选项配置
require("config.options")

-- 加载自动命令
require("config.autocmds")

-- 加载键位映射
require("config.keymaps")

-- ============================================================================
-- 自定义功能和增强配置
-- ============================================================================
-- 文件类型识别增强
vim.filetype.add({
	pattern = {
		[".*[dD][oO][cC][kK][eE][rR].*"] = "dockerfile",
	},
})

-- 6.2 文件树自动打开和焦点管理（保持原功能）
local function on_vim_enter(data)
	local ok, nvim_tree_api = pcall(require, "nvim-tree.api")
	if not ok then
		return
	end

	local opened_path = data.file or ""
	local is_directory = vim.fn.isdirectory(opened_path) == 1
	local has_file = opened_path ~= "" and not is_directory

	if is_directory or not has_file then
		if is_directory then
			vim.cmd.cd(opened_path)
		end
		nvim_tree_api.tree.open({ focus = true })
	else
		local original_win = vim.api.nvim_get_current_win()
		nvim_tree_api.tree.open({ focus = false })
		pcall(function()
			nvim_tree_api.tree.find_file(vim.fn.expand('%:p'))
		end)
		pcall(function()
			if vim.api.nvim_win_is_valid(original_win) then
				vim.api.nvim_set_current_win(original_win)
			end
		end)
	end

	-- 延迟触发 Git 状态刷新
	vim.defer_fn(function()
		vim.api.nvim_exec_autocmds("BufEnter", {
			pattern = "*",
			modeline = false,
		})

		pcall(function()
			require('nvim-tree.git').reload()
			nvim_tree_api.tree.reload()
		end)

		pcall(function()
			local highlights = require('config.highlights')
			if highlights and highlights.set_nvim_tree_highlights then
				highlights.set_nvim_tree_highlights()
				highlights.ultimate_git_color_fix()
				highlights.force_refresh_nvimtree()
			end
		end)
	end, 300)
end

-- 启动时自动命令注册
vim.api.nvim_create_autocmd({ "VimEnter" }, {
	callback = on_vim_enter,
	desc = "Auto open nvim-tree on startup",
})

-- 自定义编辑器命令
vim.api.nvim_create_user_command('Ht', function()
	vim.cmd('botright new')
	vim.cmd('resize 15')
	local help_file = vim.fn.stdpath('config') .. '/nvim-tree-marks-guide.md'
	vim.cmd('edit ' .. help_file)
	vim.bo.buftype = 'help'
	vim.bo.buflisted = false
	vim.bo.modifiable = false
	vim.bo.readonly = true
	vim.bo.filetype = 'help'
	vim.bo.modified = false
	vim.wo.number = false
	vim.wo.relativenumber = false
	vim.wo.signcolumn = 'no'
	vim.keymap.set('n', 'q', '<cmd>close<cr>', { buffer = true, silent = true })
	vim.keymap.set('n', '<Esc>', '<cmd>close<cr>', { buffer = true, silent = true })
end, {
	desc = '显示 NvimTree 快捷键帮助文档'
})

vim.cmd('cabbrev ht Ht')

-- 加载 LSP 配置（使用新 API）
require("config.lsp").setup()

-- 加载 TailwindCSS 自动配置功能
require("config.tailwind").setup()

-- 加载 NvimTree 配置
require("config.nvimtree")

-- 加载高亮配置
require("config.highlights").setup()

-- 加载主题配置
require("config.theme")