# NvimTree 高亮闪烁修复

## 🔍 问题描述
刚启动 Neovim 时，NvimTree 中当前文件所在行的高亮会出现闪烁现象。

## 🎯 根本原因
多个高亮设置函数在短时间内被重复调用，导致：
1. **启动时**: `M.setup()` 延迟 100ms 调用
2. **主题切换时**: `ColorScheme` 自动命令
3. **NvimTree 打开时**: `FileType NvimTree` 自动命令
4. **窗口切换时**: `VimEnter`, `WinEnter`, `BufEnter` 自动命令

这些调用重叠执行，造成高亮设置的"竞争条件"，导致闪烁。

## ✅ 解决方案

### 1. 统一高亮设置函数
创建统一的 `setup_all_highlights()` 函数，将所有高亮设置合并到一个调用中：

```lua
function M.setup_all_highlights()
	-- 1. 首先设置全局透明背景
	M.setup_transparency_background()

	-- 2. 然后设置自定义高亮（会覆盖一些透明设置）
	M.setup_custom_highlights()

	-- 3. 最后设置 NvimTree 特定高亮
	M.set_nvim_tree_highlights()

	-- 4. 应用 Git 颜色修复
	M.ultimate_git_color_fix()
end
```

### 2. 优化调用时机
- **启动时**: 延迟 100ms 调用统一函数
- **主题切换时**: 延迟 50ms 调用统一函数
- **NvimTree 打开时**: 延迟 25ms 调用统一函数（减少延迟）

### 3. 防抖机制
为全局自动命令添加防抖机制，避免短时间内重复调用：

```lua
-- 全局透明背景设置 - 防抖避免重复调用
local transparency_timer = nil
api.nvim_create_autocmd({ "VimEnter", "WinEnter", "BufEnter" }, {
	callback = function()
		-- 防抖机制，避免短时间内重复调用
		if transparency_timer then
			vim.fn.timer_stop(transparency_timer)
		end

		transparency_timer = vim.fn.timer_start(100, function()
			local ok, highlights = pcall(require, "config.highlights")
			if ok and highlights and highlights.setup_all_highlights then
				highlights.setup_all_highlights()
			end
		end)
	end,
})
```

## 🚀 效果

### 修复前
- ❌ 启动时 NvimTree 当前文件行高亮闪烁
- ❌ 多次重复调用高亮设置
- ❌ 视觉体验不流畅

### 修复后
- ✅ 启动时高亮平滑应用，无闪烁
- ✅ 统一的高亮设置，避免重复调用
- ✅ 防抖机制确保调用优化
- ✅ 延迟时间优化，响应更快

## 📋 技术细节

### 延迟时间优化
- **启动延迟**: 100ms → 保持不变（确保所有插件加载完成）
- **主题切换延迟**: 50ms → 新增（快速响应主题变化）
- **NvimTree 延迟**: 100ms → 25ms（减少闪烁）
- **防抖延迟**: 100ms → 新增（避免重复调用）

### 调用顺序
1. 设置透明背景（基础层）
2. 设置自定义高亮（中间层）
3. 设置 NvimTree 高亮（特定层）
4. 应用 Git 颜色修复（功能层）

## 🔧 配置文件修改

### 修改的文件
1. `/Users/dyx/.config/nvim/lua/config/highlights.lua`
   - 添加 `setup_all_highlights()` 函数
   - 优化自动命令调用
   - 减少延迟时间

2. `/Users/dyx/.config/nvim/lua/config/autocmds.lua`
   - 添加防抖机制
   - 使用统一设置函数
   - 优化调用时机

现在启动 Neovim 时，NvimTree 的高亮应该平滑应用，不再出现闪烁现象！🎉