-- ============================================================================
-- 文件模板管理器 (templates.lua)
-- 提供文件模板的创建、管理和生成功能
-- ============================================================================
--
-- 功能概述：
-- - 支持多种文件类型的预定义模板
-- - 允许用户自定义模板
-- - 提供模板选择和快速生成功能
-- - 支持模板变量替换（如日期、作者等）
--
-- 使用方法：
-- 1. 使用 :Template 命令选择模板
-- 2. 使用快捷键快速生成常用模板
-- 3. 在 ~/.config/nvim/templates/ 目录添加自定义模板
-- ============================================================================

local M = {}

-- ============================================================================
-- 1. 配置选项
-- ============================================================================

-- 模板配置
M.config = {
	-- 模板目录路径
	template_dir = vim.fn.stdpath("config") .. "/templates",
	-- 作者信息（用于模板变量替换）
	author = "Your Name",
	email = "your.email@example.com",
	-- 是否在生成模板后自动进入插入模式
	auto_insert = true,
	-- 是否显示模板选择菜单
	show_menu = true,
}

-- ============================================================================
-- 2. 预定义模板
-- ============================================================================

-- 移除内置模板，只使用自定义模板
M.templates = {}

-- ============================================================================
-- 4. 辅助函数
-- ============================================================================

-- 获取当前日期
local function get_current_date()
	return os.date("%Y-%m-%d")
end

-- 获取当前年份
local function get_current_year()
	return os.date("%Y")
end

-- 获取文件名（不含扩展名）
local function get_filename_without_ext(filepath)
	local filename = vim.fn.fnamemodify(filepath, ":t")
	return vim.fn.fnamemodify(filename, ":r")
end

-- 将字符串转换为 PascalCase
local function to_pascal_case(str)
	return str:gsub("[^%w]", " "):gsub("%f[%w]%l", string.upper):gsub("%s+", "")
end

-- 将字符串转换为 snake_case
local function to_snake_case(str)
	return str:gsub("%s+", "_"):gsub("[^%w_]", ""):lower()
end

-- 替换模板变量
local function replace_template_variables(content, filepath)
	local filename = get_filename_without_ext(filepath)
	local component_name = to_pascal_case(filename)
	local class_name = to_pascal_case(filename)
	local module_name = to_snake_case(filename):upper()

	-- 变量映射表
	local variables = {
		["{{FILENAME}}"] = vim.fn.fnamemodify(filepath, ":t"),
		["{{FILENAME_NO_EXT}}"] = filename,
		["{{COMPONENT_NAME}}"] = component_name,
		["{{COMPONENT_NAME_LOWER}}"] = component_name:lower(),
		["{{CLASS_NAME}}"] = class_name,
		["{{MODULE_NAME}}"] = module_name,
		["{{AUTHOR}}"] = M.config.author,
		["{{EMAIL}}"] = M.config.email,
		["{{DATE}}"] = get_current_date(),
		["{{YEAR}}"] = get_current_year(),
		["{{TITLE}}"] = component_name,
		["{{DESCRIPTION}}"] = "请在此处添加描述",
	}

	-- 替换所有变量
	for pattern, replacement in pairs(variables) do
		content = content:gsub(pattern, replacement)
	end

	return content
end

-- ============================================================================
-- 5. 核心功能函数
-- ============================================================================

-- 获取可用模板列表（仅自定义模板）
function M.get_available_templates()
	local templates = {}
	local template_map = {}

	-- 扫描自定义模板目录
	local template_dir = M.config.template_dir
	if vim.fn.isdirectory(template_dir) == 1 then
		local files = vim.fn.globpath(template_dir, "*", false, true)
		for _, file in ipairs(files) do
			if vim.fn.filereadable(file) == 1 then
				local filename = vim.fn.fnamemodify(file, ":t")
				local name_without_ext = vim.fn.fnamemodify(filename, ":r")
				local extension = vim.fn.fnamemodify(filename, ":e")

				template_map[name_without_ext] = {
					key = name_without_ext,
					name = name_without_ext .. " (自定义)",
					extension = extension,
					source = "custom",
					path = file
				}
			end
		end
	end

	-- 按指定顺序添加模板：html、javascript、css、python
	local preferred_order = { "html", "javascript", "css", "python" }
	for _, template_key in ipairs(preferred_order) do
		if template_map[template_key] then
			table.insert(templates, template_map[template_key])
			template_map[template_key] = nil -- 标记为已添加
		end
	end

	-- 添加其他剩余的模板
	for _, template in pairs(template_map) do
		table.insert(templates, template)
	end

	-- 如果没有找到自定义模板，提示用户创建
	if #templates == 0 then
		vim.notify("未找到自定义模板文件，请在 " .. template_dir .. " 目录下创建模板文件", vim.log.levels.WARN)
	end

	return templates
end

-- 读取模板内容（仅自定义模板）
function M.get_template_content(template_key)
	-- 只检查自定义模板
	local template_file = M.config.template_dir .. "/" .. template_key
	if vim.fn.filereadable(template_file) == 1 then
		local lines = vim.fn.readfile(template_file)
		return table.concat(lines, "\n")
	end

	-- 尝试添加常见扩展名
	local extensions = { ".txt", ".md", ".html", ".js", ".py", ".lua", ".css", ".jsx", ".ts", ".vue", ".go", ".sh",
		".json", ".yml", ".yaml" }
	for _, ext in ipairs(extensions) do
		local file_with_ext = template_file .. ext
		if vim.fn.filereadable(file_with_ext) == 1 then
			local lines = vim.fn.readfile(file_with_ext)
			return table.concat(lines, "\n")
		end
	end

	-- 如果找不到模板文件，提示用户创建
	vim.notify("模板文件不存在: " .. template_file .. "\n请创建该模板文件", vim.log.levels.ERROR)
	return nil
end

-- 生成文件模板
function M.generate_template(template_key, filepath)
	local content = M.get_template_content(template_key)
	if not content then
		vim.notify("模板 '" .. template_key .. "' 不存在", vim.log.levels.ERROR)
		return false
	end

	-- 替换模板变量
	content = replace_template_variables(content, filepath)

	-- 写入文件
	local lines = vim.split(content, "\n")
	vim.fn.writefile(lines, filepath)

	-- 打开文件（强制打开，忽略未保存的更改警告）
	vim.cmd("edit! " .. filepath)

	-- 默认保持在普通模式，不自动进入插入模式
	-- 用户可以手动按 i 或其他键进入插入模式

	vim.notify("模板 '" .. template_key .. "' 已生成到 " .. filepath, vim.log.levels.INFO)
	return true
end

-- 显示模板选择菜单
function M.show_template_menu(callback)
	local templates = M.get_available_templates()

	if #templates == 0 then
		vim.notify("没有可用的模板", vim.log.levels.WARN)
		return
	end

	-- 创建选择项
	local items = {}
	for i, template in ipairs(templates) do
		-- 自定义模板：直接显示模板名称和扩展名
		local display_name = string.format("%s (.%s)", template.name, template.extension)
		table.insert(items, display_name)
	end

	-- 显示选择菜单
	vim.ui.select(items, {
		prompt = "选择模板:",
		format_item = function(item)
			return item
		end,
	}, function(choice, idx)
		if choice and idx then
			local selected_template = templates[idx]
			if callback then
				callback(selected_template)
			end
		end
	end)
end

-- 创建新文件并应用模板
function M.new_file_with_template()
	M.show_template_menu(function(template)
		-- 获取文件路径（默认文件名不含扩展名）
		local default_name = "new_file"
		vim.ui.input({
			prompt = "文件名 (无需扩展名): ",
			default = default_name,
		}, function(filename)
			if filename and filename ~= "" then
				-- 自动添加扩展名（如果用户没有输入）
				local final_filename = filename
				if not filename:match("%.[^/\\]+$") then
					final_filename = filename .. "." .. template.extension
				end

				-- 生成完整路径
				local filepath = vim.fn.expand("%:p:h") .. "/" .. final_filename

				-- 检查文件是否已存在
				if vim.fn.filereadable(filepath) == 1 then
					vim.ui.select({ "覆盖", "取消" }, {
						prompt = "文件已存在，是否覆盖？",
					}, function(choice)
						if choice == "覆盖" then
							M.generate_template(template.key, filepath)
						end
					end)
				else
					M.generate_template(template.key, filepath)
				end
			end
		end)
	end)
end

-- 为当前文件应用模板
function M.apply_template_to_current_file()
	local current_file = vim.fn.expand("%:p")
	if current_file == "" then
		vim.notify("请先保存当前文件", vim.log.levels.WARN)
		return
	end

	M.show_template_menu(function(template)
		-- 检查当前文件是否为空
		local line_count = vim.api.nvim_buf_line_count(0)
		local is_empty = line_count == 1 and vim.api.nvim_buf_get_lines(0, 0, 1, false)[1] == ""

		if not is_empty then
			vim.ui.select({ "覆盖", "取消" }, {
				prompt = "当前文件不为空，是否覆盖内容？",
			}, function(choice)
				if choice == "覆盖" then
					M.generate_template(template.key, current_file)
				end
			end)
		else
			M.generate_template(template.key, current_file)
		end
	end)
end

-- ============================================================================
-- 5. 初始化和设置
-- ============================================================================

-- 初始化模板系统
function M.setup(opts)
	opts = opts or {}
	M.config = vim.tbl_deep_extend("force", M.config, opts)

	-- 创建模板目录（如果不存在）
	if vim.fn.isdirectory(M.config.template_dir) == 0 then
		vim.fn.mkdir(M.config.template_dir, "p")
	end

	-- 注册命令
	vim.api.nvim_create_user_command("Template", function()
		M.new_file_with_template()
	end, {
		desc = "使用模板创建新文件"
	})

	vim.api.nvim_create_user_command("TemplateApply", function()
		M.apply_template_to_current_file()
	end, {
		desc = "为当前文件应用模板"
	})

	vim.api.nvim_create_user_command("TemplateList", function()
		local templates = M.get_available_templates()
		vim.notify("可用模板:", vim.log.levels.INFO)
		for _, template in ipairs(templates) do
			vim.notify(string.format("  %s - %s (.%s)", template.key, template.name, template.extension), vim.log.levels.INFO)
		end
	end, {
		desc = "列出所有可用模板"
	})

end

return M
