-- ============================================================================
-- Conform 代码格式化配置文件 (conform.lua)
-- 配置不同文件类型的代码格式化工具和行为
-- ============================================================================

-- ============================================================================
-- 1. 基础配置初始化
-- ============================================================================
local options = {}

-- ============================================================================
-- 2. 格式化工具配置
-- ============================================================================

-- 2.1 按文件类型配置格式化工具
options.formatters_by_ft = {
	-- 为行首注释统一补空格，链在主格式化器之后
	lua = { "stylua", "comment_space" }, -- Lua: stylua + 注释空格规范
	-- CSS：使用仅处理块注释的安全变体，避免将 #id 误识别为注释
	css = { "prettier", "comment_space_blocks" },
	-- HTML：使用 prettier 进行格式化，包括 Tailwind CSS 类名排序
	-- 注意：为了确保 prettier-plugin-tailwindcss 正常工作，暂时移除后处理器
	-- 同时避免与 nvim-ts-autotag 冲突，只在手动格式化时使用
	html = { "prettier" },
	-- HTM：使用 prettier 进行格式化，包括 Tailwind CSS 类名排序
	htm = { "prettier" },
	-- ERB/ERuby：使用 prettier 进行格式化，包括 Tailwind CSS 类名排序
	eruby = { "prettier_erb" },
	-- Vue：使用 prettier 进行格式化，包括 Tailwind CSS 类名排序
	vue = { "prettier" },
	-- JSX/TSX：使用 prettier 进行格式化，包括 Tailwind CSS 类名排序
	javascriptreact = { "prettier" },
	typescriptreact = { "prettier" },
	javascript = { "prettier", "comment_space" }, -- JS: prettier + 注释空格规范（// 与 /* */）
	typescript = { "prettier", "comment_space" }, -- TS: prettier + 注释空格规范
	json = { "prettier" },                     -- JSON：无注释
	yaml = { "prettier", "comment_space" },    -- YAML: prettier + 注释空格规范（#）
	python = { "black", "comment_space" },     -- Python: black + 注释空格规范（#）
	go = { "gofmt", "comment_space" },         -- Go: gofmt + 注释空格规范（// 与 /* */）
	-- 其他常见使用 # 的文件类型：未配置主格式化器时，仅做注释空格修正
	sh = { "comment_space" },
	bash = { "comment_space" },
	zsh = { "comment_space" },
	fish = { "comment_space" },
	dockerfile = { "comment_space" },
	toml = { "comment_space" },
}

-- 2.2 格式化工具自定义配置
-- 配置各个格式化工具的缩进设置
options.formatters = {
	-- stylua 配置：使用 tab 缩进，宽度为 4
	stylua = {
		prepend_args = { "--indent-type", "Tabs", "--indent-width", "4" },
	},
	-- prettier 配置：保留用户要求的格式化参数，同时确保 prettier-plugin-tailwindcss 插件正常工作
	-- 注意：显式加载 prettier-plugin-tailwindcss 插件，确保 Tailwind CSS 类名排序生效
	prettier = {
		prepend_args = { 
			"--plugin", "prettier-plugin-tailwindcss",
			"--use-tabs", "--tab-width", "4", "--print-width", "1000", "--single-attribute-per-line=false" 
		},
	},
	prettierd = {
		prepend_args = { 
			"--plugin", "prettier-plugin-tailwindcss",
			"--use-tabs", "--tab-width", "4", "--print-width", "1000", "--single-attribute-per-line=false" 
		},
	},
	-- ERB 文件专用 prettier 配置：指定 HTML 解析器
	prettier_erb = {
		command = "prettier",
		args = { 
			"--plugin", "prettier-plugin-tailwindcss",
			"--parser", "html", "--use-tabs", "--tab-width", "4", "--print-width", "1000", "--single-attribute-per-line=false" 
		},
	},
	-- black 配置：指定完整路径和参数
	black = {
		command = vim.fn.stdpath("data") .. "/mason/bin/black",
		args = { "--line-length", "100", "--stdin-filename", "$FILENAME", "-" },
		stdin = true,
	},
	-- 自定义 HTML 后处理：移除 <head>/<body> 标签上下的空行
	-- 设计初衷：Prettier 没有“禁止在特定标签外增删空行”的开关，因此用一个只做极小变更的后处理。
	-- 执行时机：始终在 prettier 之后运行；仅当系统可用 perl 时启用。
	-- 安全性：只匹配 head/body 两类边界，不会改动正文内容或标签内部结构。
	html_noblank = {
		command = "perl",
		stdin = true,
		args = {
			"-0777",
			"-pe",
			-- 具体规则：
			-- 1) 折叠到 <head> 开始标签前的空行：  \n\s*\n(\s*<head\b)  ->  \n$1
			-- 2) 折叠到 </head> 结束标签后的空行： (</head>)\n\s*\n  ->  $1\n
			-- 3) 折叠到 <body> 开始标签前的空行：  \n\s*\n(\s*<body\b)  ->  \n$1
			-- 4) 折叠到 </body> 结束标签后的空行： (</body>)\n\s*\n  ->  $1\n
			[[s/\n\s*\n(\s*<head\b)/\n$1/g; s/(<\/head>)\n\s*\n/$1\n/g; s/\n\s*\n(\s*<body\b)/\n$1/g; s/(<\/body>)\n\s*\n/$1\n/g]],
		},
		condition = function()
			return vim.fn.executable("perl") == 1
		end,
	},
	-- 通用注释空格规范化（后处理）：
	-- 目的：在多语言中尽量统一注释标记与注释内容之间至少一个空格
	-- 范围（安全守则）：
	--   - 仅处理行首行注释（//、--、#），保留前导缩进；不处理行尾内联注释，避免误改代码/字符串中的 // 或 #
	--   - --:   仅对以 -- 起始且后面不是 [[ 的行补空格（避免破坏 Lua 长注释 --[[ 结构）；
	--   - #:    仅对以 # 起始且非 shebang(#!) 的行补空格；不会用于 CSS（CSS 走 blocks 变体）
	--   - 同时修正块注释与 HTML 注释两端空格（/* */ 与 <!-- -->）
	comment_space = {
		command = "perl",
		stdin = true,
		args = {
			"-pe",
			-- 行注释：行首补空格
			-- ^(\s*)//(\S)  ->  保留缩进 + // 后补空格
			-- ^(\s*)--(?!\[)(\S) ->  避免 --[[ 长注释，其他情况补空格
			-- ^(\s*)#(?!\!)(\S)  ->  避免 shebang 行，其他情况补空格
			-- 块/HTML 注释：两端补空格（仅在缺少时生效，不会重复）
			-- (/*)(\S)   ->  /* 后补空格； (\S)(*/) ->  */ 前补空格
			-- (<!--)(\S) ->  <!-- 后补空格； (\S)(-->) ->  --> 前补空格
			[[s/^(\s*)\/\/(\S)/$1\/\/ $2/; s/^(\s*)--(?!\[)(\S)/$1-- $2/; s/^(\s*)#(?!\!)(\S)/$1# $2/; s/(\/\*)(\S)/$1 $2/g; s/(\S)(\*\/)/$1 $2/g; s/(<!--)(\S)/$1 $2/g; s/(\S)(-->)/$1 $2/g;]],
		},
		condition = function()
			return vim.fn.executable("perl") == 1
		end,
	},
	-- 仅处理块注释的安全变体：用于 CSS 等不使用 # 作为注释的文件类型
	comment_space_blocks = {
		command = "perl",
		stdin = true,
		args = {
			"-pe",
			-- /* 后与 */ 前补空格
			[[s/(\/\*)(\S)/$1 $2/g; s/(\S)(\*\/)/$1 $2/g;]],
		},
		condition = function()
			return vim.fn.executable("perl") == 1
		end,
	},
}

-- ============================================================================
-- 3. 高级选项和自动化配置
-- ============================================================================

-- 3.1 保存时自动格式化配置
-- 注意：自动命令已移至插件配置中，确保在插件加载后执行

-- ============================================================================
-- 4. 配置导出
-- ============================================================================
-- 返回配置选项供 conform.nvim 使用
return options
