# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

This is a comprehensive Neovim configuration based on NvChad framework, providing a modern IDE-like experience with extensive customizations for web development, LSP integration, and enhanced UI/UX.

## Architecture and Structure

### Core Configuration Files
- `init.lua`: Main entry point that bootstraps lazy.nvim and loads all modules
- `lua/chadrc.lua`: NvChad configuration hub for themes, UI elements, and plugin integration
- `lua/configs/`: Core feature modules containing specialized configurations
- `lua/plugins/`: Plugin specifications and plugin-specific configurations
- `lazy-lock.json`: Plugin lockfile - commit changes after running `Lazy sync`

### Key Configuration Modules
- `configs/lspconfig.lua`: Language Server Protocol setup with Chinese diagnostics translation
- `configs/conform.lua`: Code formatting configuration with support for multiple formatters
- `configs/mappings.lua`: Custom keybindings and shortcuts
- `configs/nvimtree.lua`: File explorer configuration with Git integration
- `configs/autocmds.lua`: Auto-commands for various editor events
- `configs/tailwind-auto-config.lua`: TailwindCSS integration and auto-detection

### Plugin Management
- Uses `lazy.nvim` as the plugin manager
- Main plugins include: nvim-treesitter, nvim-lspconfig, conform.nvim, nvim-tree.lua, nvim-cmp
- Custom plugins for TailwindCSS support and HTML tag auto-completion

## Development Commands

### Plugin Management
```bash
# Install/update all plugins and update lockfile
nvim --headless "+Lazy sync" +qa

# Check plugin health and configuration
nvim --headless "+checkhealth" +qa

# Profile startup performance
nvim --startuptime startuptime.log -c qa
```

### Code Formatting and Linting
```bash
# Format Lua files (if stylua is available)
stylua lua

# Format within Neovim
:lua require('conform').format({ async = false })

# Check web assets formatting
prettier --check "**/*.{html,css,js,ts,json,md,yaml,yml}"

# Fix web assets formatting  
prettier --write "**/*.{html,css,js,ts,json,md,yaml,yml}"
```

### Testing and Validation
```bash
# Run comprehensive health checks
:checkhealth

# Test LSP functionality
# Open files of different types and verify diagnostics, completion, formatting

# Test formatting for various file types
# Open HTML/CSS/JS files and run manual formatting
```

## Key Features and Behaviors

### Auto-formatting on Save
- Files are automatically formatted when saved using `:W` or `:Wq` commands
- Uses conform.nvim with file-type specific formatters (prettier, stylua, black, etc.)

### NvimTree Integration
- Automatically opens on startup with intelligent focus management
- Git status integration with color coding
- Custom commands and keybindings for file operations

### LSP Configuration
- Multi-language support with optimized setups for web development
- Chinese translation for common diagnostic messages
- Integrated with nvim-cmp for intelligent auto-completion

### TailwindCSS Integration
- Auto-detection and configuration for TailwindCSS projects
- Class name completion and color preview
- Automatic sorting of Tailwind classes

### Custom Keybindings
- Leader key set to comma (`,`)
- Space mapped to `:` for quick command access
- Alt-based shortcuts for enhanced navigation and editing

## Coding Standards

### File Organization
- Use `lua/configs/` for feature-specific configuration modules
- Keep plugin specifications in `lua/plugins/`
- Use descriptive, hyphenated filenames (e.g., `tailwind-auto-config.lua`)

### Code Style
- Indentation: Tabs with width 4
- Extensive Chinese comments for configuration explanations
- Modular approach with clear separation of concerns

### Formatting Rules
- HTML: Prettier with TailwindCSS plugin for class sorting
- CSS: Prettier with block comment spacing
- JavaScript/TypeScript: Prettier with comment spacing normalization
- Lua: stylua with comment spacing post-processing
- Python: black with comment spacing normalization

## Common Development Tasks

### Adding New Plugins
1. Add plugin specification to `lua/configs/plugins.lua`
2. Create configuration in appropriate `configs/` file or plugin-specific config
3. Update `lazy-lock.json` by running `:Lazy sync`

### Modifying LSP Settings
- Edit `lua/configs/lspconfig.lua` for server configurations
- Language-specific settings are organized by server type
- Diagnostic translations can be extended in the Chinese mappings table

### Customizing UI/Themes
- Modify `lua/chadrc.lua` for theme and UI element changes
- Color overrides go in the `hl_override` section
- Statusline and tabufline customizations are module-based

### Working with Templates
- File templates are stored in `templates/` directory
- Template system integration is configured in `configs/templates.lua`

## Troubleshooting

### Startup Issues
- Run `:checkhealth` to identify configuration problems
- Check `lazy-lock.json` for plugin version conflicts
- Use `nvim --clean` to test without custom configuration

### Formatting Problems
- Verify formatter availability in system PATH
- Check `conform.nvim` configuration for file type mappings
- Test manual formatting with `:lua require('conform').format()`

### LSP Issues
- Verify language servers are installed via Mason
- Check server configurations in `lspconfig.lua`
- Use `:LspInfo` to debug server attachment issues

## Important Notes

- This configuration is heavily customized for Chinese-language development workflow
- Git integration is optimized for color-coded status display
- Auto-commands handle file tree focus and Git status refresh automatically
- TailwindCSS detection works across multiple project structures and configuration files