-- ============================================================================
-- 代码运行器配置文件 (code-runner.lua)
-- 提供可自定义的代码运行功能，支持编辑运行选项
-- ============================================================================

local M = {}

-- 默认运行配置
M.run_configs = {
	-- JavaScript/Node.js
	{
		name = "🟨 Node.js",
		command = "node %",
		filetypes = { "javascript", "js" }
	},
	-- Python
	{
		name = "🐍 Python",
		command = "python3 %",
		filetypes = { "python", "py" }
	},
	-- HTML (在浏览器中打开)
	{
		name = "🌐 浏览器打开",
		command = "open %", -- macOS
		filetypes = { "html", "htm" }
	},
	-- Go
	{
		name = "🚀 Go运行",
		command = "go run %",
		filetypes = { "go" }
	},
	-- Lua
	{
		name = "🌙 Lua",
		command = "lua %",
		filetypes = { "lua" }
	},
	-- Shell脚本
	{
		name = "🐚 Bash",
		command = "bash %",
		filetypes = { "sh", "bash" }
	}
}

-- 获取当前文件类型的运行配置
function M.get_configs_for_filetype(filetype)
	local configs = {}
	for _, config in ipairs(M.run_configs) do
		if vim.tbl_contains(config.filetypes, filetype) then
			table.insert(configs, config)
		end
	end
	return configs
end

-- 存储上次运行的进程信息
M.last_process = nil

-- 标记nvimtree是否在Django启动前是可见的
M.nvimtree_was_visible = false

-- 确保在编辑窗口中执行操作，而不是在nvimtree中
local function ensure_edit_window()
	local current_filetype = vim.bo.filetype
	if current_filetype == "NvimTree" then
		-- 如果当前在nvimtree窗口，切换到右侧的编辑窗口
		vim.cmd("wincmd l")
		-- 如果右侧仍然是nvimtree（或没有窗口），则切换到下一个窗口
		if vim.bo.filetype == "NvimTree" then
			vim.cmd("wincmd w")
		end
		-- 如果还是nvimtree，说明只有nvimtree窗口，创建一个新的编辑缓冲区
		if vim.bo.filetype == "NvimTree" then
			vim.cmd("wincmd l")
			if vim.bo.filetype == "NvimTree" then
				vim.cmd("enew")
			end
		end
	end
end

-- 🚀 Django启动时的nvimtree性能优化函数
local function optimize_nvimtree_for_django_startup()
	pcall(function()
		local api = require('nvim-tree.api')
		if api.tree.is_visible() then
			-- 临时折叠所有目录以减少文件系统监控负载
			api.tree.collapse_all()
			
			-- 🔥 极致优化：临时隐藏nvimtree以完全避免卡顿
			api.tree.close()
			vim.notify("📁 nvimtree已临时隐藏以避免Django启动卡顿", vim.log.levels.INFO)
			
			-- 标记需要重新显示
			M.nvimtree_was_visible = true
		end
	end)
end

-- 🔄 Django启动完成后恢复nvimtree正常状态
local function restore_nvimtree_after_startup()
	pcall(function()
		local api = require('nvim-tree.api')
		
		-- 如果之前隐藏了nvimtree，重新显示
		if M.nvimtree_was_visible then
			M.nvimtree_was_visible = false
			api.tree.open()
			
			-- 等待一小段时间让树完全加载后再刷新
			vim.defer_fn(function()
				api.tree.reload()
			end, 500)
			
		elseif api.tree.is_visible() then
			-- 如果树仍然可见，只是刷新
			api.tree.reload()
		end
	end)
end

-- 清理指定端口上的进程
local function kill_process_on_port(port)
	if not port or port == "" then
		return
	end
	
	vim.notify("🔄 检查端口 " .. port .. " 上的进程...", vim.log.levels.INFO)
	
	-- 使用 lsof 查找占用端口的进程
	local cmd = "lsof -ti:" .. port
	local result = vim.fn.system(cmd)
	
	if vim.v.shell_error == 0 and result ~= "" then
		-- 找到进程，尝试终止
		local pids = vim.split(vim.trim(result), "\n")
		for _, pid in ipairs(pids) do
			if pid ~= "" then
				vim.notify("🔄 终止端口 " .. port .. " 上的进程 (PID: " .. pid .. ")", vim.log.levels.INFO)
				vim.fn.system("kill -TERM " .. pid)
				
				-- 等待一秒，如果还在运行就强制终止
				vim.defer_fn(function()
					local check_cmd = "ps -p " .. pid .. " > /dev/null 2>&1"
					if vim.fn.system(check_cmd) == 0 then
						vim.fn.system("kill -KILL " .. pid)
						vim.notify("🔄 强制终止进程 (PID: " .. pid .. ")", vim.log.levels.WARN)
					end
				end, 1000)
			end
		end
	else
		vim.notify("✅ 端口 " .. port .. " 空闲", vim.log.levels.INFO)
	end
end

-- 自动清理组设置标志
M.autocmd_setup = false

-- 设置自动清理功能
local function setup_auto_cleanup()
	if M.autocmd_setup then
		return
	end
	
	-- 创建自动命令组
	local group = vim.api.nvim_create_augroup("CodeRunnerAutoCleanup", { clear = true })
	
	-- Neovim退出时清理进程
	vim.api.nvim_create_autocmd("VimLeavePre", {
		group = group,
		callback = function()
			if M.last_process and M.last_process.job_id then
				vim.fn.jobstop(M.last_process.job_id)
				vim.notify("🔄 Neovim退出，已清理运行进程", vim.log.levels.INFO)
			end
		end,
		desc = "清理代码运行器进程当Neovim退出时"
	})
	
	-- 终端缓冲区被删除时清理进程
	vim.api.nvim_create_autocmd("BufDelete", {
		group = group,
		callback = function(args)
			if M.last_process and M.last_process.bufnr == args.buf then
				if M.last_process.job_id then
					vim.fn.jobstop(M.last_process.job_id)
					vim.notify("🔄 终端窗口关闭，已清理运行进程", vim.log.levels.INFO)
				end
				M.last_process = nil
			end
		end,
		desc = "清理代码运行器进程当终端缓冲区删除时"
	})
	
	-- 终端作业结束时清理记录
	vim.api.nvim_create_autocmd("TermClose", {
		group = group,
		callback = function(args)
			if M.last_process and M.last_process.bufnr == args.buf then
				M.last_process = nil
				vim.notify("🔄 进程正常结束", vim.log.levels.INFO)
			end
		end,
		desc = "清理代码运行器进程记录当终端关闭时"
	})
	
	M.autocmd_setup = true
end

-- 清除上次运行的进程
local function kill_last_process()
	if M.last_process then
		-- 清理端口（如果记录了端口信息）
		if M.last_process.port then
			kill_process_on_port(M.last_process.port)
		end
		
		if M.last_process.job_id then
			-- 终止Neovim作业
			vim.fn.jobstop(M.last_process.job_id)
			vim.notify("🔄 已清除上次运行的进程 (作业ID: " .. M.last_process.job_id .. ")", vim.log.levels.INFO)
		end
		
		if M.last_process.bufnr and vim.api.nvim_buf_is_valid(M.last_process.bufnr) then
			-- 关闭终端缓冲区
			vim.api.nvim_buf_delete(M.last_process.bufnr, { force = true })
		end
		
		M.last_process = nil
	end
end

-- 运行代码（简化格式）
function M.run_project_config(project_config)
	-- 设置自动清理功能
	setup_auto_cleanup()
	
	-- 清除上次运行的进程
	kill_last_process()
	
	-- 保存当前文件（仅在可编辑缓冲区中执行）
	local buftype = vim.api.nvim_buf_get_option(0, 'buftype')
	local modifiable = vim.api.nvim_buf_get_option(0, 'modifiable')
	local modified = vim.api.nvim_buf_get_option(0, 'modified')
	
	if buftype == '' and modifiable and modified then
		vim.cmd("write")
	end
	
	-- 获取当前文件名（完整路径）
	local filename = vim.fn.expand("%:p")
	if filename == "" then
		-- 如果没有当前文件，使用项目根目录作为工作基础
		filename = project_config._project_root or vim.fn.getcwd()
	end
	
	-- 切换到项目根目录作为工作目录
	local work_directory
	if project_config._project_root then
		-- 使用找到的项目根目录
		work_directory = project_config._project_root
		if project_config.work_dir and project_config.work_dir ~= "./" then
			-- 如果配置中指定了相对工作目录，则基于项目根目录计算
			if not project_config.work_dir:match("^/") then
				work_directory = project_config._project_root .. "/" .. project_config.work_dir
			else
				work_directory = project_config.work_dir
			end
		end
	else
		-- 兼容旧版本：使用当前目录
		work_directory = vim.fn.getcwd()
		if project_config.work_dir and project_config.work_dir ~= "./" then
			work_directory = project_config.work_dir
		end
	end
	
	-- 切换到工作目录
	if work_directory ~= vim.fn.getcwd() then
		vim.cmd("cd " .. vim.fn.fnameescape(work_directory))
		vim.notify("📂 切换到工作目录: " .. work_directory, vim.log.levels.INFO)
	end
	
	-- 执行前置命令
	if project_config.pre_commands and #project_config.pre_commands > 0 then
		for _, pre_cmd in ipairs(project_config.pre_commands) do
			vim.notify("📋 执行前置命令: " .. pre_cmd, vim.log.levels.INFO)
			vim.fn.system(pre_cmd)
		end
	end
	
	-- 构建完整命令
	local command = project_config.command
	
	-- 处理Python解释器路径
	if project_config.python_path and project_config.python_path ~= "" and project_config.project_type == "python" then
		-- 直接使用指定的Python解释器完整路径
		local python_interpreter = project_config.python_path
		
		-- 检查Python解释器是否存在
		if vim.fn.filereadable(python_interpreter) == 1 then
			-- 如果命令中包含python，则替换；否则在命令前添加Python解释器
			if command:match("python3?") then
				command = command:gsub("^python3?", python_interpreter)
				command = command:gsub("python3?", python_interpreter)
			else
				-- 命令不包含python，直接在前面添加Python解释器
				command = python_interpreter .. " " .. command
			end
			
			vim.notify("🐍 使用Python解释器: " .. python_interpreter, vim.log.levels.INFO)
		else
			vim.notify("⚠️ 找不到指定的Python解释器", vim.log.levels.WARN)
			vim.notify("💡 检查路径: " .. python_interpreter, vim.log.levels.INFO)
		end
	-- 兼容旧的venv_dir配置
	elseif project_config.venv_dir and project_config.venv_dir ~= "" and project_config.project_type == "python" then
		-- venv_dir指向虚拟环境根目录，自动使用bin/python3
		local venv_python = project_config.venv_dir .. "/bin/python3"
		
		-- 检查python3是否存在，如果不存在则尝试python
		if vim.fn.filereadable(venv_python) ~= 1 then
			venv_python = project_config.venv_dir .. "/bin/python"
		end
		
		-- 检查虚拟环境中的Python是否存在
		if vim.fn.filereadable(venv_python) == 1 then
			-- 直接替换命令中的python为虚拟环境中的python（完整路径）
			command = command:gsub("^python3?", venv_python)
			command = command:gsub("python3?", venv_python)
			
			vim.notify("🐍 使用虚拟环境: " .. project_config.venv_dir, vim.log.levels.INFO)
			vim.notify("📍 Python解释器: " .. venv_python, vim.log.levels.INFO)
		else
			vim.notify("⚠️ 虚拟环境中找不到Python解释器", vim.log.levels.WARN)
			vim.notify("💡 检查路径: " .. project_config.venv_dir .. "/bin/python3", vim.log.levels.INFO)
		end
	end
	
	local args_str = ""
	if project_config.args and #project_config.args > 0 then
		args_str = " " .. table.concat(project_config.args, " ")
	end
	
	-- 替换占位符和处理文件路径
	local actual_command
	if command:find("%%") then
		-- 包含%占位符，直接替换
		actual_command = command:gsub("%%", filename) .. args_str
	else
		-- 不包含%占位符，检查是否为相对路径文件名
		if command:match("^[^/]+%.%w+$") then -- 匹配类似 "manage.py" 的文件名
			-- 构建完整路径
			local full_file_path = vim.fn.getcwd() .. "/" .. command
			if vim.fn.filereadable(full_file_path) == 1 then
				-- 使用完整文件路径，但显示时要包含原始命令结构
				local base_command = command:gsub("^[^/]+%.%w+", full_file_path)
				actual_command = base_command .. args_str
			else
				actual_command = command .. args_str
			end
		else
			actual_command = command .. args_str
		end
	end
	
	-- 设置环境变量并显示
	local env_info = {}
	if project_config.env_vars and #project_config.env_vars > 0 then
		for _, env_var in ipairs(project_config.env_vars) do
			local key, value = env_var:match("([^=]+)=(.+)")
			if key and value then
				vim.fn.setenv(key, value)
				table.insert(env_info, env_var)
			end
		end
	end
	
	-- 构建执行信息显示
	local current_dir = vim.fn.getcwd()
	local env_display = #env_info > 0 and table.concat(env_info, " ") or "使用系统默认"
	
	-- 构建在终端中显示的信息命令
	local info_lines = {
		"echo '📋 运行命令: " .. actual_command .. "'",
		"echo '📂 工作目录: " .. current_dir .. "'",
		"echo '🌍 环境变量: " .. env_display .. "'",
		"echo '----------------------------------------'"
	}
	
	-- 浏览器配置处理
	local detected_port = nil
	local should_open_browser = false
	local custom_url = nil
	local browser_path = nil
	
	-- 检查项目配置中的浏览器设置，设置默认值
	if project_config.browser then
		should_open_browser = project_config.browser.auto_open ~= false -- 默认为true
		custom_url = project_config.browser.url or "https://0.0.0.0:8888"
		browser_path = project_config.browser.path or "/Applications/Google Chrome Beta.app"
	else
		-- 没有配置浏览器时使用默认设置
		should_open_browser = true
		custom_url = "https://0.0.0.0:8888"
		browser_path = "/Applications/Google Chrome Beta.app"
	end
	
	-- 为Django项目添加自动浏览器打开和端口清理
	if project_config.project_type == "python" and 
	   (actual_command:match("manage%.py") or actual_command:match("runserver")) then
		-- 检测Django开发服务器启动命令
		local port = "8000"  -- 默认端口
		local host = "127.0.0.1"  -- 默认主机
		
		-- 从args中检查端口和主机配置
		if project_config.args then
			for _, arg in ipairs(project_config.args) do
				if arg == "runserver" then
					-- 这是runserver命令，继续检查下一个参数可能是地址
				elseif arg:match("^0%.0%.0%.0:%d+$") then
					-- 匹配 0.0.0.0:端口 格式（优先匹配）
					local matched_host, matched_port = arg:match("^([%d%.]+):(%d+)$")
					host = "127.0.0.1"  -- 浏览器使用localhost
					port = matched_port
					vim.notify("🔍 检测到端口配置: " .. arg .. " -> 浏览器使用: " .. host .. ":" .. port, vim.log.levels.INFO)
				elseif arg:match("^%d+%.%d+%.%d+%.%d+:%d+$") then
					-- 匹配其他 IP:端口 格式
					host, port = arg:match("^([%d%.]+):(%d+)$")
					vim.notify("🔍 检测到端口配置: " .. arg .. " -> " .. host .. ":" .. port, vim.log.levels.INFO)
				elseif arg:match("^:%d+$") then
					-- 匹配 :端口 格式
					port = arg:match("^:(%d+)$")
					vim.notify("🔍 检测到端口配置: " .. arg .. " -> " .. port, vim.log.levels.INFO)
				elseif arg:match("^%d+$") then
					-- 纯数字，可能是端口
					port = arg
					vim.notify("🔍 检测到端口配置: " .. arg .. " -> " .. port, vim.log.levels.INFO)
				end
			end
		end
		
		-- 也检查命令行中的端口参数
		local port_match = actual_command:match("%-%-port[=%s]+(%d+)")
		if port_match then
			port = port_match
		end
		
		detected_port = port
		
		-- 清理当前端口上的进程
		kill_process_on_port(port)
		
		-- 确定要打开的URL
		local url_to_open = custom_url or ("http://" .. host .. ":" .. port)
		
		-- 如果配置了自动打开浏览器（默认行为）
		if should_open_browser then
			M.pending_browser_config = {
				url = url_to_open,
				browser_path = browser_path
			}
		end
	elseif custom_url and should_open_browser then
		-- 其他项目类型但配置了自定义URL
		M.pending_browser_config = {
			url = custom_url,
			browser_path = browser_path
		}
	end
	
	-- 将信息显示和实际命令组合
	local full_command = table.concat(info_lines, " && ") .. " && " .. actual_command
	
	-- 根据终端模式运行命令
	local terminal_mode = project_config.terminal or "split"
	
	if terminal_mode == "none" then
		-- 不需要终端，先显示信息再执行
		vim.notify("📋 运行命令: " .. actual_command, vim.log.levels.INFO)
		vim.notify("📂 工作目录: " .. current_dir, vim.log.levels.INFO)
		vim.notify("🌍 环境变量: " .. env_display, vim.log.levels.INFO)
		vim.fn.system(actual_command)
	elseif terminal_mode == "float" then
		ensure_edit_window()
		vim.cmd("terminal " .. full_command)
		-- 记录终端缓冲区和作业ID
		M.last_process = {
			bufnr = vim.api.nvim_get_current_buf(),
			job_id = vim.b.terminal_job_id,
			port = detected_port
		}
	elseif terminal_mode == "vsplit" then
		ensure_edit_window()
		vim.cmd("vsplit")
		vim.cmd("terminal " .. full_command)
		-- 记录终端缓冲区和作业ID
		M.last_process = {
			bufnr = vim.api.nvim_get_current_buf(),
			job_id = vim.b.terminal_job_id,
			port = detected_port
		}
	elseif terminal_mode == "tab" then
		vim.cmd("tabnew")
		vim.cmd("terminal " .. full_command)
		-- 记录终端缓冲区和作业ID
		M.last_process = {
			bufnr = vim.api.nvim_get_current_buf(),
			job_id = vim.b.terminal_job_id,
			port = detected_port
		}
	else -- split
		ensure_edit_window()
		vim.cmd("split")
		vim.cmd("terminal " .. full_command)
		-- 记录终端缓冲区和作业ID
		M.last_process = {
			bufnr = vim.api.nvim_get_current_buf(),
			job_id = vim.b.terminal_job_id,
			port = detected_port
		}
	end
	
	-- 显示项目启动信息
	local desc = project_config.description and (" - " .. project_config.description) or ""
	vim.notify("🚀 " .. project_config.project_name .. desc, vim.log.levels.INFO)
	
	-- 如果有待打开的浏览器配置，延迟后启动浏览器
	if M.pending_browser_config then
		local browser_config = M.pending_browser_config
		M.pending_browser_config = nil -- 清空待处理配置
		
		vim.notify("🌐 浏览器将在3秒后自动打开: " .. browser_config.url, vim.log.levels.INFO)
		
		vim.defer_fn(function()
			-- 🚀 在启动浏览器前临时优化nvimtree性能
			pcall(function()
				local api = require('nvim-tree.api')
				if api.tree.is_visible() then
					api.tree.collapse_all() -- 折叠目录减少监控负载
				end
			end)
			
			-- 准备浏览器启动命令
			local browser_cmd
			if browser_config.browser_path then
				vim.notify("🚀 正在启动自定义浏览器: " .. browser_config.browser_path, vim.log.levels.INFO)
				if vim.fn.has('mac') == 1 then
					browser_cmd = {"open", "-a", browser_config.browser_path, browser_config.url}
				else
					browser_cmd = {browser_config.browser_path, browser_config.url}
				end
			else
				vim.notify("🚀 正在启动Chrome Beta浏览器...", vim.log.levels.INFO)
				browser_cmd = {"open", "-a", "Google Chrome Beta", browser_config.url}
			end
			
			-- 使用更轻量的启动方式
			local browser_job = vim.fn.jobstart(browser_cmd, {
				detach = true,
				on_exit = function(job_id, exit_code, event_type)
					-- 浏览器启动完成后恢复nvimtree
					vim.defer_fn(function()
						pcall(function()
							local api = require('nvim-tree.api')
							if api.tree.is_visible() then
								api.tree.reload()
							end
						end)
					end, 1500)
					
					if exit_code == 0 then
						vim.notify("✅ 浏览器已成功打开", vim.log.levels.INFO)
					else
						vim.notify("❌ 浏览器启动失败，退出码: " .. exit_code, vim.log.levels.ERROR)
					end
				end
			})
			
			if browser_job <= 0 then
				vim.notify("❌ 无法启动浏览器", vim.log.levels.ERROR)
			end
		end, 5000) -- 增加延迟，确保服务完全启动
	end
end

-- 运行简单命令（用于默认配置）
function M.run_simple_command(command)
	-- 设置自动清理功能
	setup_auto_cleanup()
	
	-- 清除上次运行的进程
	kill_last_process()
	
	-- 保存当前文件（仅在可编辑缓冲区中执行）
	local buftype = vim.api.nvim_buf_get_option(0, 'buftype')
	local modifiable = vim.api.nvim_buf_get_option(0, 'modifiable')
	local modified = vim.api.nvim_buf_get_option(0, 'modified')
	
	if buftype == '' and modifiable and modified then
		vim.cmd("write")
	end
	
	-- 获取当前文件名（完整路径）
	local filename = vim.fn.expand("%:p")
	if filename == "" then
		vim.notify("❌ 请先保存文件", vim.log.levels.ERROR)
		return
	end
	
	-- 替换占位符
	local actual_command = command:gsub("%%", filename)
	
	-- 构建执行信息显示
	local current_dir = vim.fn.getcwd()
	
	-- 构建在终端中显示的信息命令
	local info_lines = {
		"echo '📋 运行命令: " .. actual_command .. "'",
		"echo '📂 工作目录: " .. current_dir .. "'",
		"echo '🌍 环境变量: 使用系统默认'",
		"echo '----------------------------------------'"
	}
	
	-- 将信息显示和实际命令组合
	local full_command = table.concat(info_lines, " && ") .. " && " .. actual_command
	
	-- 在分屏终端中运行命令
	ensure_edit_window()
	vim.cmd("split")
	vim.cmd("terminal " .. full_command)
	
	-- 记录终端缓冲区和作业ID
	M.last_process = {
		bufnr = vim.api.nvim_get_current_buf(),
		job_id = vim.b.terminal_job_id
	}
	
	vim.notify("🚀 运行默认配置", vim.log.levels.INFO)
end

-- Django项目快速启动函数
function M.start_django_project()
	-- 查找Django项目的manage.py文件
	local django_files = {
		"manage.py",
		"./manage.py",
		vim.fn.getcwd() .. "/manage.py"
	}
	
	local manage_py_path = nil
	for _, path in ipairs(django_files) do
		if vim.fn.filereadable(path) == 1 then
			manage_py_path = path
			break
		end
	end
	
	if not manage_py_path then
		vim.notify("❌ 找不到manage.py文件，请确保在Django项目目录中", vim.log.levels.ERROR)
		return
	end
	
	-- 设置自动清理功能
	setup_auto_cleanup()
	
	-- 清除上次运行的进程
	kill_last_process()
	
	-- 检查8888端口是否被占用
	kill_process_on_port("8888")
	
	-- 构建Django启动命令
	local python_cmd = "/Users/dyx/Code/0.python-venv/mySite-django5.0-py3.12/bin/python3"
	local django_cmd = python_cmd .. " " .. manage_py_path .. " runserver 0.0.0.0:8888"
	local url = "http://localhost:8888"
	
	-- 构建Django启动命令（不包含浏览器打开）
	local django_start_command = "echo '🚀 启动Django项目...' && " .. 
		"echo '📂 工作目录: " .. vim.fn.getcwd() .. "' && " ..
		"echo '🌐 服务器地址: " .. url .. "' && " ..
		"echo '----------------------------------------' && " ..
		django_cmd
	
	-- 确保在编辑窗口中打开终端，不在nvimtree中
	ensure_edit_window()
	vim.cmd("split")
	vim.cmd("terminal " .. django_start_command)
	
	-- 记录终端缓冲区和作业ID
	M.last_process = {
		bufnr = vim.api.nvim_get_current_buf(),
		job_id = vim.b.terminal_job_id,
		port = "8888"
	}
	
	-- 🚀 Django启动时立即优化nvimtree性能
	optimize_nvimtree_for_django_startup()
	
	-- 使用Neovim的异步功能启动浏览器
	vim.notify("🚀 Django项目启动中...", vim.log.levels.INFO)
	vim.notify("🌐 浏览器将在5秒后自动打开", vim.log.levels.INFO)
	
	-- 延迟启动浏览器，给Django足够时间启动
	vim.defer_fn(function()
		vim.notify("🚀 正在启动Chrome Beta浏览器...", vim.log.levels.INFO)
		
		-- 使用轻量级方式启动浏览器
		local browser_job = vim.fn.jobstart({"open", "-a", "Google Chrome Beta", url}, {
			detach = true,
			on_exit = function(job_id, exit_code, event_type)
				-- 浏览器启动完成后，延迟恢复nvimtree正常状态
				vim.defer_fn(function()
					restore_nvimtree_after_startup()
					vim.notify("📁 nvimtree已恢复正常状态", vim.log.levels.INFO)
				end, 3000) -- 给浏览器3秒时间完全加载
				
				if exit_code == 0 then
					vim.notify("✅ Chrome Beta已成功打开", vim.log.levels.INFO)
				else
					vim.notify("❌ Chrome Beta启动失败，退出码: " .. exit_code, vim.log.levels.ERROR)
				end
			end
		})
		
		if browser_job <= 0 then
			vim.notify("❌ 无法启动Chrome Beta", vim.log.levels.ERROR)
			-- 即使浏览器启动失败，也要恢复nvimtree
			vim.defer_fn(function()
				restore_nvimtree_after_startup()
			end, 2000)
		end
	end, 6000) -- 延迟到6秒，让Django完全启动并稳定运行
end

-- 选择并运行代码
function M.select_and_run()
	local filetype = vim.bo.filetype
	local project_config = M.load_project_configs() -- 加载项目配置
	
	if project_config then
		-- 有项目配置时直接运行，不检查文件类型
		M.run_project_config(project_config)
	else
		-- 没有项目配置，使用默认配置
		local configs = M.get_configs_for_filetype(filetype)
		
		if #configs == 0 then
			vim.notify("❌ 当前文件类型 (" .. filetype .. ") 没有配置运行选项", vim.log.levels.WARN)
			vim.notify("💡 按 <leader>rc 创建项目配置文件", vim.log.levels.INFO)
			return
		end
		
		if #configs == 1 then
			-- 只有一个选项，直接运行
			M.run_simple_command(configs[1].command)
			return
		end
		
		-- 多个选项，让用户选择
		local items = {}
		for i, config in ipairs(configs) do
			table.insert(items, i .. ". " .. config.name)
		end
		
		vim.ui.select(items, {
			prompt = "选择运行方式:",
		}, function(choice, idx)
			if choice and idx then
				M.run_simple_command(configs[idx].command)
			end
		end)
	end
end

-- 编辑运行配置
function M.edit_configs()
	-- 首先尝试找到项目根目录的配置文件
	local project_root, config_file = M.find_project_root()
	
	if config_file then
		-- 找到项目配置文件，直接编辑
		vim.cmd("edit " .. config_file)
		vim.notify("📝 编辑项目运行配置: " .. vim.fn.fnamemodify(config_file, ":t"), vim.log.levels.INFO)
	else
		-- 没有找到项目配置文件，询问用户是否创建
		vim.ui.select({ "创建新的项目配置文件", "编辑默认配置文件", "取消" }, {
			prompt = "没有找到项目配置文件:",
		}, function(choice, idx)
			if idx == 1 then
				-- 创建新的项目配置文件
				M.create_config_file()
			elseif idx == 2 then
				-- 编辑默认配置文件
				local default_config_file = vim.fn.stdpath("config") .. "/lua/configs/code-runner.lua"
				vim.cmd("edit " .. default_config_file)
				vim.notify("📝 编辑默认运行配置文件", vim.log.levels.INFO)
			end
		end)
	end
end

-- 编辑默认运行配置
function M.edit_default_configs()
	local default_config_file = vim.fn.stdpath("config") .. "/lua/configs/code-runner.lua"
	
	-- 直接编辑文件，nvim-tree 现在不会自动切换根目录
	vim.cmd("edit " .. vim.fn.fnameescape(default_config_file))
	
	vim.notify("📝 编辑默认运行配置文件", vim.log.levels.INFO)
end

-- 创建新的运行配置文件
function M.create_config_file()
	-- 获取当前工作目录
	local current_dir = vim.fn.getcwd()
	local config_file = current_dir .. "/run_config.lua"
	
	-- 配置文件模板
	local template = [[-- ============================================================================
-- 项目运行配置文件 (run_config.lua)
-- 每个项目目录一个配置文件，针对单一代码类型
-- ============================================================================

return {
	-- 项目基本信息
	project_name = "我的项目",
	project_type = "python", -- 项目类型: javascript, python, go, html 等
	
	-- 工作目录（相对于配置文件位置）
	work_dir = "./",
	
	-- Python解释器完整路径（可选，针对Python项目）
	python_path = "",
	-- 示例: python_path = "/Users/username/myproject/venv/bin/python3"
	
	-- 运行命令（% 代表当前文件名）
	command = "manage.py",
	
	-- 命令参数（可选）
	args = {},
	-- 示例: args = { "runserver", "--port", "8000", "--debug" }
	
	-- 环境变量（可选）
	env_vars = {},
	-- 示例: env_vars = { "DJANGO_SETTINGS_MODULE=myproject.settings","DEBUG=1"}
	
	-- 运行前执行的命令（可选）
	pre_commands = {},
	-- 示例: pre_commands = { "pip install -r requirements.txt" }
	
	-- 终端模式: "split", "vsplit", "tab", "float", "none"
	terminal = "split",
	
	-- 浏览器配置（可选）
	browser = {
		-- 是否自动打开浏览器
		auto_open = true,
		-- 浏览器路径（默认为Google Chrome Beta）
		path = "/Applications/Google Chrome Beta.app",
		-- 自定义URL（默认为https://0.0.0.0:8888）
		url = "https://0.0.0.0:8888",
	},
	-- 示例自定义配置:
	-- browser = {
	--     auto_open = false,  -- 不自动打开浏览器
	--     path = "/Applications/Safari.app",  -- 使用Safari浏览器
	--     url = "http://localhost:3000/admin",  -- 打开管理页面
	-- },
	
	-- 描述信息（可选）
	description = "运行Python项目",
}

-- 配置说明:
-- 1. project_name: 项目名称
-- 2. project_type: 项目代码类型，用于确定适用的文件类型
-- 3. work_dir: 命令执行的工作目录
-- 4. python_path: Python解释器完整路径（Python项目专用）
-- 5. command: 要执行的主命令，% 会被替换为当前文件名
-- 6. args: 额外的命令行参数数组
-- 7. env_vars: 环境变量设置数组
-- 8. pre_commands: 运行主命令前需要执行的命令
-- 9. terminal: 终端显示模式
-- 10. browser: 浏览器相关配置
--     - auto_open: 是否自动打开浏览器（默认: true）
--     - path: 浏览器路径（默认: /Applications/Google Chrome Beta.app）
--     - url: 自定义URL（默认: https://0.0.0.0:8888）
-- 11. description: 配置的描述信息
--
-- 常见项目类型示例:
-- JavaScript: command = "node %"  或  "npm run dev"
-- Python:     command = "python %"  或  "python manage.py runserver"
--            python_path = "/path/to/venv/bin/python3"  -- 指定Python解释器
-- Go:         command = "go run %"  或  "go run ."
-- HTML:       command = "open %"  (macOS)
--
-- Python解释器说明:
-- python_path 应该指向Python解释器的完整路径
-- 系统会直接使用指定的Python解释器运行命令，无需激活虚拟环境
-- 例如: python_path = "/Users/dyx/Code/0.python-venv/mySite-django5.0-py3.12/bin/python3"
-- 实际执行: /Users/dyx/Code/0.python-venv/mySite-django5.0-py3.12/bin/python3 manage.py runserver
-- 这样确保使用正确的Python环境和依赖包
]]
	
	-- 检查文件是否已存在
	if vim.fn.filereadable(config_file) == 1 then
		-- 文件已存在，询问是否覆盖
		vim.ui.select({ "打开现有文件", "覆盖文件", "取消" }, {
			prompt = "配置文件已存在:",
		}, function(choice, idx)
			if idx == 1 then
				-- 打开现有文件
				vim.cmd("edit " .. config_file)
				vim.notify("📂 打开现有配置文件", vim.log.levels.INFO)
			elseif idx == 2 then
				-- 覆盖文件
				M.write_config_file(config_file, template)
			end
		end)
	else
		-- 文件不存在，直接创建
		M.write_config_file(config_file, template)
	end
end

-- 写入配置文件
function M.write_config_file(filepath, content)
	local file = io.open(filepath, "w")
	if file then
		file:write(content)
		file:close()
		
		-- 打开文件进行编辑
		vim.cmd("edit " .. filepath)
		vim.notify("📝 已创建运行配置文件: " .. vim.fn.fnamemodify(filepath, ":t"), vim.log.levels.INFO)
		vim.notify("💡 编辑完成后保存文件即可使用", vim.log.levels.INFO)
	else
		vim.notify("❌ 无法创建配置文件", vim.log.levels.ERROR)
	end
end

-- 查找项目根目录和配置文件（支持从任意位置运行）
function M.find_project_root()
	local current_file = vim.fn.expand("%:p:h")  -- 当前文件所在目录
	local cwd = vim.fn.getcwd()  -- 当前工作目录
	
	-- 从当前文件位置向上查找
	local search_dirs = { current_file, cwd }
	
	for _, start_dir in ipairs(search_dirs) do
		local dir = start_dir
		while dir ~= "/" and dir ~= "" do
			local config_file = dir .. "/run_config.lua"
			if vim.fn.filereadable(config_file) == 1 then
				return dir, config_file
			end
			
			-- 检查常见的项目根目录标识文件
			local markers = {
				"package.json", "requirements.txt", "go.mod", "manage.py",
				".git", "Cargo.toml", "composer.json", "pom.xml"
			}
			
			for _, marker in ipairs(markers) do
				local marker_path = dir .. "/" .. marker
				if vim.fn.filereadable(marker_path) == 1 or vim.fn.isdirectory(marker_path) == 1 then
					-- 找到项目根目录标识，检查该目录是否有配置文件
					local root_config = dir .. "/run_config.lua"
					if vim.fn.filereadable(root_config) == 1 then
						return dir, root_config
					end
				end
			end
			
			-- 向上一级目录查找
			local parent = vim.fn.fnamemodify(dir, ":h")
			if parent == dir then break end  -- 已到达根目录
			dir = parent
		end
	end
	
	return nil, nil
end

-- 加载项目特定的运行配置
function M.load_project_configs()
	local project_root, config_file = M.find_project_root()
	
	if config_file then
		-- 加载项目配置
		local ok, project_config = pcall(dofile, config_file)
		if ok and type(project_config) == "table" and project_config.command then
			-- 设置项目根目录信息
			project_config._project_root = project_root
			project_config._config_file = config_file
			return project_config
		else
			vim.notify("⚠️ 项目配置文件格式错误: " .. config_file, vim.log.levels.WARN)
		end
	end
	
	return nil
end

-- 检查项目配置是否适用于当前文件类型
function M.is_project_config_applicable(filetype, project_config)
	if not project_config or not project_config.project_type then
		return false
	end
	
	local project_type = project_config.project_type
	
	-- 定义项目类型到文件类型的映射
	local type_mapping = {
		javascript = { "javascript", "js", "json", "jsx", "ts", "tsx" },
		python = { "python", "py", "html", "css", "javascript", "json", "yaml", "yml", "toml", "cfg", "ini", "txt", "md", "sql" },
		go = { "go", "html", "css", "javascript", "json", "yaml", "yml", "toml", "txt", "md" },
		html = { "html", "htm", "css", "scss", "sass", "javascript", "js", "json" },
		lua = { "lua", "vim" },
		shell = { "sh", "bash", "zsh", "txt", "md" },
	}
	
	local applicable_types = type_mapping[project_type] or { project_type }
	return vim.tbl_contains(applicable_types, filetype)
end

return M
