// ==UserScript==
// @name         Vue网站选中文本自动复制
// @namespace    http://tampermonkey.net/
// @version      1.0
// @description  在Vue网站上选中文本时自动复制到剪贴板，支持动态内容和Vue组件
// @author       You
// @match        https://qx.ynydtk.*
// @grant        GM_setClipboard
// @run-at       document-end
// ==/UserScript==

(function() {
    'use strict';

    // 检测是否为Vue网站
    function isVueWebsite() {
        // 检查Vue相关的全局变量或DOM特征
        return !!(window.Vue ||
                 window.__VUE__ ||
                 document.querySelector('[data-v-]') ||
                 document.querySelector('.vue-') ||
                 document.querySelector('[v-]') ||
                 document.querySelector('div[id*="app"]'));
    }

    // 如果不是Vue网站，则不执行脚本
    if (!isVueWebsite()) {
        return;
    }

    let lastSelectedText = '';
    let selectionTimeout = null;

    // 创建提示元素
    function createToast(message, type = 'success') {
        const toast = document.createElement('div');
        toast.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 12px 20px;
            border-radius: 6px;
            color: white;
            font-size: 14px;
            font-family: Arial, sans-serif;
            z-index: 10000;
            opacity: 0;
            transition: opacity 0.3s ease;
            pointer-events: none;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        `;

        if (type === 'success') {
            toast.style.backgroundColor = '#4CAF50';
        } else {
            toast.style.backgroundColor = '#f44336';
        }

        toast.textContent = message;
        document.body.appendChild(toast);

        // 显示动画
        setTimeout(() => {
            toast.style.opacity = '1';
        }, 10);

        // 自动隐藏
        setTimeout(() => {
            toast.style.opacity = '0';
            setTimeout(() => {
                if (toast.parentNode) {
                    toast.parentNode.removeChild(toast);
                }
            }, 300);
        }, 2000);
    }

    // 复制文本到剪贴板
    function copyToClipboard(text) {
        try {
            // 优先使用GM_setClipboard（油猴脚本专用）
            if (typeof GM_setClipboard !== 'undefined') {
                GM_setClipboard(text);
                createToast(`已复制: ${text.length > 30 ? text.substring(0, 30) + '...' : text}`);
                return true;
            }

            // 备用方案：使用现代浏览器的Clipboard API
            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(text).then(() => {
                    createToast(`已复制: ${text.length > 30 ? text.substring(0, 30) + '...' : text}`);
                }).catch(() => {
                    fallbackCopy(text);
                });
                return true;
            }

            // 最后的备用方案
            fallbackCopy(text);
            return true;

        } catch (error) {
            console.error('复制失败:', error);
            createToast('复制失败', 'error');
            return false;
        }
    }

    // 备用复制方法
    function fallbackCopy(text) {
        const textarea = document.createElement('textarea');
        textarea.value = text;
        textarea.style.position = 'fixed';
        textarea.style.opacity = '0';
        textarea.style.left = '-9999px';
        document.body.appendChild(textarea);
        textarea.select();

        try {
            const successful = document.execCommand('copy');
            if (successful) {
                createToast(`已复制: ${text.length > 30 ? text.substring(0, 30) + '...' : text}`);
            } else {
                createToast('复制失败', 'error');
            }
        } catch (err) {
            createToast('复制失败', 'error');
        } finally {
            document.body.removeChild(textarea);
        }
    }

    // 处理文本选择
    function handleTextSelection() {
        // 清除之前的定时器
        if (selectionTimeout) {
            clearTimeout(selectionTimeout);
        }

        // 延迟处理，避免频繁触发
        selectionTimeout = setTimeout(() => {
            const selection = window.getSelection();
            const selectedText = selection.toString().trim();

            // 检查是否有选中文本且与上次不同
            if (selectedText && selectedText !== lastSelectedText && selectedText.length > 0) {
                // 过滤掉过短的选择（可能是意外选择）
                if (selectedText.length >= 2) {
                    lastSelectedText = selectedText;
                    copyToClipboard(selectedText);
                }
            }
        }, 200); // 200ms延迟
    }

    // 监听选择变化事件
    document.addEventListener('selectionchange', handleTextSelection);

    // 监听鼠标抬起事件（备用触发）
    document.addEventListener('mouseup', () => {
        setTimeout(handleTextSelection, 50);
    });

    // 监听键盘事件（处理键盘选择）
    document.addEventListener('keyup', (e) => {
        // Shift + 方向键选择文本
        if (e.shiftKey && ['ArrowLeft', 'ArrowRight', 'ArrowUp', 'ArrowDown'].includes(e.key)) {
            setTimeout(handleTextSelection, 50);
        }
        // Ctrl+A 全选
        if (e.ctrlKey && e.key === 'a') {
            setTimeout(handleTextSelection, 50);
        }
    });

    // Vue应用特殊处理：监听Vue组件更新
    function observeVueChanges() {
        // 创建MutationObserver来监听DOM变化
        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                // 如果有新节点添加，重新绑定事件
                if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                    // Vue组件可能动态添加内容，确保事件监听器仍然有效
                    // 这里不需要重新绑定，因为我们使用的是document级别的事件监听
                }
            });
        });

        // 开始观察
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }

    // 等待DOM完全加载后开始观察Vue变化
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', observeVueChanges);
    } else {
        observeVueChanges();
    }

    // 添加样式以改善用户体验
    const style = document.createElement('style');
    style.textContent = `
        /* 选中文本时的视觉反馈 */
        ::selection {
            background-color: #007bff !important;
            color: white !important;
        }

        ::-moz-selection {
            background-color: #007bff !important;
            color: white !important;
        }
    `;
    document.head.appendChild(style);

    // 初始化完成提示
    setTimeout(() => {
        createToast('Vue网站自动复制功能已启用');
    }, 1000);

    console.log('Vue网站选中文本自动复制脚本已加载');

})();