// ==UserScript==
// @name         企信-采购信息提取工具
// @namespace    http://tampermonkey.net/
// @version      1.7
// @description  提取采购信息并在控制台清晰显示，增加延迟以确保动态数据加载完成，并美化按钮交互效果。
// @author       You & AI
// @match        https://qx.ynydtk.cn:8443/buyv2/*
// @grant        GM_setClipboard
// @run-at       document-end
// ==/UserScript==

(function () {
	"use strict";

	// --- 新增配置 ---
	// 将需要延迟获取的元素选择器定义为常量，方便管理
	const delayedInputSelector = "#pane-0 > div:nth-child(2) > div.word-form.el-row > form > div:nth-child(4) > div > div > div > div.el-input.el-input--small.is-disabled.el-input--suffix > input";
	const delayDuration = 2000; // 延迟时间，2000毫秒 = 2秒

	// 全局变量存储提取的数据
	let extractedData = null;

	// 创建按钮容器
	const buttonContainer = document.createElement("div");
	buttonContainer.id = "purchase-info-button-container";
	buttonContainer.style.position = "fixed";
	buttonContainer.style.top = "120px";
	buttonContainer.style.left = "53%";
	buttonContainer.style.transform = "translateX(-50%)";
	buttonContainer.style.zIndex = "9999";
	buttonContainer.style.display = "flex";
	buttonContainer.style.gap = "20px";

	// 添加打印时隐藏按钮以及按钮交互效果的样式
	const style = document.createElement("style");
	style.textContent = `
        @media print {
            #purchase-info-button-container,
            button#extract-purchase-info-btn,
            button#create-directory-btn {
                display: none !important;
                visibility: hidden !important;
                opacity: 0 !important;
                position: absolute !important;
                left: -9999px !important;
                top: -9999px !important;
            }
        }

        /* --- 按钮悬浮和点击效果 --- */

        /* 提取信息按钮 (#409eff) */
        #extract-purchase-info-btn:hover {
            background-color: #66b1ff !important;
            transform: translateY(-2px);
            box-shadow: 0 4px 14px rgba(0, 0, 0, 0.25) !important;
        }
        #extract-purchase-info-btn:active {
            background-color: #3a8ee6 !important;
            transform: translateY(0);
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.2) !important;
        }

        /* 创建目录按钮 (#67c23a) */
        #create-directory-btn:hover {
            background-color: #85ce61 !important;
            transform: translateY(-2px);
            box-shadow: 0 4px 14px rgba(0, 0, 0, 0.25) !important;
        }
        #create-directory-btn:active {
            background-color: #5daf34 !important;
            transform: translateY(0);
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.2) !important;
        }

        /* --- 复制成功提示样式 --- */
        .copy-success-toast {
            position: fixed;
            top: 200px;
            left: 50%;
            transform: translateX(-50%);
            background-color: #67c23a;
            color: white;
            padding: 12px 24px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: bold;
            z-index: 10000;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
            opacity: 1;
            transition: opacity 0.5s ease-out;
        }
        .copy-success-toast.fade-out {
            opacity: 0;
        }
    `;
	document.head.appendChild(style);

	// 1. 创建"提取采购信息"按钮
	const extractButton = createButton("提取采购信息", "extract-purchase-info-btn", "#409eff", async function () {
		try {
			// 清空控制台
			console.clear();

			// 获取当前URL路径和页面类型
			const currentUrl = window.location.href;
			let pageType = "unknown";
			if (currentUrl.includes("/flowTodo")) {
				pageType = "flowTodo";
			} else if (currentUrl.includes("/flowDone")) {
				pageType = "flowDone";
			} else if (currentUrl.includes("/purchasedemand")) {
				pageType = "purchasedemand";
			}
			const isFlowTodo = pageType === "flowTodo";

			// 严格按照顺序定义字段
			const fieldOrder = ["是否", "是否本周", "成本费用列支", "申请部门", "申请人", "申请人电话", "项目名称", "内部编号",
				"采购审批完成时间", "月份", "项目类型", "采购模式", "采购方式", "预估金额（元）"];

			// 初始化结果对象
			const results = {};

			// 1. 是否（固定值）
			results["是否"] = "否";

			// 2. 是否本周（公式）
			results["是否本周"] = "=IF(AND([@采购审批完成时间]>=TODAY()-WEEKDAY(TODAY(),2)+1,[@采购审批完成时间]<=TODAY()-WEEKDAY(TODAY(),2)+7),\"是\",\"否\")";

			// 3. 成本费用列支（优先使用新选择器，备用原选择器）
			let costValue = "";
			// 优先使用新的选择器
			const primaryCostElement = document.querySelector('[for="amountSource"]+ div input');
			if (primaryCostElement && primaryCostElement.value) {
				costValue = primaryCostElement.value;
			} else {
				// 备用方法：通过placeholder属性获取
				const costElements = document.querySelectorAll('input[placeholder="请输入项目名称"]');
				costValue = costElements.length > 1 ? costElements[1].value : "";
			}
			// 去除所有不可见字符，包括空格、换行等
			costValue = costValue.replace(/[\s\u200B-\u200D\uFEFF]/g, "");

			// 4. 申请部门（公式）
			results["申请部门"] = "=IF([@申请人]=\"\",\"未找到对应部门\",XLOOKUP([@申请人],人员名单及部门列表!A:A,人员名单及部门列表!B:B,\"未找到对应部门\",0))";

			// 5. 申请人和8.采购审批完成时间（需要切换标签页）
			await extractApplicantAndApprovalTime(results);

			// 6. 申请人电话（公式）
			results["申请人电话"] = "=XLOOKUP([@申请人],人员名单及部门列表!A:A,人员名单及部门列表!C:C)";

			// 7. 项目名称（优先使用新选择器，备用原选择器）
			let projectName = "";
			// 优先使用新的选择器
			const primaryProjectElement = document.querySelector('[for="projectName"]+ div input');
			if (primaryProjectElement && primaryProjectElement.value) {
				projectName = primaryProjectElement.value;
			} else {
				// 备用方法：使用原选择器
				projectName = extractValue("#pane-0 > div:nth-child(2) > div.word-form.el-row > form > div:nth-child(3) > div > div > div > input");
			}
			results["项目名称"] = projectName;

			// 8. 内部编号（优先使用新选择器，备用原选择器）
			let internalId = "";
			// 优先使用新的选择器
			const primaryInternalIdElement = document.querySelector('[for="orgInnerCode"]+ div input');
			if (primaryInternalIdElement && primaryInternalIdElement.value) {
				internalId = primaryInternalIdElement.value;
			} else {
				// 备用方法：通过placeholder属性获取
				const internalIdElement = document.querySelector('input[placeholder="公司内部编号"]');
				internalId = internalIdElement ? internalIdElement.value : "";
			}

			// 处理内部编号，如果值为空或不存在，则使用"无"
			results["内部编号"] = (internalId && internalId.trim() !== "") ? internalId.trim() : "无";

			results["成本费用列支"] = costValue;

			// 9. 月份（公式）
			results["月份"] = "=TEXT(MONTH([@采购审批完成时间]),\"00\")";

			// 10. 项目类型（优先使用新选择器，备用原选择器）
			let projectType = "";
			// 优先使用新的选择器
			const primaryProjectTypeElement = document.querySelector('label[for="projectType"] + .el-form-item__content .el-radio.is-checked .el-radio__label');
			if (primaryProjectTypeElement && primaryProjectTypeElement.textContent) {
				projectType = primaryProjectTypeElement.textContent.trim();
			} else {
				// 备用方法：通过label类名判断
				projectType = extractRadioValue("#pane-0 > div:nth-child(2) > div.word-form.el-row > form > div:nth-child(7) > div > div > div > div", ["货物类",
					"服务类", "工程类"]);
			}
			results["项目类型"] = projectType;

			// 11. 采购模式（优先使用新选择器，备用原选择器）
			let purchaseMode = "";
			// 优先使用新的选择器
			const primaryPurchaseModeElement = document.querySelector('label[for="purchaseType"] + div .is-checked .el-radio__label');
			if (primaryPurchaseModeElement && primaryPurchaseModeElement.textContent) {
				purchaseMode = primaryPurchaseModeElement.textContent.trim();
			} else {
				// 备用方法：通过label类名判断
				purchaseMode = extractRadioValue("#pane-0 > div:nth-child(2) > div.word-form.el-row > form > div:nth-child(6) > div > div > div > div", ["框架",
					"专项"]);
			}
			results["采购模式"] = purchaseMode;

			// 12. 采购方式
			// 使用新的延迟函数来获取值
			let procurementMethod = await extractValueAfterDelay(delayedInputSelector, delayDuration);
			if (procurementMethod === "沿用框招结果") procurementMethod = "框架内采购";
			if (procurementMethod === "电子商城采购") procurementMethod = "电商化采购";
			results["采购方式"] = procurementMethod;

			// 13. 预估金额（元）通过prop属性获取，获取不到时使用备用选择器
			const estimatedAmountElement = document.querySelector('div[prop="estimatedAmount"] > div.el-form-item__content > div.el-input > input');
			let estimatedAmount = estimatedAmountElement?.value || "";

			// 如果通过prop属性获取不到，使用备用选择器
			if (!estimatedAmount) {
				const backupElement = document.querySelector("#pane-0 > div:nth-child(2) > div.word-form.el-row > form > div:nth-child(24) > div > div > div > input");
				estimatedAmount = backupElement?.value || "";
			}

			results["预估金额（元）"] = estimatedAmount;

			// 在控制台打印结果
			console.log("=============== 采购信息提取结果 ===============");
			console.log(`当前页面类型: ${pageType}`);
			fieldOrder.forEach(field => {
				console.log(`${field}：${results[field] || "未获取到值"}`);
			});
			console.log("=============================================");

			// 同时复制到剪贴板（制表符分隔）
			const resultString = fieldOrder.map(field => results[field] || "").join("\t");
			GM_setClipboard(resultString, "text");

			// 显示复制成功提示
			showCopySuccessToast("采购信息已复制到剪贴板");

			// 存储提取的数据到全局变量
			extractedData = results;

			// 输出复制到剪贴板的文本顺序
			console.log("============ 复制到剪贴板的文本顺序 =============");
			console.log("字段顺序：", fieldOrder.join("\t"));
			console.log("文本内容：", resultString);
			console.log("=============================================");
		} catch (error) {
			console.error("提取信息时出错:", error);
		}
	});

	// 2. 创建"创建目录"按钮
	const createDirButton = createButton("创建目录", "create-directory-btn", "#67c23a", async function () {
		try {
			// 直接获取数据，不依赖之前提取的数据
			const currentData = {};

			// 获取申请人和审批完成时间（需要切换标签页）
			await extractApplicantAndApprovalTime(currentData);

			// 获取项目名称（优先使用新选择器，备用原选择器）
			let projectName = "";
			const primaryProjectElement = document.querySelector('[for="projectName"]+ div input');
			if (primaryProjectElement && primaryProjectElement.value) {
				projectName = primaryProjectElement.value;
			} else {
				projectName = extractValue("#pane-0 > div:nth-child(2) > div.word-form.el-row > form > div:nth-child(3) > div > div > div > input");
			}
			currentData["项目名称"] = projectName;

			// 获取内部编号（优先使用新选择器，备用原选择器）
			let internalId = "";
			const primaryInternalIdElement = document.querySelector('[for="orgInnerCode"]+ div input');
			if (primaryInternalIdElement && primaryInternalIdElement.value) {
				internalId = primaryInternalIdElement.value;
			} else {
				const internalIdElement = document.querySelector('input[placeholder="公司内部编号"]');
				internalId = internalIdElement ? internalIdElement.value : "";
			}
			currentData["内部编号"] = (internalId && internalId.trim() !== "") ? internalId.trim() : "无";

			// 获取采购方式
			let procurementMethod = await extractValueAfterDelay(delayedInputSelector, delayDuration);
			if (procurementMethod === "沿用框招结果") procurementMethod = "框架内采购";
			if (procurementMethod === "电子商城采购") procurementMethod = "电商化采购";
			currentData["采购方式"] = procurementMethod;

			// 使用当前获取的数据
			const applicant = currentData["申请人"];
			const approvalDate = currentData["采购审批完成时间"];
			projectName = currentData["项目名称"];
			internalId = currentData["内部编号"];

			// 根据采购方式决定顺序
			let procurementMethodForDir = procurementMethod === "沿用框招结果" ? "框架内采购" : procurementMethod;
			if (procurementMethodForDir === "电子商城采购") procurementMethodForDir = "电商化采购";
			let directoryText;
			if (procurementMethodForDir === "零星采购") {
				directoryText = `${procurementMethodForDir} - - ${internalId} - - ${approvalDate} - - ${projectName} - - ${applicant}`;
			} else {
				directoryText = `${procurementMethodForDir} - - ${approvalDate} - - ${internalId} - - ${projectName} - - ${applicant}`;
			}

			// 复制到剪贴板
			GM_setClipboard(directoryText, "text");

			// 显示复制成功提示
			showCopySuccessToast("目录文本已复制到剪贴板");

			// 在控制台显示
			console.log("=============== 创建的目录文本 ===============");
			console.log(directoryText);
			console.log("=============================================");
		} catch (error) {
			console.error("创建目录时出错:", error);
		}
	});

	// 将按钮添加到容器
	buttonContainer.appendChild(extractButton);
	buttonContainer.appendChild(createDirButton);

	// 将容器添加到页面
	document.body.appendChild(buttonContainer);

	// 辅助函数：创建按钮
	function createButton(text, id, color, onClick) {
		const button = document.createElement("button");
		button.textContent = text;
		button.id = id;
		button.style.padding = "10px 20px";
		button.style.backgroundColor = color;
		button.style.color = "white";
		button.style.border = "none";
		button.style.borderRadius = "4px";
		button.style.cursor = "pointer";
		button.style.boxShadow = "0 2px 10px rgba(0, 0, 0, 0.2)";
		button.style.fontSize = "14px";
		button.style.fontWeight = "bold";
		// 【关键改动】增加过渡效果，使样式变化更平滑
		button.style.transition = "all 0.2s ease-in-out";
		button.addEventListener("click", onClick);
		return button;
	}

	// 辅助函数：提取申请人和审批完成时间
	async function extractApplicantAndApprovalTime(results) {
		let tab2 = null;
		// 统一通过文案匹配"流转列表"的方式查找
		const tabItems = document.querySelectorAll('.el-tabs__item');
		for (const item of tabItems) {
			if ((item.textContent || "").trim() === "流转列表") {
				tab2 = item;
				break;
			}
		}
		// 兜底：如果找不到文案匹配的，使用 id 查找
		if (!tab2) {
			tab2 = document.querySelector("#tab-2");
		}
		if (!tab2) {
			return;
		}

		// 保存当前活动标签页
		const activeTab = document.querySelector(".el-tabs__item.is-active");

		// 切换到第二个标签页（通过点击事件和直接修改class和属性）
		// 先触发点击事件
		const clickEvent = new MouseEvent('click', {
			bubbles: true,
			cancelable: true
		});
		tab2.dispatchEvent(clickEvent);

		// 移除所有标签页的激活状态
		const allTabs = document.querySelectorAll('.el-tabs__item');
		allTabs.forEach(tab => {
			tab.classList.remove('is-active');
			tab.removeAttribute('aria-selected');
			tab.setAttribute('tabindex', '-1');
		});

		// 激活目标标签页
		tab2.classList.add('is-active');
		tab2.setAttribute('aria-selected', 'true');
		tab2.setAttribute('tabindex', '0');

		// 更新活动条位置（如果存在）
		const activeBar = document.querySelector('.el-tabs__active-bar');
		if (activeBar) {
			const tabIndex = Array.from(allTabs).indexOf(tab2);
			const tabWidth = tab2.offsetWidth;
			const tabOffset = tab2.offsetLeft;
			activeBar.style.width = tabWidth + 'px';
			activeBar.style.transform = `translateX(${tabOffset}px)`;
		}

		// 等待内容加载（500ms）
		await new Promise(resolve => setTimeout(resolve, 500));

		try {
			// 获取申请人（第二个td下的div）
			const applicantRow = document.querySelector("#pane-2 > div > div.tableContainer.recordListTable > div > div.el-table__body-wrapper.is-scrolling-none > table > tbody > tr:nth-child(1)");
			if (applicantRow) {
				const secondTd = applicantRow.querySelector("td:nth-child(2)");
				if (secondTd) {
					const applicantDiv = secondTd.querySelector("div");
					if (applicantDiv) {
						// 去除"/"及后面的内容
						results["申请人"] = applicantDiv.textContent.trim().split("/")[0];
					}
				}
			}

			// 获取审批完成时间（从最后一行第三个td）
			const rows = document.querySelectorAll("#pane-2 > div > div.tableContainer.recordListTable > div > div.el-table__body-wrapper.is-scrolling-none > table > tbody > tr");
			if (rows.length > 0) {
				const lastRow = rows[rows.length - 1];
				const timeElement = lastRow.querySelector("td:nth-child(3) > div");
				let approvalDate = timeElement ? timeElement.textContent.trim() : "";
				approvalDate = approvalDate.match(/\d{4}-\d{2}-\d{2}/);
				results["采购审批完成时间"] = approvalDate ? approvalDate[0] : "";
			}
		} catch (e) {
			console.error("获取申请人或审批时间失败:", e);
		} finally {
			// 通过 getElementsByClassName 查找"表单信息"标签页
			let formInfoTab = null;
			const tabElements = document.getElementsByClassName('el-tabs__item is-top');
			console.log("找到的标签页元素数量:", tabElements.length);

			// 遍历所有标签页元素，查找文本包含"表单信息"的
			for (let i = 0; i < tabElements.length; i++) {
				const tab = tabElements[i];
				const tabText = (tab.textContent || "").trim();
				console.log(`标签页 ${i + 1} 文本:`, tabText);
				if (tabText.includes("表单信息")) {
					formInfoTab = tab;
					console.log("找到表单信息标签页:", formInfoTab);
					break;
				}
			}

			console.log("最终找到的表单信息标签页元素:", formInfoTab);

			if (formInfoTab) {
				console.log("开始切换回表单信息标签页");

				// 先触发点击事件
				const clickEventReturn = new MouseEvent('click', {
					bubbles: true,
					cancelable: true
				});
				formInfoTab.dispatchEvent(clickEventReturn);

				// 移除所有标签页的激活状态
				const allTabsReturn = document.querySelectorAll('.el-tabs__item');
				allTabsReturn.forEach(tab => {
					tab.classList.remove('is-active');
					tab.removeAttribute('aria-selected');
					tab.setAttribute('tabindex', '-1');
				});

				// 激活表单信息标签页
				formInfoTab.classList.add('is-active');
				formInfoTab.setAttribute('aria-selected', 'true');
				formInfoTab.setAttribute('tabindex', '0');

				// 更新活动条位置（如果存在）
				const activeBarReturn = document.querySelector('.el-tabs__active-bar');
				if (activeBarReturn) {
					const tabIndexReturn = Array.from(allTabsReturn).indexOf(formInfoTab);
					const tabWidthReturn = formInfoTab.offsetWidth;
					const tabOffsetReturn = formInfoTab.offsetLeft;
					activeBarReturn.style.width = tabWidthReturn + 'px';
					activeBarReturn.style.transform = `translateX(${tabOffsetReturn}px)`;
				}

				// 等待标签页切换完成
				await new Promise(resolve => setTimeout(resolve, 500));
				console.log("已完成返回表单信息标签页，当前激活标签页:", document.querySelector('.el-tabs__item.is-active'));
			} else {
				console.error("所有方法都未找到表单信息标签页，无法返回");
				// 输出所有可用的标签页供调试
				const allAvailableTabs = document.querySelectorAll('.el-tabs__item');
				console.log("当前页面所有标签页:", Array.from(allAvailableTabs).map(tab => ({
					element: tab,
					id: tab.id,
					text: tab.textContent?.trim(),
					ariaControls: tab.getAttribute('aria-controls')
				})));
			}
		}
	}

	// --- 新增辅助函数 ---
	/**
	 * 延迟一段时间后提取输入框的值。
	 * @param {string} selector - 元素选择器
	 * @param {number} delay - 延迟的毫秒数
	 * @returns {Promise<string>} - 提取到的值
	 */
	async function extractValueAfterDelay(selector, delay) {
		await new Promise(resolve => setTimeout(resolve, delay));
		const element = document.querySelector(selector);
		return element ? element.value : "";
	}

	// 辅助函数：提取输入框的值，并去除不可见字符
	function extractValue(selector) {
		const element = document.querySelector(selector);
		return element ? (element.value || "").replace(/[\s\u200B-\u200D\uFEFF]/g, "") : "";
	}

	// 辅助函数：从多个选择器中获取第一个有值的
	function getFirstValidValue(selectors, excludeTimeFormat = false) {
		for (const selector of selectors) {
			const value = extractValue(selector);
			if (value) {
				// 如果需要排除时间格式，则检查是否为时间格式
				if (excludeTimeFormat && isTimeFormat(value)) {
					continue; // 跳过时间格式，继续查找下一个
				}
				return value;
			}
		}
		return "";
	}

	// 辅助函数：检查是否为时间格式（如 2025-08-31 00:00 或 2025-08-3100:00）
	function isTimeFormat(value) {
		// 匹配日期时间格式：YYYY-MM-DD HH:MM 或 YYYY-MM-DD 或 YYYY-MM-DDHH:MM
		const timePattern = /^\d{4}-\d{2}-\d{2}(\s*\d{2}:\d{2})?$/;
		return timePattern.test(value);
	}

	// 辅助函数：提取单选按钮的值（通过检查label的is-checked类）
	function extractRadioValue(containerSelector, options) {
		const container = document.querySelector(containerSelector);
		if (!container) {
			return "";
		}

		const labels = container.querySelectorAll("label");
		for (let i = 0; i < labels.length; i++) {
			if (labels[i].classList.contains("is-checked")) {
				return options[i] || "";
			}
		}
		return "";
	}

	// 辅助函数：显示复制成功提示
	function showCopySuccessToast(message) {
		// 创建提示元素
		const toast = document.createElement("div");
		toast.className = "copy-success-toast";
		toast.textContent = message;

		// 添加到页面
		document.body.appendChild(toast);

		// 2秒后开始淡出动画
		setTimeout(() => {
			toast.classList.add("fade-out");
		}, 2000);

		// 2.5秒后移除元素
		setTimeout(() => {
			if (toast.parentNode) {
				toast.parentNode.removeChild(toast);
			}
		}, 2500);
	}
})();
