// ==UserScript==
// @name         企信-采购信息提取工具
// @namespace    http://tampermonkey.net/
// @version      1.7
// @description  提取采购信息并在控制台清晰显示，增加延迟以确保动态数据加载完成，并美化按钮交互效果。
// @author       You & AI
// @match        https://qx.ynydtk.cn:8443/buyv2/*
// @grant        GM_setClipboard
// ==/UserScript==

(function () {
	"use strict";

	// --- 新增配置 ---
	// 将需要延迟获取的元素选择器定义为常量，方便管理
	const delayedInputSelector = "#pane-0 > div:nth-child(2) > div.word-form.el-row > form > div:nth-child(4) > div > div > div > div.el-input.el-input--small.is-disabled.el-input--suffix > input";
	const delayDuration = 2000; // 延迟时间，2000毫秒 = 2秒

	// 全局变量存储提取的数据
	let extractedData = null;

	// 创建按钮容器
	const buttonContainer = document.createElement("div");
	buttonContainer.style.position = "fixed";
	buttonContainer.style.top = "120px";
	buttonContainer.style.left = "53%";
	buttonContainer.style.transform = "translateX(-50%)";
	buttonContainer.style.zIndex = "9999";
	buttonContainer.style.display = "flex";
	buttonContainer.style.gap = "20px";

	// 添加打印时隐藏按钮以及按钮交互效果的样式
	const style = document.createElement("style");
	style.textContent = `
        @media print {
            #extract-purchase-info-btn,
            #create-directory-btn {
                display: none !important;
            }
        }

        /* --- 按钮悬浮和点击效果 --- */

        /* 提取信息按钮 (#409eff) */
        #extract-purchase-info-btn:hover {
            background-color: #66b1ff !important;
            transform: translateY(-2px);
            box-shadow: 0 4px 14px rgba(0, 0, 0, 0.25) !important;
        }
        #extract-purchase-info-btn:active {
            background-color: #3a8ee6 !important;
            transform: translateY(0);
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.2) !important;
        }

        /* 创建目录按钮 (#67c23a) */
        #create-directory-btn:hover {
            background-color: #85ce61 !important;
            transform: translateY(-2px);
            box-shadow: 0 4px 14px rgba(0, 0, 0, 0.25) !important;
        }
        #create-directory-btn:active {
            background-color: #5daf34 !important;
            transform: translateY(0);
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.2) !important;
        }
    `;
	document.head.appendChild(style);

	// 1. 创建"提取采购信息"按钮
	const extractButton = createButton("提取采购信息", "extract-purchase-info-btn", "#409eff", async function () {
		try {
			// 清空控制台
			console.clear();

			// 获取当前URL路径
			const isFlowTodo = window.location.href.includes("/flowTodo");

			// 严格按照顺序定义字段
			const fieldOrder = ["是否", "是否本周", "成本费用列支", "申请部门", "申请人", "项目名称", "内部编号",
				"采购审批完成时间", "项目类型", "采购模式", "采购方式", "预估金额（元）"];

			// 初始化结果对象
			const results = {};

			// 1. 是否（固定值）
			results["是否"] = "否";

			// 2. 是否本周（公式）
			results["是否本周"] = "=IF(AND([@采购审批完成时间]>=TODAY()-WEEKDAY(TODAY(),2)+1,[@采购审批完成时间]<=TODAY()-WEEKDAY(TODAY(),2)+7),\"是\",\"否\")";

			// 3. 成本费用列支（通过placeholder属性获取）
			const costElements = document.querySelectorAll('input[placeholder="请输入项目名称"]');
			let costValue = costElements.length > 1 ? costElements[1].value : "";

			// 4. 申请部门（公式）
			results["申请部门"] = "=IF([@申请人]=\"\",\"未找到对应部门\",XLOOKUP([@申请人],人员名单及部门列表!A:A,人员名单及部门列表!B:B,\"未找到对应部门\",0))";

			// 5. 申请人和8.采购审批完成时间（需要切换标签页）
			await extractApplicantAndApprovalTime(results);

			// 6. 项目名称
			results["项目名称"] = extractValue("#pane-0 > div:nth-child(2) > div.word-form.el-row > form > div:nth-child(3) > div > div > div > input");

			// 7. 内部编号（通过placeholder属性获取）
			const internalIdElement = document.querySelector('input[placeholder="公司内部编号"]');
			let internalId = internalIdElement ? internalIdElement.value : "";
			// 调试信息
			console.log("内部编号元素:", internalIdElement);
			console.log("内部编号原始值:", internalId);
			console.log("内部编号trim后:", internalId.trim());
			// 处理内部编号，如果值为空或不存在，则使用"无"
			results["内部编号"] = (internalId && internalId.trim() !== "") ? internalId.trim() : "无";

			results["成本费用列支"] = costValue;

			// 9. 项目类型（通过label类名判断）
			results["项目类型"] = extractRadioValue("#pane-0 > div:nth-child(2) > div.word-form.el-row > form > div:nth-child(7) > div > div > div > div", ["货物类",
				"服务类", "工程类"]);

			// 10. 采购模式（通过label类名判断）
			results["采购模式"] = extractRadioValue("#pane-0 > div:nth-child(2) > div.word-form.el-row > form > div:nth-child(6) > div > div > div > div", ["框架",
				"专项"]);

			// 11. 采购方式
			// 使用新的延迟函数来获取值
			let procurementMethod = await extractValueAfterDelay(delayedInputSelector, delayDuration);
			if (procurementMethod === "沿用框招结果") procurementMethod = "框架内采购";
			if (procurementMethod === "电子商城采购") procurementMethod = "电商化采购";
			results["采购方式"] = procurementMethod;

			// 12. 预估金额（元）通过prop属性获取
			const estimatedAmountElement = document.querySelector('div[prop="estimatedAmount"] > div.el-form-item__content > div.el-input > input');
			results["预估金额（元）"] = estimatedAmountElement?.value || "";

			// 在控制台打印结果
			console.log("====== 采购信息提取结果 ======");
			console.log(`当前页面类型: ${isFlowTodo ? "flowTodo" : "flowDone"}`);
			fieldOrder.forEach(field => {
				console.log(`${field}：${results[field] || "未获取到值"}`);
			});
			console.log("=============================");

			// 同时复制到剪贴板（制表符分隔）
			const resultString = fieldOrder.map(field => results[field] || "").join("\t");
			GM_setClipboard(resultString, "text");

			// 存储提取的数据到全局变量
			extractedData = results;

			// 输出复制到剪贴板的文本顺序
			console.log("====== 复制到剪贴板的文本顺序 ======");
			console.log("字段顺序：", fieldOrder.join("\t"));
			console.log("文本内容：", resultString);
			console.log("===================================");
		} catch (error) {
			console.error("提取信息时出错:", error);
		}
	});

	// 2. 创建"创建目录"按钮
	const createDirButton = createButton("创建目录", "create-directory-btn", "#67c23a", async function () {
		try {
			// 检查是否已经提取过数据
			if (!extractedData) {
				console.log("请先点击'提取采购信息'按钮获取数据");
				return;
			}

			// 使用已提取的数据
			const procurementMethod = extractedData["采购方式"];
			const internalId = extractedData["内部编号"];
			const projectName = extractedData["项目名称"];
			const applicant = extractedData["申请人"];
			const approvalDate = extractedData["采购审批完成时间"];

			// 根据采购方式决定顺序
			let procurementMethodForDir = procurementMethod === "沿用框招结果" ? "框架内采购" : procurementMethod;
			if (procurementMethodForDir === "电子商城采购") procurementMethodForDir = "电商化采购";
			let directoryText;
			if (procurementMethodForDir === "零星采购") {
				directoryText = `${procurementMethodForDir} - - ${internalId} - - ${approvalDate} - - ${projectName} - - ${applicant}`;
			} else {
				directoryText = `${procurementMethodForDir} - - ${approvalDate} - - ${internalId} - - ${projectName} - - ${applicant}`;
			}

			// 复制到剪贴板
			GM_setClipboard(directoryText, "text");

			// 在控制台显示
			console.log("====== 创建的目录文本 ======");
			console.log(directoryText);
			console.log("===========================");
		} catch (error) {
			console.error("创建目录时出错:", error);
		}
	});

	// 将按钮添加到容器
	buttonContainer.appendChild(extractButton);
	buttonContainer.appendChild(createDirButton);

	// 将容器添加到页面
	document.body.appendChild(buttonContainer);

	// 辅助函数：创建按钮
	function createButton(text, id, color, onClick) {
		const button = document.createElement("button");
		button.textContent = text;
		button.id = id;
		button.style.padding = "10px 20px";
		button.style.backgroundColor = color;
		button.style.color = "white";
		button.style.border = "none";
		button.style.borderRadius = "4px";
		button.style.cursor = "pointer";
		button.style.boxShadow = "0 2px 10px rgba(0, 0, 0, 0.2)";
		button.style.fontSize = "14px";
		button.style.fontWeight = "bold";
		// 【关键改动】增加过渡效果，使样式变化更平滑
		button.style.transition = "all 0.2s ease-in-out";
		button.addEventListener("click", onClick);
		return button;
	}

	// 辅助函数：提取申请人和审批完成时间
	async function extractApplicantAndApprovalTime(results) {
		const tab2 = document.querySelector("#tab-2");
		if (!tab2) {
			return;
		}

		// 保存当前活动标签页
		const activeTab = document.querySelector(".el-tabs__item.is-active");

		// 切换到第二个标签页
		tab2.click();

		// 等待内容加载（200ms）
		await new Promise(resolve => setTimeout(resolve, 200));

		try {
			// 获取申请人（第二个td下的div）
			const applicantRow = document.querySelector("#pane-2 > div > div.tableContainer.recordListTable > div > div.el-table__body-wrapper.is-scrolling-none > table > tbody > tr:nth-child(1)");
			if (applicantRow) {
				const secondTd = applicantRow.querySelector("td:nth-child(2)");
				if (secondTd) {
					const applicantDiv = secondTd.querySelector("div");
					if (applicantDiv) {
						// 去除"/"及后面的内容
						results["申请人"] = applicantDiv.textContent.trim().split("/")[0];
					}
				}
			}

			// 获取审批完成时间（从最后一行第三个td）
			const rows = document.querySelectorAll("#pane-2 > div > div.tableContainer.recordListTable > div > div.el-table__body-wrapper.is-scrolling-none > table > tbody > tr");
			if (rows.length > 0) {
				const lastRow = rows[rows.length - 1];
				const timeElement = lastRow.querySelector("td:nth-child(3) > div");
				let approvalDate = timeElement ? timeElement.textContent.trim() : "";
				approvalDate = approvalDate.match(/\d{4}-\d{2}-\d{2}/);
				results["采购审批完成时间"] = approvalDate ? approvalDate[0] : "";
			}
		} catch (e) {
			console.error("获取申请人或审批时间失败:", e);
		} finally {
			// 恢复原来的活动标签页
			if (activeTab) {
				activeTab.click();
			}
		}
	}

	// --- 新增辅助函数 ---
	/**
	 * 延迟一段时间后提取输入框的值。
	 * @param {string} selector - 元素选择器
	 * @param {number} delay - 延迟的毫秒数
	 * @returns {Promise<string>} - 提取到的值
	 */
	async function extractValueAfterDelay(selector, delay) {
		console.log(`正在等待 ${delay / 1000} 秒以获取动态值...`);
		await new Promise(resolve => setTimeout(resolve, delay));
		const element = document.querySelector(selector);
		return element ? element.value : "";
	}

	// 辅助函数：提取输入框的值，并去除不可见字符
	function extractValue(selector) {
		const element = document.querySelector(selector);
		return element ? (element.value || "").replace(/[\s\u200B-\u200D\uFEFF]/g, "") : "";
	}

	// 辅助函数：从多个选择器中获取第一个有值的
	function getFirstValidValue(selectors, excludeTimeFormat = false) {
		for (const selector of selectors) {
			const value = extractValue(selector);
			if (value) {
				// 如果需要排除时间格式，则检查是否为时间格式
				if (excludeTimeFormat && isTimeFormat(value)) {
					continue; // 跳过时间格式，继续查找下一个
				}
				return value;
			}
		}
		return "";
	}

	// 辅助函数：检查是否为时间格式（如 2025-08-31 00:00 或 2025-08-3100:00）
	function isTimeFormat(value) {
		// 匹配日期时间格式：YYYY-MM-DD HH:MM 或 YYYY-MM-DD 或 YYYY-MM-DDHH:MM
		const timePattern = /^\d{4}-\d{2}-\d{2}(\s*\d{2}:\d{2})?$/;
		return timePattern.test(value);
	}

	// 辅助函数：提取单选按钮的值（通过检查label的is-checked类）
	function extractRadioValue(containerSelector, options) {
		const container = document.querySelector(containerSelector);
		if (!container) {
			return "";
		}

		const labels = container.querySelectorAll("label");
		for (let i = 0; i < labels.length; i++) {
			if (labels[i].classList.contains("is-checked")) {
				return options[i] || "";
			}
		}
		return "";
	}
})();
